/*
 * ps_mem.c -	mmap() kernel memory
 *
 * Copyright (c) 1993 Branko Lankester
 * 
 */

#include <sys/types.h>
#include <sys/mman.h>
#include <linux/mm.h>
#include <stdio.h>
#include "ps.h"
#include "psdata.h"


extern int kmemfd;

caddr_t kbase;

char *mmapped;
static unsigned long memory_start;
static unsigned long high_memory;
static unsigned short *memmap;

mmap_page(addr)
unsigned long addr;
{
    caddr_t r;
    unsigned mapnr;

    if (addr >= high_memory) {
	fprintf(stderr, "trying to mmap %x\n", addr);
	return -1;
    }
    mapnr = MAP_NR(addr);
    if (mmapped[mapnr])
	return 0;
    if (memmap[mapnr] == 0) {
	fprintf(stderr, "page not in use; %x\n", addr);
	return -1;
    }
    addr &= ~0xfff;
    r = mmap(kbase+addr, 4096, PROT_READ, MAP_SHARED|MAP_FIXED, kmemfd, addr);
    if ((int) r == -1) {
	perror("mmap_page: mmap");
	return -1;
    }
    if (memmap[mapnr] == 1) {
	munmap(kbase+addr, 4096);
	fprintf(stderr, "page not used; %x\n", addr);
	return -1;
    }
    mmapped[mapnr] = 1;
    return 0;
}

mmap_init()
{
    if (high_memory)	/* initialize only once */
	return 0;
    high_memory = get_kword(k_addr("_high_memory")) + 0xfff & ~0xfff;
    memory_start = get_kword(k_addr("_memory_start")) + 0xfff & ~0xfff;
    if ((kbase = (caddr_t) malloc(high_memory)) == NULL || ((long) kbase & 0xfff)) {
	perror("malloc failed");
	return -1;
    }
    memory_start += high_memory / 4096 * sizeof(*memmap);
    memory_start += 0xfff;
    memory_start &= ~0xfff;
    kbase = mmap(kbase, memory_start, PROT_READ, MAP_SHARED|MAP_FIXED, kmemfd, 0L);
    if (kbase == (caddr_t) -1) {
	perror("mmap_init: mmap");
	return -1;
    }
    memmap = KPTR(KWORD(k_addr("_mem_map")));
    mmapped = xmalloc(high_memory/4096);
    memset(mmapped, 0, high_memory/4096);
    memset(mmapped, 1, memory_start/4096);
    return 0;
}

/*
 * unmap unused pages
 */
mmap_cleanup()
{
    int i, pages = high_memory/4096;

    for (i = memory_start/4096; i < pages; ++i) {
	if (mmapped[i] && memmap[i] == 1) {
	    munmap(kbase + i*4096, 4096);
	    mmapped[i] = 0;
	}
    }
    return 0;
}

munmap_mem()
{
    if (munmap(kbase+memory_start, high_memory-memory_start) == -1) {
	perror("munmap");
    }
    memset(mmapped, 0, high_memory/4096);
    memset(mmapped, 1, memory_start/4096);
}
