/*
 * Copyright (c) 1995, 1996, 1997 Jeffrey A. Uphoff
 * Major rewrite by Olaf Kirch, Dec. 1996.
 *
 * NSM for Linux.
 */

#include <fcntl.h>
#include <limits.h>
#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include "statd.h"
#include "notlist.h"

#ifndef lint
static char vcid[] = "$Id: monitor.c,v 2.2 1998/06/01 19:06:43 juphoff Exp $";
#endif

notify_list *		rtnl = NULL;	/* Run-time notify list. */


/*
 * Services SM_MON requests.
 */
struct sm_stat_res *
sm_mon_1_svc(struct mon *argp, struct svc_req *rqstp)
{
	static sm_stat_res result;
	struct hostent	*hostinfo;
	char		*mon_name = argp->mon_id.mon_name,
			*my_name  = argp->mon_id.my_id.my_name;
	struct my_id	*id = &argp->mon_id.my_id;
	char            path[PATH_MAX];
	int             fd;
	notify_list	*clnt;

	/* Assume that we'll fail. */
	result.res_stat = STAT_FAIL;
	result.state = -1;	/* State is undefined for STAT_FAIL. */

	/*
	 * Check hostnames.  If I can't look them up, I won't monitor.  This
	 * might not be legal, but it adds a little bit of safety and sanity.
	 */

	/* MUST check for /'s in hostname!  See CERT's CA-96.09 for details. */
	if (strchr(mon_name, '/')) {
		log(L_CRIT, "SM_MON request for hostname containing '/': %s",
			mon_name);
		log(L_CRIT, "POSSIBLE SPOOF/ATTACK ATTEMPT!");
		goto failure;
	} else if (gethostbyname(mon_name) == NULL) {
		log(L_WARNING, "gethostbyname error for %s", mon_name);
		goto failure;
	} else if (!(hostinfo = gethostbyname(my_name))) {
		log(L_WARNING, "gethostbyname error for %s", my_name);
		goto failure;
	}

	/*
	 * Hostnames checked OK.
	 * Now check to see if this is a duplicate, and warn if so.
	 * I will also return STAT_FAIL. (I *think* this is how I should
	 * handle it.)
	 *
	 * Olaf requests that I allow duplicate SM_MON requests for
	 * hosts due to the way he is coding lockd. No problem,
	 * I'll just do a quickie success return and things should
	 * be happy.
	 */
	if (rtnl) {
		notify_list    *temp = rtnl;

		while ((temp = nlist_gethost(temp, mon_name, 0))) {
			if (!strcasecmp(NL_MY_NAME(temp), my_name) &&
				NL_MY_PROC(temp) == id->my_proc &&
				NL_MY_PROG(temp) == id->my_prog &&
				NL_MY_VERS(temp) == id->my_vers) {
				/* Hey!  We already know you guys! */
				dprintf(L_DEBUG,
					"Duplicate SM_MON request for %s "
					"from procedure on %s",
					mon_name, my_name);

				/* But we'll let you pass anyway. */
				result.res_stat = STAT_SUCC;
				result.state = MY_STATE;
				return (&result);
			}
			temp = NL_NEXT(temp);
		}
	}

	/*
	 * We're committed...ignoring errors.  Let's hope that a malloc()
	 * doesn't fail.  (I should probably fix this assumption.)
	 */
	if (!(clnt = nlist_new(my_name, mon_name, 0))) {
		log(L_WARNING, "out of memory");
		goto failure;
	}

	NL_ADDR(clnt) = *(struct in_addr *) hostinfo->h_addr;
	NL_MY_PROG(clnt) = id->my_prog;
	NL_MY_VERS(clnt) = id->my_vers;
	NL_MY_PROC(clnt) = id->my_proc;
	memcpy(NL_PRIV(clnt), argp->priv, PRIV_SIZE);

	/*
	 * Now, Create file on stable storage for host.
	 */
	snprintf(path, PATH_MAX, SM_DIR "/%s", mon_name);
	if ((fd = open(path, O_WRONLY|O_SYNC|O_CREAT, S_IRUSR|S_IWUSR)) < 0) {
		/* Didn't fly.  We won't monitor. */
		log(L_ERROR, "creat(%s) failed: %m", path);
		nlist_free(NULL, clnt);
		goto failure;
	}

	nlist_insert(&rtnl, clnt);
	close(fd);

	result.res_stat = STAT_SUCC;
	result.state = MY_STATE;
	dprintf(L_DEBUG, "MONITORING %s for %s", mon_name, my_name);
	return (&result);

failure:
	log(L_WARNING, "STAT_FAIL to %s for SM_MON of %s", my_name, mon_name);
	return (&result);
}


/*
 * Services SM_UNMON requests.
 *
 * There is no statement in the X/Open spec's about returning an error
 * for requests to unmonitor a host that we're *not* monitoring.  I just
 * return the state of the NSM when I get such foolish requests for lack
 * of any better ideas.  (I also log the offense.)
 */
struct sm_stat *
sm_unmon_1_svc(struct mon_id *argp, struct svc_req *rqstp)
{
	static sm_stat  result;
	notify_list	*clnt;
	char		*mon_name = argp->mon_name,
			*my_name  = argp->my_id.my_name;
	struct my_id	*id = &argp->my_id;

	result.state = MY_STATE;

	/* Check if we're monitoring anyone. */
	if (!(clnt = rtnl)) {
		log(L_WARNING,
			"Received SM_UNMON request from %s for %s while not "
			"monitoring any hosts.", my_name, argp->mon_name);
		return (&result);
	}

	/*
	 * OK, we are.  Now look for appropriate entry in run-time list.
	 * There should only be *one* match on this, since I block "duplicate"
	 * SM_MON calls.  (Actually, duplicate calls are allowed, but only one
	 * entry winds up in the list the way I'm currently handling them.)
	 */
	while ((clnt = nlist_gethost(clnt, mon_name, 0))) {
		if (!strcasecmp(NL_MY_NAME(clnt), my_name) &&
			NL_MY_PROC(clnt) == id->my_proc &&
			NL_MY_PROG(clnt) == id->my_prog &&
			NL_MY_VERS(clnt) == id->my_vers) {
			/* Match! */
			dprintf(L_DEBUG, "UNMONITORING %s for %s",
					mon_name, my_name);
			nlist_free(&rtnl, clnt);
			xunlink(SM_DIR, mon_name, 1);

			return (&result);
		} else
			clnt = NL_NEXT(clnt);
	}

	log(L_WARNING, "Received erroneous SM_UNMON request from %s for %s",
		my_name, mon_name);
	return (&result);
}


struct sm_stat *
sm_unmon_all_1_svc(struct my_id *argp, struct svc_req *rqstp)
{
	short int       count = 0;
	static sm_stat  result;
	notify_list	*clnt;

	result.state = MY_STATE;

	if (!(clnt = rtnl)) {
		log(L_WARNING, "Received SM_UNMON_ALL request from %s "
			"while not monitoring any hosts", argp->my_name);
		return (&result);
	}

	while ((clnt = nlist_gethost(clnt, argp->my_name, 1))) {
		if (NL_MY_PROC(clnt) == argp->my_proc &&
			NL_MY_PROG(clnt) == argp->my_prog &&
			NL_MY_VERS(clnt) == argp->my_vers) {
			/* Watch stack! */
			char            mon_name[SM_MAXSTRLEN + 1];
			notify_list	*temp;

			dprintf(L_DEBUG,
				"UNMONITORING (SM_UNMON_ALL) %s for %s",
				NL_MON_NAME(clnt), NL_MY_NAME(clnt));
			strncpy(mon_name, NL_MON_NAME(clnt), SM_MAXSTRLEN);
			temp = NL_NEXT(clnt);
			nlist_free(&rtnl, clnt);
			xunlink(SM_DIR, mon_name, 1);
			++count;
			clnt = temp;
		} else
			clnt = NL_NEXT(clnt);
	}

	if (!count) {
		dprintf(L_DEBUG, "SM_UNMON_ALL request from %s with no "
			"SM_MON requests from it.", argp->my_name);
	}

	return (&result);
}
