/*
 * Install a module in the kernel.
 *
 * See the file COPYING for your rights (GNU GPL)
 *
 * Originally by Anonymous (as far as I know...)
 * Linux version by Bas Laarhoven <bas@vimec.nl>
 * Modified by Jon Tombs.
 *
 * Support for transient and resident symbols
 * added by Bjorn Ekwall <bj0rn@blox.se> in June 1994 (C)
 *
 * Load map option conceived by Derek Atkins <warlord@MIT.EDU>
 *
 * Support for versioned kernels and symbols: Bjorn Ekwall in December 1994
 *
 * Merged in ksyms and rmmod in December 1994: Bjorn Ekwall
 *
 * Support for ELF modules: Bjorn Ekwall in December 1994 after having
 *                          mangled sources from, and been enlightened
 *                          and supported by Eric Youngdale <eric@aib.com>
 *                          (the kludges are all mine, don't blame Eric...)
 *
 * Support for (z)System.map resolving of unresolved symbols (HACKER_TOOL)
 * added from code by Hans Lermen <lermen@elserv.ffm.fgan.de>
 */

/*
#define HACKER_TOOL
*/

#include "insmod.h"
#define is_global(sp) (aout_flag?(sp->u.n.n_type & N_EXT) : \
				(ELF32_ST_BIND(sp->u.e.st_info) == STB_GLOBAL))
#define is_undef(sp) (aout_flag?((sp->u.n.n_type & ~N_EXT) == N_UNDF) : \
			    (ELF32_ST_TYPE(sp->u.e.st_info) == STT_NOTYPE))

/* hack: sizeof(struct elfhdr) > sizeof(struct exec) */
Elf32_Ehdr header;
int aout_flag;

int codesize;
char *textseg;

int nsymbols;
struct symbol *symtab;
char *stringtab;
unsigned long addr;

static int export_flag = 1; /* See comment att option handler in main() */
static int force_load = 0;
static int loadmap = 0;
static int versioned_kernel = 0;
static struct kernel_sym nullsym;
static struct symbol *symroot;
static int progsize;

void *
ckalloc(size_t nbytes)
{
	void *p;

	if ((p = malloc(nbytes)) == NULL) {
		fputs("insmod:  malloc failed\n", stderr);
		exit(2);
	}
	return p;
}


static int create_module(const char *name, unsigned long size)
{
	return syscall( __NR_create_module, name, size);
}

static int init_module(const char *name, void *code, unsigned codesize,
		struct mod_routines *routines,
		struct symbol_table *syms) {
	return syscall( __NR_init_module, name, code, codesize, routines,
		syms);
}

static int delete_module(const char *name)
{
	return syscall( __NR_delete_module, name);
}

static int get_kernel_syms(struct kernel_sym *buffer)
{
	return syscall( __NR_get_kernel_syms, buffer);
}

void check_version(int force_load)
{
	struct utsname uts_info;
	unsigned long kernel_version;
	int i;

	/* Check if module and kernel version match */
	if ((i = uname( &uts_info)) != 0) {
		fprintf( stderr, "uname call failed with code %d\n", i);
		exit( 2);
	}

	kernel_version = looksym( "_kernel_version");
	if (strcmp( (char*) textseg + kernel_version, uts_info.release)) {
		fprintf( stderr,
			"Error: module's `kernel_version' doesn't match the current kernel.\n%s\n%s\n",
			"       Check the module is correct for the current kernel, then",
			"       recompile the module with the correct `kernel_version'.");
		if (force_load)
			fprintf(stderr, "       Loading it anyway...\n");
		else
			exit( 2);
	}

}

int
v_strncmp(const char *tabentry, const char *lookfor, size_t n)
{
	int len = strlen(lookfor);
	int retval;

	if ((retval = strncmp(tabentry, lookfor, len)) != 0)
		return retval;
	/* else */
	if (strncmp(tabentry + len, "_R", 2) == 0)
		return 0;
	else
		return retval;
}

int
m_strncmp(const char *tabentry, const char *lookfor, size_t n)
{
	int retval;
	char *v;

	if ((retval = strncmp(tabentry, lookfor, n)) == 0)
		return retval;
	/* else */
	if (	(v = strrchr(lookfor, 'R')) &&
		(*(v - 1) == '_') &&
		(strlen(v - 1) == 10) &&
		(strncmp(tabentry, lookfor, (v - lookfor) + 1) == 0) &&
		(strlen(tabentry) == strlen(lookfor))
		)
		return 0; /* ! */
	else
		return retval;
}

int
main(int argc, char **argv)
{
	FILE *fp;
	struct exec *aouthdr = (struct exec *)&header;
	struct kernel_sym *curr_module = &nullsym;
	struct kernel_sym *ksymtab = NULL;
	struct kernel_sym *ksym = NULL;
	struct kernel_sym *resident_ksym_start = NULL;
	struct mod_routines routines;
	struct symbol *sp;
	unsigned long init_func, cleanup_func;
	int fatal_error;
	int i;
	int nksyms;
	int versioned_module;
	int found_resident = 0;

	struct symbol_table *newtab;
	int module_refs = 0; /* number of references modules */
	int n_symboldefs = 0; /* number of defined symbols in the module */
	int string_table_size = 0; /* size of the new symbol table string table */
	struct internal_symbol *symp;
	struct module_ref *refp;
	char *stringp;

	char *filename;
	char spare_path[200]; /* just testing... */
	char *modname = NULL;
	char *p;
#ifdef HACKER_TOOL
	char *zfile = NULL;
	int use_zSystem_local = 0;
#endif


	/* find basename */
	
	if ((p = strrchr(argv[0], '/')) != (char *)0)
		++p;
	else
		p = argv[0];

	if (strcmp(p, "rmmod") == 0)
		return rmmod(argc, argv);

	if (strcmp(p, "ksyms") == 0)
		return ksyms(argc, argv);

	/* else  this is insmod! */

	while (argc > 1 && (argv[1][0] == '-')) {
		p = &(argv[1][1]);
		while (*p) {
			switch (*p) {
			case 'o':
				modname = argv[2];
				--argc;
				++argv;
				break;

			case 'f': /* force loading */
				force_load = 1;
				break;

			case 'X': /* _do_ export externs */
				export_flag = 1;
				break;

			case 'x': /* do _not_ export externs */
				export_flag = 0;
				break;

#ifdef HACKER_TOOL
			/* Hackers:
			 *	For enhanced DOSEMU hack support,
			 *	note that -z and -Z is swapped from 1.1.67
			 */
			case 'Z': /* read from a "*System.map"-type file */
				zfile = argv[2];
				--argc;
				++argv;
				break;

			case 'z':
				zfile = "/usr/src/linux/zSystem.map";
				if (access(zfile, R_OK) < 0)
					zfile = "/usr/src/linux/System.map";
				break;
			case 'l': /* resolve using locals too */
				use_zSystem_local = 1;
				break;
#endif
			case 'm': /* generate loadmap */
				loadmap = 1;
				break;
			}
			++p;
		}
		--argc;
		++argv;
	}

	if (argc < 2) {
#ifndef HACKER_TOOL
		fputs("Usage: insmod [-f] [-x] [-o name] [-m] module "
			"[[sym=value]...]\n", stderr);
#else
		fputs("Usage:\n"
	"insmod [-f] [-x] [-o name] [-m] [-l] [-z | -Z mapfile] module [[sym=value]...]\n"
	"\n"
	"  module     Filename of a loadable kernel module (*.o)\n"
	"  -o name    Set internal modulname to name\n"
	"  -x         do *not* export externs\n"
	"  -m         generate loadmap (so crashes can be traced down)\n"
	"  -f         Force loading under wrong kernel version\n"
	"             (together with -z also disables mapfile check)\n"
	"  -z         At last use /usr/src/linux/[z]System.map to resolve\n"
	"  -Z mapfile As -z, but use specified mapfile\n"
	"  -l         Together with -z, -Z, also take local symbols from map\n"
	"             (But note, local symbols can be multiple defined, last one is taken)\n"
	, stderr);
#endif
		exit(2);
	}

	filename = argv[1];
	argv += 2;
	--argc;

	/* get the size of the current kernel symbol table */
	nksyms = get_kernel_syms(NULL);

	if (nksyms < 0) {
		fprintf(stderr, "get_kernel_sys failed: Cannot find Kernel symbols!\n");
		exit(2);
	}

	if (nksyms) {
		ksymtab = (struct kernel_sym *) ckalloc(nksyms * sizeof *ksymtab);
		/* NOTE!!! The order of the symbols is important */
		if (get_kernel_syms(ksymtab) != nksyms) {
			fprintf(stderr, "Kernel symbol problem\n");
			exit(2);
		}
	}

	/* Is this a kernel with appended CRC-versions? */
	for (ksym = ksymtab, i = 0 ; i < nksyms ; ++i, ksym++) {
		if (found_resident &&
		    (strcmp(ksym->name, "_Using_Versions") == 0)) {
		    	versioned_kernel = 1;
			/* The kernel symbols have a CRC version appended */
			break;
		}

		if (!found_resident && strcmp(ksym->name, "#") == 0) {
			resident_ksym_start = ksym;
			found_resident = i;
		}
		/* look only in residents */
	}

	/* construct the module name */
	if (modname == NULL) {
		int len;

		if ((p = strrchr(filename, '/')) != NULL)
			p++;
		else
			p = filename;
		len = strlen(p);
		if (len > 2 && strcmp(p + len - 2, ".o") == 0)
			len -= 2;
		else if (len > 4 && strcmp(p + len - 4, ".mod") == 0)
			len -= 4;

		modname = (char*) ckalloc(len + 1);
		memcpy(modname, p, len);
		modname[len] = '\0';
	}

	if ((strchr(filename, '/') == 0) && (strchr(filename, '.') == 0)) {
		struct stat dummy;
		char *path;

		/* look in default places, for now in ONE extra directory */
		/* slightly braindead, but anyway... */
		sprintf(spare_path, "./%s.o", filename);
		if (stat(spare_path, &dummy) < 0) {
			if ((path = getenv("MODPATH")) == 0)
				path = "/linux/modules";
			sprintf(spare_path, "%s/%s.o", path, filename);
		}
		filename = spare_path;
	}

	/* open file and read header */
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s\n", filename);
		exit(2);
	}

	/* sizeof(struct elfhdr) > sizeof(struct exec) */
	fread(&header, sizeof(Elf32_Ehdr), 1, fp);
	if (feof(fp) || ferror(fp)) {
		fprintf(stderr, "Could not read header of %s\n", filename);
		exit(2);
	}

	symtab = (struct symbol *)0;
	nsymbols = 0;

	if (N_MAGIC((*aouthdr)) == OMAGIC) {
		char *errstr;

		if ((errstr = load_aout(fp)) != (char *)0) {
			fprintf(stderr, "%s: %s\n", filename, errstr);
			exit(2);
		}
	}
	else if ((header.e_ident[0] == 0x7f) &&
		 (strncmp(&header.e_ident[1], "ELF",3) == 0) &&
		 (header.e_type == ET_REL) &&
		 ((header.e_machine == 3) || (header.e_machine == 6))
		) {
			char *errstr;

			if ((errstr = load_elf(fp)) != (char *)0) {
				fprintf(stderr, "%s: %s\n", filename, errstr);
				exit(2);
			}
	}
	else {
		fprintf(stderr, "%s: not an object file\n", filename);
		exit(2);
	}

	versioned_module = (findsym("_Using_Versions", NULL, strncmp))?1:0;

	/* check version info */
	if (!versioned_kernel || !versioned_module)
		check_version(force_load);


	/* get initialization and cleanup routines */
	init_func = looksym("_init_module");
	cleanup_func = looksym("_cleanup_module");

	/* bind undefined symbols (ELF-stuff hidden in defsym()) */
	defsym("_mod_use_count_", 0 - sizeof (int), N_BSS | N_EXT, TRANSIENT);

	/* First: resolve symbols using kernel transient symbols */
	/* Then: use resident kernel symbols to resolve the last ones... */
	for (ksym = ksymtab, i = 0 ; i < nksyms ; ++i, ksym++) {
		/* Magic in this version of the new get_kernel_syms:
		 * Every module is sent along as a symbol,
		 * where the module name is represented as "#module", and
		 * the adress of the module struct is stuffed into the value.
		 * The name "#" means that the symbols that follow are
		 * kernel resident.
		 */
		if (ksym->name[0] == '#') {
			curr_module = ksym;
			continue;
		}

		if (defsym(ksym->name, ksym->value, N_ABS | N_EXT,
		/* this is safe since curr_module was initialized properly */
			(curr_module->name[1]) ?  TRANSIENT : RESIDENT)) {
			/* kludge: mark referenced modules */
			if (curr_module->name[1] && /* but not the kernel */
				(curr_module->name[0] == '#')) {
				curr_module->name[0] = '+';
				++module_refs;
			}
		}
	}

#ifdef HACKER_TOOL
	/* Fake it, by reading a *System.map file */
	if (zfile) {
		FILE *zfp;
		struct kernel_sym zsym;
		int matchksyms=0;
		char line[80];
		char ztype[80];

		if ((zfp = fopen(zfile, "r")) == (FILE *)0) {
			perror(zfile);
			exit(2);
		}
		/* else */
		if (!fgets(line, 80, zfp) ||
			strcmp(line, "00100000 t startup_32\n")) {
			fprintf(stderr, ">%s<XXIllegal mapfile\n",line);
			exit(2);
		}
		/* else */

		while (fgets(line, 80, zfp)) {
			sscanf(line, "%lx %s %s",
				&(zsym.value), ztype, zsym.name);
			if (!force_load) {
				struct kernel_sym *chk;
				int n;

				chk = resident_ksym_start;
				n = found_resident;
				for (;n < nksyms; ++n, ++chk) {
					if ((chk->value == zsym.value)
						&&
					 !strcmp(chk->name, zsym.name)) {
						++matchksyms;
					 	break;
					 }
				}
			}
			switch (ztype[0]) {
			case 't':
			case 'd':
			case 'b': 
				if (!use_zSystem_local)
					break;
				/* else fall through */
			case 'T':
			case 'D':
			case 'B':
				defsym(zsym.name, zsym.value, N_ABS|N_EXT, RESIDENT);
				break;
			}
		}
		if ((!force_load) && (matchksyms != (nksyms-found_resident-1))) {
			/* We assume that *all* resident symbols can be found
			 * in System.map and that there values are matching.
			 *  Obviously this is not the case, so we are giving up.
			 */
			fprintf(stderr,"current kernel doesn't match System.map\n");
			exit(2);
		}
		/* else */
	}
#endif

	/* Change values according to the parameters to "insmod" */
	{
		struct symbol *change;
		char *param;
		char *val;
		char internal_name[SYM_MAX_NAME];
		int value;

		while (argc > 1) {
			param = *argv;
			++argv;
			--argc;

			if ((val = strchr(param, '=')) == NULL)
				continue;
			*val++ = '\0';
			if (*val < '0' || '9' < *val) {
				fprintf(stderr, "Symbol '%s' has an illegal value: %s\n", param, val);
				exit(2);
			}

			if (val[0] == '0') {
				if (val[1] == 'x')
					sscanf(val, "%x", &value);
				else
					sscanf(val, "%o", &value);
			}
			else
				sscanf(val, "%d", &value);

			sprintf(internal_name, "_%s", param);
			if (versioned_module)
				change = findsym(internal_name, NULL, v_strncmp);
			else
				change = findsym(internal_name, NULL, strncmp);
			if (change == NULL) {
				fprintf(stderr, "Symbol '%s' not found\n", param);
				exit(2);
			}
			*((int *)(textseg + symvalue(change))) = value;
		}
	}

	/* allocate space for "common" symbols */
	/* and check for undefined symbols */
	fatal_error = 0;
	progsize = (codesize + 3) & ~3;
	for (sp = symtab ; sp < symtab + nsymbols ; sp++) {
		if ((symname(sp) == (char *)0) || (*symname(sp) == '\0'))
			continue;

		if ( ((aout_flag && (sp->u.n.n_type == (N_UNDF | N_EXT))) ||
			(!aout_flag && is_global(sp) && is_undef(sp)))
			&& (symvalue(sp) != 0)) {
			int len;

			if (aout_flag) {
				sp->u.n.n_type = N_BSS | N_EXT;
				len = symvalue(sp);
			}
			else { /* I'm not sure I understand this... */
				sp->u.e.st_info = (STB_GLOBAL << 4)| STT_OBJECT;
				len = (sp->u.e.st_size)?(sp->u.e.st_size):4;
			}

			symvalue(sp) = progsize;
			progsize += len;
			progsize = (progsize + 3) & ~3;
		} else if (is_undef(sp)) {
			if ((versioned_module) &&
				findsym(symname(sp), NULL, m_strncmp)) {
				char *v = strrchr(symname(sp), '_');
				if (v && strncmp(v, "_R", 2) == 0)
					*v = '\0';
				fprintf(stderr, "%s: wrong version\n",
					symname(sp));
				if (v)
					*v = '_';
			}
			else
				fprintf(stderr, "%s undefined\n", symname(sp));
			fatal_error = 1;
		}
	}
	if (fatal_error)
		exit(2);

	/* create the module */
	errno = 0;
	/* We add "sizeof (int)" to skip over the use count */
	addr = create_module(modname, progsize) + sizeof (int);

	switch (errno) {
	case EEXIST:
		fprintf(stderr, "A module named %s already exists\n", modname);
		exit(2);
	case ENOMEM:
		fprintf(stderr, "Cannot allocate space for module\n");
		exit(2);
	case 0:
		break;
	default:
		perror("create_module");
		exit(2);
	}

	/* perform relocation */
	if (aout_flag)
		relocate_aout(fp);
	else
		relocate_elf(fp);

	init_func += addr;
	cleanup_func += addr;

	/* "hide" the module specific symbols, not to be exported! */
	hidesym("_cleanup_module");
	hidesym("_init_module");
	hidesym("_kernel_version");
	hidesym("_mod_use_count_");

	/* Build the module symbol table */
	/* abuse of *_other:
	 * 0	refer kernel resident
	 * 1	define module symbol, inserted in kernel syms
	 * 2	refer module symbol
	 * 3	won't happen (define resolved by other module?)
	 */

	/*
	 * Get size info for the new symbol table
	 * we already have module_refs
	 */
	for (sp = symtab ; export_flag && (sp < symtab + nsymbols) ; sp++) {
		if (is_global(sp) && (symother(sp) & DEF_BY_MODULE)) {
			string_table_size += strlen(symname(sp)) + 1;
			if (!aout_flag)
				++string_table_size;
			++n_symboldefs;
		}
	}
	newtab = (struct symbol_table *)ckalloc(sizeof(struct symbol_table) +
		n_symboldefs * sizeof(struct internal_symbol) +
		module_refs * sizeof(struct module_ref) +
		string_table_size);

	newtab->size = sizeof(struct symbol_table) +
		n_symboldefs * sizeof(struct internal_symbol) +
		module_refs * sizeof(struct module_ref) +
		string_table_size;

	newtab->n_symbols = n_symboldefs;
	newtab->n_refs = module_refs;

	symp = &(newtab->symbol[0]);
	stringp = ((char *)symp) + 
		n_symboldefs * sizeof(struct internal_symbol) +
		module_refs * sizeof(struct module_ref);

	/* update the string pointer (to a string index) in the symbol table */
	for (sp = symtab ; export_flag && (sp < symtab + nsymbols) ; sp++) {
		if (is_global(sp) && (symother(sp) & DEF_BY_MODULE)) {
			symp->addr = (void *)(symvalue(sp) + addr);
			symp->name = (char *)(stringp - (char *)newtab);
			if (!aout_flag) {
				strcpy(stringp, "_");
				stringp += 1;
			}
			strcpy(stringp, symname(sp));
			stringp += strlen(symname(sp)) + 1;
			++symp;
		}
	}

	refp = (struct module_ref *)symp;
	for (ksym = ksymtab, i = 0 ; i < nksyms ; ++i, ksym++) {
		if (ksym->name[0] == '+') {
			refp->module = (struct module *)ksym->value;
			++refp;
		}
	}

	/* load the module into the kernel */
	/* NOTE: the symbol table == NULL if there are no defines/updates */
	routines.init = (int (*)(void)) init_func;
	routines.cleanup = (void (*)(void)) cleanup_func;

	if (init_module(modname, textseg, codesize, &routines,
		((n_symboldefs + module_refs)? newtab : NULL)) < 0) {

		if (errno == EBUSY) {
			fprintf(stderr, "Initialization of %s failed\n", modname);
		} else {
			perror("init_module");
		}
		delete_module(modname);
		exit(1);
	}
#ifdef DEBUG_DISASM
dis(textseg, codesize);
#endif
	/*
	 * Print a loadmap so that kernel panics can be identified.
	 * Load map option conceived by Derek Atkins <warlord@MIT.EDU>
	 */
	for (sp = symtab; loadmap && (sp < symtab + nsymbols) ; sp++) {
		char symtype = 'u';

		if (aout_flag) {
			if ((sp->u.n.n_type & ~N_EXT) == N_ABS)
				continue;

			switch (sp->u.n.n_type & ~N_EXT) {
			case N_TEXT | N_EXT: symtype = 'T'; break;
			case N_TEXT: symtype = 't'; break;
			case N_DATA | N_EXT: symtype = 'D'; break;
			case N_DATA: symtype = 'd'; break;
			case N_BSS | N_EXT: symtype = 'B'; break;
			case N_BSS: symtype = 'b'; break;
			}
		}

		printf("%08lx %c %s\n",
			symvalue(sp) + addr, symtype, symname(sp));
	}

	if (nksyms > 0)
		free(ksymtab); /* it has done its job */

	exit(0);
}

void
hidesym(const char *name)
{
	struct symbol *sp;

	if (!aout_flag)
		++name; /* Horrible kludge */

	if ((sp = findsym(name, NULL, strncmp)) != NULL) {
		if (aout_flag)
			sp->u.n.n_type &= ~N_EXT;
		else
			sp->u.e.st_info = (STB_LOCAL << 4) |
		    			  (ELF32_ST_TYPE(sp->u.e.st_info));
	}
}

int
defsym(const char *name, unsigned long value, int type, int source)
{
	struct symbol *sp;
	/*
	 * field "*_other" abused in the symbol table
	 * in order to discriminate between defines and references
	 * and resolves by kernel resident or transient symbols...
	 */

	if (!aout_flag)
		++name; /* Horrible kludge */

	if ((sp = findsym(name, NULL, strncmp)) == NULL)
		return 0; /* symbol not used */
	/* else */

	/*
	 * Multiply defined?
	 * Well, this is what we _want_!
	 * I.e. a module _should_ be able to replace a kernel resident
	 * symbol, or even a symbol defined by another module!
	 * NOTE: The symbols read from the kernel (the transient ones)
	 * MUST be handled in the order:
	 *   last defined, ..., earlier defined, ...
	 */
	if (aout_flag) {
		if (sp->u.n.n_type != (N_UNDF | N_EXT))
			return 0; /* symbol not used */
		sp->u.n.n_type = type;
	}
	else { /* elf */
		if ((ELF32_ST_BIND(sp->u.e.st_info) != STB_GLOBAL) ||
		    (ELF32_ST_TYPE(sp->u.e.st_info) != STT_NOTYPE))
			return 0; /* symbol not used */
		/* hack follows... */
		if (type & N_ABS)
			sp->u.e.st_info = (STB_GLOBAL << 4);
		else
			sp->u.e.st_info = (STB_LOCAL << 4);
		sp->u.e.st_info |= STT_OBJECT; /* or whatever */
	}

	symother(sp) |= source;
	symvalue(sp) = value;

	return 1; /* symbol used */
}


/*
 * Look up an name in the symbol table.  If "add" is not null, add a
 * the entry to the table.  The table is stored as a splay tree.
 */
struct symbol *
findsym(const char *key, struct symbol *add,
	int (*pfi) (const char *, const char *, size_t))
{
	struct symbol *left, *right;
	struct symbol **leftp, **rightp;
	struct symbol *sp1, *sp2, *sp3;
	int cmp;
	int path1, path2;

	if (add) {
		if (aout_flag)
			add->u.n.n_un.n_name = (char *)key;
		else
			add->u.e.st_name = (Elf32_Word)key;
	}
	sp1 = symroot;
	if (sp1 == NULL)
		return add? symroot = add : NULL;
	leftp = &left, rightp = &right;
	for (;;) {
		cmp = (*pfi)(symname(sp1), key, SYM_MAX_NAME);
		if (cmp == 0)
			break;
		if (cmp > 0) {
			sp2 = sp1->child[0];
			path1 = 0;
		} else {
			sp2 = sp1->child[1];
			path1 = 1;
		}
		if (sp2 == NULL) {
			if (! add)
				break;
			sp2 = add;
		}
		cmp = (*pfi)(symname(sp2), key, SYM_MAX_NAME);
		if (cmp == 0) {
one_level_only:
			if (path1 == 0) {	/* sp2 is left child of sp1 */
				*rightp = sp1;
				rightp = &sp1->child[0];
			} else {
				*leftp = sp1;
				leftp = &sp1->child[1];
			}
			sp1 = sp2;
			break;
		}
		if (cmp > 0) {
			sp3 = sp2->child[0];
			path2 = 0;
		} else {
			sp3 = sp2->child[1];
			path2 = 1;
		}
		if (sp3 == NULL) {
			if (! add)
				goto one_level_only;
			sp3 = add;
		}
		if (path1 == 0) {
			if (path2 == 0) {
				sp1->child[0] = sp2->child[1];
				sp2->child[1] = sp1;
				*rightp = sp2;
				rightp = &sp2->child[0];
			} else {
				*rightp = sp1;
				rightp = &sp1->child[0];
				*leftp = sp2;
				leftp = &sp2->child[1];
			}
		} else {
			if (path2 == 0) {
				*leftp = sp1;
				leftp = &sp1->child[1];
				*rightp = sp2;
				rightp = &sp2->child[0];
			} else {
				sp1->child[1] = sp2->child[0];
				sp2->child[0] = sp1;
				*leftp = sp2;
				leftp = &sp2->child[1];
			}
		}
		sp1 = sp3;
	}
	/*
	 * Now sp1 points to the result of the search.  If cmp is zero,
	 * we had a match; otherwise not.
	 */
	*leftp = sp1->child[0];
	*rightp = sp1->child[1];
	sp1->child[0] = left;
	sp1->child[1] = right;
	symroot = sp1;
	return cmp == 0? sp1 : NULL;
}


unsigned long
looksym(const char *name)
{
	struct symbol *sp;

	if (!aout_flag)
		++name; /* Horrible kludge */

	sp = findsym(name, NULL, strncmp);
	if (sp == NULL) {
		fprintf(stderr, "%s undefined\n", name);
		exit(2);
	}
	return symvalue(sp);
}

/*
 * This is (was) rmmod, now merged with insmod!
 *
 * Original author: Jon Tombs <jon@gtex02.us.es>,
 * and extended by Bjorn Ekwall <bj0rn@blox.se> in 1994 (C).
 * See the file COPYING for your rights (GNU GPL)
 */

#define WANT_TO_REMOVE 1
#define CAN_REMOVE 2

struct ref {
	int modnum;
	struct ref *next;
};

struct mod {
	int status; /* or of: WANT_TO_REMOVE, CAN_REMOVE */
	char name[MOD_MAX_NAME];
	struct ref *ref;
};

struct mod *loaded;
int current = -1;

/* build the references as shown in /proc/ksyms */
void
get_stacks()
{
	FILE *fp;
	struct ref *rp;
	int i;
	char line[200]; /* or whatever... */
	char *p;
	char *r;

	if ((fp = fopen("/proc/modules", "r")) == (FILE *)0) {
		perror("/proc/modules");
		exit(1);
	}

	while (fgets(line, 200, fp)) {
		if (loaded) {
			++current;
			loaded = (struct mod *)realloc(loaded,
				(current + 1) * sizeof(struct mod));
		} else {
			current = 0;
			loaded = (struct mod *)ckalloc(sizeof(struct mod));
		}

		loaded[current].status = 0;
		loaded[current].ref = (struct ref *)0;

		/* save the module name */
		p = strchr(line, ' ');
		*p = '\0';
		strcpy(loaded[current].name, line);

		/* any references? */
		if ((p = strchr(p + 1, '['))) {
			*(strrchr(p + 1, ']')) = '\0';
			do {
				r = p + 1;
				if ((p = strchr(r, ' ')))
					*p = '\0';
				for (i = 0; i < current; ++i) {
					if (strcmp(loaded[i].name, r) == 0)
						break;
				}

				if (i == current) { /* not found! */
					fprintf(stderr,
					"Strange reference in "
					"/proc/modules: '%s' used by '%s'?\n",
					loaded[current].name, r);
					exit(1);
				}

				rp = (struct ref *)ckalloc(sizeof(struct ref));
				rp->modnum = i;
				rp->next = loaded[current].ref;
				loaded[current].ref = rp;
			} while (p);
		}
	}

	fclose(fp);
}

int
rmmod(int argc, char **argv)
{
	struct ref *rp;
	int i, j;
	int count;
	char **list;
	char *p;

	if (argc == 1) {
		fprintf(stderr, "usage: rmmod [-r] module ...\n");
		exit(1);
	}
	/* else */

	if (strcmp(argv[1], "-r") != 0) {
		while (argc > 1) {
			if ((p = strrchr(argv[1], '.')) &&
				((strcmp(p, ".o") == 0) ||
				 (strcmp(p, ".mod") == 0))) 
					*p = '\0';
			if (delete_module(argv[1]) < 0) {
				perror(argv[1]);
			}
			--argc;
			++argv;
		}
		exit(0);
	}
	/* else recursive removal */

	count = argc - 2;
	list = &(argv[2]);
	if (count <= 0) {
		fprintf(stderr, "usage: rmmod [-r] module ...\n");
		exit(1);
	}

	get_stacks();

	for (i = 0; i < count; ++i) {
		for (j = 0; j <= current; ++j) {
			if (strcmp(loaded[j].name, list[i]) == 0) {
				loaded[j].status = WANT_TO_REMOVE;
				break;
			}
		}
		if (j > current)
			fprintf(stderr, "module '%s' not loaded\n", list[i]);
	}

	for (i = 0; i <= current; ++i) {
		if (loaded[i].ref || (loaded[i].status == WANT_TO_REMOVE))
			loaded[i].status |= CAN_REMOVE;

		for (rp = loaded[i].ref; rp; rp = rp->next) {
			switch (loaded[rp->modnum].status) {
			case CAN_REMOVE:
			case WANT_TO_REMOVE | CAN_REMOVE:
				break;

			case WANT_TO_REMOVE:
				if (loaded[rp->modnum].ref == (struct ref *)0)
					break;
				/* else fallthtough */
			default:
				loaded[i].status &= ~CAN_REMOVE;
				break;
			}
		}

		switch (loaded[i].status) {
		case CAN_REMOVE:
		case WANT_TO_REMOVE | CAN_REMOVE:
			if (delete_module(loaded[i].name) < 0) {
				perror(loaded[i].name);
			}
			break;

		case WANT_TO_REMOVE:
			fprintf(stderr, "module '%s' is in use!\n",
				loaded[i].name);
			break;
		}
	}

	return 0;
}


/*
 * Used to be ksyms.c, but merged as well...
 *
 * Get kernel symbol table(s).
 *
 * Bjorn Ekwall <bj0rn@blox.se> in 1994 (C)
 * See the file COPYING for your rights (GNU GPL)
 */

int
ksyms(int argc, char **argv)
{
	struct kernel_sym *ksymtab = NULL;
	struct kernel_sym *ksym = NULL;
	struct module module_struct;
	int nksyms;
	int i;
	int kmem = 0;
	int allsyms = 0;
	int show_header = 1;
	char *p;
	char *module_name = "";

	while (argc > 1 && (argv[1][0] == '-')) {
		p = &(argv[1][1]);
		while (*p) {
			switch (*p) {
			case 'a':
				allsyms = 1;
				break;

			case 'm':
				if ((kmem = open("/dev/kmem", O_RDONLY)) < 0) {
					perror("/dev/kmem");
					exit(2);
				}
				break;

			case 'h':
				show_header = 0;
				break;
			}
			++p;
		}
		--argc;
		++argv;
	}

	if (argc < 1) {
		fputs("Usage: ksyms [-a] [-h]\n", stderr);
		exit(2);
	}

	/* get the size of the current kernel symbol table */
	nksyms = get_kernel_syms(NULL);

	if (nksyms < 0) {
		fprintf(stderr, "get_kernel_sys failed: Cannot find Kernel symbols!\n");
		exit(2);
	}

	if (nksyms) {
		ksymtab = (struct kernel_sym *) ckalloc(nksyms * sizeof *ksymtab);
		/* NOTE!!! The order of the symbols is important */
		if (get_kernel_syms(ksymtab) != nksyms) {
			fprintf(stderr, "Kernel symbol problem\n");
			exit(2);
		}
	}

	if (show_header)
		printf("Address  Symbol    \tDefined by\n");

	for (ksym = ksymtab, i = 0 ; i < nksyms ; ++i, ksym++) {
		/* Magic in this version of the new get_kernel_syms:
		 * Every module is sent along as a symbol,
		 * where the module name is represented as "#module", and
		 * the adress of the module struct is stuffed into the value.
		 * The name "#" means that the symbols that follow are
		 * kernel resident.
		 */
		if (ksym->name[0] == '#') {
			module_name = ksym->name + 1;
			if (!allsyms && (*module_name == '\0'))
				break;
			/* else */
			if (kmem) {
				if (
		(lseek(kmem, (off_t)ksym->value, SEEK_SET) > 0) &&
		(read(kmem, (char *)&module_struct, sizeof(struct module)) ==
		sizeof(struct module))) {
					printf("%08lx --- (%dk) ---\t[%s]\n",
					(long)module_struct.addr,
					module_struct.size * 4,
					module_name);
				}
				else {
					perror("/dev/kmem");
					exit(2);
				}
			}
			continue;
		}

		printf("%08lx %s", ksym->value, ksym->name);
		if (*module_name)
			printf("\t[%s]", module_name);
		printf("\n");
	}

	return 0;
}
