#include "basic.h"
#include "errlog.h"
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>

static void (*errorlog) (const char *str);
static int ErrLogLevel;

/*****************************************************************************/
/*                                                                           */
/*  ErrLogInit                                                               */
/*                                                                           */
/*****************************************************************************/
void 
ErrLogInit(void (*errlog)(const char *str), int level)
{
  errorlog = errlog;
  ErrLogLevel = level;
}

#ifdef __GNUC__

/*****************************************************************************/
/*                                                                           */
/*  _ErrLog                                                                  */
/*                                                                           */
/*****************************************************************************/
void
_ErrLog(const char *filename, int line, int level, const char *s, ...)
{
  va_list ap;
  char buf[2048];

  if (level > ErrLogLevel)
    return;
  
  sprintf(buf, "%s:%d: ", filename, line);
 
  va_start(ap, s);
  vsprintf(buf + strlen(buf), s, ap);
  va_end(ap);

  if (errorlog != NULL)
    (*errorlog)(buf);
}
  
#else /* __GNUC__ */

/*****************************************************************************/
/*                                                                           */
/*  ErrLog                                                                   */
/*                                                                           */
/*****************************************************************************/
void
ErrLog(int level, const char *s, ...)
{
  va_list ap;
  char buf[2048];

  if (level > ErrLogLevel) return;
  
  if (errorlog != NULL) {
    va_start(ap, s);
    vsprintf(buf, s, ap);
    va_end(ap);
    (*errorlog)(buf);
  }
}

#endif /* __GNUC__ */

#if DEBUG > 0

/*****************************************************************************/
/*                                                                           */
/*  DebugPrintf  */
/*                                                                           */
/*****************************************************************************/
void
DebugPrintf(const char *s, ...)
{
  va_list ap;
  char buf[1024];               /* Not clean, but this should be enough      */

  MemGarbage(buf, 1024 * sizeof(char));
  
  va_start(ap, s);
  vsprintf(buf, s, ap);
  va_end(ap);

  ErrLog(ERR_INTERNAL, buf);
}

/*****************************************************************************/
/*                                                                           */
/*  AssertPrintf                                                             */
/*                                                                           */
/*****************************************************************************/
void
AssertPrintf(const char *file, int line, const char *s, ...)
{
  va_list ap;
  char buf[1024];               /* Not clean, but this should be enough      */
  
  ErrLog(ERR_INTERNAL, "%s:%d:\n", file, line);

  va_start(ap, s);
  MemGarbage(buf, 1024*sizeof(char));
  vsprintf(buf, s, ap);
  va_end(ap);

  ErrLog(ERR_INTERNAL, buf);
}

/*****************************************************************************/
/*                                                                           */
/*  AssertFailed                                                             */
/*  In a debugger, you should put a breakpoint here.                         */
/*                                                                           */
/*****************************************************************************/
void 
AssertFailed(const char *s, int l)
{
  static Bool do_exit = True;
  DebugPrintf("Hey, this should not happen!\n");
  if (do_exit)
    exit(1);
}


/*****************************************************************************/
/*                                                                           */
/*  AssertStr                                                                */
/*                                                                           */
/*****************************************************************************/
void
_AssertStr(const char *f, int l, const char *s)
{
  if (!s) {
    AssertPrintf(f,l, "String is NULL\n");
    AssertFailed(f,l);
  }
  if (strlen(s) > 0x10000) {
    AssertPrintf(f,l, "String > 64k\n");
    AssertFailed(f,l);
  }
}
    
/*****************************************************************************/
/*                                                                           */
/*  AssertPrintStr                                                           */
/*                                                                           */
/*****************************************************************************/
void     
_AssertPrintStr(const char *f, int l, const char *s)
{
  const char *p;
  
  _AssertStr(f,l,s);

  if (!s) return;

  for (p=s; *p; p++) {
    if (!isprint(*p)) {
      AssertPrintf(f,l, "Char should be printable: %d\n", *p);
      AssertFailed(f,l);
    }
  }
}
/*****************************************************************************/
/*                                                                           */
/*  _AssertPtr                                                                */
/*                                                                           */
/*****************************************************************************/
void
_AssertPtr(const char *f, int l, const void *const p)
{
  if (!p) {
    AssertPrintf(f,l, "Pointer is NULL\n");
    AssertFailed(f,l);
  }
}

#endif /* DEBUG > 0 */
