/*
-- ----------------------------------------------------------------------------
--
--     Object Name : boot.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver code for booting and initialising the transputer.
--
--     NOTES
--         This module should be changed if porting to a new host or transputer
--         board.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include "version.h"
#include "afserver.h"

/* External procedures */

extern long int peek_word();
extern int Link;

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : load_code
--
--     Input Parameters :
--         char *boot_file_name - name of the boot file.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int                  - booting error code.
--
--     DESCRIPTION
--         Boot the root transputer, ie. reset the root transputer, open the
--         input file and send it to the tansputer as a stream of bytes.
--
-- ----------------------------------------------------------------------------
*/

static int load_code (boot_file_name)
    char *boot_file_name;
{
    FILE *boot_file;
    int result = BOOT_FILE_OK;

    if ((boot_file = fopen(boot_file_name, OPEN_READ_BINARY)) != NULL)
    {
        int count = 0;
        char file_buffer[BOOT_BUFFER_LENGTH];

        while (result == BOOT_FILE_OK)
        {
            count = fread(file_buffer, 1, BOOT_BUFFER_LENGTH, boot_file);
            if (count > 0)
            {
		if ( write( Link, &file_buffer, count ) != count )
                	result = BOOT_FILE_SEND_ERR;
            }
            else if (feof(boot_file) == 0)
                result = BOOT_FILE_READ_ERR;
            else
                break;
        }
        if (result == BOOT_FILE_OK)
        {
            if (fclose(boot_file) != 0)
                result = BOOT_FILE_CLOSE_ERR;
        }
        else
            (void) fclose(boot_file);
    }
    else
        result = BOOT_FILE_OPEN_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : boot_root
--
--     Input Parameters :
--         char *boot_file_name - name of the boot file.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int                  - result of operation.
--
--     DESCRIPTION
--         Load the transputer with the contents of the given boot file and
--         report on the result.
--
-- ----------------------------------------------------------------------------
*/

static int boot_root (boot_file_name)
    char *boot_file_name;                            
{
    int result;

    result = load_code(boot_file_name);
    switch (result)
    {
        case BOOT_FILE_OK :
            break;
        case BOOT_FILE_OPEN_ERR :
            fprintf(stderr, "\nFailed to open boot file : %s\n", boot_file_name);
            break;
        case BOOT_FILE_READ_ERR :
            fprintf(stderr, "\nFailed to read from boot file : %s\n", boot_file_name);
            break;
        case BOOT_FILE_SEND_ERR :
            fprintf(stderr, "\nFailed to transmit boot file : %s\n", boot_file_name);
            break;
        case BOOT_FILE_CLOSE_ERR :
            fprintf(stderr, "\nFailed to close boot file : %s\n", boot_file_name);
            break;
        case BOOT_FILE_PROTOCOL_ERR :
            fprintf(stderr, "\nUnknown protocol tag in boot file : %s\n", boot_file_name);
            break;
        case BOOT_FILE_TARGET_ERR :
            fprintf(stderr, "\nUnknown machine target in boot file : %s\n",
                    boot_file_name);
            break;
        case BOOT_FILE_VERSION_ERR :
            fprintf(stderr, "\nUnknown version number in boot file : %s\n",
                    boot_file_name);
            break;
        case BOOT_FILE_SCALER_ERR :
            fprintf(stderr, "\nInvalid scaler workspace size in boot file : %s\n",
                    boot_file_name);
            break;
        case BOOT_FILE_VECTOR_ERR :
            fprintf(stderr, "\nInvalid vector workspace size in boot file : %s\n",
                    boot_file_name);
            break;
        case BOOT_FILE_CODE_ERR :
            fprintf(stderr, "\nInvalid code size in boot file : %s\n", boot_file_name);
            break;
        case DEBUG_FILE_OPEN_ERR :
            fprintf(stderr, "\nFailed to open debug file : %s\n", DEBUG_FILE);
            break;
        case DEBUG_FILE_CLOSE_ERR :
            fprintf(stderr, "\nFailed to close debug file : %s\n", DEBUG_FILE);
            break;
    }
    if (result != BOOT_FILE_OK)
        return(T_BAD_BOOT);
    return(T_TERMINATED);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : peek_base_memory
--
--     Input Parameters :
--         int size           - amount of memory to be peeked (in bytes).
--
--     Output Parameters :
--         char core_memory[] - peeked memory from the transputer
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Peek from MININT of the transputer the given size of memory (in
--         bytes).
--
-- ----------------------------------------------------------------------------
*/

static void peek_base_memory (size, core_memory)
    int size;
    char core_memory[];
{
    int i;

    for (i = 0; i < size; i += INT_LEN)
    {
        long int value;

        value = peek_word(i);
        core_memory[i] = value & 0xff;
        core_memory[i + 1] = (value >> 8) & 0xff;
        core_memory[i + 2] = (value >> 16) & 0xff;
        core_memory[i + 3] = (value >> 24) & 0xff;
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : root_init
--
--     Input Parameters :
--         char *boot_file_name    - name of the boot file.
--         int analyse_host        - analyse transputer flag.
--         int reset_host          - reset transputer flag.
--
--     Output Parameters :
--         long int *version_state - method used for transputer initialisation.
--         char *core_memory       - peeked memory from transputer (if any).
--
--     Result :
--         int - initialisation error code.
--
--     DESCRIPTION
--         Initialise the root transputer board by resetting it and sending the
--         boot file to it.
--
-- ----------------------------------------------------------------------------
*/

int root_init (boot_file_name, analyse_host, reset_host,
               version_state, core_memory)
    long int *version_state;
    int analyse_host, reset_host;
    char *boot_file_name, *core_memory;
{
    if (init_root())
    {
        if (boot_file_name != NULL)
        {
            int result;

            if (! reset_host)
            {
                *version_state = NORESET_STATE;
            }
            else if (analyse_host)
            {
                reset_analyse_root();
                *version_state = ANALYSE_STATE;
                fprintf(stderr, "Peeking root transputer ...");
                peek_base_memory(CORE_MEMORY_SIZE, core_memory);
                fprintf(stderr, "\r                           \r");
            }
            else
            {
                reset_root();
                *version_state = RESET_STATE;
            }
            fprintf(stderr, "Booting root transputer ...");
            result = boot_root(boot_file_name);
            fprintf(stderr, "\r                           \r");
            return(result);
        }
        else
        {
            *version_state = NOBOOT_STATE;
            return(T_TERMINATED);
        }
    }
    return(T_BAD_HOST_INIT);
}
