/*
-- ----------------------------------------------------------------------------
--
--     Object Name : commands.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver code for executing the commands coming from transputer.
--
--     NOTES
--         None.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--         20-Apr-1988    Antony King    Fixed bug in testing of i/o streams
--                                       for errors - now use ferror().
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include "version.h"
#include "afserver.h"

/* External variables */

extern char core_memory[];

extern long int option_flag, board_size_flag;

extern int transputer_result, tcominptr, tcomoutptr;

extern long int version_number, version_date, version_state;

/* Static variables */

static struct STREAM_DESC filers[N_STREAMS];               /* Stream data */

static char *open_types[MAX_OPEN_MODE+1][MAX_EXIST_MODE+1] /* Open modes */
                       [MAX_ACCESS_METHOD+1] = OPEN_TYPES;

/* External procedures */

extern long int ftell();

extern char *mktemp(), *fgets();

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : find_stream
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int - number of first stream which is not in use (if any).
--
--     DESCRIPTION
--         Search for the first stream which is not in use.
--
-- ----------------------------------------------------------------------------
*/

static int find_stream ()
{
    int i = 0, found = FALSE;

    while ((i < N_STREAMS) && (found == FALSE))
    {
        if ((filers[i].inuse) == FALSE)
            found = TRUE;
        else
            i++;
    }
    if (found)
        return(i);
    else
        return(INVALID_STREAM);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : valid_stream
--
--     Input Parameters :
--         int n - number of the stream to be tested.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int   - TRUE if the stream open, otherwise FALSE.
--
--     DESCRIPTION
--         Test given stream number to see if corresponds to an opened stream.
--
-- ----------------------------------------------------------------------------
*/

static int valid_stream (n)
    int n;
{
    if ((n >= 0) && (n < N_STREAMS))
        if (filers[n].inuse)
            return(TRUE);
    return(FALSE);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : make_file_name
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         char *name - file name created for a temporary file.
--
--     Result :
--         int        - TRUE if the file name created, otherwise FALSE.
--
--     DESCRIPTION
--         Create a unique file name for the creation of a temporary file.
--
-- ----------------------------------------------------------------------------
*/

static int make_file_name (name)
    char *name;
{
    char *str, template[10];

    strcpy(template, "afsXXXXXX");
    str = mktemp(template);
    strcpy(name, str);
    if (*name == '\0')
        return(FALSE);
    return(TRUE);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : seek_file_length
--
--     Input Parameters :
--         FILE *fptr       - the file stream to be used.
--
--     Output Parameters :
--         long int *length - length of the given file stream.
--
--     Result :
--         int              - result of operation.
--
--     DESCRIPTION
--         Find the size (i.e. length) of the given file stream in bytes.
--
-- ----------------------------------------------------------------------------
*/

static int seek_file_length (fptr, length)
    FILE *fptr;
    long int *length;
{
    long int curpos;
    int result = OPERATION_OK;

    /* Remember where we are */
    if ((curpos = ftell(fptr)) == -1L)
        result = OPERATIONFAILED_ERR;
    else
    {
        /* Go to the end to find where it is */
        if (fseek(fptr, 0L, 2))
            result = OPERATIONFAILED_ERR;
        else
        {
            if ((*length = ftell(fptr)) == -1L)
                result = OPERATIONFAILED_ERR;
            else
            {
                /* Go back to where we were */
                if (fseek(fptr, curpos, 0))
                    result = OPERATIONFAILED_ERR;
            }
        }
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : filer_close_file
--
--     Input Parameters :
--         struct STREAM_DESC *streamp - pointer to stream to be closed.
--         int close_mode              - close mode (leave or delete).
--
--     Output Parameters :
--         None.
--
--     Result :
--         int                         - result of operation.
--
--     DESCRIPTION
--         Close the given file stream, deleting the file if specified.
--
-- ----------------------------------------------------------------------------
*/

static int filer_close_file (streamp, close_mode)
    int close_mode;
    struct STREAM_DESC *streamp;
{
    int res = OPERATION_OK;

    if ((close_mode >= 0) && (close_mode <= MAX_CLOSE_OPTION))
    {
        if (streamp->lifetime_m != SPECIAL)
        {
            if (fclose(streamp->fileptr) == 0)
            {
                if ((close_mode == CLOSEDEL_OPTION) || (streamp->lifetime_m == TEMPORARY))
                {
                    if (delete_file(streamp->name) != 0)
                        res = OPERATIONFAILED_ERR;
                }
                streamp->inuse = FALSE;
            }
            else
            {
                res = OPERATIONFAILED_ERR;
                /* Remove our reference anyway */
                streamp->inuse = FALSE;
            }
        }
        else
            streamp->inuse = FALSE; /* Just remove our reference */
    }
    else
        res = INVALIDCLOSEOPTION_ERR;
    return(res);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : make_af_err
--
--     Input Parameters :
--         int c_err - system error number.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int       - a unique error number.
--
--     DESCRIPTION
--         Create a unique error number from system error number.
--
-- ----------------------------------------------------------------------------
*/

int make_af_err (c_err)
    int c_err;
{
    return(HOST_ERROR_BASE - c_err);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : filer_init
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Initialises the afserver's file i/o streams to be not in use.
--
-- ----------------------------------------------------------------------------
*/

void filer_init ()
{
    int i;

    for (i = 0; i < N_STREAMS; i++)
        filers[i].inuse = FALSE;
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : filer_close
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Close all streams that are still open.
--
-- ----------------------------------------------------------------------------
*/

void filer_close ()
{
    int i;

    for (i = 0; i < N_STREAMS; i++)
        if (filers[i].inuse)
        {
            (void) filer_close_file(&filers[i], CLOSE_OPTION);
            filers[i].inuse = FALSE;
        }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : terminate_call
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int      - result of operation.
--
--     DESCRIPTION
--         Performs afserver commands AlienTerminate.Cmd and Terminate.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int terminate_call ()
{
    return(OPERATION_OK);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_temp_call
--
--     Input Parameters :
--         int access_method - access method (text or binary).
--         int record_length - record length (not used).
--
--     Output Parameters :
--         int *stream_id    - stream number if operation ok.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver command OpenTemp.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int open_temp_call (access_method, record_length, stream_id)
    int access_method, record_length, *stream_id;
{
    int result = OPERATION_OK;

    *stream_id = INVALID_STREAM;
    if ((access_method >= 0) && (access_method <= MAX_ACCESS_METHOD))
    {
        *stream_id = find_stream();
        if (*stream_id >= 0)
        {
            struct STREAM_DESC *streamp = &filers[*stream_id];

            if (make_file_name(streamp->name))
            {
                FILE *fptr;
                char *type = open_types[UPDATE_MODE][NEW_FILE][access_method];

                if ((fptr = fopen(streamp->name, type)) != NULL)
                {
                    streamp->inuse = TRUE;
                    streamp->fileptr = fptr;
                    streamp->open_m = UPDATE_MODE;
                    streamp->exist_m = NEW_FILE;
                    streamp->access_m = access_method;
                    streamp->lifetime_m = TEMPORARY;
                    streamp->result = OPERATION_OK;
                }
                else
                    result = OPERATIONFAILED_ERR;
            }
            else
                result = OPERATIONFAILED_ERR;
        }
        else
            result = NOFREECHANNEL_ERR;
    }
    else
        result = INVALIDACCESSMETHOD_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_file_call
--
--     Input Parameters :
--         char filename[]   - name of file to be opened.
--         int length        - length of file name.
--         int access_method - access method (text or binary).
--         int open_mode     - open mode (read, write or update).
--         int exist_mode    - exist mode (old or new).
--         int record_length - record length (not used).
--         int lifetime      - lifetime of file (always permanent).
--
--     Output Parameters :
--         int *stream_id    - stream number if operation ok.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver command OpenFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int open_file_call (filename, length, access_method, open_mode, exist_mode,
                    record_length, stream_id, lifetime)
    char filename[];
    int *stream_id, lifetime;
    int length, access_method, open_mode, exist_mode, record_length;
{
    int result = OPERATION_OK;

    *stream_id = INVALID_STREAM;
    if ((length >= 0) && (length <= FILE_NAME_LENGTH))
    {
        if ((access_method >= 0) && (access_method <= MAX_ACCESS_METHOD))
        {
            if ((open_mode >= 0) && (open_mode <= MAX_OPEN_MODE))
            {
                if ((exist_mode >= 0) && (exist_mode <= MAX_EXIST_MODE))
                {
                    *stream_id = find_stream();
                    if (*stream_id >= 0)
                    {
                        FILE *fptr;
                        struct STREAM_DESC *streamp = &filers[*stream_id];
                        char *type = open_types[open_mode][exist_mode][access_method];

                        filename[length] = '\0';
                        strcpy(streamp->name, filename);
#ifdef DEBUG
			printf("DEBUG: afserver: opening file %s\n" , filename);
#endif
                        if ((fptr = fopen(streamp->name, type)) != NULL)
                        {
                            streamp->inuse = TRUE;
                            streamp->fileptr = fptr;
                            streamp->open_m = open_mode;
                            streamp->exist_m = exist_mode;
                            streamp->access_m = access_method;
                            streamp->lifetime_m = lifetime;
                            streamp->result = OPERATION_OK;
                        }
                        else
                            result = OPERATIONFAILED_ERR;
                    }
                    else
                        result = NOFREECHANNEL_ERR;
                }
                else
                    result = INVALIDEXISTMODE_ERR;
            }
            else
                result = INVALIDOPENMODE_ERR;
        }
        else
            result = INVALIDACCESSMETHOD_ERR;
    }
    else
        result = INVALIDFILENAMELENGTH_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_stream_call
--
--     Input Parameters :
--         int std_stream_no - standard stream type (screen, keyboard, etc).
--         int open_mode     - open mode (input or ouptut).
--
--     Output Parameters :
--         int *stream_id    - stream number if operation ok.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver commands OpenInputStream.Cmd and
--         OpenOutputStream.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int open_stream_call (std_stream_no, stream_id, open_mode)
    int std_stream_no, *stream_id, open_mode;
{
    int result = OPERATION_OK;

    *stream_id = INVALID_STREAM;
    if (open_mode == S_INPUT)
    {
        switch (std_stream_no)
        {
            case 0:
                *stream_id = find_stream();
                if (*stream_id >= 0)
                {
                    struct STREAM_DESC *streamp = &filers[*stream_id];

                    streamp->inuse = TRUE;
                    strcpy(streamp->name, "stdin");
                    streamp->fileptr = stdin;
                    streamp->open_m = READ_MODE;
                    streamp->exist_m = OLD_FILE;
                    streamp->access_m = TEXTBYTESTREAM_ACCESS;
                    streamp->lifetime_m = SPECIAL;
                    streamp->result = OPERATION_OK;
                }
                else
                    result = NOFREECHANNEL_ERR;
                break;
            case 1:
                *stream_id = PARAM_STREAM;
                break;
            default:
                result = INVALIDSTDSTREAM_ERR;
                break;
        }
    }
    else if (open_mode == S_OUTPUT)
    {
        switch (std_stream_no)
        {
            case 0:
                *stream_id = find_stream();
                if (*stream_id >= 0)
                {
                    struct STREAM_DESC *streamp = &filers[*stream_id];

                    streamp->inuse = TRUE;
                    strcpy(streamp->name, "stdout");
                    streamp->fileptr = stdout;
                    streamp->open_m = WRITE_MODE;
                    streamp->exist_m = OLD_FILE;
                    streamp->access_m = TEXTBYTESTREAM_ACCESS;
                    streamp->lifetime_m = SPECIAL;
                    streamp->result = OPERATION_OK;
                }
                else
                    result = NOFREECHANNEL_ERR;
                break;
            case 1:
                *stream_id = find_stream();
                if (*stream_id >= 0)
                {
                    struct STREAM_DESC *streamp = &filers[*stream_id];

                    streamp->inuse = TRUE;
                    strcpy(streamp->name, "stderr");
                    streamp->fileptr = stderr;
                    streamp->open_m = WRITE_MODE;
                    streamp->exist_m = OLD_FILE;
                    streamp->access_m = TEXTBYTESTREAM_ACCESS;
                    streamp->lifetime_m = SPECIAL;
                    streamp->result = OPERATION_OK;
                }
                else
                    result = NOFREECHANNEL_ERR;
                break;
            default:
                result = INVALIDSTDSTREAM_ERR;
                break;
        }
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : close_stream_call
--
--     Input Parameters :
--         int stream_id  - number of stream to be closed
--         int close_mode - close mode (leave or delete).
--
--     Output Parameters :
--         None.
--
--     Result :
--         int            - result of operation.
--
--     DESCRIPTION
--         Performs afserver command CloseStream.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int close_stream_call (stream_id, close_mode)
    int stream_id, close_mode;
{
    int result = OPERATION_OK;

    if (valid_stream(stream_id))
        result = filer_close_file(&(filers[stream_id]), close_mode);
    else
    {
        if (stream_id == PARAM_STREAM)
            tcomoutptr = 0;
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_block_call
--
--     Input Parameters :
--         int stream_id     - number of the stream to be read from.
--         int record_length - length of data block read from the stream.
--
--     Output Parameters :
--         char buffer[]     - data block read.
--         int *bytes_read   - actual length of data read.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReadBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int read_block_call (stream_id, record_length, buffer, bytes_read)
    char buffer[];
    int stream_id, record_length, *bytes_read;
{
    int result = OPERATION_OK;
#ifdef DEBUG
	printf("DEBUG: performing ReadBlock.Cmd\n");
#endif

    *bytes_read = 0;
    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        if (streamp->open_m != WRITE_MODE)
        {
            if ((record_length >= 0) && (record_length <= RECORD_LENGTH))
            {
                if (isatty(fileno(streamp->fileptr)))
                {
                    result = read_terminal(buffer, record_length, streamp, bytes_read);
                }
                else
                {
                    *bytes_read = fread(buffer, 1, record_length, streamp->fileptr);
                    if (ferror(streamp->fileptr))
                    {
                        streamp->result = make_af_err(ferror(streamp->fileptr));
                        result = OPERATIONFAILED_ERR;
                    }
                    else if (feof(streamp->fileptr))
                    {
                        streamp->result = ENDOFFILE;
                        result = ENDOFFILE;
                    }
                }
            }
            else
                result = INVALIDRECORDLENGTH_ERR;
        }
        else
            result = INVALIDSTREAMID_ERR;
    }
    else
    {
        if (stream_id == PARAM_STREAM)
        {
            if ((record_length >=0) && (record_length <= RECORD_LENGTH))
            {
                *bytes_read = tcomread(buffer, record_length);
                if (*bytes_read < record_length)
                    result = ENDOFFILE;
            }
            else
                result = INVALIDRECORDLENGTH_ERR;
        }
        else
           result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_block_call
--
--     Input Parameters :
--         int stream_id      - number of the stream to be written to.
--         char buffer[]      - data block to be written.
--         int record_length  - length of data block written to the stream.
--
--     Output Parameters :
--         int *bytes_written - actual length of data written.
--
--     Result :
--         int                - result of operation.
--
--     DESCRIPTION
--         Performs afserver command WriteBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int write_block_call (stream_id, buffer, record_length, bytes_written)
    char buffer[];
    int stream_id, record_length, *bytes_written;
{
    int result = OPERATION_OK;

    *bytes_written = 0;
    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        if (streamp->open_m != READ_MODE)
        {
            if ((record_length >= 0) && (record_length <= RECORD_LENGTH))
            {
                if (isatty(fileno(streamp->fileptr)))
                {
                    result = write_terminal(buffer, record_length, streamp, bytes_written);
                }
                else
                {
                    *bytes_written = fwrite(buffer, 1, record_length, streamp->fileptr);
                    if (ferror(streamp->fileptr))
                    {
                        streamp->result = make_af_err(ferror(streamp->fileptr));
                        result = OPERATIONFAILED_ERR;
                    }
                }
            }
            else
                result = INVALIDRECORDLENGTH_ERR;
        }
        else
            result = INVALIDSTREAMID_ERR;
    }
    else
        result = INVALIDSTREAMID_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_seek_call
--
--     Input Parameters :
--         int stream_id   - number of the stream to be seeked upon.
--         long int offset - stream offset to be seeked too.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command Seek.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_seek_call (stream_id, offset)
    int stream_id;
    long int offset;
{
    int result = OPERATION_OK;

    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        if (isatty(fileno(streamp->fileptr)))
            result = NOSEEKPOSSIBLE_ERR;
        else
        {
            if (offset >= 0)
            {
                if (fseek(streamp->fileptr, offset, 0))
                {
                    if (ferror(streamp->fileptr))
                    {
                        streamp->result = make_af_err(ferror(streamp->fileptr));
                        result = OPERATIONFAILED_ERR;
                    }
                }
            }
            else
                result = INVALIDSEEKOFFSET_ERR;
        }
    }
    else
    {
        if (stream_id == PARAM_STREAM)
            result = NOSEEKPOSSIBLE_ERR;
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_file_call
--
--     Input Parameters :
--         int stream_id   - number of the stream to be tested.
--
--     Output Parameters :
--         char filename[] - file name associated with the stream.
--         int *length     - length of the file name.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command StreamFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_file_call (stream_id, filename, length)
    char filename[];
    int stream_id, *length;
{
    int result = OPERATION_OK;

    *length = 0;
    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        strcpy(filename, streamp->name);
        *length = strlen(filename);
    }
    else
    {
        if (stream_id == PARAM_STREAM)
        {
            strcpy(filename, "stdcmd");
            *length = strlen(filename);
        }
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_length_call
--
--     Input Parameters :
--         int stream_id    - number of the stream to be tested.
--
--     Output Parameters :
--         long int *length - length of the stream (e.g. file length).
--
--     Result :
--         int              - result of operation.
--
--     DESCRIPTION
--         Performs afserver command StreamLength.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_length_call (stream_id, length)
    int stream_id;
    long int *length;
{
    int result = OPERATION_OK;

    *length = -1L;
    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        if (isatty(fileno(streamp->fileptr)))
            result = INVALIDSTREAMID_ERR;
        else
            result = seek_file_length(streamp->fileptr, length);
    }
    else
    {
        if (stream_id == PARAM_STREAM)
            *length = (long int) tcominptr;
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_access_call
--
--     Input Parameters :
--         int stream_id      - number of the stream to be tested.
--
--     Output Parameters :
--         int *access_method - access method used for the stream.
--
--     Result :
--         int                - result of operation.
--
--     DESCRIPTION
--         Performs afserver command StreamAccess.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_access_call (stream_id, access_method)
    int stream_id, *access_method;
{
    int result = OPERATION_OK;

    *access_method = -1;
    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        *access_method = streamp->access_m;
    }
    else
    {
        if (stream_id == PARAM_STREAM)
            *access_method = TEXTBYTESTREAM_ACCESS;
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_status_call
--
--     Input Parameters :
--         int stream_id      - number of the stream to be tested.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int                - result of operation.
--
--     DESCRIPTION
--         Performs afserver command StreamStatus.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_status_call (stream_id)
    int stream_id;
{
    int result;

    if (valid_stream(stream_id))
    {
        struct STREAM_DESC *streamp = &filers[stream_id];

        result = streamp->result;
    }
    else
    {
        if (stream_id == PARAM_STREAM)
        {
            if (tcominptr == tcomoutptr)
                result = ENDOFFILE;
            else
                result = OPERATION_OK;
        }
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_connect_call
--
--     Input Parameters :
--         int stream_id         - number of the stream to be tested.
--
--     Output Parameters :
--         int *device_connected - type of device connected to the stream.
--
--     Result :
--         int                   - result of operation.
--
--     DESCRIPTION
--         Performs afserver command StreamConnect.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int stream_connect_call (stream_id, device_connected)
    int stream_id, *device_connected;
{
    int result = OPERATION_OK;

    *device_connected = -1;
    if (valid_stream(stream_id))
    {
        struct stat buffer;
        struct STREAM_DESC *streamp = &filers[stream_id];

        if (fstat(fileno(streamp->fileptr), &buffer) == 0)
        {
            if (buffer.st_mode & S_IFCHR)
            {
                /* We are connected to a device */
                if (streamp->fileptr == stdin)
                    *device_connected = KEYBOARD_USE;
                else if ((streamp->fileptr == stderr) || (streamp->fileptr == stdout))
                    *device_connected = SCREEN_USE;
                else
                    result = OPERATIONFAILED_ERR;
            }
            else if (buffer.st_mode & S_IFREG)
            {
                /* We are connected to a file */
                if (streamp->lifetime_m == TEMPORARY)
                    *device_connected = TEMP_USE;
                else
                    *device_connected = FILE_USE;
            }
            else
                result = OPERATIONFAILED_ERR;
        }
        else
            result = OPERATIONFAILED_ERR;
    }
    else
    {
        if (stream_id == PARAM_STREAM)
            *device_connected = PARAMETER_USE;
        else
            result = INVALIDSTREAMID_ERR;
    }
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : set_result_call
--
--     Input Parameters :
--         int value - value to be returned by the afserver.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int       - result of operation.
--
--     DESCRIPTION
--         Performs afserver command SetResult.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int set_result_call (value)
    int value;
{
    transputer_result = value;
    return(OPERATION_OK);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : run_time_data_call
--
--     Input Parameters :
--         int option      - option to be read (option or board size flags).
--
--     Output Parameters :
--         long int *value - value of the option.
--
--     Result :
--         int             - result of operation.
--
--     DESCRIPTION
--         Performs afserver command RunTimeData.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int run_time_data_call (option, value)
    int option;
    long int *value;
{
    int result = OPERATION_OK;

    *value = -1;
    switch (option)
    {
        case 0:
            *value = option_flag;
            break;
        case 1:
            *value = board_size_flag;
            break;
        default:
            result = INVALIDDATAOPTION_ERR;
            break;
    }
 
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_core_dump_call
--
--     Input Parameters :
--         int base_offset   - offset from base of core dump to read from.
--         int record_length - length of data block to be read.
--
--     Output Parameters :
--         char buffer[]     - data block read from core dump.
--         int *bytes_read   - actual length of data block read.
--
--     Result :
--         int               - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ReadCoreDump.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int read_core_dump_call (base_offset, record_length, buffer, bytes_read)
    char buffer[];
    int base_offset, record_length, *bytes_read;
{
    int count, result = OPERATION_OK;

    *bytes_read = 0;
    if ((base_offset >= 0) && (base_offset < CORE_MEMORY_SIZE))
    {
        if ((record_length >= 0) && (record_length <= RECORD_LENGTH))
        {
            if ((version_state & STATE_MASK) == ANALYSE_STATE)
            {
                if (CORE_MEMORY_SIZE >= (base_offset + record_length))
                {
                    *bytes_read = record_length;
                    for (count = 0; count < *bytes_read; count++)
                        buffer[count] = core_memory[base_offset + count];
                }
                else
                {
                    *bytes_read = CORE_MEMORY_SIZE - base_offset;
                    for (count = 0; count < *bytes_read; count++)
                        buffer[count] = core_memory[base_offset + count];
                    result = ENDOFFILE;
                }
            }
            else
                result = OPERATIONFAILED_ERR;
        }
        else
            result = INVALIDRECORDLENGTH_ERR;
    }
    else
        result = INVALIDCOREOFFSET_ERR;
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : server_version_call
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         long int *number - afserver version number.
--         long int *date   - afserver version data.
--         long int *state  - loading state of transputer.
--
--     Result :
--         int              - result of operation.
--
--     DESCRIPTION
--         Performs afserver command ServerVersion.Cmd.
--
-- ----------------------------------------------------------------------------
*/

int server_version_call (number, date, state)
    long int *number, *date, *state;
{
    *number = version_number;
    *date = version_date;
    *state = version_state;
    return(OPERATION_OK);
}
