/*
-- ----------------------------------------------------------------------------
--
--     Object Name : process.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver code for processing the commands coming from transputer.
--
--     NOTES
--         None.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include "version.h"
#include "afserver.h"

/* External variables */

extern int terminate_code;

/* Static variables */

static int command = -1;    /* Last server command */

static int running = TRUE;  /* Server is running flag */

/* External procedures */


/*
-- ----------------------------------------------------------------------------
--
--     Function Name : terminate_server
--
--     Input Parameters :
--         int t_code - terminate code of the afserver.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Set the flags to terminate the afserver and the afserver termination
--         code.
--
-- ----------------------------------------------------------------------------
*/

static void terminate_server (t_code)
    int t_code;
{
    running = FALSE;
    terminate_code = t_code;
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : tag_from_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         int - protocol tag received from the transputer.
--
--     DESCRIPTION
--         Receives an afserver protocol tag from the transputer.
--
-- ----------------------------------------------------------------------------
*/

static int tag_from_link ()
{
#ifdef REV_A_FIX
    return(word_from_link());
#else
    return(byte_from_link());
#endif
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_integer
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         int *v - value received from the transputer.
--
--     Result :
--         int    - TRUE if tag value is int32.value, otherwise FALSE.
--
--     DESCRIPTION
--         Receives a tagged integer value from the transputer.
--
-- ----------------------------------------------------------------------------
*/

static int read_integer (v)
    int *v;
{
    if (tag_from_link() == INT32_VALUE)
    {
        *v = word_from_link();
        return(TRUE);
    }
    else
    {
        terminate_server(T_BAD_INT32);
        fprintf(stderr, "Last command = %d\n", command);
        return(FALSE);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_long_integer
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         long int *v - value received from the transputer.
--
--     Result :
--         int         - TRUE if tag value is int32.value, otherwise FALSE.
--
--     DESCRIPTION
--         Receives a tagged integer value from the transputer.
--
-- ----------------------------------------------------------------------------
*/

static int read_long_integer (v)
    long int *v;
{
    if (tag_from_link() == INT32_VALUE)
    {
        *v = word_from_link();
        return(TRUE);
    }
    else
    {
        terminate_server(T_BAD_INT32);
        fprintf(stderr, "Last command = %d\n", command);
        return(FALSE);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_record
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         int *len     - length of the record received.
--         char *buffer - record received from the transputer.
--
--     Result :
--         int          - TRUE if tag value is record32.value or
--                        nilrecord.value, otherwise FALSE.
--
--     DESCRIPTION
--         Receives a tagged record and length count from the transputer.
--
-- ----------------------------------------------------------------------------
*/

static int read_record (len, buffer)
    int *len;
    char *buffer;
{
    int t;


    t = tag_from_link();
    if (t == NILRECORD_VALUE)
    {
        *len = 0;
        return(TRUE);
    }
    else if (t == RECORD32_VALUE)
    {
        *len = trunc_slice_from_link(RECORD_LENGTH, buffer);
        return(TRUE);
    }
    else
    {
        terminate_server(T_BAD_RECORD);
        fprintf(stderr, "Last command = %d\n", command);
        return(FALSE);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : tag_to_link
--
--     Input Parameters :
--         int tag - protocol tag to be sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Sends an afserver protocol tag to the transputer.
--
-- ----------------------------------------------------------------------------
*/

static void tag_to_link (tag)
    int tag;
{
#ifdef REV_A_FIX
    word_to_link(tag);
#else
    byte_to_link(tag);
#endif
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_integer
--
--     Input Parameters :
--         int n - value to be sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Sends a tagged integer value to the transputer.
--
-- ----------------------------------------------------------------------------
*/

static void write_integer (n)
    int n;
{
    tag_to_link(INT32_VALUE);
    word_to_link(n);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_long_integer
--
--     Input Parameters :
--         long int n - value to be sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Sends a tagged integer value to the transputer.
--
-- ----------------------------------------------------------------------------
*/

static void write_long_integer (n)
    long int n;
{
    tag_to_link(INT32_VALUE);
    word_to_link(n);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_record
--
--     Input Parameters :
--         int len      - length of the record sent.
--         char *buffer - record sent to the transputer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Sends a tagged record and length count to the transputer.
--
-- ----------------------------------------------------------------------------
*/

static void write_record (len, buffer)
    int len;
    char *buffer;
{
    if (len == 0)
        tag_to_link(NILRECORD_VALUE);
    else
    {
        tag_to_link(RECORD32_VALUE);
        slice_to_link(len, buffer);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : terminate
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver commands AlienTerminate.Cmd
--         and Terminate.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void terminate ()
{
    int result;

    result = terminate_call();
    write_integer(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_temp
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command OpenTemp.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void open_temp ()
{
    int access_method, record_length, result, stream_id;

    if (read_integer(&access_method))
    if (read_integer(&record_length))
    {
        result = open_temp_call(access_method, record_length, &stream_id);
        write_integer(stream_id);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_file
--
--     Input Parameters :
--         int lifetime - lifetime of file (always permanent).
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command OpenFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void open_file (lifetime)
    int lifetime;
{
    int result, stream_id;
    char filename[RECORD_LENGTH + 1];
    int length, access_method, open_mode, exist_mode, record_length;

    if (read_record(&length, filename))
    if (read_integer(&access_method))
    if (read_integer(&open_mode))
    if (read_integer(&exist_mode))
    if (read_integer(&record_length))
    {
        result = open_file_call(filename, length, access_method, open_mode, exist_mode,
                                record_length, &stream_id, lifetime);
        write_integer(stream_id);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : open_stream
--
--     Input Parameters :
--         int open_mode - open mode (input or ouptut).
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver commands
--         OpenInputStream.Cmd and OpenOutputStream.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void open_stream (open_mode)
    int open_mode;
{
    int std_stream_no, result, stream_id;

    if (read_integer(&std_stream_no))
    {
        result = open_stream_call(std_stream_no, &stream_id, open_mode);
        write_integer(stream_id);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : close_stream
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command CloseStream.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void close_stream ()
{
    int result, close_mode, stream_id;

    if (read_integer(&stream_id))
    if (read_integer(&close_mode))
    {
        result = close_stream_call(stream_id, close_mode);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_block
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ReadBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_block ()
{
    char buffer[RECORD_LENGTH + 1];
    int record_length, stream_id, result, bytes_read;

    if (read_integer(&stream_id))
    if (read_integer(&record_length))
    {
        result = read_block_call(stream_id, record_length, buffer, &bytes_read);
        write_record(bytes_read, buffer);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : write_block
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command WriteBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void write_block ()
{
    char buffer[RECORD_LENGTH];
    int record_length, stream_id, result, bytes_written;

    if (read_integer(&stream_id))
    if (read_record(&record_length, buffer))
    {
        result = write_block_call(stream_id, buffer, record_length, &bytes_written);
        write_integer(bytes_written);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_seek
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command Seek.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_seek ()
{
    int offset;
    int stream_id, result;

    if (read_integer(&stream_id))
    if (read_integer(&offset))
    {
        result = stream_seek_call(stream_id, offset);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_file
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command StreamFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_file ()
{
    int stream_id, length, result;
    char filename[RECORD_LENGTH + 1];

    if (read_integer(&stream_id))
    {
        result = stream_file_call(stream_id, filename, &length);
        write_record(length, filename);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_length
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command StreamLength.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_length ()
{
    int length;
    int stream_id, result;

    if (read_integer(&stream_id))
    {
        result = stream_length_call(stream_id, &length);
        write_integer(length);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_access
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command StreamAccess.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_access ()
{
    int stream_id, result, access_method;

    if (read_integer(&stream_id))
    {
        result = stream_access_call(stream_id, &access_method);
        write_integer(access_method);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_status
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command StreamStatus.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_status ()
{
    int stream_id, result;

    if (read_integer(&stream_id))
    {
        result = stream_status_call(stream_id);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : stream_connect
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command StreamConnect.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void stream_connect ()
{
    int stream_id, result, device_connected;

    if (read_integer(&stream_id))
    {
        result = stream_connect_call(stream_id, &device_connected);
        write_integer(device_connected);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : set_result
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command SetResult.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void set_result ()
{
    int value, result;

    if (read_integer(&value))
    {
        result = set_result_call(value);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : run_time_data
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command RunTimeData.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void run_time_data ()
{
    int value;
    int option, result;

    if (read_integer(&option))
    {
        result = run_time_data_call(option, &value);
        write_integer(value);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_core_dump
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ReadCoreDump.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_core_dump ()
{
    char buffer[RECORD_LENGTH];
    int base_offset, record_length, result, bytes_read;

    if (read_integer(&base_offset))
    if (read_integer(&record_length))
    {
        result = read_core_dump_call(base_offset, record_length, buffer, &bytes_read);
        write_record(bytes_read, buffer);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : server_version
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ServerVersion.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void server_version ()
{
    int result;
    int number, date, state;

    result = server_version_call(&number, &date, &state);
    write_integer(number);
    write_integer(date);
    write_integer(state);
    write_integer(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_key
--
--     Input Parameters :
--         int wait - flag to determine wether to wait for a key to be pressed.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver commands ReadKey.Cmd and
--         ReadKeyWait.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_key (wait)
    int wait;
{
    int result, chr;

    result = read_key_call(&chr, wait);
    write_integer(chr);
    write_integer(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_time
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ReadTime.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_time ()
{
    int result;
    int seconds;

    result = read_time_call(&seconds);
    write_integer(seconds);
    write_integer(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : rename_file
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command RenameFile.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void rename_file ()
{
    int old_length, new_length, result;
    char old_name[RECORD_LENGTH + 1], new_name[RECORD_LENGTH + 1];

    if (read_record(&old_length, old_name))
    if (read_record(&new_length, new_name))
    {
        result = rename_file_call(old_name, old_length, new_name, new_length);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : run_command
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command RunCommand.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void run_command ()
{
    int length, result;
    char buffer[RECORD_LENGTH + 1];

    if (read_record(&length, buffer))
    {
        result = run_command_call(buffer, length);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_environment
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command
--         ReadEnvironment.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_environment ()
{
    int old_length, new_length, result;
    char old_name[RECORD_LENGTH + 1], new_name[RECORD_LENGTH + 1];

    if (read_record(&old_length, old_name))
    {
        result = read_environment_call(old_name, old_length, new_name, &new_length);
        write_record(new_length, new_name);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : send_block
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command SendBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void send_block ()
{
    int destination;
    char buffer[RECORD_LENGTH];
    int length, result, bytes_written;

    if (read_integer(&destination))
    if (read_record(&length, buffer))
    {
        result = send_block_call(destination, buffer, length, &bytes_written);
        write_integer(bytes_written);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : receive_block
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ReceiveBlock.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void receive_block ()
{
    int source;
    char buffer[RECORD_LENGTH];
    int result, bytes_read, length;

    if (read_integer(&source))
    if (read_integer(&length))
    {
        result = receive_block_call(source, length, buffer, &bytes_read);
        write_record(bytes_read, buffer);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_regs
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command ReadRegs.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void read_regs ()
{
    int result, length;
    char buffer[RECORD_LENGTH];

    result = read_regs_call(buffer, &length);
    write_record(length, buffer);
    write_integer(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : call_interrupt
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command CallInterrupt.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void call_interrupt ()
{
    char input[RECORD_LENGTH], output[RECORD_LENGTH];
    int intrpt_no, in_length, out_length, flag, result;

    if (read_integer(&intrpt_no))
    if (read_record(&in_length, input))
    {
        result = call_interrupt_call(intrpt_no, input, in_length,
                                     &flag, output, &out_length);
        write_integer(flag);
        write_record(out_length, output);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : port_read
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command PortRead.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void port_read ()
{
    int address, value, result;

    if (read_integer(&address))
    {
        result = port_read_call(address, &value);
        write_integer(value);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : port_write
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Reads and writes parameters for afserver command PortWrite.Cmd.
--
-- ----------------------------------------------------------------------------
*/

static void port_write ()
{
    int address, value, result;

    if (read_integer(&address))
    if (read_integer(&value))
    {
        result = port_write_call(address, value);
        write_integer(result);
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_link
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Polls the transputer for afserver commands.
--
-- ----------------------------------------------------------------------------
*/

static void read_link ()
{
    if (read_integer(&command))
    {
        switch (command)
        {
            case TERMINATE_CMD:
                terminate();
                terminate_server(T_TERMINATED);
                break;
            case ALIENTERMINATE_CMD:
                terminate();
                break;
            case OPENFILE_CMD:
                open_file(PERMANENT);
                break;
            case OPENTEMP_CMD:
                open_temp();
                break;
            case OPENINPUTSTREAM_CMD:
                open_stream(S_INPUT);
                break;
            case OPENOUTPUTSTREAM_CMD:
                open_stream(S_OUTPUT);
                break;
            case STREAMACCESS_CMD:
                stream_access();
                break;
            case STREAMSTATUS_CMD:
                stream_status();
                break;
            case STREAMFILE_CMD:
                stream_file();
                break;
            case STREAMLENGTH_CMD:
                stream_length();
                break;
            case STREAMCONNECT_CMD:
                stream_connect();
                break;
            case CLOSESTREAM_CMD:
                close_stream();
                break;
            case READBLOCK_CMD:
                read_block();
                break;
            case WRITEBLOCK_CMD:
                write_block();
                break;
            case SEEK_CMD:
                stream_seek();
                break;

            case SETRESULT_CMD:
                set_result();
                break;
            case RUNTIMEDATA_CMD:
                run_time_data();
                break;
            case READCOREDUMP_CMD:
                read_core_dump();
                break;
            case SERVERVERSION_CMD:
                server_version();
                break;

            case READTIME_CMD:
                read_time();
                break;
            case RUNCOMMAND_CMD:
                run_command();
                break;
            case RENAMEFILE_CMD:
                rename_file();
                break;
            case READKEY_CMD:
                read_key(FALSE);
                break;
            case READKEYWAIT_CMD:
                read_key(TRUE);
                break;
            case READENVIRONMENT_CMD:
                read_environment();
                break;
            case RECEIVEBLOCK_CMD:
                receive_block();
                break;
            case SENDBLOCK_CMD:
                send_block();
                break;
            case CALLINTERRUPT_CMD:
                call_interrupt();
                break;
            case READREGS_CMD:
                read_regs();
                break;
            case PORTREAD_CMD:
                port_read();
                break;
            case PORTWRITE_CMD:
                port_write();
                break;

            default:
                terminate_server(T_BAD_COMMAND);
                break;
        }
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : server_main
--
--     Input Parameters :
--         None.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Main loop of the afserver, it is always waiting for the next server
--         command until the afserver is terminated.
--
-- ----------------------------------------------------------------------------
*/

void server_main ()
{
    while (running)
    {
        error_test();
        read_link();
    }
}
