/*
-- ----------------------------------------------------------------------------
--
--  Object name : server.c
--  Revision    : 1.0
--
--  Copyright (c) Inmos Ltd, 1987.
--  All Rights Reserved.
--
--  DESCRIPTION
--    TDS Server main body.
--
--  NOTES
--
--  HISTORY
--     ibm-pc  6-Apr-86  CREATION.
--     ibm-pc  23-Feb-88 RJO - Changed ReadLink to read Byte from Link instead
--             of Word... the upgrade goes on.
--     ibm-pc  24-Feb-88 RJO - Improved the tds.suspend facility following the
--             example in "Advanced MS-DOS" book (p.195).
--     ibm-pc  29-Feb-88 RJO - Added LOCKFILE generation to stop multiple TDSs
--             being run on the B004.
--     ibm-pc   3-Mar-88 RJO - Added new command line options 'x','n'.
--     ibm-pc  21-Apr-88 RJO - Fixed unitiliased variable (TcomOutPtr) in main.
-- ----------------------------------------------------------------------------
*/
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <linux/types.h>
#include <linux/termios.h>
#include <errno.h>
#include "srvconst.h"
#include "inmos.h"
#include <signal.h>
#include <setjmp.h>

#define DEBUG_NO
#define FIND_KEYS_NO

/*
-- ----------------------------------------------------------------------------
-- Exit Reboot Key values.
-- ----------------------------------------------------------------------------
*/

#define EXIT_KEY		ESCAPE
#define REBOOT_KEY		SPACE
#define LINK_NAME		"/dev/link"

/*
-- ----------------------------------------------------------------------------
-- Server global variables.
-- ----------------------------------------------------------------------------
*/

BYTE TopLevelFile[FILE_NAME_LENGTH];
BOOL Running;                   /* Server is Running flag                  */
BYTE Tcom[COMMAND_LINE_LENGTH]; /* Transputer command line buffer          */
                                /* Contains parameters from the server     */
                                /* command line which the server does not  */
                                /* recognise.                              */
int TcomInPtr = 0,              /* Pointers to the TCL buffer              */
    TcomOutPtr = 0;
BOOL ReBoot = FALSE;            /* Flags whether we are rebooting or not   */
BOOL InitialBootFlag = TRUE;    /* Set FALSE when '-n' on command line     */
BOOL AnalyseFlag = FALSE;       /* Set TRUE  when '-x' on command line     */
BOOL BootFlag = FALSE;          /* Set TRUE  when '-b' on command line     */
/*
-- ----------------------------------------------------------------------------
-- Server static variables
-- ----------------------------------------------------------------------------
*/

static int TerminateCode;       /* Server result                           */
static BYTE Version[] = "Server14,C version ";
/*
-- ----------------------------------------------------------------------------
-- External variables.
-- ----------------------------------------------------------------------------
*/

extern int Link;
extern struct termios OrgMode, CurMode;
extern int LAST_SCREEN_COL;
extern int LAST_SCREEN_LINE;

/*
-- ----------------------------------------------------------------------------
-- BOOL CreateTopLevelFile - Create a .TOP file.
--
-- Input Parameters:
--    (BYTE *) ToplevelFile  - Name of toplevel file.
-- Output Parameters:
--    (BOOL) True or False.
--
-- ----------------------------------------------------------------------------
*/
BOOL CreateTopLevelFile (TopLevelFile)
BYTE *TopLevelFile;
{
  FILE *fil_ptr;

  if ((fil_ptr = fopen (TopLevelFile, "w")) != NULL)
    {
      if (fwrite(TOP_LEVEL_FILE_CONTENTS, sizeof(char),
                 TOP_LEVEL_FILE_LENGTH, fil_ptr) == TOP_LEVEL_FILE_LENGTH)
        {
          if (fclose(fil_ptr) != 0)
            {
              printf ("Cannot create top level file\n");
              return (FALSE);
            }
        }
      else
        {
          printf ("Cannot create top level file\n");
          return(FALSE);
        }
    }
  else
    {
      fprintf (stderr, "Cannot create top level file\n");
      return (FALSE);
    };
  return (TRUE);
}
/*
-- ----------------------------------------------------------------------------
-- BOOL SetUpToLevelFile
--
-- Input Parameters:
--    (BYTE *) ToplevelFile  - Name of toplevel file.
--    (BYTE *) FileNamePtr
--
-- Output Parameters:
--    (BOOL) True or False.
--
-- ----------------------------------------------------------------------------
*/
BOOL SetUpTopLevelFile (TopLevelFile, FileNamePtr)
BYTE *TopLevelFile;
BYTE *FileNamePtr;
{
  BYTE *ExtnPtr;
  if ((ExtnPtr = strchr(FileNamePtr, '.')) != NULL)
    {
      if (strcmp(ExtnPtr, GENERIC_TOP_LEVEL_NAME_EXTN) == 0)
        strcpy(TopLevelFile, FileNamePtr);
      else
        {
          fprintf (stderr, "Illegal top level filename extension\n");
          return (FALSE);
        }
    }
  else
    {
      fprintf (stderr,"No filename extension in top level filename\n");
      return(FALSE);
    };
  return (TRUE);
}
/*
-- ----------------------------------------------------------------------------
-- void TerminateServer - Set flags to terminate the server.
--
-- Input Parameters:
--    (int) TCode - Value of termination code.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void TerminateServer (TCode)
int TCode;
{
  TerminateCode = TCode;
  Running = FALSE;
  ioctl(0, TCSETS, &OrgMode);
}
/*
-- ----------------------------------------------------------------------------
-- void TerminateMessage - Report the server's reason for termination.
--
-- Input Parameters:
--    (int) TCode - Value of termination code.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void TerminateMessage (TCode)
int TCode;
{
  switch (TCode)
    {
      case T_OK: printf("Server finished\n"); break;
      case T_USR_STOP:
      case T_USR_BRK:
        printf ("Server aborted by user\n"); break;
      case T_INVALID_TOPFILE_READ:
        printf ("Server terminated: cannot read top level file\n"); break;
      case T_TRANSPUTER_ERR:
        printf ("Server terminated: error in transputer system\n"); break;
      case T_BAD_BOOT:
        printf ("Server terminated: cannot boot root transputer\n"); break;
      case T_SCREEN_CLOSED:
        printf ("Server terminated: screen message received after closing \
                     screen channel\n"); break;
      case T_FILER_CLOSED:
        printf ("Server terminated: filer message received after closing \
                     filer channel\n"); break;
      case T_ILLEGAL_LINK_CHANNEL:
        printf ("Server terminated: message recieved on illegal channel\n");
            break;
      case T_ILLEGAL_SCREEN_P_TOKEN:
        printf ("Server terminated: illegal token received on screen.p \
                     channel\n"); break;
      case T_NO_BOARD:
        printf ("Server terminated: No transputer board connected\n"); break;
      case T_BAD_FILER_COMMAND:
        printf ("Server terminated: illegal filer command received\n");break;
      case T_BAD_COMMAND_LINE:
        printf ("Server terminated: bad command line\n"); break;
      case T_BOARD_IN_USE:
        printf ("Server Terminated: The B004 is already in use\n"); break;
      default:
        printf ("Server terminated: Unknown terminate code %d\n",TCode);
        break;
    };
}
/*
-- ----------------------------------------------------------------------------
-- void ServerInit - Initialise the server.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void ServerInit ()
{
  Running = TRUE;
  TerminateCode = 0;
}
/*
-- ----------------------------------------------------------------------------
-- int RootInit -Initialise the root transputer board by resetting it and
--               sending the boot file to it.
--
-- Input Parameters:
--   (BYTE) boot_file_name[] - Name of the bootfile
--   (int)  analyse_flag;    - Analyse or not to analyse...
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
int RootInit (BootFileName, AnalyseFlag )
BYTE BootFileName[];
int AnalyseFlag;
{
  printf ( "Booting root transputer ...\n" );     /* The famous message ... */
  return (BootRoot (BootFileName));
}
/*
-- ----------------------------------------------------------------------------
-- void ReadLink - Read a message coming down the link.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void ReadLink ()
{
  int Channel;
  Channel = ByteFromLink();

  switch (Channel)
    {
      case TO_TERM_P_CHAN:
        {
          int TermPResult;
          TermPResult = DoTermPChan();
          if (TermPResult != T_OK)
            TerminateServer (TermPResult);
          break;
        }
      case TO_FILER_0_CHAN:
      case TO_FILER_1_CHAN:
      case TO_FILER_2_CHAN:
      case TO_FILER_3_CHAN:
      case TO_FILER_4_CHAN:
      case TO_FILER_5_CHAN:
      case TO_FILER_6_CHAN:
      case TO_FILER_7_CHAN:
	{
          int FilerResult;
          FilerResult = DoFilerChan (Channel - TO_FILER_0_CHAN);
          if (FilerResult != T_OK)
            TerminateServer (FilerResult);
          break;
	}
      case TO_KERNEL_CHAN:
          DoKernelChan();
          break;
      default:
          printf ("The illegal channel no is %d\n",Channel);
          TerminateServer (T_ILLEGAL_LINK_CHANNEL);
          break;
    }
}
/*
-- ----------------------------------------------------------------------------
-- void RootError - Process an error in the transputer system.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void RootError ()
{
  TerminateServer (T_TRANSPUTER_ERR);
}
/*
-- ----------------------------------------------------------------------------
-- int UsrBrkHandler - Process a user break ( control-break on the keyboard).
--
-- Input Parameters:
--    (int) sig - Signal number.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void UsrBrkHandler (Sig)
int Sig;
{
  TerminateServer (T_USR_BRK);
}
/*
-- ----------------------------------------------------------------------------
-- int ReadNumber - Read in a number !
--
-- Input Parameters:
--    (BYTE *) Str - Number string.
--
-- Output Parameters:
--    (int) Result - The actual number.
--
-- ----------------------------------------------------------------------------
*/
int ReadNumber (Str)
BYTE *Str;
{
  int result = 0;
  int base;
  int carry_on;
  int c;

  if (*Str == '#')
    {
      base = 16;
      Str++;
    }
  else
    base = 10;

  carry_on = TRUE;
  while (carry_on)
    {
      c = *Str++;
      if ((c >= '0') && (c <= '9'))
        result = (result * base) + c - '0';
      else
        {
          if (base == 16)
            {
              if ((c >= 'a') && (c <= 'f'))
                result = (result * base) + 10 + c - 'a';
              else
                {
                  if ((c >= 'A') && (c <= 'F'))
                    result = (result * base) + 10 + c - 'A';
                  else
                    carry_on = FALSE;
                }
            }
          else
            carry_on = FALSE;
        }
    };
  return (result);
}
/*
-- ----------------------------------------------------------------------------
-- void AddToTcom - Add string to Transputer Command Line (TCL).
--
-- Input Parameters:
--    (BYTE *) Str - Command Line string.
--
-- Output Parameters:
--    (int) Result - The actual number.
--
-- ----------------------------------------------------------------------------
*/
void AddToTcom (Str)
BYTE *Str;
{
  BYTE C;
  while ((C = *Str++) != '\0')
    Tcom[TcomInPtr++] = C;
  Tcom[TcomInPtr++] = ' ';
}
/*
-- ----------------------------------------------------------------------------
-- BOOL ParseCommand - Parse the command line typed in by the user.
--
-- Input Parameters:
--    int argc     - Number of CLI args.
--    (BYTE *) argv[] - CLI args themselves.
--    (BYTE *) BootFileName - Name of boot file.
--    (BYTE *) TopLevelFile - Toplevel file name.
--
-- Output Parameters:
--    (BOOL) Result - Worked or it didnt !
-- ----------------------------------------------------------------------------
*/
BOOL ParseCommand (argc, argv, BootFileName, TopLevelFile)
int argc;
BYTE *argv[];
BYTE *BootFileName;
BYTE *TopLevelFile;
{
  BYTE *Ch;
  int i = 1;

  InitialBootFlag = TRUE;   /* Initialise variables     */
  AnalyseFlag = FALSE;
  strcpy (TopLevelFile, DEFAULT_TOP_LEVEL);
  strcpy (BootFileName, DEFAULT_BOOT_NAME);

  while (i < argc)
    {
      Ch = argv[i++];
      switch (*Ch++)
        {
          case '-':
            switch (*Ch++)
              {
                case 'b':
                  if (i < argc)
                    {
                      BootFlag = TRUE;                    /* We have a boot flag */
                      strcpy (BootFileName, argv[i++]);
                    }
                  else
                    {
                      printf ("Missing boot file name parameter\n");
                      return (FALSE);
                    };
                  break;
                case 't':
                  if (i < argc)
                    {
                      if (!(SetUpTopLevelFile (TopLevelFile, argv[i++])))
                        return (FALSE);
                    }
                  else
                    {
                      printf ("Missing top level file parameter\n");
                      return (FALSE);
                    };
                  break;
                case 'x':
                  AnalyseFlag = TRUE;
                  break;
                case 'n':
                  InitialBootFlag = FALSE;
                  break;
                default:
                  AddToTcom ( argv[i-1] );
                  break;
              };
            break;
          default:                        /* Add this unknown command to the TCL */
            AddToTcom ( argv[i-1] );
            break;
        }
    }

  /* Make sure top level file exists */

  if ( !(FileExists (TopLevelFile)))
    {
      if ( ! (CreateTopLevelFile (TopLevelFile)))
        {
          printf ("Cannot create top level file\n");
          return(FALSE);
        }
    }
  return (TRUE);
}

/*
-- ----------------------------------------------------------------------------
-- BOOL BoardInUse - Test if any other TDS has been loaded on the B004.
--
-- Input Parameters:
--     (int *)  Result - Means to return the result code.
-- Output Parameters:
--     (BOOL)   Return result.
-- ----------------------------------------------------------------------------
*/
BOOL BoardInUse (Result)
int *Result;
{
  struct winsize win;
  /* I think this is a good place to get the current screen geometry */

  ioctl( fileno(stdout), TIOCGWINSZ, &win );

  if ( !win.ws_col || !win.ws_row )
  {
    printf("Can not determine your terminal's screen size\n");
    return(TRUE);
  }

  LAST_SCREEN_COL = win.ws_col - 1;
  LAST_SCREEN_LINE = win.ws_row - 1;
  
  Link = open( LINK_NAME, O_RDWR);
  if (Link == -1)
  {
    perror("Opening Link");
    return(TRUE);
  }
  else
    return(FALSE);
}
/*
-- ----------------------------------------------------------------------------
-- void ServerMain - Main body of the Server where it all happens.
-- All the links, error flag etc are polled here.
-- Input Parameters:
--    None.
-- Output Parameters:
--    None.
--
-- Notes:
-- There is also an assembly version of this available in Quickio.asm.
-- ----------------------------------------------------------------------------
*/
void ServerMain ()
{
  while (Running)
    {
      if (LinkInTest()) 
      {
#ifdef DEBUG
	printf("DEBUG: Reading from Link\n");
#endif
	ReadLink();
      }
      if (KeyTest())
      {
#ifdef DEBUG
	printf("DEBUG: Reading a key\n");
#endif
	ReadKey();
      }
      if (ErrorTest()) 
      {
#ifdef DEBUG
	printf("DEBUG: calling RootError()\n");
#endif
	RootError();
      }
    }
}
/*
-- ----------------------------------------------------------------------------
-- void main - The real main body of the Server where it all happens.
--
-- Input Parameters:
--    None.
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void main (argc, argv, envp)
int argc;
BYTE *argv[];
BYTE *envp[];
{
  int AnalyseFlag = FALSE,   /* Boot transputer in analyse mode flag */
      InitFlag;              /* Boot transputer flag                 */
  BYTE BootFileName[FILE_NAME_LENGTH];
  int Result = F_OK;         /* Clear error flag                     */

  ReBoot = FALSE;
  printf (Version);
  printf ("\nCopyright INMOS Limited, 1985; Christoph Niemann, 1993\n");
  InitRoot();

    
  if (!BoardInUse (&TerminateCode))     /* test if another TDS is running */
    {
#ifdef DEBUG
      printf("DEBUG: Board not in use, ok\n");
#endif      
      /* Parse the command line */
      if (!ParseCommand(argc, argv, BootFileName, TopLevelFile))
        {
          TerminateCode = T_BAD_COMMAND_LINE;
          InitFlag = FALSE;
        }
      else
        InitFlag = TRUE;
#ifdef DEBUG
      printf("DEBUG: Parsed command line, ok\n");
#endif

      while (InitFlag)
        {
#ifdef DEBUG
	  printf("DEBUG: Begin initialisation of all server services & functions\n");
#endif
          /* Begin initialisation of all server services and functions */
          InitFlag = FALSE;
          TcomOutPtr = 0;      /* Start at beginning of Transputer Command Line */
          TermPInit ();
          ServerInit();
          FilerInit();

          Result = RootInit (BootFileName);
          if (Result == T_OK)
            {
#ifdef DEBUG
	      printf("DEBUG: RootInit ok, starting ServerMain()\n");
#endif
              signal (SIGINT, UsrBrkHandler);
              /* Cause a user break ( Control-Break on the keyboard */
              /* to pass control to the routine usr_brk_handler     */
#ifdef FIND_KEYS
  while (1) {
    int Key;
    printf("Press a Function-Key with ESC-prefix\n");
    Key = GetKey();
    if (Key == 27) 
    {
      Key = GetKey();
      printf("2nd Code = %d\n", Key);
      while ( KeyTest() ) 
        printf("Next Code = %d\n", GetKey());
     }
   }
#endif 

              ServerMain ();

              switch (TerminateCode)
                {
                  case T_TRANSPUTER_ERR:
                    {
                      int key;
                      fprintf (stderr,"Error in transputer system\n");

                      FilerClose();
                      AnalyseFlag = TRUE;
                      fprintf (stderr,"Press [SPACE] to reboot, [ESC] to quit\n");
                      do
                        key = GetKey();
                      while ( ( key != EXIT_KEY ) &&
                          ( key != REBOOT_KEY ) );
                      if ( key == REBOOT_KEY )
                        {
                         InitFlag = TRUE;
                         ReBoot = TRUE;
                        }
                      break;
                    }
                  case T_USR_BRK:
                    {
                      int key;

                      FilerClose();
                      AnalyseFlag = TRUE;
                      fprintf (stderr,"Press [SPACE] to reboot, [ESC] to quit\n");
                      do
                        key = GetKey();
                      while ( ( key != EXIT_KEY ) &&
                          ( key !=REBOOT_KEY ) );
                      if ( key == REBOOT_KEY )
                        {
                         InitFlag = TRUE;
                         ReBoot = TRUE;
                        }
                      break;
                    }
                  default:       break;
                }
            }
          else
            TerminateCode = Result;
        }
    }

  close( Link );
  ioctl(0, TCSETS, &OrgMode);
  TerminateMessage (TerminateCode);
  exit(0);
}
