/************************************************************************
*									*
*	Device Driver for INMOS-B004 compatible Link-Interfaces		*
*									*
*	Copyright (C) 1993 by Christoph Niemann				*
*			      niemann@swt.ruhr-uni-bochum.de		*
*	based on /linux/kernel/chr_drv/lp.c				*
*									*
*	Version 0.4, for Linux 0.99pl11 and above, November 1993.	*
*									*
*************************************************************************/

/* Copyright (C) 1992 by Jim Weigand, Linus Torvalds, and Michael K. Johnson */

#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/b004.h>

#include <asm/io.h>
#include <asm/segment.h>

/*
 * Define these symbols if you want to debug the code.
 */
#undef B004_DEBUG 
#undef B004_DEBUG_MORE

/* If your link interface does not work with the normal i/o-commands use
   the slower ones. But I think this should be no problem for any board
   that uses INMOS's C012 chip.
   Defining the symbol B004_REALLY_SLOW_C012 will decrease the data-
   transfer-rate from about 400 KBytes/sec to 200 KBytes/sec
   (on my noname 486/33/16, assembler version).
*/
#undef B004_REALLY_SLOW_C012

/*
 * Do not define this symbol if you want to use the C-version of
 * the read/write functions (much slower)
 */
#define USE_ASM


#ifdef B004_REALLY_SLOW_C012
#define out		outb_p
#define in		inb_p
#define B004_SLOW_DOWN	"outb %%al,$0x80\n\t"
#else
#define out		outb
#define in		inb
#define B004_SLOW_DOWN	""
#endif

#ifdef B004_DEBUG
#define DEB(x) x
#define NO_DEB(x) /* */
#else
#define DEB(x) /* */
#define NO_DEB(x) x
#endif

#ifdef B004_DEBUG_MORE
#define DEB_MORE(x) x
#else
#define DEB_MORE(x) /* */
#endif


static struct b004_struct b004_table[B004_NO] =
	{
		 { B004_WRITE_ABORT, 0, 0, 0, 0, B004_INIT_TIME,
		 B004_INIT_READ_TIMEOUT, B004_INIT_WRITE_TIMEOUT },
		 { B004_WRITE_ABORT, 0, 0, 0, 0, B004_INIT_TIME,
		 B004_INIT_READ_TIMEOUT, B004_INIT_WRITE_TIMEOUT },
		 { B004_WRITE_ABORT, 0, 0, 0, 0, B004_INIT_TIME,
		 B004_INIT_READ_TIMEOUT, B004_INIT_WRITE_TIMEOUT },
	};

static int b004_devices = 0;		/* number of devices dectected */
static int use_schedule = 0;            /* don't use schedule() in init() */

/*
 * At these addresses the driver will search for B004-compatible boards
 */
static int b004_base_addresses[B004_NO] = { 0x150, 0x200, 0x300 };

static void b004_delay( void )
{
	if ( use_schedule )
	{
		DEB(printk("b004_delay: Using schedule()\n");)
		current->state = TASK_INTERRUPTIBLE;
		current->timeout = jiffies + 10;   /* 100ms */
		schedule();
	}
	else
	{		/* only for detection in b004_init() */
		int i;
		DEB(printk("b004_delay: Using idle loop\n");)
		for (i = 0; i < 10000; )
			i++;
	}

}


/****************************************************************************
 *
 * static void b004_reset(): reset transputer network.
 *
 ****************************************************************************/

static void b004_reset( const int minor )
{

	DEB(printk("B004 resetting transputer at link %d.\n", minor);)

	out( B004_DEASSERT_ANALYSE, B004_BASE(minor) + B004_ANALYSE_OFFSET );
	b004_delay();

	out( B004_DEASSERT_RESET, B004_BASE(minor) + B004_RESET_OFFSET );
	b004_delay();

	out( B004_ASSERT_RESET, B004_BASE(minor) + B004_RESET_OFFSET );
	b004_delay();

	out( B004_DEASSERT_RESET, B004_BASE(minor) + B004_RESET_OFFSET );
	b004_delay();

	DEB(printk("B004 reset done.\n");)

}

/****************************************************************************
 *
 * static void b004_analyse(): switch transputer network to analyse mode.
 *
 ****************************************************************************/

static void b004_analyse( const int minor )
{

	DEB(printk("B004 switching transputer at link %d to analyse-mode.\n",
		minor);)

	out( B004_DEASSERT_ANALYSE, B004_BASE(minor) + B004_ANALYSE_OFFSET );
	b004_delay();

	out( B004_ASSERT_ANALYSE, B004_BASE(minor) + B004_ANALYSE_OFFSET );
	b004_delay();

	out( B004_ASSERT_RESET, B004_BASE(minor) + B004_RESET_OFFSET );
	b004_delay();

	out( B004_DEASSERT_RESET, B004_BASE(minor) + B004_RESET_OFFSET );
	b004_delay();

	out( B004_DEASSERT_ANALYSE, B004_BASE(minor) + B004_ANALYSE_OFFSET );
	b004_delay();

	DEB(printk("B004 switching to analyse-mode done.\n");)

} /* b004_analyse() */

	DEB(
		unsigned int b004_total_bytes_read = 0;
		unsigned int b004_read_last_call = 0;
	)

/****************************************************************************
 *
 * static int b004_read() - read bytes from the link interface.
 *
 * At first, the driver checks if the link-interface is ready to send a byte
 * to the PC. If not, this check is repeated up to B004_MAXTRY times.
 * If the link-interface is not ready after this loop, the driver sleeps for
 * B004_TIME jiffies and then checks the link-interface again.
 * If the interface is still not ready, the driver sleeps for B004_READ_TIMEOUT
 * jiffies. Then it checks the interface again. If it is not ready and
 * the flag B004_READ_ABORT is set, this function returns with the number
 * of bytes read or with an error if no byte was read.
 * If the B004_READ_ABORT flag is not set, the driver sleeps again for
 * B004_READ_ABORT jiffies and so on.
 *
 * By default, B004_READ_ABORT is not set.
 *
 *****************************************************************************/

static int b004_read(struct inode * inode, struct file * file, char * buf, int count)
{

	char 	*temp = buf;
	unsigned int minor = MINOR(inode->i_rdev);
#ifndef USE_ASM
	int	l_count;
	char	buffer[B004_MAX_BYTES];
	int	end;
	int	i;
#endif

	DEB(
		if (jiffies-b004_read_last_call > B004_TIME(minor)) 
			b004_total_bytes_read = 0;
		b004_read_last_call = jiffies;
	)
	DEB_MORE(printk("B004 reading %d bytes from link %d.\n", count, minor);)

	if ( count < 0)
	{
		DEB(printk("B004: invalid argument for reading: count = %d.\n", count);)
		return -EINVAL;
	}

#ifdef USE_ASM
	__asm__("cld": :);

	while ( 1 )
	{
		/*
		 * This is very ugly! For a description of this algorithm
		 * see the C-version below.
		 */
		__asm__(
			"movl %4,%0\n\t"
			"push %%es\n\t"
			"movw %%fs, %%ax\n\t"
			"movw %%ax, %%es\n"
			"9:\tmovl %0,%4\n\t"
			"movb $2, %%ah\n"
			"cmpl %7,%4\n\t"
			"jc 6f\n\t"
			"movl %7,%4\n"
			"6:\tmovl %4,%%esi\n"
			"5:\tinb %w2,%%al\n\t"
			B004_SLOW_DOWN
			"andb %6,%%al\n\t"
			"jnz 1f\n\t"
			"pushl %4\n\t"
			"movl %5,%4\n"
			"3:\tinb %w2,%%al\n\t"
			B004_SLOW_DOWN
			"andb %6,%%al\n\t"
			"jnz 2f\n\t"
			"decl %4\n\t"
			"jnz 3b\n\t"
			"popl %4\n\t"
			"jmp 7f\n"		/* timeout */
			"2:\tpopl %4\n"
			"1:\tsubb %%ah,%b2\n\t"
			"insb\n\t"
			B004_SLOW_DOWN
			"addb %%ah,%b2\n\t"
			"decl %4\n\t"
			"jnz 5b\n\t"
			"subl %%esi,%0\n\t"
			"jz 8f\n\t"		/* all transfered */
			"cmpl %4,_need_resched\n\t"	/* ecx == 0 */
			"je 9b\n\t"		/* go on */
			"pop %%es\n\t"
			"pushl %%edx\n\t"
			"call _schedule\n\t"
			"popl %%edx\n\t"
			"push %%es\n\t"
			"movw %%fs,%%ax\n\t"
			"movw %%ax,%%es\n\t"
			"jmp 9b\n"
			"7:\tsubl %4,%%esi\n\t"
			"subl %%esi,%0\n"
			"8:\tpop %%es"
			:"=b" (count),
			 "=D" (temp)
			:"d" ((unsigned short) B004_ISR(minor)),
			 "1" (temp),
			 "c" (count),
			 "i" (B004_MAXTRY),
			 "i" (B004_READBYTE),
			 "i" (B004_MAX_BYTES)
			:"ax", "di", "cx", "dx", "si");
		DEB(b004_total_bytes_read += temp - buf);

		if (!count)
			return temp - buf;
#else
	while ( count )
	{
		end = (B004_MAX_BYTES > count) ? count : B004_MAX_BYTES;
		l_count = 0;
		do
		{
			if (!in(B004_ISR(minor)) & B004_READBYTE)
			{
				i = B004_MAXTRY;
				while (!(in(B004_ISR(minor)) & B004_READBYTE) && i)
					i--;
				if (!i)
					break;
			}
			buffer[l_count++] = (char) in(B004_IDR(minor));
			DEB(b004_total_bytes_read++;)
		} while (--end);

		if (l_count)
		{
			memcpy_tofs(temp, buffer, l_count);
			count -= l_count;
			temp += l_count;
			if ( need_resched ) 
				schedule();
		}
#endif /* USE_ASM */
		else
		{
			/* check for signals before going to sleep. */
			if (current->signal & ~current->blocked)
				return temp - buf ? temp - buf : -EINTR;

			DEB(
				printk("B004 sleeping at %d bytes for %d jiffies (reading).\n",
					b004_total_bytes_read, B004_TIME(minor) );
				b004_total_bytes_read=0;
			)

			current->state = TASK_INTERRUPTIBLE;
			current->timeout = jiffies + B004_TIME(minor);
			schedule();

			/* If nothing is coming from the C012
			   for a considerable length of time,
			   someone oughtta know.  */
			while ( !( in(B004_ISR(minor)) & B004_READBYTE ) )
			{

				/* check for signals before going to sleep. */
				if (current->signal & ~current->blocked)
					return temp - buf ? temp - buf : -EINTR;

				current->state = TASK_INTERRUPTIBLE;
				current->timeout = jiffies + B004_READ_TIMEOUT(minor);
				schedule();
				DEB(printk("B004: timeout reading from link %d.\n", minor);)
				if ( !( in(B004_ISR(minor)) & B004_READBYTE )
					&& (B004_F(minor) & B004_READ_ABORT) )
				{
					printk("B004: timeout reading from link %d.\n", minor);
					return temp - buf ? temp - buf : -EFAULT;
				}
				
			}
		}
	}
	DEB_MORE(printk("B004 reading successful.\n");)

	return temp - buf;

} /* B004_read() */

#ifdef minor
#undef minor
#endif


	DEB(
		unsigned int b004_total_bytes_written = 0;
		unsigned int b004_write_last_call = 0;
	)

/****************************************************************************
 *
 * static int b004_write() - write to the link interface.
 *
 *****************************************************************************/

static int b004_write(struct inode * inode, struct file * file, char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	char	*temp = buf;
#ifndef USE_ASM
	int	l_count;
	int	end;
	char	buffer[B004_MAX_BYTES];
	int	i;
#endif

	DEB(
		if (jiffies-b004_write_last_call > B004_TIME(minor)) 
			b004_total_bytes_written = 0;
		b004_write_last_call = jiffies;
	)
	DEB_MORE(printk("B004 writing %d bytes to link %d.\n", count, minor);)

	if ( count < 0)
	{
		DEB(printk("B004 invalid argument for writing: count = %d.\n", count);)
		return -EINVAL;
	}

#ifdef USE_ASM
		__asm__("cld": :);

	while ( 1 )
	{
		/*
		 * this is very ugly!
		 */
		__asm__(
			"movl %4,%0\n\t"
			"push %%ds\n\t"
			"push %%fs\n\t"
			"movw %%fs,%%ax\n\t"
			"movw %%ds,%w4\n\t"
			"movw %%ax,%%ds\n"
			"movw %w4,%%fs\n\t"
			"9:\tmovl %0,%4\n\t"
			"movb $2,%%ah\n"
			"cmpl %7,%4\n\t"
			"jc 6f\n\t"
			"movl %7,%4\n"
			"6:\tmovl %4,%%edi\n"
			"5:\tinb %w2, %%al\n\t"
			B004_SLOW_DOWN
			"andb %6,%%al\n\t"
			"jnz 1f\n\t"
			"pushl %4\n\t"
			"movl %5,%4\n"
			"3:\tinb %w2, %%al\n\t"
			B004_SLOW_DOWN
			"andb %6,%%al\n\t"
			"jnz 2f\n\t"
			"decl %4\n\t"
			"jnz 3b\n\t"
			"popl %4\n\t"
			"jmp 7f\n"
			"2:\tpopl %4\n"
			"1:\tsubb %%ah,%b2\n\t"
			"outsb\n\t"
			B004_SLOW_DOWN
			"addb %%ah,%b2\n\t"
			"decl %4\n\t"
			"jnz 5b\n\t"
			"subl %%edi,%0\n\t"
			"jz 8f\n\t"
			"fs; cmpl %4,_need_resched\n\t"
			"je 9b\n\t"
			"pop %%fs\n\t"
			"pop %%ds\n\t"
			"pushl %%edx\n\t"
			"call _schedule\n\t"
			"popl %%edx\n\t"
			"push %%ds\n\t"
			"push %%fs\n\t"
			"movw %%fs,%%ax\n\t"
			"movw %%ds,%w4\n\t"
			"movw %%ax,%%ds\n"
			"movw %w4,%%fs\n"
			"jmp 9b\n"
			"7:\tsubl %4,%%edi\n\t"
			"subl %%edi,%0\n"
			"8:\tpop %%fs\n\t"
			"pop %%ds"
			:"=b" (count),
			 "=S" (temp)
			:"d" ((unsigned short) B004_OSR(minor)),
			 "1" (temp),
			 "c" (count),
			 "i" (B004_MAXTRY),
			 "i" (B004_WRITEBYTE),
			 "i" (B004_MAX_BYTES)
			:"ax", "si", "dx", "cx", "di");
		DEB(b004_total_bytes_written += temp - buf);

		if (!count)
			return temp - buf;
#else
	while ( count )
	{
		end = (B004_MAX_BYTES > count) ? count : B004_MAX_BYTES;
		memcpy_fromfs(buffer, temp, end);
		l_count = 0;
		do
		{
			if (!in(B004_OSR(minor)) & B004_WRITEBYTE)
			{
				i = B004_MAXTRY;
				while (!(in(B004_OSR(minor)) & B004_WRITEBYTE) && i)
					i--;
				if (!i)
					break;
			}
			out(buffer[l_count++], B004_ODR(minor));
			DEB(b004_total_bytes_written++;)
		} while (--end);

		if (l_count)
		{
			temp += l_count;
			count -= l_count;

			if ( need_resched )
				schedule();
		}
#endif	/* USE_ASM */
		else
		{
			/* check for signals before going to sleep. */
			if (current->signal & ~current->blocked)
				return temp - buf ? temp - buf : -EINTR;
			DEB(
				printk("B004 sleeping at %d bytes for %d jiffies (writing).\n",
					b004_total_bytes_written, B004_TIME(minor) );
				b004_total_bytes_written=0;
			)
			current->state = TASK_INTERRUPTIBLE;
			current->timeout = jiffies + B004_TIME(minor);
			schedule();

			/* If nothing is getting to the C012
			   for a considerable length of time,
			   someone oughtta know.  */
			while ( !( in(B004_OSR(minor)) & B004_WRITEBYTE ) )
			{
				if (current->signal & ~current->blocked)
					return temp - buf ? temp - buf : -EINTR;
				current->state = TASK_INTERRUPTIBLE;
				current->timeout = jiffies + B004_WRITE_TIMEOUT(minor);
				schedule();
				if ( !( in(B004_OSR(minor)) & B004_WRITEBYTE )
					&& (B004_F(minor) & B004_WRITE_ABORT) )
				{
					printk("B004: timeout writing to link %d.\n", minor);
					return temp - buf ? temp - buf : -EFAULT;
				}
			}
		}
	}

	DEB_MORE(printk("B004 writing successful.\n");)

	return temp-buf;
} /* b004_write() */

/****************************************************************************
 *
 * static int b004_lseek()
 *
 ***************************************************************************/
static int b004_lseek(struct inode * inode, struct file * file,
		    off_t offset, int origin)
{
	return -ESPIPE;
}

/****************************************************************************
 *
 * static int b004_open()
 *
 * open the link-device.
 *
 ***************************************************************************/

static int b004_open(struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR(inode->i_rdev);

	if (minor >= b004_devices)
	{
		DEB(printk("B004 not opened, minor device number >= %d.\n", b004_devices);)
		return -ENODEV;
	}
	if ((B004_F(minor) & B004_EXIST) == 0)
	{
		DEB(printk("B004 not opened, B004-board does not exist (minor = %d).\n", minor);)
		return -ENODEV;
	}
	if (B004_F(minor) & B004_BUSY)
	{
		DEB(printk("B004 not opened, B004-board busy (minor = %d).\n", minor);)
		return -EBUSY;
	}

	B004_F(minor) |= B004_BUSY;

	DEB(printk( "B004 opened, minor = %d.\n", minor );)
	return 0;

} /* b004_open() */


/****************************************************************************
 *
 * static int b004_release()
 *
 * close the link device.
 *
 ****************************************************************************/

static void b004_release(struct inode * inode, struct file * file)
{
	unsigned int minor = MINOR(inode->i_rdev);

	if (minor >= b004_devices)
	{
		DEB(printk("B004 not released, minor device number >= %d.\n", b004_devices);)
		return;
	}

	B004_F(minor) &= ~B004_BUSY;

	DEB(printk("B004(%d) released.\n", minor );)

} /* b004_release() */


/****************************************************************************
 *
 * static int b004_ioctl()
 *
 * This function performs the various ioctl() functions: resetting the
 * transputer, switching to analyse-mode, testing the status, changing
 * timeouts etc.
 *
 *****************************************************************************/

static int b004_ioctl(struct inode *inode, struct file *file,
		    unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	int result = arg;

	DEB(printk("B004(%d) ioctl, cmd: 0x%x, arg: 0x%x.\n", minor, cmd, arg);)

	if (minor >= b004_devices)
	{
		DEB(printk("B004 ioctl exit, minor >= %d.\n", b004_devices );)
		return -ENODEV;
	}

	if ((B004_F(minor) & B004_EXIST) == 0)
	{
		DEB(printk("B004 ioctl exit, (B004_F & B004_EXIST) == 0.\n" );)
		return -ENODEV;
	}

	switch ( cmd )
	{
		case B004TIME:		/* set the time that the driver sleeps
					   before retrying to read/write. */
			B004_TIME(minor) = arg;
			break;
		case B004RESET:		/* reset transputer */
			b004_reset(minor);
			break;
		case B004WRITEABLE:	/* can we write a byte to the C012 ? */
	 		result = ( ( in(B004_OSR(minor)) & B004_WRITEBYTE) != 0 ); 
			break;
		case B004READABLE:	/* can we read a byte from C012 ? */
	 		result = ( ( in(B004_ISR(minor)) & B004_READBYTE) != 0 ); 
			break;
		case B004ANALYSE:	/* switch transputer to analyse mode */
			b004_analyse(minor);
			break;
		case B004ERROR:		/* test error-flag */
			result = ( in(B004_BASE(minor) + B004_ERROR_OFFSET) & B004_TEST_ERROR) ? 0 : 1;
			break;
		case B004READTIMEOUT:	/* set timeout for reading*/
			result = B004_READ_TIMEOUT(minor);
			B004_READ_TIMEOUT(minor) = arg;
			break;
		case B004WRITETIMEOUT:	/* set timeout for writing */
			result = B004_WRITE_TIMEOUT(minor);
			B004_WRITE_TIMEOUT(minor) = arg;
			break;
		case B004READABORT:	/* abort after a timeout ? */
			if ( arg )
				B004_F(minor) |= B004_READ_ABORT;
			else
				B004_F(minor) &= ~B004_READ_ABORT;
			break;
		case B004WRITEABORT:	/* abort after a timeout ? */
			if ( arg )
				B004_F(minor) |= B004_WRITE_ABORT;
			else
				B004_F(minor) &= ~B004_WRITE_ABORT;
			break;
		default: result = -EINVAL;
	}

	DEB(printk("B004(%d) ioctl done.\n\n", minor);)

	return result;

} /* b004_ioctl() */


static struct file_operations b004_fops = {
	b004_lseek,
	b004_read,
	b004_write,
	NULL,		/* b004_readdir */
	NULL,		/* b004_select */
	b004_ioctl,
	NULL,		/* b004_mmap */
	b004_open,
	b004_release
};

/****************************************************************************
 *
 * long b004_init()
 *
 * This function initializes the driver. It tries to detect the hardware
 * and sets up all relevant data-structures.
 *
 ****************************************************************************/

long b004_init(long kmem_start)
{
	unsigned int test, i;

	if ( register_chrdev( B004_MAJOR, B004_NAME, &b004_fops ) )
	{
		printk("b004_init: unable to get major %d for link interface.\n",
			B004_MAJOR );
		return kmem_start;
	}

	/* After a reset it should be possible to write a byte to
	   the B004. So let'S do a reset and then test the output status
	   register
	*/
	for (test = 0; test < B004_NO; test++)
	{
		b004_delay();
		B004_BASE(b004_devices) = b004_base_addresses[test];
		B004_ODR(b004_devices) = B004_BASE(b004_devices) + B004_ODR_OFFSET;
		B004_ISR(b004_devices) = B004_BASE(b004_devices) + B004_ISR_OFFSET;
		B004_OSR(b004_devices) = B004_BASE(b004_devices) + B004_OSR_OFFSET;
		b004_reset(b004_devices);
		b004_delay();

		for (i = 0; i < B004_MAXTRY; i++)
			if ( in(B004_OSR(b004_devices)) == B004_WRITEBYTE)
			{
				B004_F(b004_devices) |= B004_EXIST;
				out(B004_BASE(b004_devices) + B008_INT_OFFSET, 0);
				b004_delay();
				if (in(B004_BASE(b004_devices) + B008_INT_OFFSET) & 0x0f == 0)
					B004_BOARDTYPE(b004_devices) = B008;
				else
					B004_BOARDTYPE(b004_devices) = B004;
				printk("link%d at 0x0%x (polling) is a B00%s\n",
					b004_devices,B004_IDR(b004_devices),
					(B004_BOARDTYPE(b004_devices) == B004) ? "4" : "8");
				b004_devices++;
				break;
			}

		}

	use_schedule = 1;

	if (!b004_devices)
		printk("b004_init(): no B004-board found.\n");

	return kmem_start;

} /* b004_init() */
