/* --------------------------------- views.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Show all visual displays
*/

#include "plane.h"


static MENU FAR MenuView[] = {
	{'f', "front"},		/*  0 */
	{'n', "none"},		/*  1 */
	{'a', "rear"},		/*  2 */
	{'m', "map"},		/*  3 */
	{'r', "radar"},		/*  4 */
	{'t', "target"},	/*  5 */
	{'p', "pan"},		/*  6 */
	{'g', "gaze"},		/*  7 */
	{'c', "chase"},		/*  8 */
	{'o', "follow"},	/*  9 */
	{'h', "hud"},		/* 10 */
	{'u', "up-front"},	/* 11 */
	{'P', "panel"},		/* 12 */
	{'R', "right"},		/* 13 */
	{'L', "left"},		/* 14 */
	{'s', "stores"},	/* 15 */
	{'l', "lamps"},		/* 16 */
	{'M', "mirror"},	/* 17 */
{'\0', 0}};

/* Select external view mode
*/

extern int FAR
menu_view (int sel)
{
	sel = menu_open (MenuView, sel);
	if (MENU_FAILED != sel)
		menu_close ();
	return (sel);
}

LOCAL_FUNC void NEAR
ident (int id, int orgx, int orgy, int maxx, int maxy, int ss)
{
	int	bss, i, l;
	char	*p;

	if (st.flags1 & SF_IDENT) {
		i = id ? st.hdd[id-1].type : st.extview;
		p = MenuView[i].text;
		l = strlen (p);
		bss = ss*4;
		if (bss > 64)
			bss = 64;
		if (bss > maxx/l)
			bss = maxx/l;
		stroke_num (orgx-bss*l/2, orgy, (long)id, bss, ST_INFO);
		stroke_str (orgx-bss*l/2, orgy+bss, p, bss, ST_INFO);
	} else if (!(st.flags & SF_BLANKER) && (st.flags1 & SF_WIDENT)) {
		i = id ? st.hdd[id-1].type
		       : (st.flags1&SF_EXTVIEW ? st.extview: -1);
		if (i >= 0) {
			p = MenuView[i].text;
			l = stroke_size (p, ss);
			stroke_str (orgx+maxx-l, orgy-maxy+ss, p, ss, ST_HFG);
		}
	}
}

/* Get window dimentions.
*/
extern void FAR
get_area (VIEW *view, int *orgx, int *orgy, int *maxx, int *maxy)
{
	int		px;
	VIEWPORT	*vp;

/* Calculate paralax.
*/
	if (F(vp = VP))
		vp = CVIEW->viewport;
	px = muldiv (vp->z, vp->shift, vp->distz);	/* normalized */
	px = muldiv (VS->sizex, px, vp->maxx) / 2;	/* pixels */
	px = fmul (px, VW->maxx) + (px<0);

	if (orgx)
		*orgx = fmul (VW->orgx, VS->sizex) + VS->minx + px;
	if (orgy)
		*orgy = fmul (VW->orgy, VS->sizey) + VS->miny;
	if (maxx)
		*maxx = fmul (VW->maxx, VS->sizex) - 1 - iabs(px);
	if (maxy)
		*maxy = fmul (VW->maxy, VS->sizey) - 1;
}

/* Get largest square that fits in window.
*/
extern void FAR
get_square (VIEW *view, int maxx, int maxy, int *sizex, int *sizey)
{
	int	sx, sy;

	sx = muldiv (VS->device->lengx, maxx, VS->device->sizex);
	sy = muldiv (VS->device->lengy, maxy, VS->device->sizey);

	if (sx > sy) {		/* wide window */
		sx = muldiv (maxx, sy, sx);
		sy = maxy;
	} else {		/* high window */
		sy = muldiv (maxy, sx, sy);
		sx = maxx;
	}
	*sizex = sx;
	*sizey = sy;
}

/* this is the main display dispatcher.
*/
extern void FAR
show_main (VIEW *view, OBJECT *pov, int type, int special)
{
	int	orgx, orgy, maxx, maxy, ss;
	OBJECT	*p;

	if (IS_PLANE(CV))
		p = CV;
	else
		p = CC;
	if (!p)
		return;

	get_area (view, &orgx, &orgy, &maxx, &maxy);
	if (maxx <= 0)
		return;

	ss = fmul (maxx, FCON (0.03));
	ss = muldiv (ss, st.StFontSize, 8);
	ss = (ss+1)&~1;					/* make it even */
	if (ss < 8)
		ss = 8;

	if (special & 1)
		show_hud (view, pov, p, orgx, orgy, maxx, maxy, type);

/* digital panel
*/
	if ((HDT_FRONT==type && (st.flags & SF_MAIN) && (EX->hud1 & HUD_PANEL))
	    || HDT_PANEL == type)
		show_panel (view, p, maxx, maxy, orgx, orgy, ss);

/* status info. This text is shown over the whole screen. It will not show if
 * there is an active UPFRONT HDD.
*/
	screen_info (view, orgx, orgy, maxx, maxy, ss, type);

	if (st.flags & SF_MAIN)
		ident (0, orgx, orgy, maxx, maxy, ss);
}

extern int FAR
scenery (int type)
{
	return (HDT_NONE != type && HDT_HUD != type && HDT_UPFRONT != type &&
		HDT_PANEL != type && HDT_LAMPS != type);
}

LOCAL_FUNC void NEAR
show_view (VIEW *cv, int type, int special)
{
	OBJECT	*pov;
	E_PLANE	*ecv;
	int	t, vr, wr, smaxx, smaxy;
	Ushort	shud, shud1;

	if (!cv)
		return;

	t = (HDT_HUD == type) ? HDT_FRONT : type;
	if (F(pov = get_viewer (t)) || !pov->viewport)
		return;

	ecv = IS_PLANE(CV) ? EE(CV) : 0;
	cv->viewport = pov->viewport;
	smaxx = cv->viewport->maxx;
	smaxy = cv->viewport->maxy;
	if (ecv) {
		shud  = ecv->hud;
		shud1 = ecv->hud1;
		ecv->hud  &= ~(HUD_ON|HUD_CURSOR|HUD_RETICLE);
		ecv->hud1 &= ~(HUD_VALARM|HUD_PANEL);
	} else
		shud = shud1 = 0;	/* avoid compiler warning */

/* Adjust aspect ratio. All viewports use the same aspect ratio as the main
 * window, now adjust the viewport for the actual window.
 * The 1/4 scaling is done to allow handling high aspect ratios.
*/
	if (!(st.flags & SF_MAIN)) {
		vr = muldiv (FONE/4, cv->viewport->maxx, cv->viewport->maxy);
		wr = muldiv (FONE/4, cv->window->maxx,   cv->window->maxy);
		wr = muldiv (wr, cv->screen->device->lengx,
						cv->screen->device->lengy);
#if 0
		if (vr > wr)	/* viewport too wide */
			cv->viewport->maxx = muldiv (cv->viewport->maxx,
							wr, vr);
		else		/* viewport too hight */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
#else
		if (vr > wr)	/* viewport wider than window */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
		else		/* viewport higher than window */
			cv->viewport->maxy = muldiv (cv->viewport->maxy,
							vr, wr);
#endif
	}

	if (scenery (type)) {
		show_sky (cv, pov);
		objects_show (0, cv, pov, 0, 0, 0);
	}

	show_main (cv, pov, type, special);

	cv->viewport->maxx = smaxx;
	cv->viewport->maxy = smaxy;
	if (ecv) {
		ecv->hud  = shud;
		ecv->hud1 = shud1;
	}

	cv->viewport = 0;
}

extern void FAR
show_hdd (void)
{
	int	i, orgx, orgy, maxx, maxy, ss, type;
	long	t27;
	VIEW	*v;

	t27 = STATS_TIME3D;
	Tm->Interval (TMR_START, 0L);
	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;

		v = &st.hdd[i].view;
		orgx = fmul (v->window->orgx, v->screen->sizex)
			+ v->screen->minx;
		orgy = fmul (v->window->orgy, v->screen->sizey)
			+ v->screen->miny;
		maxx = fmul (v->window->maxx, v->screen->sizex) - 1;
		maxy = fmul (v->window->maxy, v->screen->sizey) - 1;
		ss = fmul (maxx, FCON (0.04));
		ss = muldiv (ss, st.StFontSize, 8);
		ss = (ss+1)&~1;		/* make it even */
		if (ss < 8)
			ss = 8;

		switch (type = st.hdd[i].type) {
		case HDT_NONE:
			break;
		case HDT_UPFRONT:
			if (hdd_menu (v, orgx, orgy, maxx, maxy))
				screen_info (v, orgx, orgy, maxx, maxy, ss,
					st.hdd[i].type);
			break;
		case HDT_PANEL:
			show_panel (v, CC, maxx, maxy, orgx, orgy, ss);
			break;
		case HDT_STORES:
			show_stores (v, CC, maxx, maxy, orgx, orgy, ss);
			break;
		case HDT_LAMPS:
			show_lamps (v, CC, maxx, maxy, orgx, orgy, ss);
			break;
		default:
			show_view (v, st.hdd[i].type, 1);
			break;
		}
		ident (i+1, orgx, orgy, maxx, maxy, ss);
		buffer_close ();
		st.hdd[i].bufs[1-st.which_buffer] = st.buf[HEAD];
		st.buf[HEAD] = st.buf[TAIL] = 0;
	}
	STATS_TIMEHDD += Tm->Interval (TMR_STOP, 10000L)
						- (STATS_TIME3D - t27);
}
