/****************************************************************************
 *
 * Class:   AttributeList implementation
 * Author:  Mark Roseman
 * 
 * Maintain a list of attribute / value pairs.
 * 
 * Revision History:
 * 
 * Date     Modifier  Description
 * -------- --------- -------------------------------------------------------
 * 10/02/92 MR        initial version
 * 
 ****************************************************************************/
 
/*
 *  This file is part of GroupKit.
 *
 *  (c) Copyright 1992 Department of Computer Science, University of
 *      Calgary, Calgary, Alberta, Canada.  All rights reserved.
 *    
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted, provided
 *  that the above copyright notice appears in all copies.  The University
 *  of Calgary makes no representations about the suitability of this
 *  software for any purpose.  It is provided "as is" without express or
 *  implied warranty.
 */

#include <string.h>
#include <malloc.h>
#include <stdlib.h>
#include <gk/attrlist.h>


/**************************************************************************
 * 
 * AVPair - copy strings
 *
 **************************************************************************/

AVPair::AVPair(const char* at, const char* vl) { 
  attr = strdup(at); val = strdup(vl); 
}
AVPair::~AVPair() { free(attr); free(val); }



/**************************************************************************
 * 
 * AttributeList - constructor and destructor, return internal list
 *
 **************************************************************************/

implementPtrList(AVPairList,AVPair)

AttributeList::AttributeList() { list_ = new AVPairList(); }

AttributeList::~AttributeList() { 
  long i = list_->count();
  while(i > 0) { list_->remove(0); i--; }
  delete list_; 
}

AVPairList* AttributeList::list() { return list_; }


/**************************************************************************
 * 
 * add an attribute to the list
 *
 **************************************************************************/

void AttributeList::attribute(const char* attr, const char* name) { 
  remove_attribute(attr);
  list_->append( new AVPair(attr,name) );
}


/**************************************************************************
 * 
 * remove an attribute from the list by name
 *
 **************************************************************************/

void AttributeList::remove_attribute(const char* attr) { 
  long i;
  if( (i = find(attr)) !=-1)
    list_->remove(i);
}


/**************************************************************************
 * 
 * find attribute in list, return true if found
 *
 **************************************************************************/

boolean AttributeList::find_attribute(const char* attr, char* value) { 
  long i;
  if( (i=find(attr)) != -1) {
    strcpy(value, list_->item(i)->val);
    return true;
  } else return false;
}


/**************************************************************************
 * 
 * construct an attribute list from the given string (produced by write,
 * see below)
 *
 **************************************************************************/

AttributeList* AttributeList::read(const char* inp) { 
  AttributeList* a = new AttributeList();
  const char* s = inp;
  char attr[80]; char val[80]; char* p = attr;
  *attr = 0; *val = 0; 
  boolean readingAttr = true;
  while(*s != 0) {
    if(readingAttr) 
      if(*s=='=') 
	{ readingAttr = false; s++; *p = 0; p = val; }
      else 
	{ *p++ = *s++; }
    else 
      if(*s==':') 
	{ readingAttr = true; s++; *p=0; a->attribute( attr, val ); p = attr;}
      else if(*s=='\\') 
        { s++; *p++ = *s++; } 
      else 
	{ *p++ = *s++; }
  }
  if(!readingAttr) {
    *p = 0;
    a->attribute( attr, val );
  }
  return a;
}


/**************************************************************************
 * 
 * convert an attribute list to a string suitable for transmission.
 * the list will look like:   
 *
 *      a1=v1:a2=v2:... 
 *
 * with any ":", "&", "\" in the values escaped with a "\".
 *
 **************************************************************************/

void AttributeList::write(char* outp) { 
  *outp = 0;
  char *p, *s;
  for( ListItr(AVPairList) i(*list_); i.more(); i.next()) {
    if (strlen(outp) != 0) strcat(outp, ":");
    strcat( outp, i.cur()->attr );
    strcat( outp, "=" );
    for( p=i.cur()->val, s=outp+strlen(outp); *p!=0; ) {
      if( (*p==':') || (*p=='&') || (*p=='\\') ) *s++ = '\\';
      *s++ = *p++;
    }
    *s = 0;
  }
}

/**************************************************************************
 * 
 * find an attribute in the list and return its index in the list or -1
 *
 **************************************************************************/

long AttributeList::find(const char* attr) {
  long i;
  for( i = 0L; i < list_->count(); i++ )
    if ( strcmp(list_->item(i)->attr, attr)==0)
      return i;
  return -1;
}


/**************************************************************************
 * 
 * attribute list table
 *
 **************************************************************************/
implementTable(AttrLstTbl, int, AttributeList*);

AttrListTable::AttrListTable(int x) : AttrLstTbl(x) {}


/**************************************************************************
 * 
 * write out an attribute list table - same as writing out an attribute
 * list but with "&" between each list, i.e.
 * 
 *     a1=v1:a2=v2&a1=v1:a2=v2&...
 *
 **************************************************************************/

void AttrListTable::write(char* outp) {
  char s[1000];
  *outp = 0;
  for ( TableIterator(AttrLstTbl) i(*this); i.more(); i.next()) {
    i.cur_value()->write(s);
    if(strlen(outp)>0)
      strcat(outp, "&");
    strcat(outp, s);
  }
}

/**************************************************************************
 * 
 * read an attribute list table (produced by write).  one of the attributes
 * is specified as the key for this table (i.e. lists will be inserted into
 * the table based on the key).  this key must:
 * 
 *     - correspond to an attribute whose value is an integer (the table
 *           is based on integer keys)
 *     - correspond to an attribute present in every list read in (otherwise
 *           the list can't be added to the table)
 *
 **************************************************************************/

AttrListTable* AttrListTable::read(const char* inp, const char* key_attr, int size) {
  char one[1000], val[80];
  const char *s = inp; char *t = one;
  AttrListTable* a = new AttrListTable(size);
  while (*s != 0) {
    if(*s == '\\') 
      { *t++ = *s++; *t++ = *s++; }
    else if (*s == '&') { 
      *t= 0; s++; 
      AttributeList* alst = AttributeList::read(one);
      if( alst->find_attribute( key_attr, val))
	a->insert( atoi(val), alst );
      t = one;
    } else
      *t++ = *s++;
  }
  *t = 0;
  AttributeList* alst2 = AttributeList::read(one);
  if( alst2->find_attribute( key_attr, val))
    a->insert( atoi(val), alst2 );
  return a;
}


/**************************************************************************
 * 
 * table of attribute list tables
 *
 **************************************************************************/

implementTable(UserListTbl, int, AttrListTable*)
