/* morph3d.h  --  header file for image morphology program morph */

/* morph3d version 4.0    1 June 1993              */
/* 3D image morphology program                     */ 
/*                                                 */ 
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */

/* For a nice tutorial on image morphology, see                             */
/* Haralick, R.M, S.R. Sternberg, and X. Zhuang, "Image analysis using      */
/* mathematical morphology," IEEE Trans. PAMI, vol. 9, No. 4, July 1987,    */
/* pp. 532-550.                                                             */
/* or                                                                       */
/* Maragos, P. and R. Shaffer,                                              */
/* "Morphological systems for multidimensional Signal Processing",          */ /* Proc. IEEE, vol. 78, no. 4, April 1990.                                  */

/* For additional info on the hit-or-miss transform, see J. Serra, Image    */
/* Analysis and Mathematical Morphology, Academic Press, London, 1982.      */

/* usage:

morph3d <FileList [>LogFile] -m e|d|o|c|r|a|b|l|m|n 
       [-i g|b] [-s g|b] [-o s|f] [-l kkk lll] 
       [-t nnn [mmm]] [-r med|nnn] [-z] [-f slope inter]
       [-n mnemonic] -k SEFile | 3x3x3 | 5x5x5 | plus | 
       [cone | cylinder | sphere x y z [v]]

where: FileList is an ascii file containing the filenames of the individual
    frames of the image sequence. Each frame must be a sun rasterfile
    of identical format.
    (Type <FileList so the file is read in through stdin.)
    If the evironment variable IMGPATHI is set, its value is prepended to
    each filename in FileList, to create a complete input path name.
    Morph3d creates output file names from input file names as follows:

        input file name - suffix + mnemonic + index + suffix

    So, if the 54th input file name was BrainSlice.0236.ras and a
    mnemonic (see below) of .cl555 was specified, then the corresponding
    output file name is BrainSlice.0236.cl555.054.ras.
    [The length of the index field is determined by the constant IXLGTH
    in morph3d.h.]
    If the evironment variable IMGPATHO is set, its value is prepended to
    each output filename, to create a complete output path name.

    LogFile is the pathname of a file for a log the program's activity.
    If this is ommitted, the log is displayed in the csh window.
    (Type >LogFile so the file is output through stdout.)

    The letter following -m indicates the morphological operation:
        e - erode
        d - dilate
        o - open
        c - close
        r - rank filter
        a - minmax
        b - maxmin
        l - general LUM (lower-upper-middle) filter
        m - LUM smoothing filter
        n - LUM sharpening filter
        p - I && !E (pixelwise)  where I = original image; E = eroded image;
        q - D && !I (pixelwise)        D = dilated image;
    one of these letters must be specified; there is no default.


    l is the lower-upper-middle filter as defined by Hardie, R. C., and 
    C. G. Boncelet, "LUM filters: A class of rank-order-based filters for 
    smoothing and sharpening," IEEE Trans. Signal Processing, vol. SP-41, 
    No. 3, March 1993.    
    m is the LUM smoothing filter defined therein, and n is the LUM 
    sharpening filter.                                    
    These filters compare the center pixel in a neighborhood defined by 
    an SE to upper and lower order statistics in the neighborhood.
    Depending on the ordering either the center pixel or one of the order 
    stats is output.                                       
    Note that we define order statistics opposite Hardie and Boncelet
    Whereas OS(1) is the minimum for them OS(1) is the maximum in     
    these routines.   

    p with a 3D binary hit-or-miss structuring element will delete in a binary 
    image sequence, white features with the 3D shape of the "hit" portion of 
    the SE. (e.g. one can easily devise a SE to delete isolated pixels).
    p with a gray-level SE will delete the "interiors" from 3D sets of white
    pixels in a binary image sequence.
    q with a 3D binary hit-or-miss structuring element will delete in a binary 
    image sequence, black features with the 3D shape of the "hit" portion of 
    the SE. (e.g. one can easily devise a SE to delete isolated pixels).
    q with a 3D gray-level SE will delete the "interiors" from 3D sets of black
    pixels in a binary image sequence.

    With two exceptions, each of these operations occurs over a 3D data set 
    within a 3D neighborhood (nbhd) defined by the 3D shape of the structuring
    element (SE). The definitions are exactly the same as for their 2D 
    counterparts.  (Mathematical morphology is defined in n dimesions.)
    The exceptions are maxmin, 'b', and minmax, 'a', which have no counterparts 
    in 2D. These operators were designed for use with time sequences. Consider 
    the 3D SE to be a stack of z 2D SE's. The 2D SE's trace nbhds in adjacent 
    sequential images. Maxmin finds the min in each 2D nbhd and takes the 
    maximum of these. Minmax takes the smallest maximum from among the 2D 
    nbhds. 
    The idea behind minmax is this:
    Applied to a time sequence, a 3D structuring element demarcates
    an area (a nbhd) in successive images. (The nbhd in one 2D image
    may or may not be the same the nbhds in succesive 2D images.)
    If a bright, moving particle stays within the successive nbhds
    the result of the minmax will be a bright pixel. If, on the other
    hand, the particle moves out of one of the nbhds, the result will
    be a darker pixel. Thus, when used with minmax, an appropriately 
    shaped SE can act as a velocity filter. Maxmin is the same with
    with respect to dark particles.

    Switch -i indicates that the next letter tells the image sequence type:
    either g for a gray-level image sequence or b for a binary image seq. 
    If -i is not included, the default is gray-level.

    Switch -s  indicates that the next letter tells the structuring 
    element type: either b for a binary SE or g for a gray-level SE. 
    If -s is not included, the default is binary.

    The letter following -o, either s or f, indicates that the
    operation is either a set operation or a function operation.
    (See reference.) If -sf is not included, the default is set op.

    -t nnn [mmm] indicates that a threshold of value nnn from below (and 
    mmm from above; if unspecified mmm == 255) will be used on the
    input if the following 2 criteria are true: 
    the input is a gray-level image AND the operation is a set operation. 
    If the two criteria are true and -t nnn [mmm] is not included, 
    the operation is treated as a function and set processing (FSP) 
    operation (See ref.). If the criteria are not true and -t nnn is 
    specified anyway, it is ignored.

    Switch -r is meaningful only if -m r is specified. Then the
    field following -r indicates the order of the 3D order statistic 
    filter. If the letters "med" are in the field, a median filter is 
    used. If the field contains a number, nnn, then that value is used. 
    If -m r is specified and -r med|nnn is not, the rank filter defaults 
    to a 3D median filter.

    Switch -l is meaningful only if -m l, -m m, or -m n is specified.
    Then kkk and lll are integers that correspond to the the values in 
    Hardie, R. C., and C. G. Boncelet, 
    "LUM filters: A class of rank-order-based filters for smoothing and 
    sharpening," IEEE Trans. Signal Processing, vol. SP-41, No. 3, 
    March 1993. 
    .
    If -m l is specified, the general LUM filter is selected, then 
    integers kkk and lll in the statement -l kkk lll must satisfy
    1 <= kkk <= lll <= med == int(SESupport / 2) + 1, where SESupport is
    the number of active pixels in the structuring element.
    .
    If -m m is specified, the LUM smoothing filter is selected, then 
    integer kkk in the statement -l kkk lll must satisfy
    1 <= kkk <= med == int(SESupport / 2) + 1. The value of lll is ignored.
    .
    If -m n is specified, the LUM sharpening filter is selected, then 
    integer lll in the statement -l kkk lll must satisfy
    1 <= lll <= med == int(SESupport / 2) + 1. The value of kkk is ignored.
    .
    To have morph tell you the value of int(SESupport / 2) + 1,
    execute the program with -l 0 0 as well as the -k SE spec.

    The presence of switch -z tells the program NOT to zero-pad the
    boundary of the image. With this option, the output image has a
    border of zeros inside it the width and height of the SE. That is,
    the transformed area of the output image is smaller than the actual
    image dimensions. This is, in a sense, a more accurate result than
    the zero padded default. To zero pad the input permits the program
    to transform the border region, but it does this on the assumption
    that the original scene was black outside the image. This, of course,
    is almost nver true. Thus, the border region is inaccurately 
    transformed. Use this switch if accuracy is more important than having
    an image that is "colored in" out to the boundary.
    Morph3d _always_ zero pads in the z-dimension. That is it always catenates 
    enough zero images to the beginning and and of the sequence so that the 
    number of images output equals the number of images input.


    Switch -f tells the program to scale the output of the operation
    by multiplying each output pixel by slope and adding inter.
    Instead of the words "slope" and "inter", the user supplies two
    floating point numbers (or integers). 

    The string following -n is a filename mnemonic. By string, I mean the 
    contiguous non-whitespace characters following -n. For example:
    -n .op3358 tells morph3d to use .op3358 as a filename mnemonic.
    See above for the formula morph3d uses to create output file names.

    Switch -k indicates that the next field is one of 7 things:

    (1) SEFile -- This is the filename of a SE file. If the environment 
    variable SEPATH is defined, the value of it is prepended to the filename
    prior to opening the file. The exact structure of a SE file is given
    below.

    (2) 3x3x3  -- specifies the structuring element to be a 3 by 3 by 3
    cube of voxels with origin at the center. 

    (3) 5x5x5  -- Extend each face of the 3x3x3 cube outward by one layer
    of voxels and you get this SE. I looks like a fat, short-armed
    3D "+".

    (4) plus   -- This SE has a voxel at its origin and another attached
    to each face of the origin voxel for as total of 7 voxel elements.
    It looks like a "+" in orthographic projection from any x, y or
    z, direction.

    (5) cone x y z [v]: morph3d makes an elliptical cone SE as follows: The 
    origin is a single voxel at the center of the SE. The axis of the cone 
    is parallel the z-axis (it is perpendicular to the image/voxel planes). 
    Each end of the cone is an x by y elliptical disk (circular disk, if 
    x = y). x, y, and z must be odd and greater than or equal to 1. Each end 
    of the cone is (z-1)/2 image planes away from the plane of the origin. 
    For example, if z = 3, the ends of the cone are in the planes adjacent 
    the origin plane.  If z=5 there is one image plane between the end plane 
    and the origin plane at each end. The slope of the cone in the xz plane 
    is x/z and in the yz plane the slope is y/z. SE planes between the origin 
    and the ends are ellipical disks with semimajor and semiminor axes 
    determined by the slopes. The cone is actually an hourglass in shape. 
    This shape can track pixels in a time sequence that have a maximum 
    velocity in (x,y) given by (x/z,y/z).  If a graylevel value, v > 0, is 
    requested, the voxels along the cone axis are given value v. The 
    graylevels assigned to other voxels decrease as the square root of the 
    radius to zero at the edges of the cone.

    (6) cylinder x y z [v]: morph3d makes an elliptical cylinder as follows: 
    First, it makes an elliptical disk with a diameter of x pixels horizontally 
    and y pixels vertically.  x and y must be odd and greater than or equal 
    to 1. v is the gray level of the center voxel. If v > 0 is requested the 
    gray level of the voxels decrease as the square root of the radius to zero 
    at the edge of the disk. Then, the disk is copied z times. The origin of 
    the SE is placed at its very center (center of the disk at z/2). A 
    cylindrical SE can track shapes in a time sequence larger than its own 
    cross-section that are not moving over z frames. A cylinder of diameter 1 
    can track stationary pixels.

    (7) sphere x y z [v]: This is a spheroid of dimensions x by y by z It is 
    made by GetSE in a fashion analogous to the construction of a single 
    identical to one slice of the cylinder. If v is > 0, v is the value of the 
    center pixel and the gray-values decrease as the square root of the radius 
    to zero at the edges. The graylevel sphere will approximate densities in a 
    3D voxel image.

    Note: when one of 3x3x3, 5x5x5, plus, cone, cylinder, or sphere is
    chosen, switches -s and -o are ignored. Morph3d automatically sets
    flag -s to -s g. Flag -o is set to -o s if 3x3x3, 5x5x5, or plus is 
    chosen or if v is 0 or unspecified with cone, cylinder, or sphere. 
    If cone, cylinder, or sphere is chosen with v > 0, then flag -o is 
    set to -o f.

    A structuring element file is an ASCII file of integers separated by 
    spaces. The first three numbers, x, y, z are the  horizontal, vertical, 
    and depth dimensions in pixels of the smallest 3D box that will cover the 
    structuring element.  Numbers x, y and z must be > 0.  The next three 
    numbers, i, j, k are the horizontal, vertical, and depth coordinates, 
    respectively, of the SE origin.  
    IMPORTANT: The origin is expected to be in the covering box.
    If not, morph3d aborts.
    The front upper left hand corner of the box has coordinates 
    (0,0,0); the back lower right is (x-1,y-1z-1). Following the 
    first six integers are x*y*z integers separated by spaces. 
    These numbers are the SE elements. Their interpretation depends 
    on the morphological operation being performed.

    Negative SE elements are ALWAYS treated as logical DON'T CAREs. 
    That is, when the operation is in progress, image pixels under 
    negative SE elements are ignored. Thus, the support of the SE 
    is limited to those elments that are nonnegative. This permits 
    the creation of odd-shaped and multiply connected SE's or the
    placement of the SE origin outside the body of the SE.
    If -i b is specified, (i.e. pixels grouped as zero and not 
    zero), and if -s b is selected, then the SE is used to perform 
    a hit-or-miss transform. In this case, zero SE elements cover 
    the "miss" support and positive (nonzero) elements cover the 
    "hit" support. The actual gray-levels are ignored.

    If -i b is given, and the SE is gray (-s g) then the nonnegative 
    (both zero and greater than zero) SE elements determine the 
    support of a "hit-only" transform.  That is, the nonnegative 
    suport is used as a standard set-type SE for set (binary) 
    morphology. (Of course, the other gray-level info is ignored.)
    Note: if the image is binary, then a set operation is performed
    by default (SorF is ignored).

    The interpretation of the SE elements for -i g depends
    on the -s and -o flags:

    -i g  -s b  -o s  Function-set morphology on support of strictly
        greater than zero SE elements.

    -i g  -s b  -o f  Same as above.

    -i g  -s g  -o s  Function-set morphology on support of nonnegative
        (greater than or equal to zero) SE elements.

    -i g  -s g  -o f  Function-function morphology on support of 
        nonnegative SE elements.

*/


/* includes */

#include <ctype.h>
#include <math.h>
#include <malloc.h>            
#include <stdio.h>    
#include <stdlib.h>
#include <strings.h>


/* defs */

#define SFMASK  0x0400     /* set / function mask */
#define SET     0x0000     /* set operation */
#define FUNCT   0x0400     /* function operation */

#define IMMASK  0x0200     /* image type bit of MorphOp */
#define GRAIMG  0x0000     /* gray-level image flag */
#define BINIMG  0x0200     /* binary image flag */

#define SEMASK  0x0100     /* structuring element (SE) type bit */
#define GRASE   0x0000     /* gray-level SE */
#define BINSE   0x0100     /* binary SE */

#define OPMASK  0x007F     /* morph operation type bits of MorphOp */
#define ERODE   0x0001     /* erode flag  */
#define DILATE  0x0002     /* dilate flag */
#define OPEN    0x0004     /* open flag   */
#define CLOSE   0x0008     /* close flag  */
#define RANK    0x0010     /* rank filter flag */
#define LUM     0x0011     /* Lower-Upper-Middle filter */
#define LUMSMO  0x0012     /* LUM smoothing filter */
#define LUMSHA  0x0014     /* LUM sharpening filter */
#define MAXMIN  0x0020     /* maxmin flag */
#define MINMAX  0x0040     /* minmax flag */
#define NOTFLG  0x0080     /* "not" flag for binary erodes and dilates */

#define SEPATH  "SEPATH"   /* structuring element directory env. var */
#define INPATH  "IMGPATHI" /* input image directory environment var. */
#define OUTPATH "IMGPATHO" /* output image directory environment var. */
#define FNLGTH  128        /* maximum filename length */
#define IXLGTH  3          /* file index length (must be strictly < 10) */
#define SXLGTH  8          /* maximum number of chars in file name suffix */
#define FTLGTH  8          /* max length of format string for output proc */

#define PLUS     1         /* use 3 by 3 "+" shaped SE */
#define S3X3X3   2         /* use 3 by 3 square SE */
#define S5X5X5   3         /* use 5 by 5 quasi disk SE */
#define CONE     4         /* make a conical SE */
#define CYLINDER 5         /* make cylindrical SE */
#define SPHERE   6         /* make spherical SE */

#define BYTEPX   1         /* pixels are 1 byte long */
#define WORDPX   2         /* pixels are 2 bytes long */

#define BLACK    0
#define WHITE  255
#define TRUE     1
#define FALSE    0
#define ZERO     0
#define NSIZE  256
#define ERROR   -1
#define NOERR    0

#define MIN(a,b) ((a<b)?(a):(b))
#define MAX(a,b) ((a>b)?(a):(b))

/* typedefs */

#ifndef BYTE
typedef unsigned char byte;
#define BYTE 1
#endif

#ifndef WORD
typedef short int word;
#define WORD 1
#endif

/* global functions */

extern byte **AllocBuf();
extern void   DeAllocBuf();
extern void   BlockMove();
extern void   OptErr();
extern void   ScrollBuffer();
extern int    SetUpOp();
extern void   ThresholdImg();
extern int   *GetSE();

extern void BinBinErode();
extern void BinGrayErode();
extern void GrayBinErode();
extern void GrayGraySetErode();
extern void GrayGrayFctErode();
extern void BinBinDilate();
extern void BinGrayDilate();
extern void GrayBinDilate();
extern void GrayGraySetDilate();
extern void GrayGrayFctDilate();

extern void BinSetMaxMin();
extern void GraySetMaxMin();
extern void GrayFctMaxMin();
extern void BinSetMinMax();
extern void GraySetMinMax();
extern void GrayFctMinMax();

extern int  GetSupport();
extern void BinRankFilt();
extern void GrayRankFilt();


/* global variables */
extern int SEplus[];         /* internal 3x3x3 "+" shaped kernel */
extern int SE3x3x3[];        /* internal 3x3x3 square kernel */
extern int SE5x5x5[];        /* internal 5x5x5 quasi disk kernel */
