/* avg3d -- performs a pixelwise moving combination of an image sequence */
/*          optionally will blur the result in a 4 or 8 connected nbhd.  */
/*                                                                       */
/* operations: average (avg)                                             */
/*             minimum (min)                                             */
/*             maximum (max)                                             */
/*             median  (med)                                             */
/*             rank or order statistic (os) nnn (rank 1 == max)          */
/*             ordinary linear combination (olc) coefficients a1 ... aN  */
/*             rank-ordered linear comb. (rlc) coefficients a1 ... aN    */


/* version 1.0     11 Aug 1992 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */


#include <math.h>
#include <memory.h>		
#include <stdio.h>
#include <strings.h>
#include <values.h>
#include <sys/types.h>
#include "rasterio.h"
#include "avg3d.h"


main( argc, argv ) 
   unsigned int argc;    /* count of arguments parsed by operating system    */
   char *argv[];         /* pointers to arguments parsed by operating system */
   {
   FILE *fptr;
   char filename[80];
   char compname[80];
   char *basename;
   char *outbase;
   char *slash;
   byte **imgptr;
   float *coefflist;
   float *c;
   float f;
   struct rasterfile H; /* rasterfile headers */
   byte *cmap;          /* colormaps */
   byte *B,*I,*O,*T;    /* image pointers */
   byte *u,*v,*w,*z;   	/* image pointers */
   int X,Y,N;           /* image dimensions */
   byte *pix;
   int i,j,k;           /* indices */
   int r,s;
   int order;
   int InIndex;
   int OutIndex;
   int frames;
   int ncoeffs;
   int n = argc;
   int mode;
   int rank;
   int done;
   int first;
   int blur;

   char Index[IXLGTH+2];
   char Suffix[SXLGTH+1];
   char Format[FTLGTH];



   if (argc < 5)
      {
      fprintf(stderr,
         "usage: avg3d -i basename\n\
             -o output_basename  [-r nnn] [-s nnn]  [-n nnn]\n\
             -c avg | min | max | med | os nnn | olc[|rlc] a1 ... aN\n\
             [-b plus | square]\n");
      exit(0);
      }

   basename = (char *)NULL;
   outbase = (char *)NULL;
   frames = -1;
   InIndex = 1;
   blur = 0;
   mode = MAVG;
   rank = 0;
   order = DFLTORD;

   /* get arguments */
   while ( --n )
      {
      if ( argv[n][0] == '-' )          /* then this is a switch */
         {
         if ( argv[n][1] == 'i' )       /* basename follows */
            {
            basename = argv[n+1];
            }
         else if ( argv[n][1] == 'o' )       /* output basename follows */
            {
            outbase = argv[n+1];
            }
         else if ( argv[n][1] == 'r' )       /* order of average follows */
            {
            order = atoi(argv[n+1]);
            }
         else if ( argv[n][1] == 's' )       /* starting index follows */
            {
            InIndex = atoi(argv[n+1]);
            }
         else if ( argv[n][1] == 'n' )       /* number of frames follows */
            {
            frames = atoi(argv[n+1]);
            }
         else if ( argv[n][1] == 'b' )       /* blur mode follows */
            {
            if ( !strcasecmp( argv[n+1], "plus" ) )
               {
               blur = PLUS;
               }
            else if ( !strcasecmp( argv[n+1], "square" ) )
               {
               blur = SQUARE;
               }
            }
         else if ( argv[n][1] == 'c' )       /* combination mode follows */
            {
            if ( !strcasecmp( argv[n+1], "avg" ) )
               {
               mode = MAVG;
               }
            else if ( !strcasecmp( argv[n+1], "min" ) )
               {
               mode = MMIN;
               }
            else if ( !strcasecmp( argv[n+1], "max" ) )
               {
               mode = MMAX;
               }
            else if ( !strcasecmp( argv[n+1], "med" ) )
               {
               mode = MMED;
               }
            else if ( !strcasecmp( argv[n+1], "os" ) )
               {
               mode = MOS;
               rank = atoi( argv[n+2] );
               }
            else if ( !strcasecmp( argv[n+1], "olc" ) )
               {
               mode = MOLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            else if ( !strcasecmp( argv[n+1], "rlc" ) )
               {
               mode = MRLC;
               ncoeffs = 0;
               while ( (n+2+ncoeffs<argc) && (argv[n+2+ncoeffs][0] != '-') ) 
                 ++ncoeffs;
               if ( !(coefflist = 
                     (float *)calloc( 1, ncoeffs*(sizeof(float)) )) )
                  {
                  fprintf(stderr,"error allocating coefficient list\n");
                  exit( 0 );
                  }
               for ( i=0; i<ncoeffs; ++i)
                  {
                  *(coefflist+i) = atof(argv[n+2+i]);
                  /*fprintf(stderr,"coeff[%d]: %f\n",i,*(coefflist+i));*/
                  }
               }
            }
         }
      }

   OutIndex = InIndex;

   fprintf(stderr,"Order of combination = %d\n",order);

   if ( !basename || !outbase )
      {
      fprintf(stderr,"-i and -o must be specified\n");
      exit( 0 );
      }

   if ( ((mode == MOLC) || (mode == MRLC)) && (order != ncoeffs) )
      {
      fprintf(stderr,"Number of coefficients must equal order of combination\n");
      exit( 0 );
      }

   if ( mode == MMED )
      {
      rank = order/2+1;
      }
   else if ( mode == MOS )
      {
      if ( (rank < 0) || (rank > order) )
         {
         fprintf(stderr,"Rank must be between 1 and %d\n",order);
         exit( 0 );
         }
      else if ( rank == 1 )
         {
         mode == MMAX;
         }
      else if ( rank == order )
         {
         mode == MMIN;
         }
      }

      

   if ( !(imgptr = (byte **)calloc( order, sizeof(byte *) )) )
      {
      fprintf(stderr,"error allocating image pointer list\n");
      exit( 0 );
      }

   /* get size of images */

   /* write index field width in format string */
   sprintf( Format, "%%0%1dd", IXLGTH );

   /* create index field for output file name */
   Index[0] = '.';
   sprintf( Index+1, Format, InIndex );

   /* make output file name from basename + index + suffix */
   strcpy( filename, basename);
   strcat( filename, Index );
   strcat( filename, ".ras" );

   /* try to open the file */
   fptr = OpenFile( filename, "", "r" );
   if (  ReadRasterFile( fptr, &H, &cmap, &I, 0, 0, 0, 0, ALLOC )  ) exit( 0 );
   fclose( fptr );


   /* get size info */
   X = H.ras_width;
   Y = H.ras_height;
   N = X*Y;

   /* allocate image buffers */

   *imgptr = I;
   for ( i=1; i<order; ++i)
      {
      if ( !(*(imgptr+i) = (byte *)calloc( N, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating image %d\n", i);
         exit( 0 );
         }
      }
   if ( !(O = (byte *)calloc( N, sizeof(byte)) ) )
      {
      fprintf(stderr,"error allocating output image \n");
      exit( 0 );
      }

   if ( blur )
      {
      if ( !(B = (byte *)calloc( N, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating blurred image \n");
         exit( 0 );
         }
      }

   if ( !( (mode == MAVG)  || (mode == MOLC) ) )
      if ( !(pix = (byte *)calloc( order, sizeof(byte)) ) )
         {
         fprintf(stderr,"error allocating pixel sort array \n");
         exit( 0 );
         }



   /* main loop */
   done = FALSE;
   first = TRUE;
   while ( frames )
      {
      /* get the next image */
      if ( !first )
         {
         T = *imgptr;
         for ( i=0; i<order-1; ++i )
            {
            *(imgptr+i) = *(imgptr+i+1);
            }
         *(imgptr+i) = T;

         /* make the input file name index */
         sprintf( Index+1, Format, InIndex );

         if ( frames - (order>>1) )
            {
            strcpy( filename, basename );
            strcat( filename, Index );
            strcat( filename, ".ras" );

            fptr = OpenFile( filename, "", "r" );
            if (  ReadRasterFile( fptr, &H, &cmap, imgptr+i, 0, 0, 0, 0, 0 )  ) 
               {
               done = TRUE;
               }
            else
               {
               fclose(fptr);
               /* remap the image */
               ExtractLuminance( &H, cmap, *(imgptr+i), 0, 0, 0, 0 );
               }
            }
         else
            {
            done = TRUE;
            }
         if ( done )
            {
            memset( *(imgptr+i), '\000', N );
            }
         ++InIndex;
         }
      else
         {
         for ( i=(order>>1); i<order; ++i)
            {
            /* make the input file name index */
            sprintf( Index+1, Format, InIndex );
            strcpy( filename, basename );
            strcat( filename, Index );
            strcat( filename, ".ras" );
            fptr = OpenFile( filename, "", "r" );
            if (  ReadRasterFile( fptr, &H, &cmap, imgptr+i, 0, 0, 0, 0, 0 )  ) 
               exit( 0 );
            /* remap the image */
            ExtractLuminance( &H, cmap, *(imgptr+i), 0, 0, 0, 0 );
            ++InIndex;
            }
         first = FALSE;
         }

      /* do the various combinations */
      if ( mode == MAVG)
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = 0;
            for ( i=0; i<order; i++ )
               r += *(*(imgptr+i)+j);
            *(z++) = r / order;
            }
         }
       else if ( mode == MOLC )  /* ordinary linear combination */
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            f = 0;
            c = coefflist;
            for ( i=0; i<order; i++ )
               f += *(c++) * (float)(*(*(imgptr+i)+j));
            *(z++) = (byte)(f > WHITE) 
                   ? WHITE : ( (f < BLACK) ? BLACK : f );
            }
         }
      else if ( mode == MMAX )
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = BLACK;
            for ( i=0; i<order; i++ )
               {
               s = *(*(imgptr+i)+j);
               if ( s > r )
                  r = s;
               }
            *(z++) = r;
            }
         }
      else if ( mode == MMIN )
         {
         z=O;
         for ( j=0; j<N; ++j )
            {
            r = WHITE;
            for ( i=0; i<order; i++ )
               {
               s = *(*(imgptr+i)+j);
               if ( s < r )
                  r = s;
               }
            *(z++) = r;
            }
         }
      else if ( (mode == MMED) || (mode == MOS) || (mode == MRLC) )
         {
         z=O;
         for ( j=0; j<N; ++j ) /* process loop */
            {
            w = pix;
            for ( i=0; i<order; i++ )  /* get pixels */
               {
               *(w++) = *(*(imgptr+i)+j);
               }

            for ( i=1; i<order; i++ )  /* sort loop */
               {
               v = pix+i;
               u = v-1;
               if ( *u > *v )  /* order loop */
                  {
                   r = *u;
                  *u = *v;
                  *v =  r;
                  j=i-1;
                  while ( j )  /* bubble loop */
                     {
                     v = pix+j;
                     u = v-1;
                     if ( *u <= *v ) /* are ok */
                        {
                        break;       /* go to next order */
                        }
                     else            /* swap them and */
                        {
                         r = *u;
                        *u = *v;
                        *v =  r;
                        }
                     --j;            /* check next bubble */
                     }  /* bubble loop */
                  }  /* order loop */
               }  /* sort loop */

            if ( mode != MRLC) /* (mode == MMED) || (mode == MOS) */
               {
               *(z++) = *(pix + order - rank);
               }
            else /* mode == MRLC*/
               {
               f = 0;
               u = pix;
               c = coefflist + ncoeffs;
               for ( i=0; i<order; i++ )
                  {
                  f += *(--c) * (float)*(u++);
                  }
               *(z++) = (byte)(f > WHITE) 
                      ? WHITE : ( (f < BLACK) ? BLACK : f );
               }

            }  /* process loop */
         }  /* mode == MMED, MOS, or MRLC*/

      /* if blurring is called for, do it */
      if ( blur )
         {
         v = O;
         z = B;
         if ( blur == PLUS )
            {
            /* do the corners */
            *z = (*v + *(v+1) + *(v+X)) / 3;
            *(z+X-1) = (*(v+X-1) + *(v+X-2) + *(v+2*X-1)) / 3;
            *(z+(Y-1)*X) = (*(v+(Y-1)*X) + *(v+(Y-1)*X+1) + *(v+(Y-2)*X)) / 3;
            *(z+Y*X-1) = (*(v+Y*X-1) + *(v+Y*X-2) + *(v+(Y-1)*X-1)) / 3;

            /* do the first row */
            w = O+X;
            for ( i=1; i<X-1; ++i )
               {
               ++v;
               *(++z) = (*(v-1) + *v + *(v+1) + *(++w)) / 4;
               }

            /* do the interior rows */
            u = O;
            v = O + X;
            w = O + 2*X;
            z = B + X;
            for ( j=1; j<Y-1; ++j )
               {
               *(z++) = (*(u++) + *v + *(v+1) + *(w++)) / 4;
               ++v;
               for ( i=1; i<X-1; ++i )
                  {
                  *(z++) = (*(u++) + *(v-1) + *v + *(v+1) + *(w++)) / 5;
                  ++v;
                  }
               *(z++) = (*(u++) + *(v-1) + *v + *(w++)) / 4;
               ++v;
               }
 
            /* do the last row */
            ++u;
            ++v;
            ++z;
            for ( i=1; i<X-1; ++i )
               {
               *(++z) = (*(u++) +*(v-1) + *v + *(v+1)) / 4;
               ++v;
               }
            }
         else if ( blur == SQUARE )
            {
            /* do the corners */
            *z = (*v + *(v+1) + *(v+X) + *(v+X+1)) / 4;
            *(z+X-1) = (*(v+X-1) + *(v+X-2) + *(v+2*X-1) + *(v+2*X-2)) / 4;
            *(z+(Y-1)*X) = (*(v+(Y-1)*X) + *(v+(Y-1)*X+1) + *(v+(Y-2)*X) + *(v+(Y-2)*X+1)) / 4;
            *(z+Y*X-1) = (*(v+Y*X-1) + *(v+Y*X-2) + *(v+(Y-1)*X-1) + *(v+(Y-1)*X-2)) / 4;

            /* do the first row */
            w = O+X;
            for ( i=1; i<X-1; ++i )
               {
               ++v;
               ++w;
               *(++z) = (*(v-1) + *v + *(v+1) + *(w-1) + *w + *(w+1)) / 6;
               }

            /* do the interior rows */
            u = O;
            v = O + X;
            w = O + 2*X;
            z = B + X;
            for ( j=1; j<Y-1; ++j )
               {
               *(z++) = (*u + *(u+1) + *v + *(v+1) + *w + *(w+1)) / 6;
               ++u;
               ++v;
               ++w;
               for ( i=1; i<X-1; ++i )
                  {
                  *(z++) = (*(u-1) + *u + *(u+1) + *(v-1) + *v + *(v+1) + *(w-1) + *w + *(w+1)) / 9;
                  ++u;
                  ++v;
                  ++w;
                  }
               *(z++) = (*(u-1) + *u + *(v-1) + *v + *(w-1) + *w) / 6;
               ++u;
               ++v;
               ++w;
               }

            /* do the last row */
            ++z;
            ++v;
            for ( i=1; i<X-1; ++i )
               {
               *(++z) = (*(u-1) + *u + *(u+1) + *(v-1) + *v + *(v+1)) / 6;
               ++u;
               ++v;
               }
            }
         O = B;
         }

      /* make the output file */

      /* make the file name index */
      sprintf( Index+1, Format, OutIndex );
      strcpy( filename, outbase );
      strcat( filename, Index );
      strcat( filename, ".ras" );
      fptr = OpenFile( filename, "", "w" );
      WriteRasterFile( fptr, &H, cmap, O, 0, 0, 0, 0 ); 
      fclose( fptr );
      fprintf( stderr, "%s\n", filename );

      ++OutIndex;
      --frames;
      }
               
   }

