/* rasinfo.c  --  program to dispaly information about a sun rasterfile. */

/* includes: main */

/* depends on: ReadRasterFile, WriteRasterFile, ExtractLuminance */

/* version 2.0     14 Jan 1993 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */


#include "stdio.h"
#include "malloc.h"
#include "rasterio.h"




main( argc, argv )
   unsigned int argc;
   byte *argv[];
   {
   struct rasterfile RasHd;	/* raster file header */
   FILE *fp;
   byte *Img;
   byte *CMap;
   byte *r,*g,*b;
   int gray;
   int lgth;
   int ncol;
   int min, max;
   int i;

   /* check args */

   if ( argc < 2  )
      {
      fprintf(stderr, "usage:  rasinfo raster_file_name\n");
      exit(0);
      }

   /* open the input raster file */
   fp = OpenFile( argv[1], NULL, "r" );

   /* get the header */
   if ( fread( &RasHd, sizeof(int), 8, fp ) != 8 )
      {
      fprintf(stderr,"Error reading image header\n");
      exit(0);
      }
 
   /* make sure it is a rasterfile */
   if ( RasHd.ras_magic != RAS_MAGIC )
      {
      fprintf(stderr,"This is not a sun rasterfile.\n");
      exit(0);;
      }

   /* print the header info */
   fprintf( stderr, "%s:   width %d   height %d   bits/pixel %d   length %d\n", 
      argv[1], RasHd.ras_width, RasHd.ras_height, RasHd.ras_depth, 
      RasHd.ras_length );

   switch( RasHd.ras_type )
      {
      case RT_OLD :
         fprintf( stderr, "type:  RT_OLD, Raw pixrect image in 68000 byte order\n" );
         break;
      case RT_STANDARD :
         fprintf( stderr, "type:  RT_STANDARD, Raw pixrect image in 68000 byte order\n" );
         break;
      case RT_BYTE_ENCODED :
         fprintf( stderr, "type:  RT_BYTE_ENCODED, Run-length compression of bytes\n" );
         break;
      case RT_FORMAT_RGB :
         fprintf( stderr, "type:  RT_FORMAT_RGB, XRGB (32-bit) or RGB (24-bit)\n" );
         break;
      case RT_FORMAT_TIFF :
         fprintf( stderr, "type:  RT_FORMAT_TIFF, tiff format rasterfile\n" );
         break;
      case RT_FORMAT_IFF :
         fprintf( stderr, "type:  RT_FORMAT_IFF, TAAC format XBGR (32-bit) or BGR (24-bit)\n" );
         break;
      case RT_EXPERIMENTAL :
         fprintf( stderr, "type:  RT_EXPERIMENTAL, Reserved for testing\n" );
         break;
      }

   /* Get the colormap if there is one */
   if ( GetRasColorMap( fp, &RasHd, &CMap, ALLOC ) )
      {
      fprintf(stderr,"Error reading colormap\n");
      exit(0);
      }

   gray = FALSE;

   /* if there is one, analyze the colormap. */
   if ( RasHd.ras_maplength )
      {
      ncol = RasHd.ras_maplength / 3;
      fprintf( stderr, "colormap length %d   # of colors in map %d   ", RasHd.ras_maplength, ncol);
      r = CMap;
      g = r + ncol;
      b = g + ncol;
      i=ncol;
      gray = TRUE;
      while ( i && gray )
         {
         gray = ( (*r == *g) && (*g == *b) );
         ++r;
         ++g;
         ++b;
         --i;
         }
      if ( gray )
         {
         fprintf( stderr, "grayscale\n");
         }
      else
         {
         fprintf( stderr, "color\n");
         }
      }
   else
      {
      fprintf( stderr, "No colormap  ");
      if ( RasHd.ras_depth == 1 )
         {
         fprintf( stderr, "binary\n");
         }
      else if ( RasHd.ras_depth == 8 )
         {
         fprintf( stderr, "grayscale\n");
         gray = TRUE;
         }
      else if ( RasHd.ras_depth == 24 )
         {
         fprintf( stderr, "true color\n");
         }
      else if ( RasHd.ras_depth == 32 )
         {
         fprintf( stderr, "true color with alpha\n");
         }
      }


   lgth  = RasHd.ras_width * RasHd.ras_depth;
   lgth  = (lgth % 8) ? (lgth / 8) + 1 : (lgth / 8);
   lgth *= RasHd.ras_height;

   if ( lgth != RasHd.ras_length )
      {
      if ( RasHd.ras_length == (RasHd.ras_width+1) * RasHd.ras_height )
         {
         fprintf( stderr, "note: the image data structure is padded for an even # of bytes per line\n");
         }
      else
         {
         fprintf( stderr, "Warning: actual image length is %d\n", lgth );
         }
      }


   /* only do this image analysis on a grayscale image of type RT_STANDARD */
   if ( gray  &&  (RasHd.ras_type == RT_STANDARD) )
      {
      if ( GetRasImage( fp, &RasHd, &Img, 0, 0, 0, 0, ALLOC ) )
         {
         fprintf(stderr,"Error reading image\n");
         exit(0);
         }

      /* close the input file */
      fclose( fp );

      /* remap the image data */
      ExtractLuminance( &RasHd, CMap, Img, 0, 0, 0, 0 );

      /* find the extrema */
      min = 255;
      max = 0;
      for ( i=0; i<lgth; ++i )
         {
         min = min < *(Img+i) ? min : *(Img+i);
         max = max > *(Img+i) ? max : *(Img+i);
         }

      fprintf( stderr, "min %d  max %d\n", min, max );
      }
   else
      {
      /* close the input file */
      fclose( fp );
      }

   free(Img);
   if ( CMap ) free(CMap);
   }




 
