/* xmesa.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: xmesa.c,v 1.39 1995/06/22 15:50:12 brianp Exp $

$Log: xmesa.c,v $
 * Revision 1.39  1995/06/22  15:50:12  brianp
 * more X SHM extension error checking!
 *
 * Revision 1.38  1995/06/22  14:41:49  brianp
 * initiaiize mindist to 0.0 instead of 0, just to be explicit
 *
 * Revision 1.37  1995/06/12  15:43:20  brianp
 * changed color arrays to GLubyte
 *
 * Revision 1.36  1995/06/07  16:03:14  brianp
 * added free(allcolors) to setup_pseudocolor()
 *
 * Revision 1.35  1995/06/07  14:46:47  brianp
 * better dist<mindist test in setup_pseudocolor()
 *
 * Revision 1.34  1995/05/31  19:33:50  brianp
 * removed references to MAX_VERTICES
 *
 * Revision 1.33  1995/05/31  14:59:40  brianp
 * cleaner shared memory support
 *
 * Revision 1.32  1995/05/26  14:39:48  brianp
 * incorporated shared memory changes per Frederic Devernay
 *
 * Revision 1.31  1995/05/25  20:41:11  brianp
 * more default color, clearcolor bug fixes
 * removed alloc_back_buffer() calls from XMesaBind*() functions
 *
 * Revision 1.30  1995/05/24  18:16:28  brianp
 * improved initialization of default drawing color and background color
 * preallocate gray needed for GLUT menus in 8-bit RGB mode
 *
 * Revision 1.29  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.28  1995/05/19  15:48:42  brianp
 * fixed dd_clear() size problem for monochrome visual
 *
 * Revision 1.27  1995/05/18  14:47:53  brianp
 * fixed dangling backpixmap references in alloc_back_buffer()
 *
 * Revision 1.26  1995/05/15  16:04:58  brianp
 * changed color scaling to 255.0 for monochrome mode
 * suppress GraphicsExpose/NoExpose events from swapbuffers() per Michael Pichler
 *
 * Revision 1.25  1995/04/21  15:34:23  brianp
 * attempted to implement byte swapping logic
 *
 * Revision 1.24  1995/04/20  17:32:08  brianp
 * fixed small bug in 8-bit PseudoColor color matching code
 *
 * Revision 1.23  1995/04/19  13:35:39  brianp
 * added XMesaGetBackBuffer() function
 * fixed XMesaBindWindow() and XMesaBindPixmap() per Thorsten Ohl
 *
 * Revision 1.22  1995/04/18  15:49:03  brianp
 * fixed distance2 macro to prevent warnings on Suns
 *
 * Revision 1.21  1995/04/17  14:39:08  brianp
 * added Pixmap rendering support
 * API changes:  XMesaCreateContext(), XMesaBindPixmap(), XMesaBindWindow()
 *
 * Revision 1.20  1995/04/12  15:37:44  brianp
 * cleaned up 8-bit dithering
 * implemented pointers to point, line and polygon rendering functions
 *
 * Revision 1.19  1995/04/11  20:52:13  brianp
 * incorporated Bob Mercier's 8-bit dithering
 *
 * Revision 1.18  1995/04/11  18:12:02  brianp
 * added missing comma to XMesaCreatePixmapContext declaration
 *
 * Revision 1.17  1995/04/11  14:06:59  brianp
 * introduced DD function pointer struct
 * removed hardcoded colormap sizes
 * fixed a few small bugs
 *
 * Revision 1.16  1995/03/31  17:48:21  brianp
 * fixed a new bug in dd_draw_line
 *
 * Revision 1.15  1995/03/31  17:02:00  brianp
 * fixed dd_clear bugs
 *
 * Revision 1.14  1995/03/30  21:10:34  brianp
 * rewritten to use pointers to span and pixel output functions
 *
 * Revision 1.13  1995/03/24  19:23:47  brianp
 * new functions for TrueColor, PseudoColor, monochrome setup
 * new XAllocColor logic for 8-bit PseudoColor
 *
 * Revision 1.12  1995/03/13  20:56:41  brianp
 * fixed color scaling bug in 8-bit RGB mode
 *
 * Revision 1.11  1995/03/08  15:25:02  brianp
 * replaced gc1 w/ cleargc in XMesaSwapBuffers()
 *
 * Revision 1.10  1995/03/08  15:10:02  brianp
 * added support for dd_logicop
 * added dd_clear_index and dd_clear_color
 *
 * Revision 1.9  1995/03/07  19:01:39  brianp
 * added dd_read_index_pixels() and dd_read_color_pixels()
 *
 * Revision 1.8  1995/03/07  14:21:20  brianp
 * updated for new XSetForeground/GC scheme
 *
 * Revision 1.7  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.6  1995/03/04  19:17:46  brianp
 * new 8-bit PseudoColor colormap logic
 *
 * Revision 1.5  1995/03/02  19:32:07  brianp
 * added defines for FRONT_PIXMAP, BACK_PIXMAP, and BACK_XIMAGE
 *
 * Revision 1.4  1995/02/28  21:32:21  brianp
 * fixed comments and typos
 *
 * Revision 1.3  1995/02/28  21:23:39  brianp
 * moved xmesa_context struct to xmesaP.h to share with glx.c
 *
 * Revision 1.2  1995/02/27  22:49:13  brianp
 * modified for PB
 *
 * Revision 1.1  1995/02/24  14:28:31  brianp
 * Initial revision
 *
 */


/*
 * Mesa/X11 interface.
 */


/*
 * NOTES:
 *
 * The window coordinate system origin (0,0) is in the lower-left corner
 * of the window.  X11's window coordinate origin is in the upper-left
 * corner of the window.  Therefore, most drawing functions in this
 * file have to flip Y coordinates.
 *
 * Define SHM in the Makefile with -DSHM if you want to compile in support
 * for the MIT Shared Memory extension.  If enabled, when you use an Ximage
 * for the back buffer in double buffered mode, the "swap" operation will
 * be faster.  You must also link with -lXext.
 *
 * As of revision 1.21 the X/Mesa interface was reorganized a bit to
 * separate the "bind to window" operation from CreateContext.  This allows
 * one X/Mesa context to be used with multiple windows or off-screen pixmaps.
 * Two new functions were introduced: XMesaBindWindow() and XMesaBindPixmap()
 * which bind an X/Mesa context to either a window or pixmap.  Also, the
 * XMesaCreateContext was changed: instead of passing in a window argument
 * you must pass an XVisualInfo argument.
 */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef SHM
#  include <sys/ipc.h>
#  include <sys/shm.h>
#  include <X11/extensions/XShm.h>
#endif
#include "GL/xmesa.h"
#include "xmesaP.h"
#include "bresenhm.h"
#include "context.h"
#include "dd.h"
#include "quikpoly.h"
#include "xform.h"



/* Values for Current->dest: */
#define FRONT_PIXMAP	1
#define BACK_PIXMAP	2
#define BACK_XIMAGE	4


/* Values for Current->pixelformat: */
#define PF_INDEX	1
#define PF_COLOR	2
#define PF_8A8B8G8R	3
#define PF_8BIT		4
#define PF_1BIT		5


/*
 * If pixelformat==PF_COLOR:
 */
#define PACK_RGB( R, G, B )		\
	(   ((R) << Current->rshift)	\
	  | ((G) << Current->gshift)	\
	  | ((B) << Current->bshift) )
#define PACK_RGBA( R, G, B, A )		\
	(   ((R) << Current->rshift)	\
	  | ((G) << Current->gshift)	\
	  | ((B) << Current->bshift)	\
	  | ((A) << Current->ashift) )


/*
 * If pixelformat==PF_1BIT:
 */
static int kernel1[4][4] = {  0*47,  9*47,  4*47, 12*47,
			      6*47,  2*47, 14*47,  8*47,
			     10*47,  1*47,  5*47, 11*47,
			      7*47, 13*47,  3*47, 15*47 };
#define DITHER_1BIT( X, Y, R, G, B )	\
	( ((R)+(G)+(B)) > kernel1[(X)&3][(Y)&3] )


/*
 * If pixelformat==PF_8BIT:
 *
 * Improved 8-bit RGB dithering code contributed by Bob Mercier
 * (mercier@hollywood.cinenet.net).  Thanks Bob!
 */
#define _R	5
#define _G	9
#define _B	5
#define _DX	4
#define _DY	4
#define _D	(_DX*_DY)
#define _MIX(r,g,b)	(((r)*_G+(g))*_B+(b))
/*#define _DITH(C,c,d)	(((unsigned)((_D*(C-1)+1)*c+d))/(_D*256))*/
#define _DITH(C,c,d)	(((unsigned)((_D*(C-1)+1)*c+d)) >> 12)

#define MAXC	256
static int kernel8[_DY][_DX] = {
   { 0 * MAXC,  8 * MAXC,  2 * MAXC, 10 * MAXC},
   {12 * MAXC,  4 * MAXC, 14 * MAXC,  6 * MAXC},
   { 3 * MAXC, 11 * MAXC,  1 * MAXC,  9 * MAXC},
   {15 * MAXC,  7 * MAXC, 13 * MAXC,  5 * MAXC},
};
static int __d;
#define DITHER_8BIT( X, Y, R, G, B )			\
	(__d = kernel8[(Y)&(_DY-1)][(X)&(_DX-1)],	\
	 Current->color_table[_MIX(_DITH(_R, (R), __d),	\
				   _DITH(_G, (G), __d),	\
				   _DITH(_B, (B), __d))])




/*
 * Useful macros:
 */
#define OFFSET1( X, Y )   ((Current->height-(Y)-1) * Current->backimage->bytes_per_line + (X))
#define OFFSET4( X, Y )   ((Current->height-(Y)-1) * Current->width + (X))
#define FLIP(Y)  (Current->height-(Y)-1)

#define SWAP( A, B, T )   { T = A;  A = B;  B = T; }



static XMesaContext Current = NULL;


static void select_write_function( void );




/**********************************************************************/
/*****                      Private Functions                     *****/
/**********************************************************************/


/*
 * X/Mesa Error reporting function:
 */
static void error( const char *msg )
{
   fprintf( stderr, "X/Mesa error: %s\n", msg );
}


/*
 * Return the host's byte order as LSBFirst or MSBFirst ala X.
 */
static int host_byte_order( void )
{
   int i = 1;
   char *cptr = (char *) &i;
   return (*cptr==1) ? LSBFirst : MSBFirst;
}



/*
 * Error handling.
 */
static int mesaXErrorFlag = 0;

static int mesaHandleXError( Display *dpy, XErrorEvent *event )
{
    mesaXErrorFlag = 1;
    return 0;
}


/*
 * Check if the X Shared Memory extension is available.
 * Return:  0 = not available
 *          1 = shared XImage support available
 *          2 = shared Pixmap support available also
 */
static int check_for_xshm( Display *display )
{
#ifdef SHM
   int major, minor, ignore;
   Bool pixmaps;

   if (XQueryExtension( display, "MIT-SHM", &ignore, &ignore, &ignore )) {
      if (XShmQueryVersion( display, &major, &minor, &pixmaps )==True) {
	 return (pixmaps==True) ? 2 : 1;
      }
      else {
	 return 0;
      }
   }
   else {
      return 0;
   }
#else
   /* Can't compile XSHM support */
   return 0;
#endif
}


/*
 * Allocate a shared memory XImage back buffer for the given context.
 * Return:  GL_TRUE if success, GL_FALSE if error
 */
static GLboolean alloc_shm_back_buffer( XMesaContext c )
{
#ifdef SHM
   /*
    * We have to do a _lot_ of error checking here to be sure we can
    * really use the XSHM extension.  It seems different servers trigger
    * errors at different points if the extension won't work.  Therefore
    * we have to be very careful...
    */
   GC gc;
   c->backimage = XShmCreateImage( c->display, c->visual, c->depth,
				   ZPixmap, NULL, &c->shminfo,
				   c->width, c->height );
   if (c->backimage == NULL) {
      error( "alloc_back_buffer: Shared memory error (XShmCreateImage), disabling." );
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.shmid = shmget( IPC_PRIVATE, c->backimage->bytes_per_line
			     * c->backimage->height, IPC_CREAT|0777 );
   if (c->shminfo.shmid < 0) {
      perror("alloc_back_buffer");
      XDestroyImage( c->backimage );
      c->backimage = NULL;
      error( "alloc_back_buffer: Shared memory error (shmget), disabling." );
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.shmaddr = c->backimage->data
                      = (char*)shmat( c->shminfo.shmid, 0, 0 );
   if (c->shminfo.shmaddr == (char *) -1) {
      perror("alloc_back_buffer");
      XDestroyImage( c->backimage );
      c->backimage = NULL;
      error("alloc_back_buffer: Shared memory error (shmat), disabling.");
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.readOnly = False;
   mesaXErrorFlag = 0;
   XSetErrorHandler( mesaHandleXError );
   /* This may trigger the X protocol error we're ready to catch: */
   XShmAttach( c->display, &c->shminfo );
   XSync( c->display, False );

   if (mesaXErrorFlag) {
      /* we are on a remote display, this error is normal, don't print it */
      XFlush( c->display );
      mesaXErrorFlag = 0;
      XDestroyImage( c->backimage );
      shmdt( c->shminfo.shmaddr );
      shmctl( c->shminfo.shmid, IPC_RMID, 0 );
      c->backimage = NULL;
      c->shm = 0;
      return GL_FALSE;
   }

   shmctl( c->shminfo.shmid, IPC_RMID, 0 ); /* nobody else needs it */

   /* Finally, try an XShmPutImage to be really sure the extension works */
   gc = XCreateGC( c->display, c->frontbuffer, 0, NULL );
   XShmPutImage( c->display, c->frontbuffer, gc,
		 c->backimage, 0, 0, 0, 0, 1, 1 /*one pixel*/, False );
   XSync( c->display, False );
   XFreeGC( c->display, gc );
   XSetErrorHandler( NULL );
   if (mesaXErrorFlag) {
      XFlush( c->display );
      mesaXErrorFlag = 0;
      XDestroyImage( c->backimage );
      shmdt( c->shminfo.shmaddr );
      shmctl( c->shminfo.shmid, IPC_RMID, 0 );
      c->backimage = NULL;
      c->shm = 0;
      return GL_FALSE;
   }

   return GL_TRUE;
#else
   /* Can't compile XSHM support */
   return GL_FALSE;
#endif
}



/*
 * Setup an off-screen pixmap or Ximage to use as the back buffer.
 * Input:  c - the X/Mesa context.
 */
static void alloc_back_buffer( XMesaContext c )
{
   if (c->db_state==BACK_XIMAGE) {
      char *img;

      /* Deallocate the old backimage, if any */
      if (c->backimage) {
#ifdef SHM
	 if (c->shm) {
	    XShmDetach( c->display, &c->shminfo );
	    XDestroyImage( c->backimage );
	    shmdt( c->shminfo.shmaddr );
	 }
	 else
#endif
	   XDestroyImage( c->backimage );
	 c->backimage = NULL;
      }

      /* Allocate new back buffer */
      if (c->shm==0 || alloc_shm_back_buffer(c)==GL_FALSE) {
	 /* Allocate a regular XImage for the back buffer. */
	 /* TODO: this number of bytes computation is a hack! */
	 img = (char *) malloc( c->width * c->height * 4 );
	 c->backimage = XCreateImage( c->display, c->visual, c->depth,
				      ZPixmap, 0,   /* format, offset */
				      img, c->width, c->height,
				      8, 0 );  /* pad, bytes_per_line */
	 if (!c->backimage) {
	    /* TODO: more related error checks are needed throughout xmesa.c */
	    error("alloc_back_buffer: XCreateImage failed.");
	 }
      }
      c->backpixmap = None;
   }
   else if (c->db_state==BACK_PIXMAP) {
      Pixmap old_pixmap = c->backpixmap;
      /* Free the old back pixmap */
      if (c->backpixmap) {
	 XFreePixmap( c->display, c->backpixmap );
      }
      /* Allocate new back pixmap */
      c->backpixmap = XCreatePixmap( c->display, c->frontbuffer,
				     c->width, c->height, c->depth );
      c->backimage = NULL;
      /* update other references to backpixmap */
      if (c->drawable==old_pixmap) {
	 c->drawable = c->backpixmap;
      }
      if (c->readable==old_pixmap) {
	 c->readable = c->backpixmap;
      }
   }
}



/*
 * Read a pixel from an X drawable.
 */
static unsigned long read_pixel( Display *dpy, Drawable d, int x, int y )
{
   XImage *pixel;
   unsigned long p;

   pixel = XGetImage( dpy, d, x, y, 1, 1, AllPlanes, ZPixmap );
   if (pixel) {
      p = XGetPixel( pixel, 0, 0 );
      XDestroyImage( pixel );
   }
   else {
      p = 0;
   }

   return p;
}



/*
 * Return the width and height of the given drawable.
 */
static void get_drawable_size( Display *dpy, Drawable d,
			       unsigned int *width, unsigned int *height )
{
   Window root;
   int x, y;
   unsigned int bw, depth;

   XGetGeometry( dpy, d, &root, &x, &y, width, height, &bw, &depth );
}



/*
 * Setup RGB rending for a window with a True/DirectColor visual.
 */
static int setup_truecolor( XMesaContext c )
{
   /* TODO: force scaling factors to 255, replace _mult with _mask */


   unsigned long rmask, gmask, bmask;

   /*
    * Use the red, green, and blue mask values in the visual structure
    * to compute the multiplier and shift values for converting RGB
    * triplets from floats in [0,1] to packed pixel values.
    */

   /* Red */
   c->rshift = 0;
   rmask = c->visual->red_mask;
   while ((rmask & 1)==0) {
      c->rshift++;
      rmask = rmask >> 1;
   }
   c->rmult = (GLint) rmask;

   /* Green */
   c->gshift = 0;
   gmask = c->visual->green_mask;
   while ((gmask & 1)==0) {
      c->gshift++;
      gmask = gmask >> 1;
   }
   c->gmult = (GLint) gmask;

   /* Blue */
   c->bshift = 0;
   bmask = c->visual->blue_mask;
   while ((bmask & 1)==0) {
      c->bshift++;
      bmask = bmask >> 1;
   }
   c->bmult = (GLint) bmask;

   /* Alpha */
   c->ashift = 24;
   c->amult = 255;

   if (host_byte_order() != ImageByteOrder(c->display)) {
      /* must reverse order of bytes in each word */
      /* TODO: need a more elegant solution for other depths */
      if (c->depth==24) {
	 GLint tmp;
/*
	 SWAP( c->rshift, c->ashift, tmp );
	 SWAP( c->gshift, c->bshift, tmp );
	 SWAP( c->rmult, c->amult, tmp );
	 SWAP( c->gmult, c->bmult, tmp );
*/
      }
   }

   c->gl_ctx->RedScale   = (GLfloat) c->rmult;
   c->gl_ctx->GreenScale = (GLfloat) c->gmult;
   c->gl_ctx->BlueScale  = (GLfloat) c->bmult;
   c->gl_ctx->AlphaScale = (GLfloat) c->amult;

   if (c->rshift==0 && c->gshift==8 && c->bshift==16) {
      /* a common case */
      c->pixelformat = PF_8A8B8G8R;
      c->pixel = 0xffffffff;  /* white */
      c->clearpixel = 0x0;    /* black */
   }
   else {
      c->pixelformat = PF_COLOR;
      c->pixel = c->visual->red_mask | c->visual->green_mask
	         | c->visual->blue_mask;  /* white */
      c->clearpixel = 0x0;    /* black */
   }

   return 1;
}




/*
 * Setup RGB rending for a window with a Pseudo/StaticColor visual.
 * This version uses Bob Mercier's dithering.
 */
static int setup_pseudocolor( XMesaContext c, Window window )
{
   int r, g, b, i;
   int colorsfailed = 0;
   XColor xcol;
   XColor *allcolors = NULL, *acptr;
   Colormap cmap;
   XWindowAttributes attr;
   static Colormap PseudoRGBmap = 0;

   /* Need to know the colormap in this case */
   XGetWindowAttributes( c->display, window, &attr );
   cmap = attr.colormap;

   xcol.red = xcol.green = xcol.blue = 0;
   if (!XAllocColor( c->display, cmap, &xcol)) {
      /* This colormap must have been allocated w/ AllocAll (ala GLUT).
       * If so, we can't use it and will now try to make our own.
       */
      /* TODO: Remove this ugliness if/when GLUT is updated to handle */
      /* PseudoColor visuals in RGB mode. */

      if (PseudoRGBmap) {
	 /* share the previously allocated one. */
	 cmap = PseudoRGBmap;
      }
      else {
	 Screen *scr = DefaultScreenOfDisplay(c->display);
	 int scrnum = DefaultScreen(c->display);
	 /* get a new colormap */
	 if (MaxCmapsOfScreen(scr)==1   /* TODO: is this test needed? */
	     && c->visual==DefaultVisual(c->display,scrnum)) {
	    cmap = DefaultColormap(c->display,scrnum);
	 }
	 else {
	    cmap = XCreateColormap( c->display, window, c->visual, AllocNone );
	 }
	 if (!cmap) {
	    error("XMesaCreateContext: couldn't allocate colormap.");
	    return 0;
	 }
	 PseudoRGBmap = cmap;
      }
      XSetWindowColormap( c->display, window, cmap );
      XInstallColormap( c->display, cmap );
   }

   /* This is a hack for GLUT: pre-allocate the gray needed for pop-up menus */
   xcol.red = xcol.green = xcol.blue = 0xaa00;
   XAllocColor( c->display, cmap, &xcol );

#define distance2(r1,g1,b1,r2,g2,b2)  ( ((r2)-(r1)) * ((r2)-(r1)) + \
    ((g2)-(g1)) * ((g2)-(g1)) + ((b2)-(b1)) * ((b2)-(b1)) )

   /* Allocate X colors and initialize color_table[], red_table[], etc */
   for (r = 0; r < _R; r++) {
      for (g = 0; g < _G; g++) {
	 for (b = 0; b < _B; b++) {
	    xcol.red   = r * 65535 / (_R-1);
	    xcol.green = g * 65535 / (_G-1);
	    xcol.blue  = b * 65535 / (_B-1);
	    if (!XAllocColor(c->display, cmap, &xcol)) {
	       /* Search for best match (contributed by Michael Pichler) */
	       int p, bestmatch;
	       double dist, mindist;  /* 3*2^16^2 exceeds long int precision */
	       int cmap_size = c->visual->map_entries;

	       /* query whole colormap if not yet done */
	       if (!allcolors) {
		  allcolors = (XColor *) malloc( cmap_size * sizeof(XColor) );
		  for (p = 0;  p < cmap_size;  p++)
		    allcolors[p].pixel = p;
		  XQueryColors (c->display, cmap, allcolors, cmap_size);
	       }
 
	       /* find best match */
	       mindist = 0.0;
	       bestmatch = -1;
	       p = cmap_size;
	       while (p--) {
		  acptr = allcolors + p;
		  dist = distance2( (double)xcol.red, (double)xcol.green,
				    (double)xcol.blue, (double)acptr->red,
				    (double)acptr->green, (double)acptr->blue);
		  if (bestmatch < 0 || dist < mindist)
		     mindist = dist, bestmatch = p;
	       }
	       xcol.pixel = bestmatch;
	       colorsfailed++;
	    }
	    i = _MIX( r, g, b );
	    c->color_table[i] = xcol.pixel;
	    c->red_table[xcol.pixel]   = r * 255 / (_R-1);
	    c->green_table[xcol.pixel] = g * 255 / (_G-1);
	    c->blue_table[xcol.pixel]  = b * 255 / (_B-1);
	 }
      }
   }

#undef distance2

   if (allcolors) {
      free(allcolors);
   }

   if (colorsfailed) {
      char msg[1000];
      sprintf( msg,
	       "xmesa: %d out of 225 allocated colors do not match exactly.",
	       colorsfailed );
      error(msg);
   }

   c->rmult = 255;
   c->gmult = 255;
   c->bmult = 255;
   c->amult = 255;
   c->gl_ctx->RedScale   = 255.0;
   c->gl_ctx->GreenScale = 255.0;
   c->gl_ctx->BlueScale  = 255.0;
   c->gl_ctx->AlphaScale = 255.0;
   c->pixelformat = PF_8BIT;
   c->pixel = c->color_table[_MIX(_R-1,_G-1,_B-1)];  /* white */
   c->clearpixel = c->color_table[_MIX(0,0,0)];      /* black */

   return 1;
}



/*
 * Setup RGB rending for a window with a monochrome visual.
 */
static int setup_monochrome( XMesaContext c )
{
   /* Simulate RGB in monochrome */
   c->rmult = 255;  /* was 15 */
   c->gmult = 255;
   c->bmult = 255;
   c->amult = 255;  /* was 1 */
   c->gl_ctx->RedScale   = 255.0;  /* was 15 */
   c->gl_ctx->GreenScale = 255.0;
   c->gl_ctx->BlueScale  = 255.0;
   c->gl_ctx->AlphaScale = 255.0;  /* was 1.0 */
   c->pixelformat = PF_1BIT;
   c->pixel = 1;        /* white */
   c->clearpixel = 0;  /* black */

   return 1;
}



/*
 * When a context is "made current" for the first time, we can finally
 * finish initializing the context.
 * Input:  c - the XMesaContext to initialize
 *         window - the window we're rendering into if RGB mode & PseudoColor
 *
 */
static void initialize_context( XMesaContext c, Window window )
{
   XGCValues gcvalues;

   assert( c->initialized==GL_FALSE );

   if (c->gl_ctx->RGBAflag==GL_FALSE) {
      /* COLOR-INDEXED WINDOW:
       * Even if the visual is TrueColor or DirectColor we treat it as
       * being color indexed.  This is weird but might be useful to someone.
       */
      c->pixelformat = PF_INDEX;
      c->pixel = 1;
      c->clearpixel = 0;
   }
   else {
      /* RGB WINDOW:
       * If the visual is TrueColor or DirectColor we're all set.  Other-
       * wise, we simulate RGB mode using a color-mapped window.
       */
      if (c->visual->class==TrueColor || c->visual->class==DirectColor) {
	 (void) setup_truecolor( c );
      }
      else if ((c->visual->class==PseudoColor ||
	        c->visual->class==StaticColor) && c->depth>=8) {
	 (void) setup_pseudocolor( c, window );
      }
      else if (c->visual->class==StaticGray && c->depth==1) {
	 (void) setup_monochrome( c );
      }
      else {
	 error("XMesaCreateContext: can't simulate RGB mode with given visual.");
	 /*return NULL;*/
      }
   }

   get_drawable_size( c->display, window, &c->width, &c->height );

   if (c->db_state) {
      /* Double buffered */
      alloc_back_buffer( c );
      c->gl_ctx->Color.DrawBuffer = GL_BACK;
      c->gl_ctx->Pixel.ReadBuffer = GL_BACK;
      c->dest = c->db_state;
      if (c->db_state==BACK_PIXMAP) {
	 c->readable = c->backpixmap;
      }
      else {
	 c->readable = 0;
      }
      c->gl_ctx->DBflag = GL_TRUE;
   }
   else {
      /* Single Buffered */
      c->gl_ctx->Color.DrawBuffer = GL_FRONT;
      c->gl_ctx->Pixel.ReadBuffer = GL_FRONT;
      c->readable = c->frontbuffer;
      c->dest = FRONT_PIXMAP;
      c->gl_ctx->DBflag = GL_FALSE;
   }

   /* X11 graphics context */
   c->gc1 = XCreateGC( c->display, window, 0, NULL );
   XSetForeground( c->display, c->gc1, c->pixel );
   XSetBackground( c->display, c->gc1, c->pixel );
   XSetFunction( c->display, c->gc1, GXcopy );
   c->gc2 = XCreateGC( c->display, window, 0, NULL );
   XSetForeground( c->display, c->gc2, c->pixel );
   XSetBackground( c->display, c->gc2, c->pixel );
   XSetFunction( c->display, c->gc2, GXcopy );
   /*
    * Don't generate Graphics Expose/NoExpose events in swapbuffers().
    * Patch contributed by Michael Pichler May 15, 1995.
    */
   gcvalues.graphics_exposures = False;
   c->cleargc = XCreateGC( c->display, window, GCGraphicsExposures, &gcvalues);
   XSetForeground( c->display, c->cleargc, c->clearpixel );
   XSetBackground( c->display, c->cleargc, c->clearpixel );
   XSetFunction( c->display, c->cleargc, GXcopy );

   c->initialized = GL_TRUE;
}




/**********************************************************************/
/*****                       Public Functions                     *****/
/**********************************************************************/


/*
 * Create a new XMesaContext for rendering into an X11 window.
 *
 * Input:  display - X11 display
 *         visinfo - X11 VisualInfo describing the colorbuffer we'll use
 *         rgb_flag - GL_TRUE = RGB mode,
 *                    GL_FALSE = color index mode
 *         db_flag - GL_TRUE = double-buffered,
 *                   GL_FALSE = single buffered
 *         ximage_flag - GL_TRUE = use an XImage for back buffer,
 *                       GL_FALSE = use an off-screen pixmap for back buffer
 *         share_list - another XMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XMesaContext or NULL if error.
 */
XMesaContext XMesaCreateContext( Display *display,
				 XVisualInfo *visinfo,
				 GLboolean rgb_flag,
				 GLboolean db_flag,
				 GLboolean ximage_flag,
				 XMesaContext share_list )
{
   XMesaContext c;

   /* allocate xmesa_context struct initialized to zeros */
   c = (struct xmesa_context *) calloc( 1, sizeof(struct xmesa_context) );
   if (!c) {
      return NULL;
   }

   /* allocate a new Mesa context, possible share display lists */
   c->gl_ctx = gl_new_context( share_list ? share_list->gl_ctx : NULL );

   /* X stuff */
   c->display = display;
   c->visual = visinfo->visual;
   c->depth = visinfo->depth;
   c->shm = check_for_xshm( display );

   /* Color buffer mode */
   if (rgb_flag) {
      c->gl_ctx->RGBAflag = GL_TRUE;
   }
   else {
      c->gl_ctx->RGBAflag = GL_FALSE;
   }

   /* Double buffering configuration */
   if (db_flag) {
      if (ximage_flag) {
	 c->db_state = BACK_XIMAGE;
      }
      else {
	 c->db_state = BACK_PIXMAP;
      }
   }
   else {
      c->db_state = 0;
   }

   /*XSynchronize( display, 1 );*/    /* This makes debugging X easier */

   return c;
}




void XMesaDestroyContext( XMesaContext c )
{
   gl_destroy_context( c->gl_ctx );

   XFreeGC( c->display, c->gc1 );
   XFreeGC( c->display, c->gc2 );
   XFreeGC( c->display, c->cleargc );

   if (c->backimage) {
#ifdef SHM
       if (c->shm) {
	   XShmDetach( c->display, &c->shminfo );
	   XDestroyImage( c->backimage );
	   shmdt( c->shminfo.shmaddr );
       }
       else
#endif
	   XDestroyImage( c->backimage );
   }
   if (c->backpixmap) {
      XFreePixmap( c->display, c->backpixmap );
   }

   free( c );
}



/*
 * Bind an X/Mesa context to an X window.
 */
GLboolean XMesaBindWindow( XMesaContext c, Window w )
{
   unsigned int newwidth, newheight;

   c->frontbuffer = w;
   if (!c->initialized) {
      initialize_context( c, w );
   }
#ifdef LEAVEOUT
   get_drawable_size( c->display, w, &newwidth, &newheight );
   if (c->width!=newwidth || c->height!=newheight) {
      /* if buffer size changed since last binding, reallocate back buffer */

      if (newwidth!=c->gl_ctx->Viewport.Width ||
	  newheight!=c->gl_ctx->Viewport.Height) {
	 printf("Inconsistant size, w=%d x %d   v=%d x %d\n",
		newwidth, newheight, c->gl_ctx->Viewport.Width,
		c->gl_ctx->Viewport.Height );
      }
      c->width = newwidth;
      c->height = newheight;
      alloc_back_buffer( c );
   }
#endif
   return GL_TRUE;
}



/*
 * Bind an X/Mesa context to an X pixmap.
 */
GLboolean XMesaBindPixmap( XMesaContext c, Pixmap p )
{
   unsigned int newwidth, newheight;

   c->frontbuffer = p;
   if (!c->initialized) {
      if (c->gl_ctx->RGBAflag && c->visual->class==PseudoColor) {
	 /* Error, there isn't enough information carried in a pixmap */
	 /* to initialize the context.  Specifically, we need to know */
	 /* the colormap at this point for this configuration. */
	 return GL_FALSE;
      }
      initialize_context( c, p );
   }
#ifdef LEAVEOUT
   get_drawable_size( c->display, p, &newwidth, &newheight );
   if (c->width!=newwidth || c->height!=newheight) {
      /* if buffer size changed since last binding, reallocate back buffer */
      c->width = newwidth;
      c->height = newheight;
      alloc_back_buffer( c );
   }
#endif
   return GL_TRUE;
}



GLboolean XMesaMakeCurrent( XMesaContext c )
{
   if (c) {
      gl_set_context( c->gl_ctx );
      Current = c;

      if (Current->gl_ctx->Viewport.Width==0) {
	 /* initialize viewport to window size */
	 gl_viewport( 0, 0, Current->width, Current->height );
	 CC.Scissor.Width = Current->width;
	 CC.Scissor.Height = Current->height;
      }

      select_write_function();
   }
   else {
      /* Detach */
      Current = NULL;
   }
   return GL_TRUE;
}



XMesaContext XMesaGetCurrentContext( void )
{
   return Current;
}



/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.
 */
void XMesaSwapBuffers( void )
{
   if (Current->db_state) {
      if (Current->backimage) {
	 /* Copy Ximage from host's memory to server's window */
#ifdef SHM
	 if (Current->shm) {
	    XShmPutImage( Current->display, Current->frontbuffer,
			  Current->cleargc,
			  Current->backimage, 0, 0,
			  0, 0, Current->width, Current->height, False );
	    /* wait for finished event??? */
	 }
	 else
#endif
	     XPutImage( Current->display, Current->frontbuffer, Current->cleargc,
			Current->backimage, 0, 0,
			0, 0, Current->width, Current->height );
      }
      else {
	 /* Copy pixmap to window on server */
	 XCopyArea( Current->display,
		    Current->backpixmap,   /* source drawable */
		    Current->frontbuffer,  /* dest. drawable */
		    Current->cleargc,
		    0, 0, Current->width, Current->height,  /* source region */
		    0, 0                 /* dest region */
		   );
      }
   }

   XSync( Current->display, False );
}



/*
 * Return a pointer to the XMesa backbuffer Pixmap or XImage.  This function
 * is a way to get "under the hood" of X/Mesa so one can manipulate the
 * back buffer directly.
 * Output:  pixmap - pointer to back buffer's Pixmap, or 0
 *          ximage - pointer to back buffer's XImage, or NULL
 * Return:  GL_TRUE = context is double buffered
 *          GL_FALSE = context is single buffered
 */
GLboolean XMesaGetBackBuffer( Pixmap *pixmap, XImage **ximage )
{
   if (Current->db_state) {
      if (pixmap)  *pixmap = Current->backpixmap;
      if (ximage)  *ximage = Current->backimage;
      return GL_TRUE;
   }
   else {
      *pixmap = 0;
      *ximage = NULL;
      return GL_FALSE;
   }
}


/**********************************************************************/
/*****           Miscellaneous Device Driver Functions            *****/
/**********************************************************************/


/*
 * Finish all pending operations and synchronize.
 */
void dd_flush( void )
{
   if (Current) {
      XSync( Current->display, False );
   }
}



/*
 * Return information about the current buffer.
 * This function should also be called by the glViewport function because
 * glViewport is often called when the window gets resized.  We need to
 * update some X/Mesa stuff when that happens too.
 * Output:  width - width of buffer in pixels.
 *          height - height of buffer in pixels.
 *          depth - In Color Index mode, return bits/pixel
 *                - In RGBA mode, return bits/component
 */
void dd_buffer_info( GLuint *width, GLuint *height, GLuint *depth )
{
   Window root;
   int winx, winy;
   unsigned int winwidth, winheight;
   unsigned int bw, d;

   XGetGeometry( Current->display, Current->frontbuffer, &root,
		 &winx, &winy, &winwidth, &winheight, &bw, &d );

   *width = winwidth;
   *height = winheight;
   *depth = Current->depth;

   if (winwidth!=Current->width || winheight!=Current->height) {
      Current->width = winwidth;
      Current->height = winheight;
      alloc_back_buffer( Current );
   }
}



GLenum dd_read_buffer( GLenum mode )
{
   GLenum m;

   switch (mode) {
      case GL_FRONT:
      case GL_FRONT_LEFT:
      case GL_LEFT:
         m = mode;
	 Current->readable = Current->frontbuffer;
	 break;
      case GL_BACK:
      case GL_BACK_LEFT:
	 m = mode;
	 if (Current->backpixmap) {
	    Current->readable = Current->backpixmap;
	 }
	 else if (Current->backimage) {
	    Current->readable = None;
	 }
	 break;
      case GL_AUX0:
      case GL_FRONT_RIGHT:
      case GL_RIGHT:
      case GL_BACK_RIGHT:
      default:
	 return GL_FALSE;
   }
   return m;
}



GLenum dd_draw_buffer( GLenum mode )
{
   GLenum m;

   switch (mode) {
      case GL_NONE:
         m = GL_NONE;
         Current->dest = 0;
	 break;
      case GL_FRONT:
      case GL_FRONT_LEFT:
	 m = mode;
	 Current->dest = FRONT_PIXMAP;
	 break;
      case GL_BACK:
      case GL_BACK_LEFT:
	 m = mode;
	 if (Current->backpixmap) {
	    Current->dest = BACK_PIXMAP;
	 }
	 else if (Current->backimage) {
	    Current->dest = BACK_XIMAGE;
	 }
	 else {
	    Current->dest = 0;
	 }
	 break;
      case GL_LEFT:
      case GL_FRONT_AND_BACK:
	 m = mode;
	 if (Current->backpixmap) {
	    Current->dest = FRONT_PIXMAP | BACK_PIXMAP;
	 }
	 else if (Current->backimage) {
	    Current->dest = FRONT_PIXMAP | BACK_XIMAGE;
	 }
	 else {
	    Current->dest = FRONT_PIXMAP;
	 }
	 break;
      case GL_RIGHT:
      case GL_FRONT_RIGHT:
      case GL_BACK_RIGHT:
      case GL_AUX0:
      default:
	 /* Don't change drawbuffer, return error signal */
	 return GL_FALSE;
   }
   select_write_function();
   return m;
}




/*
 * Set the color index used to clear the color buffer.
 * This implements glClearIndex().
 */
void dd_clear_index( GLuint index )
{
   unsigned long p = (unsigned long) index;

   Current->clearpixel = p;
   XSetForeground( Current->display, Current->cleargc, p );
/*   XSetWindowBackground( Current->display, Current->window, p );*/
}



/*
 * Set the color used to clear the color buffer.
 * This implements glClearColor().
 */
void dd_clear_color( const GLfloat color[4] )
{
   int r, g, b, a;
   unsigned long p;

   r = color[0] * Current->rmult;
   g = color[1] * Current->gmult;
   b = color[2] * Current->bmult;
   a = color[3] * Current->amult;

   switch (Current->pixelformat) {
      case PF_8A8B8G8R:
         p =  (a << 24) | (b << 16) | (g << 8) | r;
	 break;
      case PF_8BIT:
	 p = DITHER_8BIT( 0, 0, r, g, b );
	 break;
      case PF_1BIT:
	 p = (r+g+b) > 382;
	 break;
      default:
	 p = PACK_RGB( r, g, b );
   }

   Current->clearpixel = p;
   XSetForeground( Current->display, Current->cleargc, p );
}



/*
 * Clear the color buffer using the last color given to dd_clear_color()
 * or dd_clear_index().  This is used to implement glClear().
 * Input:  all - GL_TRUE = clear whole buffer
 *               GL_FLASE = clear only the specified region
 *         x, y, width, height - region to clear if all=GL_FALSE
 */
void dd_clear( GLboolean all, GLint x, GLint y, GLint width, GLint height )
{
   if (all) {
      /* Clear whole buffer */
      if (Current->dest & FRONT_PIXMAP) {
/*
	 XSetWindowBackground( Current->display, Current->window,
			       Current->clearpixel );
	 XClearWindow( Current->display, Current->window );
*/
	 XFillRectangle( Current->display, Current->frontbuffer,
			 Current->cleargc,
			 0, 0, Current->width+1, Current->height+1 );
      }
      if (Current->dest & BACK_PIXMAP) {
	 XFillRectangle( Current->display, Current->backpixmap,
			 Current->cleargc,
			 0, 0, Current->width+1, Current->height+1 );
      }
      else if (Current->dest & BACK_XIMAGE) {
	 if (Current->backimage->bits_per_pixel==8) {
	    size_t n;
	    n = Current->backimage->bytes_per_line*Current->backimage->height;
	    memset( Current->backimage->data, Current->clearpixel, n );
	 }
	 else if (Current->backimage->bits_per_pixel==16) {
	    register GLuint i, n;
	    register GLushort *ptr;
	    n = Current->backimage->bytes_per_line/2 * Current->height;
	    ptr = (GLushort *) Current->backimage->data;
	    for (i=0;i<n;i++) {
	       *ptr++ = (GLushort) Current->clearpixel;
	    }
	 }
	 else if (Current->backimage->bits_per_pixel==32) {
	    register GLuint i, n, *ptr;
	    n = Current->width * Current->height;
	    ptr = (GLuint *) Current->backimage->data;
	    for (i=0;i<n;i++) {
	       *ptr++ = (GLuint) Current->clearpixel;
	    }
	 }
	 else {
	    register int i, j;
	    for (j=0;j<Current->height;j++) {
	       for (i=0;i<Current->width;i++) {
		  XPutPixel( Current->backimage, i, j, Current->clearpixel );
	       }
	    }
	 }
      }
   }
   else {
      /* Only clear the region defined by x,y,width,height */
      if (Current->dest & FRONT_PIXMAP) {
/*
	 XClearArea( Current->display, Current->frontbuffer,
		     x, Current->height - y - height, width, height, False );
*/
	 XFillRectangle( Current->display, Current->frontbuffer, Current->cleargc,
			 x, Current->height - y - height, width, height );
      }
      if (Current->dest & BACK_PIXMAP) {
	 XFillRectangle( Current->display, Current->backpixmap,
			 Current->cleargc,
			 x, Current->height - y - height, width, height );
      }
      else if (Current->dest & BACK_XIMAGE) {
	 /* clear back ximage */
	 register int i, j;
	 y = FLIP(y);
	 for (j=0;j<height;j++) {
	    for (i=0;i<width;i++) {
	       XPutPixel( Current->backimage, x+i, y-j, Current->clearpixel );
	    }
	 }
      }
   }
}



/*
 * Set the pixel logic operation.  Return GL_TRUE if the device driver
 * can perform the operation, otherwise return GL_FALSE.
 */
GLboolean dd_logicop( GLenum op )
{
   int func;

   if (!Current)  return GL_FALSE;

   if (Current->dest & BACK_XIMAGE) {
      /* X can't do logic ops in Ximages */
      return GL_FALSE;
   }

   switch (op) {
      case GL_CLEAR:		func = GXclear;		break;
      case GL_SET:		func = GXset;		break;
      case GL_COPY:		func = GXcopy;		break;
      case GL_COPY_INVERTED:	func = GXcopyInverted;	break;
      case GL_NOOP:		func = GXnoop;		break;
      case GL_INVERT:		func = GXinvert;	break;
      case GL_AND:		func = GXand;		break;
      case GL_NAND:		func = GXnand;		break;
      case GL_OR:		func = GXor;		break;
      case GL_NOR:		func = GXnor;		break;
      case GL_XOR:		func = GXxor;		break;
      case GL_EQUIV:		func = GXequiv;		break;
      case GL_AND_REVERSE:	func = GXandReverse;	break;
      case GL_AND_INVERTED:	func = GXandInverted;	break;
      case GL_OR_REVERSE:	func = GXorReverse;	break;
      case GL_OR_INVERTED:	func = GXorInverted;	break;
      default:  return GL_FALSE;
   }

   XSetFunction( Current->display, Current->gc1, func );
   XSetFunction( Current->display, Current->gc2, func );
   return GL_TRUE;
}



/**********************************************************************/
/*****                Simple rendering functions                  *****/
/**********************************************************************/

/*
 * Set current color index.
 */
void dd_index( GLuint index )
{
   register unsigned long p = (unsigned long) index;

   if (Current->pixel != p) {
      Current->pixel = p;
      XSetForeground( Current->display, Current->gc1, p );
   }
}



/* Set the index mode bitplane mask. */
void dd_index_mask( GLuint mask )
{
   XSetPlaneMask( Current->display, Current->gc1, (unsigned long) mask );
   XSetPlaneMask( Current->display, Current->gc2, (unsigned long) mask );
}



/*
 * Set current RGBA color.
 */
void dd_color( const GLfloat color[4] )
{
   register int r, g, b, a;
   register unsigned long p, q;

   r = color[0] * Current->rmult;
   g = color[1] * Current->gmult;
   b = color[2] * Current->bmult;
   a = color[3] * Current->amult;

   switch (Current->pixelformat) {
      case PF_8A8B8G8R:
         q = p = (a << 24) | (b << 16) | (g << 8) | r;
	 break;
      case PF_8BIT:
         p = (a << 24) | (b << 16) | (g << 8) | r;
	 q = DITHER_8BIT( 0, 0, r, g, b );
	 break;
      case PF_1BIT:
	 q = p = (r+g+b) > 382;
	 break;
      default:
	 q = p = PACK_RGB( r, g, b );
   }

   if (Current->pixel != p) {
      Current->pixel = p;
      XSetForeground( Current->display, Current->gc1, q );
   }
}


void dd_color_mask( GLboolean rmask, GLboolean gmask,
		    GLboolean bmask, GLboolean amask )
{
   /*TODO*/
}




/**********************************************************************/
/***     Render points, lines and polygons.                         ***/
/**********************************************************************/


/*
 * Write a pixel of the current color to a pixmap.
 */
static void draw_pixel_pixmap( GLint x, GLint y )
{
   XDrawPoint( Current->display, Current->drawable, Current->gc1,
	       (int)x, (int) FLIP(y) );
}



/*
 * Write a pixel of the current color to an XImage.
 */
static void draw_pixel_ximage( GLint x, GLint y )
{
   unsigned long q;
   q = Current->pixel;
   y = FLIP(y);
   if (Current->pixelformat == PF_8BIT) {
      q = DITHER_8BIT( x, y, (q&0xff), (q>>8)&0xff, (q>>16)&0xff );
   }
   XPutPixel( Current->backimage, x, y, q);
}



/*
 * General purpose pixel draw.
 */
static void draw_pixel( GLint x, GLint y )
{
   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      draw_pixel_pixmap( x, y );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      draw_pixel_pixmap( x, y );
   }
   else if (Current->dest & BACK_XIMAGE) {
      draw_pixel_ximage( x, y );
   }
}



/*
 * Draw a line segment into an XImage using current color.
 */
static void draw_line_ximage( GLint x0, GLint y0, GLint x1, GLint y1 )
{
   y0 = FLIP(y0);
   y1 = FLIP(y1);
   if (Current->pixelformat == PF_8BIT) {
      unsigned long q = Current->pixel;
      unsigned char red = (q&0xff);
      unsigned char green = (q>>8)&0xff;
      unsigned char blue = (q>>16)&0xff;
#define BRESENHAM_PLOT(X,Y) \
      XPutPixel(Current->backimage,X,Y,DITHER_8BIT(X,Y,red,green,blue));
      BRESENHAM( x0, y0, x1, y1 );
#undef BRESENHAM_PLOT
   }
   else {
#define BRESENHAM_PLOT(X,Y)  XPutPixel(Current->backimage,X,Y,Current->pixel);
      BRESENHAM( x0, y0, x1, y1 );
#undef BRESENHAM_PLOT
   }
}



/*
 * Draw a line segment into an X Pixmap using the current color.
 */
static void draw_line_pixmap( GLint x0, GLint y0, GLint x1, GLint y1 )
{
   y0 = FLIP(y0);
   y1 = FLIP(y1);
   XDrawLine( Current->display, Current->drawable, Current->gc1,
	     (int) x0, (int) y0, (int) x1, (int) y1 );
}



/*
 * General purpose line draw.
 */
static void draw_line( GLint x0, GLint y0, GLint x1, GLint y1 )
{
   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      draw_line_pixmap( x0, y0, x1, y1 );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      draw_line_pixmap( x0, y0, x1, y1 );
   }
   else if (Current->dest & BACK_XIMAGE) {
      draw_line_ximage( x0, y0, x1, y1 );
   }
}



/*
 * Draw a filled polygon into an XImage.
 */
static void draw_polygon_ximage( GLuint n, GLint x[], GLint y[] )
{
   gl_quick_polygon( n, x, y );  /* don't flip Y */
}



/*
 * Draw a filled polygon into a pixmap.
 */
static void draw_polygon_pixmap( GLuint n, GLint x[], GLint y[] )
{
   XPoint p[1000];
   register GLuint i;

   for (i=0;i<n;i++) {
      p[i].x = x[i];
      p[i].y = FLIP(y[i]);
   }

   XFillPolygon( Current->display, Current->drawable, Current->gc1,
		       p, n, Convex, CoordModeOrigin );
}



/*
 * General purpose polygon draw.
 */
static void draw_polygon( GLuint n, GLint x[], GLint y[] )
{
   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      draw_polygon_pixmap( n, x, y );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      draw_polygon_pixmap( n, x, y );
   }
   else if (Current->dest & BACK_XIMAGE) {
      draw_polygon_ximage( n, x, y );
   }
}



/*
 * The Mesa library needs to be able to draw pixels in a number of ways:
 *   1. RGB vs Color Index
 *   2. as horizontal spans (polygons, images) vs random locations (points, lines)
 *   3. different color per-pixel or same color for all pixels
 *
 * Furthermore, the X driver needs to support rendering to 3 possible "buffers",
 * usually one, but sometimes two at a time:
 *   1. The front buffer as an X window
 *   2. The back buffer as a Pixmap
 *   3. The back buffer as an XImage
 *
 * Finally, if the back buffer is an XImage, we can avoid using XPutPixel and
 * optimize common cases such as 24-bit and 8-bit modes.
 *
 * By multiplication, there's at least 48 possible combinations of the above.
 *
 * Below are implementations of the most commonly used combinations.  They are
 * accessed through function pointers which get initialized here and are used
 * directly from the Mesa library.  The 8 function pointers directly correspond
 * to the first 3 cases listed above.
 *
 *
 * The function naming convention is:
 *
 *   write_[span|pixels]_[mono]_[format]_[pixmap|ximage]
 *
 * New functions optimized for specific cases can be added without too much
 * trouble.  An example might be the 24-bit TrueColor mode 8A8R8G8B which is
 * found on IBM RS/6000 X servers.
 */




/**********************************************************************/
/*** Write COLOR SPAN functions                                     ***/
/**********************************************************************/


#define COLOR_SPAN_ARGS	GLuint n, GLint x, GLint y,		\
			const GLubyte red[], const GLubyte green[],	\
			const GLubyte blue[], const GLubyte alpha[],\
			const GLubyte mask[]

/*
 * Write a span of PF_8A8B8G8R-format pixels to a pixmap.
 */
static void write_span_8A8B8G8R_pixmap( COLOR_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP( y );
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = (alpha[i]<<24) | (blue[i]<<16) | (green[i]<<8) | red[i];
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2, (int) x, (int) y );
      }
   }
}



/*
 * Write a span of PF_8BIT-format pixels to a pixmap.
 */
static void write_span_8BIT_pixmap( COLOR_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP( y );
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 register GLint r, g, b, d;
	 register unsigned long p;
	 p = DITHER_8BIT( x, y, red[i], green[i], blue[i] );
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2, (int) x, (int) y );
      }
   }
}



/*
 * Write a span of PF_1BIT-format pixels to a pixmap.
 */
static void write_span_1BIT_pixmap( COLOR_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP( y );
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = DITHER_1BIT( x, y, red[i], green[i], blue[i] );
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2, (int) x, (int) y );
      }
   }
}



/*
 * Write a span of PF_8A8B8G8R-format pixels to an ximage.
 */
static void write_span_8A8B8G8R_ximage( COLOR_SPAN_ARGS )
{
   register GLuint i;
   register GLuint *ptr;
   ptr = (GLuint *) Current->backimage->data + OFFSET4(x,y);
   for (i=0;i<n;i++,ptr++) {
      if (mask[i]) {
	 *ptr = (alpha[i] << 24) | (blue[i] << 16) | (green[i] << 8) | red[i];
      }
   }
}



/*
 * Write a span of PF_8BIT pixels to an XImage.
 */
static void write_span_8BIT_ximage( COLOR_SPAN_ARGS )
{
   register GLuint i;
   register GLubyte *ptr;
   ptr = (GLubyte *) Current->backimage->data + OFFSET1(x,y);
   for (i=0;i<n;i++,x++,ptr++) {
      if (mask[i]) {
	 register GLint r, g, b, d;
	 *ptr = DITHER_8BIT( x, y, red[i], green[i], blue[i] );
      }
   }
}



/*
 * Write a span of PF_1BIT pixels to an XImage.
 */
static void write_span_1BIT_ximage( COLOR_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP(y);
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = DITHER_1BIT( x, y, red[i], green[i], blue[i] );
	 XPutPixel( Current->backimage, x, y, p );
      }
   }
}



/*
 * Write a span of colored pixels.  Use this function when no specialty
 * function will work.
 */
static void write_span_color( COLOR_SPAN_ARGS )
{
   register GLuint i;
   register unsigned long p;

   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_span_8A8B8G8R_pixmap(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_span_8BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_span_1BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* i.e. PF_COLOR: */
	    y = FLIP(y);
	    for (i=0;i<n;i++,x++) {
	       if (mask[i]) {
		  p = PACK_RGB( red[i], green[i], blue[i] );
		  XSetForeground( Current->display, Current->gc2, p );
		  XDrawPoint( Current->display, Current->frontbuffer, Current->gc2,
			      (int) x, (int) y );
	       }
	    }
      }
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_span_8A8B8G8R_pixmap(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_span_8BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_span_1BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* i.e. PF_COLOR: */
	    y = FLIP(y);
	    for (i=0;i<n;i++,x++) {
	       if (mask[i]) {
		  p = PACK_RGB( red[i], green[i], blue[i] );
		  XSetForeground( Current->display, Current->gc2, p );
		  XDrawPoint( Current->display, Current->backpixmap, Current->gc2,
			      (int) x, (int) y );
	       }
	    }
      }
   }
   else if (Current->dest & BACK_XIMAGE) {
      /* back ximage */
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_span_8A8B8G8R_ximage(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_span_8BIT_ximage( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_span_1BIT_ximage( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* PF_COLOR */
	    y = FLIP(y);
	    for (i=0;i<n;i++,x++) {
	       if (mask[i]) {
		  p = PACK_RGB( red[i], green[i], blue[i] );
		  XPutPixel( Current->backimage, x, y, p );
	       }
	    }
      }
   }
}


/**********************************************************************/
/*** Write COLOR PIXEL functions                                    ***/
/**********************************************************************/


#define COLOR_PIXEL_ARGS   GLuint n, const GLint x[], const GLint y[],	\
			   const GLubyte red[], const GLubyte green[],	\
			   const GLubyte blue[], const GLubyte alpha[],	\
			   const GLubyte mask[]

/*
 * Write an array of PF_8A8B8G8R-format pixels to a pixmap.
 */
static void write_pixels_8A8B8G8R_pixmap( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = (alpha[i]<<24) | (blue[i]<<16) | (green[i]<<8) | red[i];
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2,
	             (int) x[i], (int) FLIP(y[i]) );
      }
   }
}



/*
 * Write an array of PF_8BIT-format pixels to a pixmap.
 */
static void write_pixels_8BIT_pixmap( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = DITHER_8BIT( x[i], y[i], red[i], green[i], blue[i] );
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2,
	             (int) x[i], (int) FLIP(y[i]) );
      }
   }
}



/*
 * Write an array of PF_1BIT-format pixels to a pixmap.
 */
static void write_pixels_1BIT_pixmap( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = DITHER_1BIT( x[i], y[i], red[i], green[i], blue[i] );
	 XSetForeground( Current->display, Current->gc2, p );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2,
	             (int) x[i], (int) FLIP(y[i]) );
      }
   }
}



/*
 * Write an array of PF_8A8B8G8R-format pixels to an ximage.
 */
static void write_pixels_8A8B8G8R_ximage( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 GLuint *ptr;
	 ptr = (GLuint *) Current->backimage->data + OFFSET4(x[i],y[i]);
	 *ptr = (alpha[i] << 24) | (blue[i] << 16) | (green[i] << 8) | red[i];
      }
   }
}



/*
 * Write an array of PF_8BIT pixels to an XImage.
 */
static void write_pixels_8BIT_ximage( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register GLubyte *ptr;
	 ptr = (GLubyte *) Current->backimage->data + OFFSET1(x[i],y[i]);
	 *ptr = DITHER_8BIT( x[i], y[i], red[i], green[i], blue[i] );
      }
   }
}



/*
 * Write an array of PF_1BIT pixels to an XImage.
 */
static void write_pixels_1BIT_ximage( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register unsigned long p;
	 p = DITHER_1BIT( x[i], y[i], red[i], green[i], blue[i] );
	 XPutPixel( Current->backimage, x[i], FLIP(y[i]), p );
      }
   }
}



/*
 * Write an array of colored pixels.  Use this function when no specialty
 * function will work.
 */
static void write_pixels_color( COLOR_PIXEL_ARGS )
{
   register GLuint i;
   register unsigned long p;

   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_pixels_8A8B8G8R_pixmap(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_pixels_8BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_pixels_1BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* i.e. PF_COLOR: */
	    for (i=0;i<n;i++) {
	       if (mask[i]) {
		  p = PACK_RGB( red[i], green[i], blue[i] );
		  XSetForeground( Current->display, Current->gc2, p );
		  XDrawPoint( Current->display, Current->frontbuffer, Current->gc2,
			      (int) x[i], (int) FLIP(y[i]) );
	       }
	    }
      }
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_pixels_8A8B8G8R_pixmap(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_pixels_8BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_pixels_1BIT_pixmap( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* i.e. PF_COLOR: */
	    for (i=0;i<n;i++) {
	       if (mask[i]) {
		  p = PACK_RGB( red[i], green[i], blue[i] );
		  XSetForeground( Current->display, Current->gc2, p );
		  XDrawPoint( Current->display, Current->backpixmap, Current->gc2,
			      (int) x[i], (int) FLIP(y[i]) );
	       }
	    }
      }
   }
   else if (Current->dest & BACK_XIMAGE) {
      /* back ximage */
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_pixels_8A8B8G8R_ximage(n, x, y, red, green, blue, alpha, mask);
	    break;
	 case PF_8BIT:
	    write_pixels_8BIT_ximage( n, x, y, red, green, blue, alpha, mask );
	    break;
	 case PF_1BIT:
	    write_pixels_1BIT_ximage( n, x, y, red, green, blue, alpha, mask );
	    break;
	 default:  /* PF_COLOR */
	    for (i=0;i<n;i++) {
	       if (mask[i]) {
		  p = PACK_RGBA( red[i], green[i], blue[i], alpha[i] );
		  XPutPixel( Current->backimage, x[i], FLIP(y[i]), p );
	       }
	    }
      }
   }
}


/**********************************************************************/
/*** Write MONO COLOR SPAN functions                                ***/
/**********************************************************************/

#define MONO_SPAN_ARGS  GLuint n, GLint x, GLint y, const GLubyte mask[]


/*
 * Write a span of identical pixels to a pixmap.  The pixel value is
 * the one set by dd_color() or dd_index().
 */
static void write_span_mono_pixmap( MONO_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP( y );
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 XDrawPoint( Current->display, Current->drawable, Current->gc1, (int) x, (int) y );
      }
   }
}


/*
 * Write a span of identical pixels to an XImage.  The pixel value is
 * the one set by dd_color() or dd_index().
 */
static void write_span_mono_ximage( MONO_SPAN_ARGS )
{
   register GLuint i;
   register unsigned long p = Current->pixel;
   y = FLIP( y );
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 XPutPixel( Current->backimage, x, y, p );
      }
   }
}


/*
 * Write a span if identical 8A8B8G8R pixels to an XImage.  The pixel
 * value is the one set by dd_color().
 */
static void write_span_mono_8A8B8G8R_ximage( MONO_SPAN_ARGS )
{
   register GLuint i, p, *ptr;
   p = (GLuint) Current->pixel;
   ptr = (GLuint *) Current->backimage->data + OFFSET4(x,y);
   for (i=0;i<n;i++,ptr++) {
      if (mask[i]) {
	 *ptr = p;
      }
   }
}


/*
 * Write a span if identical 8BIT pixels to an XImage.  The pixel
 * value is the one set by dd_color().
 */
static void write_span_mono_8BIT_ximage( MONO_SPAN_ARGS )
{
   register GLuint i;
   unsigned long p = Current->pixel;
   register GLbyte *ptr = (GLbyte *) Current->backimage->data + OFFSET1(x,y);
   register GLubyte r, g, b;
   r = p&0xff;
   g = (p>>8)&0xff;
   b = (p>>16)&0xff;
   for (i=0;i<n;i++,ptr++,x++) {
      if (mask[i]) {
	 *ptr = DITHER_8BIT( x, y, r, g, b );
      }
   }
}


/*
 * Write a span of identical pixels.  This generic function is used
 * when none of the above speciality funcions can be used.
 */
static void write_span_mono( MONO_SPAN_ARGS )
{
   register GLuint i;
   register unsigned long p;

   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      write_span_mono_pixmap( n, x, y, mask );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      write_span_mono_pixmap( n, x, y, mask );
   }
   else if (Current->dest & BACK_XIMAGE) {
      /* back ximage */
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_span_mono_8A8B8G8R_ximage( n, x, y, mask );
	    break;
	 case PF_8BIT:
	    write_span_mono_8BIT_ximage( n, x, y, mask );
	    break;
	 default:
	    y = FLIP(y);
	    for (i=0;i<n;i++,x++) {
	       if (mask[i]) {
		  XPutPixel( Current->backimage, x, y, Current->pixel );
	       }
	    }
      }
   }
}


/**********************************************************************/
/*** Write MONO COLOR PIXELS functions                              ***/
/**********************************************************************/

#define MONO_PIXEL_ARGS		GLuint n, const GLint x[], const GLint y[], \
				const GLubyte mask[]

/*
 * Write an array of identical pixels to a pixmap.  The pixel value is
 * the one set by dd_color() or dd_index.
 */
static void write_pixels_mono_pixmap( MONO_PIXEL_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 XDrawPoint( Current->display, Current->drawable, Current->gc1,
		    (int) x[i], (int) FLIP(y[i]) );
      }
   }
}


/*
 * Write an array of identical 8A8B8G8R pixels to an XImage.  The pixel value is
 * the one set by dd_color().
 */
static void write_pixels_mono_8A8B8G8R_ximage( MONO_PIXEL_ARGS )
{
   register GLuint i;
   register GLuint p = (GLuint) Current->pixel;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register GLuint *ptr;
	 ptr = (GLuint *) Current->backimage->data + OFFSET4(x[i],y[i]);
	 *ptr = p;
      }
   }
}


/*
 * Write an array of identical 8BIT pixels to an XImage.  The pixel value is
 * the one set by dd_color().
 */
static void write_pixels_mono_8BIT_ximage( MONO_PIXEL_ARGS )
{
   register GLuint i;
   register unsigned long p = Current->pixel;
   register GLubyte r, g, b;
   r = p&0xff;
   g = (p>>8)&0xff;
   b = (p>>16)&0xff;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 register GLubyte *ptr;
	 ptr = (GLubyte *) Current->backimage->data + OFFSET1(x[i],y[i]);
	 *ptr = DITHER_8BIT( x[i], y[i], r, g, b );
      }
   }
}


/*
 * Write an array of identical pixels to an XImage.  The pixel value is
 * the one set by dd_color() or dd_index.
 */
static void write_pixels_mono_ximage( MONO_PIXEL_ARGS )
{
   register GLuint i;
   register unsigned long p = Current->pixel;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 XPutPixel( Current->backimage, x[i], FLIP(y[i]), p );
      }
   }
}



/*
 * Write an array of identical pixels.  This generic function is used
 * when none of the above speciality funcions can be used.
 */
static void write_pixels_mono( MONO_PIXEL_ARGS )
{
   register GLuint i;
   register unsigned long p;

   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      write_pixels_mono_pixmap( n, x, y, mask );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      write_pixels_mono_pixmap( n, x, y, mask );
   }
   else if (Current->dest & BACK_XIMAGE) {
      /* back ximage */
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    write_pixels_mono_8A8B8G8R_ximage( n, x, y, mask );
	    break;
	 case PF_8BIT:
	    write_pixels_mono_8BIT_ximage( n, x, y, mask );
	    break;
	 default:
	    for (i=0;i<n;i++) {
	       if (mask[i]) {
		  XPutPixel( Current->backimage, x[i], FLIP(y[i]), Current->pixel );
	       }
	    }
      }
   }
}


/**********************************************************************/
/*** Write INDEX SPAN functions                                     ***/
/**********************************************************************/

#define INDEX_SPAN_ARGS   GLuint n, GLint x, GLint y, const GLuint index[], \
			  const GLubyte mask[]


/*
 * Write a span of CI pixels to a Pixmap.
 */
static void write_span_index_pixmap( INDEX_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP(y);
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 XSetForeground( Current->display, Current->gc2,
			 (unsigned long) index[i] );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2, (int) x, (int) y );
      }
   }
}


/*
 * Write a span of CI pixels to an XImage.
 */
static void write_span_index_ximage( INDEX_SPAN_ARGS )
{
   register GLuint i;
   y = FLIP(y);
   for (i=0;i<n;i++,x++) {
      if (mask[i]) {
	 XPutPixel( Current->backimage, x, y, (unsigned long) index[i] );
      }
   }
}


/*
 * General case of writing a span of CI pixels.
 */
static void write_span_index( INDEX_SPAN_ARGS )
{
   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      write_span_index_pixmap( n, x, y, index, mask );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      write_span_index_pixmap( n, x, y, index, mask );
   }
   else if (Current->dest & BACK_XIMAGE) {
      write_span_index_ximage( n, x, y, index, mask );
   }
}



/**********************************************************************/
/*** Write INDEX PIXELS functions                                   ***/
/**********************************************************************/

#define INDEX_PIXELS_ARGS	GLuint n, const GLint x[], const GLint y[], \
				const GLuint index[], const GLubyte mask[]


/*
 * Write an array of CI pixels to a Pixmap.
 */
static void write_pixels_index_pixmap( INDEX_PIXELS_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 XSetForeground( Current->display, Current->gc2,
			 (unsigned long) index[i] );
	 XDrawPoint( Current->display, Current->drawable, Current->gc2, 
		     (int) x[i], (int) FLIP(y[i]) );
      }
   }
}


/*
 * Write an array of CI pixels to an XImage.
 */
static void write_pixels_index_ximage( INDEX_PIXELS_ARGS )
{
   register GLuint i;
   for (i=0;i<n;i++) {
      if (mask[i]) {
	 XPutPixel( Current->backimage, x[i], FLIP(y[i]), (unsigned long) index[i] );
      }
   }
}



/*
 * General case of writing an array of CI pixels.
 */
static void write_pixels_index( INDEX_PIXELS_ARGS )
{
   if (Current->dest & FRONT_PIXMAP) {
      Current->drawable = Current->frontbuffer;
      write_pixels_index_pixmap( n, x, y, index, mask );
   }
   if (Current->dest & BACK_PIXMAP) {
      Current->drawable = Current->backpixmap;
      write_pixels_index_pixmap( n, x, y, index, mask );
   }
   else if (Current->dest & BACK_XIMAGE) {
      write_pixels_index_ximage( n, x, y, index, mask );
   }
}




/**********************************************************************/
/*** Setup function                                                 ***/
/**********************************************************************/


static void die( char *s )
{
   fprintf( stderr, "%s\n", s );
   abort();
}


/*
 * This function sets the function pointers used for drawing pixels depending
 * on the current output buffer and pixel format.
 */
static void select_write_function( void )
{
   if ((Current->dest & FRONT_PIXMAP)
       && (Current->dest & (BACK_PIXMAP|BACK_XIMAGE))) {
      /* Writing to front AND back buffers -> use general functions */
      DD.draw_pixel   = draw_pixel;
      DD.draw_line    = draw_line;
      DD.draw_polygon = draw_polygon;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	 case PF_8BIT:
	 case PF_1BIT:
	 case PF_COLOR:
	    DD.write_color_span        = write_span_color;
	    DD.write_monocolor_span    = write_span_mono;
	    DD.write_color_pixels      = write_pixels_color;
	    DD.write_monocolor_pixels  = write_pixels_mono;
	    break;
	 case PF_INDEX:
	    DD.write_index_span        = write_span_index;
	    DD.write_monoindex_span    = write_span_mono;
	    DD.write_index_pixels      = write_pixels_index;
	    DD.write_monoindex_pixels  = write_pixels_mono;
	    break;
	 default:
	    die("Bad pixel format in select_write_function (1)");
      }
   }
   else if ((Current->dest & FRONT_PIXMAP) || (Current->dest & BACK_PIXMAP)) {
      /* Writing to front pixmap OR back pixmap */
      if (Current->dest & FRONT_PIXMAP) {
	 Current->drawable = Current->frontbuffer;
      }
      else {
	 Current->drawable = Current->backpixmap;
      }
      DD.draw_pixel   = draw_pixel_pixmap;
      DD.draw_line    = draw_line_pixmap;
      DD.draw_polygon = draw_polygon_pixmap;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    DD.write_color_span       = write_span_8A8B8G8R_pixmap;
	    DD.write_monocolor_span   = write_span_mono_pixmap;
	    DD.write_color_pixels     = write_pixels_8A8B8G8R_pixmap;
	    DD.write_monocolor_pixels = write_pixels_mono_pixmap;
	    break;
	 case PF_8BIT:
	    DD.write_color_span       = write_span_8BIT_pixmap;
	    DD.write_monocolor_span   = write_span_mono_pixmap;
	    DD.write_color_pixels     = write_pixels_8BIT_pixmap;
	    DD.write_monocolor_pixels = write_pixels_mono_pixmap;
	    break;
	 case PF_1BIT:
	    DD.write_color_span       = write_span_1BIT_pixmap;
	    DD.write_monocolor_span   = write_span_mono_pixmap;
	    DD.write_color_pixels     = write_pixels_1BIT_pixmap;
	    DD.write_monocolor_pixels = write_pixels_mono_pixmap;
	    break;
	 case PF_COLOR:
	    /* Generic RGB */
	    DD.write_color_span       = write_span_color;
	    DD.write_monocolor_span   = write_span_mono_pixmap;
	    DD.write_color_pixels     = write_pixels_color;
	    DD.write_monocolor_pixels = write_pixels_mono_pixmap;
	    break;
	 case PF_INDEX:
	    DD.write_index_span       = write_span_index_pixmap;
	    DD.write_monoindex_span   = write_span_mono_pixmap;
	    DD.write_index_pixels     = write_pixels_index_pixmap;
	    DD.write_monoindex_pixels = write_pixels_mono_pixmap;
	    break;
	 default:
	    die("Bad pixel format in select_write_function (2)");
      }
   }
   else if (Current->dest & BACK_XIMAGE) {
      DD.draw_pixel   = draw_pixel_ximage;
      DD.draw_line    = draw_line_ximage;
      DD.draw_polygon = draw_polygon_ximage;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    DD.write_color_span       = write_span_8A8B8G8R_ximage;
	    DD.write_monocolor_span   = write_span_mono_8A8B8G8R_ximage;
	    DD.write_color_pixels     = write_pixels_8A8B8G8R_ximage;
	    DD.write_monocolor_pixels = write_pixels_mono_8A8B8G8R_ximage;
	    break;
	 case PF_8BIT:
	    DD.write_color_span       = write_span_8BIT_ximage;
	    DD.write_monocolor_span   = write_span_mono_8BIT_ximage;
	    DD.write_color_pixels     = write_pixels_8BIT_ximage;
	    DD.write_monocolor_pixels = write_pixels_mono_8BIT_ximage;
	    break;
	 case PF_1BIT:
	    DD.write_color_span       = write_span_1BIT_ximage;
	    DD.write_monocolor_span   = write_span_mono_ximage;
	    DD.write_color_pixels     = write_pixels_1BIT_ximage;
	    DD.write_monocolor_pixels = write_pixels_mono_ximage;
	    break;
	 case PF_COLOR:
	    /* Generic RGB */
	    DD.write_color_span       = write_span_color;
	    DD.write_monocolor_span   = write_span_mono_ximage;
	    DD.write_color_pixels     = write_pixels_color;
	    DD.write_monocolor_pixels = write_pixels_mono_ximage;
	    break;
	 case PF_INDEX:
	    DD.write_index_span       = write_span_index_ximage;
	    DD.write_monoindex_span   = write_span_mono_ximage;
	    DD.write_index_pixels     = write_pixels_index_ximage;
	    DD.write_monoindex_pixels = write_pixels_mono_ximage;
	    break;
	 default:
	    die("Bad pixel format in select_write_function (3)");
      }
   }
}






/**********************************************************************/
/*****                      Pixel reading                         *****/
/**********************************************************************/



/*
 * Read a horizontal span of color-index pixels.
 */
void dd_read_index_span( GLuint n, GLint x, GLint y, GLuint index[] )
{
   int i;

   y = FLIP(y);

   if (Current->readable) {
      XImage *span;
      span = XGetImage( Current->display, Current->readable,
		        x, y, n, 1, AllPlanes, ZPixmap );
      if (span) {
	 for (i=0;i<n;i++) {
	    index[i] = (GLuint) XGetPixel( span, i, 0 );
	 }
	 XDestroyImage( span );
      }
      else {
	 error("dd_read_index_span: read index xgetimage failed.");
	 /* return 0 pixels */
	 for (i=0;i<n;i++) {
	    index[i] = 0;
	 }
      }
   }
   else if (Current->backimage) {
      for (i=0;i<n;i++,x++) {
	 index[i] = (GLuint) XGetPixel( Current->backimage, x, y );
      }
   }
}



/*
 * Read a horizontal span of color pixels.
 */
void dd_read_color_span( GLuint n, GLint x, GLint y,
			 GLubyte red[], GLubyte green[],
			 GLubyte blue[], GLubyte alpha[] )
{
   register GLuint i;
   register unsigned long p;
   register GLuint p4, *ptr4;  /* 4-byte type */

   y = FLIP(y);

   if (Current->readable) {
      XImage *span;
      span = XGetImage( Current->display, Current->readable,
		        x, y, n, 1, AllPlanes, ZPixmap );
      if (span) {
	 switch (Current->pixelformat) {
	    case PF_8A8B8G8R:
	       ptr4 = (GLuint *) span->data;
	       for (i=0;i<n;i++) {
		  p4 = *ptr4++;
		  red[i]   = (GLubyte) ( p4        & 0xff);
		  green[i] = (GLubyte) ((p4 >> 8)  & 0xff);
		  blue[i]  = (GLubyte) ((p4 >> 16) & 0xff);
		  alpha[i] = (GLubyte) ((p4 >> 24) & 0xff);   /* 255?? */
	       }
	       break;
	    case PF_8BIT:
	       for (i=0;i<n;i++) {
		  p = XGetPixel( span, i, 0 );
		  /* TODO: is this it? */
		  red[i]   = (GLubyte) Current->red_table[p];
		  green[i] = (GLubyte) Current->green_table[p];
		  blue[i]  = (GLubyte) Current->blue_table[p];
		  alpha[i] = 255;
	       }
	       break;
	    case PF_1BIT:
	       for (i=0;i<n;i++) {
		  p = XGetPixel( span, i, 0 );
		  red[i]   = (GLubyte) (p * 255);  /* was 15 */
		  green[i] = (GLubyte) (p * 255);
		  blue[i]  = (GLubyte) (p * 255);
		  alpha[i] = 255;
	       }
	       break;
	    default:
	       for (i=0;i<n;i++) {
		  p = XGetPixel( span, i, 0 );
		  red[i]   = (GLubyte) ((p >> Current->rshift) & Current->rmult);
		  green[i] = (GLubyte) ((p >> Current->gshift) & Current->gmult);
		  blue[i]  = (GLubyte) ((p >> Current->bshift) & Current->bmult);
		  alpha[i] = (GLubyte) ((p >> Current->ashift) & Current->amult);
	       }
	 }
	 XDestroyImage( span );
      }
      else {
	 /* return black pixels */
	 for (i=0;i<n;i++) {
	    red[i] = green[i] = blue[i] = alpha[i] = 0;
	 }
      }
   }
   else if (Current->backimage) {
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    ptr4 = (GLuint *)
		      Current->backimage->data + y * Current->width + x;
	    for (i=0;i<n;i++) {
	       p4 = *ptr4++;
	       red[i]   = (GLubyte) ( p4        & 0xff);
	       green[i] = (GLubyte) ((p4 >> 8)  & 0xff);
	       blue[i]  = (GLubyte) ((p4 >> 16) & 0xff);
	       alpha[i] = (GLint) ((p4 >> 24) & 0xff);
	    }
	    break;
	 case PF_8BIT:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x, y );
	       /* TODO: is this right? */
	       red[i]   = (GLubyte) Current->red_table[p];
	       green[i] = (GLubyte) Current->green_table[p];
	       blue[i]  = (GLubyte) Current->blue_table[p];
	       alpha[i] = 255;
	    }
	    break;
	 case PF_1BIT:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x, y );
	       red[i]   = (GLubyte) (p * 255); /* was 15 */
	       green[i] = (GLubyte) (p * 255);
	       blue[i]  = (GLubyte) (p * 255);
	       alpha[i] = 255;
	    }
	    break;
	 default:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x, y );
	       /* extract r,g,b,a from pixel values */
	       red[i]   = (GLubyte) ((p >> Current->rshift) & Current->rmult);
	       green[i] = (GLubyte) ((p >> Current->gshift) & Current->gmult);
	       blue[i]  = (GLubyte) ((p >> Current->bshift) & Current->bmult);
	       alpha[i] = (GLubyte) ((p >> Current->ashift) & Current->amult);
	    }
      }
   }
}



/*
 * Read an array of color index pixels.
 */
void dd_read_index_pixels( GLuint n, const GLint x[], const GLint y[],
			   GLuint indx[] )
{
   register GLuint i;

   if (Current->readable) {
      for (i=0;i<n;i++) {
	 indx[i] = (GLuint) read_pixel( Current->display, Current->readable,
				        x[i], FLIP(y[i]) );
      }
   }
   else if (Current->backimage) {
      for (i=0;i<n;i++) {
	 indx[i] = (GLuint) XGetPixel( Current->backimage, x[i], FLIP(y[i]) );
      }
   }
}



void dd_read_color_pixels( GLuint n, const GLint x[], const GLint y[],
			   GLubyte red[], GLubyte green[],
			   GLubyte blue[], GLubyte alpha[] )
{
   register GLuint i;
   register unsigned long p;

   if (Current->readable) {
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    for (i=0;i<n;i++) {
	       p = read_pixel( Current->display, Current->readable,
			       x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) ( p        & 0xff);
	       green[i] = (GLubyte) ((p >> 8)  & 0xff);
	       blue[i]  = (GLubyte) ((p >> 16) & 0xff);
	       alpha[i] = (GLubyte) ((p >> 24) & 0xff);   /* 255?? */
	    }
	    break;
	 case PF_8BIT:
	    for (i=0;i<n;i++) {
	       p = read_pixel( Current->display, Current->readable,
			       x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) Current->red_table[p];
	       green[i] = (GLubyte) Current->green_table[p];
	       blue[i]  = (GLubyte) Current->blue_table[p];
	       alpha[i] = 255;
	    }
	    break;
	 case PF_1BIT:
	    for (i=0;i<n;i++) {
	       p = read_pixel( Current->display, Current->readable,
			       x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) (p * 255);  /* was 15 */
	       green[i] = (GLubyte) (p * 255);
	       blue[i]  = (GLubyte) (p * 255);
	       alpha[i] = 255;
	    }
	    break;
	 default:
	    for (i=0;i<n;i++) {
	       p = read_pixel( Current->display, Current->readable,
			       x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) ((p >> Current->rshift) & Current->rmult);
	       green[i] = (GLubyte) ((p >> Current->gshift) & Current->gmult);
	       blue[i]  = (GLubyte) ((p >> Current->bshift) & Current->bmult);
	       alpha[i] = Current->amult;
	    }
      }
   }
   else if (Current->backimage) {
      GLuint *ptr4, p4;
      switch (Current->pixelformat) {
	 case PF_8A8B8G8R:
	    for (i=0;i<n;i++) {
	       ptr4 = (GLuint *) Current->backimage->data
		        + FLIP(y[i]) * Current->width + x[i];
	       p4 = *ptr4;
	       red[i]   = (GLubyte) ( p4        & 0xff);
	       green[i] = (GLubyte) ((p4 >> 8)  & 0xff);
	       blue[i]  = (GLubyte) ((p4 >> 16) & 0xff);
	       alpha[i] = (GLubyte) ((p4 >> 24) & 0xff);
	    }
	    break;
	 case PF_8BIT:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) Current->red_table[p];
	       green[i] = (GLubyte) Current->green_table[p];
	       blue[i]  = (GLubyte) Current->blue_table[p];
	       alpha[i] = 255;
	    }
	    break;
	 case PF_1BIT:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x[i], FLIP(y[i]) );
	       red[i]   = (GLubyte) (p * 255); /* was 15 */
	       green[i] = (GLubyte) (p * 255);
	       blue[i]  = (GLubyte) (p * 255);
	       alpha[i] = 255;
	    }
	    break;
	 default:
	    for (i=0;i<n;i++,x++) {
	       p = XGetPixel( Current->backimage, x[i], FLIP(y[i]) );
	       /* extract r,g,b,a from pixel values */
	       red[i]   = (GLubyte) ((p >> Current->rshift) & Current->rmult);
	       green[i] = (GLubyte) ((p >> Current->gshift) & Current->gmult);
	       blue[i]  = (GLubyte) ((p >> Current->bshift) & Current->bmult);
	       alpha[i] = Current->amult;
	    }
      }
   }
}

