/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Eero Tamminen
 *
 * Getstring widget with scrolling and copy&paste for fixed width fonts.
 *
 * Pasting doesn't scroll for the obvious reason that the widget
 * would not receive any events after the mouse goes out of the
 * editing field (make horizontal borders bigger?).
 *
 * Mouse has to be in the text widget for input. Container widgets
 * / W server don't have the concept of clickable keyboard focus
 * (yet?).
 *
 * Before realizing the widget tree, options:
 *	WT_STRING_ADDRESS	- initialized(!) string buffer on user space
 *	WT_STRING_LENGTH	- String buffer size (- terminating zero)
 *	WT_STRING_WIDTH		- String editing width in chars
 *	WT_MODE			- GetstrModeNoBorder, GetstrModeWithBorder
 *
 * _have_ to be set as these are needed by the widget functions and
 * some widget internals are initialized only when these are set.
 *
 * ATM WT_WIDTH and WT_HEIGHT can only be queried.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

/* Copy&Paste buffer, should really be in the toolkit so that
 * other text widgets could use it too...
 */
#define BUFFER_SIZE	64
static char Buffer[BUFFER_SIZE];
static int BufferLenght;		/* whether there's something... */

/* special keys */
#define KEY_DEL		'\177'
#define KEY_LEFT	('B' & 0x1f)	/* emacs keys */
#define KEY_RIGHT	('F' & 0x1f)
#define KEY_HOME	('A' & 0x1f)
#define KEY_END		('E' & 0x1f)
#define KEY_CLEAR	'\33'		/* ESC clears the text */

/* default font */
#define FONT_NAME	"fixed10.wfnt"
/* horizontal and vertical border offsets */
#define XBORDER		2
#define YBORDER		(2*XBORDER)

typedef struct
  {
    widget_t w;
    short is_open;
    short is_realized;
    char fontname[32];
    WFONT *font;
    short mode;			/* GetstrNoBorder, GetstrWithBorder */
    short xborder, yborder;	/* offsets from left, top */
    char *text;			/* user buffer on which text is processed */
    int maxlen;			/* the string buffer lenght */
    int textlen;		/* set / input text lenght */
    int showlen;		/* the string lenght on screen */
    int char_width;		/* fixed width character width */
    int cursor;			/* cursor postion in characters (in the
				   edit field, not in the string) */
    int textpos;		/* string index to first char in edit box */
    int copypos;		/* cursor 'postion' when painting chars */
    int paint;			/* painting in process */
    long timer;			/* for copy&paste mouse checking */
    void (*ok_cb) (char *text, int cursor_pos);	/* called with CR/LF */
  } getstring_widget_t;


/* show the character places on empty edit positions */
static void fill_sublines(getstring_widget_t *w)
{
  int idx = w->textlen - w->textpos;

  if (idx < w->showlen)
  {
    w_setmode(w->w.win, M_CLEAR);
    w_pbox(w->w.win,
      w->xborder + w->char_width * idx, w->yborder,
      w->char_width * (w->showlen - idx), w->font->height);
    w_setmode(w->w.win, M_INVERS);
    while(idx < w->showlen)
    {
      w_hline (w->w.win,
        w->xborder + w->char_width * idx + 1, w->yborder + w->font->height - 1,
	w->xborder + w->char_width * (idx+1) - 1);
      idx++;
    }
  }
}

/* draw the initial (everything postioned to zero)
 * text field, default graphics mode is M_INVERS
 */
static void
getstring_draw (getstring_widget_t * w)
{
  char bak, idx;

  if(w->showlen > w->maxlen)
    w->showlen = w->maxlen;
  idx = w->textpos + w->showlen;
  idx = MIN(w->maxlen, idx);
  bak = w->text[idx];
  w->text[idx] = 0;
  w_printstring (w->w.win, w->xborder, w->yborder, &w->text[w->textpos]);
  w->text[idx] = bak;

  if (w->mode == GetstrModeWithBorder) {
    wt_box3d (w->w.win, 0, 0, w->w.w, w->w.h);
    wt_box3d_press (w->w.win, 0, 0, w->w.w, w->w.h, 0);
  }

  fill_sublines(w);
  w_setmode (w->w.win, M_INVERS);
}

static void
draw_to_endpos(getstring_widget_t * w)
{
  if((w->textpos = w->textlen - w->showlen) > 0)
    w->cursor = w->showlen;
  else
  {
    w->textpos = 0;
    w->cursor = w->textlen;
  }
  w->copypos = w->cursor;
  getstring_draw(w);
}

/* return total widget size */
static long
getstring_query_geometry (widget_t * _w, long *xp, long *yp, long *wdp, long *htp)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;

  *xp = w->w.x;
  *yp = w->w.y;
  *wdp = w->char_width * w->showlen + 2 * w->xborder;
  *htp = w->font->height + 2 * w->yborder;
  return 0;
}

/* scroll the text field contents to left */
static void
scroll_left(getstring_widget_t *w)
{
  w->textpos++;
  w_bitblk(w->w.win, 
    w->xborder + w->char_width, w->yborder,
    w->char_width * (w->showlen - 1), w->font->height,
    w->xborder, w->yborder);
  w_printchar(w->w.win,
    w->xborder + w->char_width * (w->showlen-1), w->yborder,
    w->text[w->textpos + w->showlen - 1]);
}

/* scroll the text field contents to right */
static void
scroll_right(getstring_widget_t *w)
{
  w->textpos--;
  w_bitblk(w->w.win, 
    w->xborder, w->yborder,
    w->char_width * (w->showlen - 1), w->font->height,
    w->xborder + w->char_width, w->yborder);
  w_printchar(w->w.win, w->xborder, w->yborder, w->text[w->textpos]);
}

/* delete one character from the text string left of the cursor
 * and update the text field accordingly (scroll if needed).
 * check for whether there's something to delete is done elsewhere!
 */
static void
backspace_key(getstring_widget_t *w)
{
  int idx;

  /* move the rest of the characters */
  idx = w->textpos + w->cursor;
  while(idx <= w->textlen)
  {
    /* overlap, so better do manually, copy zero too... */
    w->text[idx-1] = w->text[idx];
    idx++;
  }
  w->textlen--;

  if(w->cursor <= 0)
  {
    w->textpos--;
    return;
  }
  /* need to move editfield contents? */
  if(w->textpos + w->cursor <= w->textlen && w->cursor < w->showlen)
  {
    idx = w->textlen - w->textpos - w->cursor + 1;
    if(idx > w->showlen - w->cursor)
      idx = w->showlen - w->cursor;
    w_bitblk(w->w.win, 
      w->xborder + w->char_width * w->cursor, w->yborder,
      w->char_width * idx, w->font->height,
      w->xborder + w->char_width * (w->cursor - 1), w->yborder);
  }
  /* need to replace last character with a new one? */
  if(w->textlen >= w->textpos + w->showlen)
    w_printchar(w->w.win,
      w->xborder + w->char_width * (w->showlen-1), w->yborder,
      w->text[w->textpos + w->showlen - 1]);
  else
  {
    /* or with a 'line'? */
    w_setmode(w->w.win, M_CLEAR);
    w_pbox(w->w.win,
      w->xborder + w->char_width * (w->textlen - w->textpos), w->yborder,
      w->char_width, w->font->height);
    w_setmode(w->w.win, M_INVERS);
    w_hline (w->w.win,
      w->xborder + w->char_width * (w->textlen - w->textpos) + 1,
      w->yborder + w->font->height - 1,
      w->xborder + w->char_width * (w->textlen - w->textpos + 1) - 1);
  }
  w->cursor--;
}

/* add one character to the text string right of the cursor
 * and update the text field accordingly (scroll if needed)
 */
static void
add_key(getstring_widget_t *w, int key)
{
  int idx;

  if(w->textlen >= w->maxlen)				/* enough space? */
    return;
 
  idx = w->textlen;
  while(idx >= w->textpos + w->cursor)
  {
    /* overlap, so better do manually, copy zero too... */
    w->text[idx+1] = w->text[idx];
    idx--;
  }
  w->text[w->textpos + w->cursor] = key;
  w->textlen++;

  if(w->cursor >= w->showlen)	/* at the end of editfield -> just scroll */
  {
    /* have to trick because editfield and string don't correlate before */
    scroll_left(w);
    return;
  }
  /* not at the text end? */
  if(w->textpos + w->cursor < w->textlen)
  {
    idx = w->textlen - w->textpos - w->cursor - 1;
    if(idx >= w->showlen - w->cursor)
      idx = w->showlen - w->cursor - 1;
    w_bitblk(w->w.win, 
      w->xborder + w->char_width * w->cursor, w->yborder,
      w->char_width * idx, w->font->height,
      w->xborder + w->char_width * (w->cursor + 1), w->yborder);
  }
  w_printchar(w->w.win, w->xborder + w->char_width * w->cursor, w->yborder, key);
  w->cursor++;
}

/* delegate key events */
static void
process_key(getstring_widget_t *w, int key)
{
  switch(key)
  {
    /* escape sequences */
    case '\b':
      if(w->cursor < 1 && w->textpos < 1)		/* nothing to do? */
        return;
      backspace_key(w);
      break;

    case KEY_DEL:
      if(w->textpos + w->cursor >= w->textlen)		/* nothing to do? */
        return;
      if(w->cursor < w->showlen)
      {
	w->cursor++;
	backspace_key(w);
      }
      else
      {
	/* no need to move editfield contents... */
	key = w->textpos + w->cursor;
	do
	{
	  w->text[key] = w->text[key+1];
	} while(++key < w->textlen);
	w->textlen--;
      }
      break;

    case KEY_LEFT:
      if(w->cursor > 0 || w->textpos > 0)
        if(w->cursor <= 0)
	  scroll_right(w);
	else
	  w->cursor--;
      break;

    case KEY_RIGHT:
      if(w->textpos + w->cursor < w->textlen)
        if(w->cursor >= w->showlen)
	  scroll_left(w);
	else
	  w->cursor++;
      break;

    case KEY_CLEAR:
      if(w->textlen < 1)				/* nothing to do? */
        return;
      memcpy(Buffer, w->text, w->textlen);		/* copy to buffer */
      w->text[0] = w->textpos = w->cursor = w->textlen = 0;
      fill_sublines(w);
      break;

    case KEY_HOME:
      if(w->textpos == 0)
        w->cursor = 0;
      /*
       * this was w->cursor > 0. but w->cursor is the position in the visible
       * edit field, not in the string! kay.
       */
      if(w->textpos > 0)
      {
      	w->textpos = w->cursor = 0;
	getstring_draw(w);
      }
      break;

    case KEY_END:
      if(w->textpos + w->cursor < w->textlen)		/* something to do? */
        if(w->textpos + w->showlen < w->textlen)
          draw_to_endpos(w);
	else
	  w->cursor = w->textlen - w->textpos;
      break;

    case '\r':
    case '\n':
      if(w->ok_cb)
	w->ok_cb(w->text, w->textpos + w->cursor);
      break;

    default:
      /* printable? */
      if(key >= ' ')
        add_key(w, key);
  }
}

/* paint text between x and copypos. set copypos to x */
static void
paint_text(getstring_widget_t *w, int x)
{
  if(x != w->copypos && x <= w->textlen)
  {
    if(x > w->copypos)
      w_pbox(w->w.win, w->xborder + w->char_width * w->copypos,
         w->yborder, w->char_width * (x - w->copypos), w->font->height);
    else
      w_pbox(w->w.win, w->xborder + w->char_width * x,
         w->yborder, w->char_width * (w->copypos - x), w->font->height);
    w->copypos = x;
  }
}

/* process painting with mouse at intervals */
static void
timer_callback (long _w)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;
  short mx, my;

  if (w_querymousepos (w->w.win, &mx, &my))
    return;

  mx = (mx - w->xborder) / w->char_width;
  paint_text(w, mx);
  w->timer = wt_addtimeout (100, timer_callback, _w);
}

/* show/remove cursor */
static inline void cursor_invert(getstring_widget_t *w)
{
  int d = (w->mode == GetstrModeWithBorder) ? 3 : 2;
  w_vline (w->w.win, w->xborder + w->char_width * w->cursor, d, w->w.h - d -1);
}

/* check cursor position etc. */
inline static void cursor_position(getstring_widget_t *w, int x)
{
  if(x + w->textpos > w->textlen)
    x = w->textlen - w->textpos;
  w->copypos = w->cursor = x;
}

/* process widget events */
static long
getstring_event (widget_t * _w, WEVENT * ev)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;
  int len;

  switch (ev->type)
    {
    case EVENT_MPRESS:
      /* remove selection if any */
      cursor_invert(w);
      paint_text(w, w->cursor);

      /* new cursor postions */
      cursor_position(w, (ev->x - w->xborder) / w->char_width);
      cursor_invert(w);

      /* text painting (selection...) */
      if(ev->key == BUTTON_LEFT)
	timer_callback((long)_w);
      else
        /* paste buffer to cursor position */
        if(ev->key == BUTTON_RIGHT && BufferLenght)
        {
	  cursor_invert(w);
	  len = -1;
	  while(++len < BufferLenght)
	    add_key(w, Buffer[len]);
	  cursor_position(w, w->cursor);
	  cursor_invert(w);
        }
      break;

    case EVENT_ACTIVE:
      cursor_invert(w);
      break;

    case EVENT_INACTIVE:
      cursor_invert(w);
    case EVENT_MRELEASE:
      /* end painting and copy selected area to buffer */
      if(w->copypos != w->cursor && ev->key == BUTTON_LEFT)
      {
        /* copy painted text into internal (static) buffer */
	if(w->cursor > w->copypos)
	  len = w->cursor - w->copypos;
	else
	  len = w->copypos - w->cursor;
	if(len >= BUFFER_SIZE)
	  len = BUFFER_SIZE-1;
        memcpy(Buffer, &w->text[w->textpos + MIN(w->cursor, w->copypos)], len);
	BufferLenght = len;
	Buffer[len] = 0;
      }
      if(w->timer)
        wt_deltimeout(w->timer);
      break;

    case EVENT_KEY:
      /* remove selection if any */
      cursor_invert(w);
      if(w->copypos != w->cursor)
      {
	if(w->cursor > w->copypos)
	  len = w->cursor - w->copypos;
	else
	{
	  len = w->copypos - w->cursor;
	  w->cursor = w->copypos;
	}
	while(len--)
          backspace_key(w);

	/* just remove the selection? */
        if((ev->key & 0xff) == '\b' || (ev->key & 0xff) == KEY_DEL)
	{
          cursor_position(w, w->cursor);
	  cursor_invert(w);
	  return 0;
	}
      }
      /* insert a key (changes cursor position) */
      process_key(w, ev->key & 0xff);
      cursor_position(w, w->cursor);
      cursor_invert(w);
      break;
    }
  return 0;
}

/* ------------------------- */

static long
getstring_init (void)
{
  return 0;
}

static int
getstring_loadfont (getstring_widget_t *w, char *fname)
{
  WFONT *fp = w_loadfont (fname);
  if (!fp)
    return -1;
  w->font = fp;
  strncpy (w->fontname, fname, sizeof (w->fontname));
  w->fontname[sizeof (w->fontname) - 1] = '\0';
  w->char_width = w_strlen (w->font, " ");
  return 0;
}

static widget_t *
getstring_create (widget_class_t * cp)
{
  getstring_widget_t *wp = malloc (sizeof (getstring_widget_t));
  if (!wp)
    return NULL;
  memset (wp, 0, sizeof (getstring_widget_t));
  wp->w.class = wt_getstring_class;
  if (getstring_loadfont (wp, FONT_NAME))
    {
      free (wp);
      return NULL;
    }
  wp->xborder = XBORDER;
  wp->yborder = YBORDER;

  return (widget_t *) wp;
}

static long
getstring_delete (widget_t * _w)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;

  if (w->is_realized)
    w_delete (w->w.win);
  w_unloadfont (w->font);
  free (w);
  return 0;
}

static long
getstring_close (widget_t * _w)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;

  if (w->is_realized && w->is_open)
    {
      w_close (w->w.win);
      w->is_open = 0;
    }
  return 0;
}

static long
getstring_open (widget_t * _w)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;

  if (w->is_realized && !w->is_open)
    {
      w_open (w->w.win, w->w.x, w->w.y);
      w->is_open = 1;
    }
  return 0;
}

static long
getstring_addchild (widget_t * parent, widget_t * w)
{
  return -1;
}

static long
getstring_delchild (widget_t * parent, widget_t * w)
{
  return -1;
}

static long
getstring_realize (widget_t * _w, WWIN * parent)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;
  long x, y, wd, ht;

  if (w->is_realized || !(w->text && w->showlen && w->maxlen))
    return -1;

  getstring_query_geometry (_w, &x, &y, &wd, &ht);
  w->w.x = x;
  w->w.y = y;
  w->w.w = wd;
  w->w.h = ht;

  w->w.win = wt_create_window (parent, wd, ht,
	W_NOBORDER | W_MOVE | EV_KEYS | EV_MOUSE | EV_ACTIVE);
  if (!w->w.win)
    return -1;
  w->w.win->user_val = (long) w;

  w_setfont (w->w.win, w->font);
  /* cursor to last character & redraw the widget */
  draw_to_endpos (w);
  w->is_realized = 1;
  w->is_open = 1;
  w_open (w->w.win, w->w.x, w->w.y);
  return 0;
}

static long
getstring_setopt (widget_t * _w, long key, void *val)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;
  int needredraw = 0;
  short mask = 0;

  switch (key)
    {
    case WT_XPOS:
      w->w.x = *(long *) val;
      mask |= WT_CHANGED_POS;
      if (w->is_realized)
	  w_move (w->w.win, w->w.x, w->w.y);
      break;

    case WT_YPOS:
      w->w.y = *(long *) val;
      mask |= WT_CHANGED_POS;
      if (w->is_realized)
	  w_move (w->w.win, w->w.x, w->w.y);
      break;

    case WT_FONT:
      if (getstring_loadfont (w, (char *) val))
	  return -1;
      w_setfont (w->w.win, w->font);
      needredraw = 1;
      break;

    case WT_STRING_ADDRESS:
      w->text = (char *) val;
      w->textlen = strlen (w->text);
      needredraw = 1;
      break;

    case WT_STRING_LENGTH:
      w->maxlen = *(long *) val;
      if(w->showlen > w->maxlen)
        w->showlen = w->maxlen;
      needredraw = 1;
      break;

    case WT_STRING_WIDTH:
      w->showlen = *(long *) val;
      if(w->maxlen && w->showlen > w->maxlen)
        w->showlen = w->maxlen;
      needredraw = 1;
      break;

    case WT_ACTION_CB:
      w->ok_cb = val;
      break;

    case WT_MODE:
      if (w->is_realized)
        return -1;
      w->mode = *(long *)val;
      w->xborder = XBORDER;
      w->yborder = YBORDER;
      if (w->mode == GetstrModeWithBorder) {
        w->xborder += 2;
        w->yborder += 2;
      }
      break;

    default:
      return -1;
    }
  if (mask && w->is_realized)
    wt_change_notify (_w, mask);
  if (w->is_realized && needredraw)
    getstring_draw (w);

  return 0;
}

static long
getstring_getopt (widget_t * _w, long key, void *val)
{
  getstring_widget_t *w = (getstring_widget_t *) _w;

  switch (key)
    {
    case WT_XPOS:
      *(long *) val = w->w.x;
      break;

    case WT_YPOS:
      *(long *) val = w->w.y;
      break;

    case WT_WIDTH:
      *(long *) val = w->w.w;
      break;

    case WT_HEIGHT:
      *(long *) val = w->w.h;
      break;

    case WT_FONT:
      (char *) val = w->fontname;
      break;

    case WT_STRING_ADDRESS:
      (char *) val = w->text;
      break;

    case WT_STRING_LENGTH:
      *(long *) val = w->maxlen;
      break;

    case WT_STRING_WIDTH:
      *(long *) val = w->showlen;
      break;

    case WT_ACTION_CB:
      *(long *)val = (long)w->ok_cb;
      break;

    case WT_MODE:
      *(long *)val = w->mode;
      break;

    default:
      return -1;
    }
  return 0;
}

static long
getstring_changes (widget_t * w, widget_t * w2, short changes)
{
  return 0;
}

static widget_class_t _wt_getstring_class =
{
  "getstring", 0,
  getstring_init,
  getstring_create,
  getstring_delete,
  getstring_close,
  getstring_open,
  getstring_addchild,
  getstring_delchild,
  getstring_realize,
  getstring_query_geometry,
  getstring_setopt,
  getstring_getopt,
  getstring_event,
  getstring_changes,
  getstring_changes
};

widget_class_t *wt_getstring_class = &_wt_getstring_class;
