/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * incomplete HTML widget.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

/*
 * we have two font types, a proportional font (FONT_N*) for normal text and
 * and a monospaced one (FONT_C*) for <TT> and the like.
 *
 * for each of them are four variants: roman, bold, italic and bold-italic.
 *
 * Ooops. With 8 fonts the per-client-font-limit is reached! Should be
 * increased definitely!
 */
#define FONT_N_DEF	"lucidat12"
#define FONT_C_DEF	"cour12"

#define FONT_BFLAG	1
#define FONT_IFLAG	2

#define FONT_N		(0)
#define FONT_NB		(FONT_N | FONT_BFLAG)
#define FONT_NI		(FONT_N | FONT_IFLAG)
#define FONT_NBI	(FONT_N | FONT_BFLAG | FONT_IFLAG)

#define FONT_C		(4)
#define FONT_CB		(FONT_C | FONT_BFLAG)
#define FONT_CI		(FONT_C | FONT_IFLAG)
#define FONT_CBI	(FONT_C | FONT_BFLAG | FONT_IFLAG)

#define FONT_MAX	(8)

/*
 * how deep lists can be nested
 */
#define LIST_STACK_SZ	(8)

/*
 * tokens for the text parser.
 */
#define TOK_SPACE	' '
#define TOK_TAB		'\t'
#define TOK_NL		'\n'
#define TOK_LT		'<'
#define TOK_GT		'>'
#define TOK_AMP		'&'
#define TOK_SEMICO	';'
#define TOK_QUOT	'"'
#define TOK_EQU		'='
#define TOK_EOF		(-1)

/************************* Lots of Typedefs **************************/

/*
 * list stack entry
 */
typedef struct {
	short n;
	short y;
	short indent;
} liststack_t;

/*
 * a rectangle
 */
typedef struct _rect_t {
	short x, y, wd, ht;
	short ismap;
	struct _rect_t *next;
} rect_t;

/*
 * an anchor
 */
typedef struct _anchor_t {
	short is_href;
	char *url;
	rect_t *recs;
	struct _anchor_t *next;
} anchor_t;

/*
 * html widget
 */
typedef struct {
	widget_t w;
	short is_open;
	short is_realized;

	short mode;

	WFONT *fonts[FONT_MAX];
	short cur_f;
	short textstyle;

	char *text, *cp, *cp2;
	uchar last_char_was_space;

	uchar preformat, underline, italic, bold, tt;
	short line_dist, line_ht;

	liststack_t lstack[LIST_STACK_SZ];
	short lsp;

	short area_x, area_y, area_w;
	short cur_x, cur_y;
	short indent;

	anchor_t *anchors, *cur_a, *sel_a;
	short sel_x, sel_y;

	long (*img_query_cb) (widget_t *w, char *url, int *wd, int *ht);
	void (*img_place_cb) (widget_t *w, long handle, int x, int y);
	void (*img_inval_cb) (widget_t *w, int discard);

	void (*anchor_cb) (widget_t *w, char *url, int x, int y, int pressed);
} html_widget_t;

/*
 * entry of table of special (ie escaped) characters
 */
typedef struct {
	char *name;
	short val;
} special_char_t;

/*
 * entry of argument vector for tag functions
 */
typedef struct {
	char *arg;
	char *val;
} arg_t;

/*
 * entry of tag function table
 */
typedef struct {
	char *name;
	long (*fn) (html_widget_t *, int argc, arg_t *argv);
} ftab_t;

/*************** Tag Function And Special Char Tables ***************/

/*
 * tag function prototypes
 */
static long tag_h	(html_widget_t *, int argc, arg_t *);
static long tag_not_h	(html_widget_t *, int argc, arg_t *);

static long tag_hr	(html_widget_t *, int argc, arg_t *);
static long tag_br	(html_widget_t *, int argc, arg_t *);
static long tag_p	(html_widget_t *, int argc, arg_t *);

static long tag_pre	(html_widget_t *, int argc, arg_t *);
static long tag_not_pre	(html_widget_t *, int argc, arg_t *);
static long tag_b	(html_widget_t *, int argc, arg_t *);
static long tag_not_b	(html_widget_t *, int argc, arg_t *);
static long tag_i	(html_widget_t *, int argc, arg_t *);
static long tag_not_i	(html_widget_t *, int argc, arg_t *);
static long tag_u	(html_widget_t *, int argc, arg_t *);
static long tag_not_u	(html_widget_t *, int argc, arg_t *);
static long tag_tt	(html_widget_t *, int argc, arg_t *);
static long tag_not_tt	(html_widget_t *, int argc, arg_t *);

static long tag_ul	(html_widget_t *, int argc, arg_t *);
static long tag_not_ul	(html_widget_t *, int argc, arg_t *);
static long tag_li	(html_widget_t *, int argc, arg_t *);
static long tag_dt	(html_widget_t *, int argc, arg_t *);
static long tag_dd	(html_widget_t *, int argc, arg_t *);

static long tag_area	(html_widget_t *, int argc, arg_t *);
static long tag_font	(html_widget_t *, int argc, arg_t *);

static long tag_img	(html_widget_t *, int argc, arg_t *);

static long tag_a	(html_widget_t *, int argc, arg_t *);
static long tag_not_a	(html_widget_t *, int argc, arg_t *);

/*
 * tag function table. must be alphabetically sorted.
 */
static const ftab_t tag_funcs[] = {
	{ "/a",		tag_not_a },
	{ "/b",		tag_not_b },
	{ "/dir",	tag_not_ul },
	{ "/dl",	tag_not_ul },
	{ "/h1",	tag_not_h },
	{ "/h2",	tag_not_h },
	{ "/h3",	tag_not_h },
	{ "/h4",	tag_not_h },
	{ "/h5",	tag_not_h },
	{ "/h6",	tag_not_h },
	{ "/i",		tag_not_i },
	{ "/menu",	tag_not_ul },
	{ "/ol",	tag_not_ul },
	{ "/pre",	tag_not_pre },
	{ "/tt",	tag_not_tt },
	{ "/u",		tag_not_u },
	{ "/ul",	tag_not_ul },

	{ "a",		tag_a },
	{ "area",	tag_area },
	{ "b",		tag_b },
	{ "br",		tag_br },
	{ "dd",		tag_dd },
	{ "dir",	tag_ul },
	{ "dl",		tag_ul },
	{ "dt",		tag_dt },
	{ "font",	tag_font },
	{ "h1",		tag_h },
	{ "h2",		tag_h },
	{ "h3",		tag_h },
	{ "h4",		tag_h },
	{ "h5",		tag_h },
	{ "h6",		tag_h },
	{ "hr",		tag_hr },
	{ "i",		tag_i },
	{ "img",	tag_img },
	{ "li",		tag_li },
	{ "menu",	tag_ul },
	{ "ol",		tag_ul },
	{ "p",		tag_p },
	{ "pre",	tag_pre },
	{ "tt",		tag_tt },
	{ "u",		tag_u },
	{ "ul",		tag_ul }
};

/*
 * special character table. a bit imcomplete at the moment...
 * must be alphabetically sorted.
 */
static const special_char_t special_chars[] = {
	{ "amp",	'&' },
	{ "gt",		'>' },
	{ "lt",		'<' },
	{ "quot",	'"' }
};

/************************ Utility Funtions ********************************/

static void
setfont (html_widget_t *w)
{
	short idx;

	idx = w->tt ? FONT_C : FONT_N;
	if (w->bold)
		idx |= FONT_BFLAG;
	if (w->italic)
		idx |= FONT_IFLAG;

	w->cur_f = idx;
	w_setfont (w->w.win, w->fonts[idx]);
}

static void
settextstyle (html_widget_t *w)
{
	short style;

	style = w->underline ? F_UNDERLINE : 0;
	w->textstyle = style;
	w_settextstyle (w->w.win, style);
}

static int
loadfonts (html_widget_t *w, char *fbase, WFONT **fp)
{
	char fname[64];

	sprintf (fname, "%s.wfnt", fbase);
	if (!(fp[FONT_N] = w_loadfont (fname)))
		return -1;

	sprintf (fname, "%sb.wfnt", fbase);
	if (!(fp[FONT_NB] = w_loadfont (fname)))
		fp[FONT_NB] = fp[FONT_N];

	sprintf (fname, "%si.wfnt", fbase);
	if (!(fp[FONT_NI] = w_loadfont (fname)))
		fp[FONT_NI] = fp[FONT_N];

	sprintf (fname, "%sbi.wfnt", fbase);
	if (!(fp[FONT_NBI] = w_loadfont (fname)))
		fp[FONT_NBI] = fp[FONT_NB];

	return 0;
}

static void
unloadfonts (WFONT **fp)
{
	int i, k;

	for (i = 0; i < 4; ++i) {
		for (k = 0; k < i; ++k) {
			if (fp[k] == fp[i])
				break;
		}
		if (k >= i) {
			w_unloadfont (fp[i]);
		}
	}
}

/*
 * find the entry in tag_funcs the belongs to the tag 'name'.
 */
static const ftab_t *
lookup_tag_fn (char *name)
{
	int l, r, m, i;

	l = 0;
	r = sizeof (tag_funcs) / sizeof (*tag_funcs) - 1;
	do {
		m = (l+r) >> 1;
		i = strcmp (name, tag_funcs[m].name);
		if (i == 0)
			return &tag_funcs[m];
		if (i < 0) {
			r = m-1;
		} else {
			l = m+1;
		}
	} while (l <= r);
	return NULL;
}

/*
 * find the entry in special_chars the belongs to the special character
 * 'name'.
 */
static const special_char_t *
lookup_special_char (char *name)
{
	int l, r, m, i;

	l = 0;
	r = sizeof (special_chars) / sizeof (*special_chars) - 1;
	do {
		m = (l+r) >> 1;
		i = strcmp (name, special_chars[m].name);
		if (i == 0)
			return &special_chars[m];
		if (i < 0) {
			r = m-1;
		} else {
			l = m+1;
		}
	} while (l <= r);
	return NULL;
}

/******************** Anchor Support Functions **********************/

/*
 * add a new anchor to 'w'.
 */
static void
anchor_add (html_widget_t *w, char *url, int is_href)
{
	anchor_t *ap;

	ap = malloc (sizeof (anchor_t));
	if (ap) {
		ap->url = strdup (url);
		if (!url) {
			free (ap);
			return;
		}
		ap->is_href = is_href;
		ap->recs = NULL;
		ap->next = w->anchors;
		w->cur_a = w->anchors = ap;
	}
}

/*
 * add a rectangle to the current anchor (w->cur_a)
 */
static void
anchor_add_rect (html_widget_t *w, int x, int y, int wd, int ht, int ismap)
{
	rect_t *rp;

	if (!w->cur_a)
		return;
	if ((rp = w->cur_a->recs) && !ismap && !rp->ismap &&
	    rp->x + rp->wd == x && rp->y == y && rp->ht == ht) {
		/*
		 * can join the rectangles
		 */
		rp->wd += wd;
		return;
	}
	rp = malloc (sizeof (rect_t));
	if (rp) {
		rp->x  = x;
		rp->y  = y;
		rp->wd = wd;
		rp->ht = ht;
		rp->ismap = ismap;
		rp->next = w->cur_a->recs;
		w->cur_a->recs = rp;
	}
}

/*
 * free all the anchors of the html widget 'w'
 */
static void
anchor_free_all (html_widget_t *w)
{
	anchor_t *ap, *next_ap;
	rect_t *rp, *next_rp;

	for (ap = w->anchors; ap; ap = next_ap) {
		for (rp = ap->recs; rp; rp = next_rp) {
			next_rp = rp->next;
			free (rp);
		}
		next_ap = ap->next;
		free (ap->url);
		free (ap);
	}
	w->anchors = NULL;
	w->cur_a = NULL;
	w->sel_a = NULL;
}

/*
 * Find the position of a NAME-anchor that matches the partial URL (#something)
 * int 'url'.
 */
static anchor_t *
anchor_position (html_widget_t *w, char *url, int *x, int *y)
{
	int xmin, ymin;
	anchor_t *ap;
	rect_t *rp;
	char *cp;

	if (!(url = strchr (url, '#'))) {
		return NULL;
	}
	for (ap = w->anchors; ap; ap = ap->next) {
		if (!ap->is_href) {
			cp = strchr (ap->url, '#');
			if (!strcmp (cp, url))
				break;
		}
	}
	if (ap && ap->recs) {
		xmin = INT_MAX;
		ymin = INT_MAX;
		for (rp = ap->recs; rp; rp = rp->next) {
			if (rp->x < xmin)
				xmin = rp->x;
			if (rp->y < ymin)
				ymin = rp->y;
		}
		*x = xmin;
		*y = ymin;
		return ap;
	}
	return NULL;
}

/*
 * find the HREF-anchor (and rectangle) that contains point (x,y)
 */
static anchor_t *
anchor_find (html_widget_t *w, int x, int y, rect_t **rect)
{
	anchor_t *ap;
	rect_t *rp;

	for (ap = w->anchors; ap; ap = ap->next) {
		if (!ap->is_href)
			continue;
		for (rp = ap->recs; rp; rp = rp->next) {
			if ((ushort)(x - rp->x) < (ushort)rp->wd &&
			    (ushort)(y - rp->y) < (ushort)rp->ht) {
				if (rect)
					*rect = rp;
				return ap;
			}
		}
	}
	return NULL;
}

/*
 * highlight the anchor (highlighting it twice must return to
 * non-highlighted state).
 */
static void
anchor_highlight (html_widget_t *w, anchor_t *ap)
{
	rect_t *rp;

	w_setmode (w->w.win, M_INVERS);
	for (rp = ap->recs; rp; rp = rp->next) {
		w_pbox (w->w.win, rp->x, rp->y, rp->wd, rp->ht);
	}
}

/************************* HTML Parser ********************************/

/*
 * get the next lookahead-token from the input string
 */
static int
gettoken (html_widget_t *w, int preformat)
{
	char *cp = w->cp;

	if (!*cp) {
		return TOK_EOF;
	}
	if (!isspace (*cp) || preformat) {
		w->cp2 = w->cp + 1;
		return (uchar)*cp;
	}
	while (*cp && isspace (*cp)) {
		++cp;
	}
	w->cp2 = cp;
	return TOK_SPACE;
}

/*
 * eat the lookahead token
 */
static inline void
eattoken (html_widget_t *w)
{
	w->cp = w->cp2;
}

/*
 * parse an encoded character:
 *	&something;
 */
static int
parse_special (html_widget_t *w)
{
	char name[128];
	const special_char_t *spc;
	char *cp = name;
	int c;

	while ((c = gettoken (w, 0)) != TOK_EOF && c != TOK_SEMICO) {
		eattoken (w);
		if (cp - name < sizeof (name)-1)
			*cp++ = tolower (c);
	}
	eattoken (w);
	*cp = '\0';

	if ((spc = lookup_special_char (name)))
		return spc->val;
	/*
	 * no such special char
	 */
	return TOK_SPACE;
}

static inline int
convert_special (html_widget_t *w, int c)
{
	return (c != TOK_AMP) ? c : parse_special (w);
}

/*
 * parse a tag name/argument:
 *	"something with spaces"
 *	something
 */
static char *
parse_name (html_widget_t *w, char *cp, int sz)
{
	int c, end;

	if (sz <= 0)
		return NULL;

	while ((c = gettoken (w, 0)) == TOK_SPACE)
		eattoken (w);

	switch (c) {
	case TOK_EOF:
		eattoken (w);
		return NULL;

	case TOK_GT:
		return cp;

	case TOK_QUOT:
		eattoken (w);
		end = TOK_QUOT;
		break;

	default:
		eattoken (w);
		*cp++ = convert_special (w, c);
		--sz;
		end = TOK_SPACE;
		break;
	}
	while ((c = gettoken (w, 0)) != TOK_EOF &&
	       c != TOK_GT && c != TOK_EQU && c != end) {
		eattoken (w);
		c = convert_special (w, c);
		if (sz > 0) {
			*cp++ = c;
			--sz;
		}
	}
	if (c != TOK_GT && c != TOK_EQU)
		eattoken (w);

	if (sz > 0) {
		*cp++ = '\0';
		return cp;
	}
	return NULL;
}

#define ARGBUF	1024
#define ARGMAX	10

/*
 * parse and evaluate a tag: <tag name1[=val1] name2[=val2] ...>
 */
static int
parse_tag (html_widget_t *w)
{
	char buf[ARGBUF];
	arg_t args[ARGMAX], *argp = args;
	char *cp2, *cp = buf;
	const ftab_t *ftab;
	int c, i;

	for (i = 0; i < ARGMAX; ++argp) {
		argp->arg = cp;
		argp->val  = NULL;

		if (!(cp2 = parse_name (w, cp, ARGBUF-(cp-buf))))
			goto bad;

		if (cp2 != cp) {
			cp = cp2;
			++i;
		}
		while ((c = gettoken (w, 0)) == TOK_SPACE)
			eattoken (w);

		if (c == TOK_EOF)
			goto bad;
		if (c == TOK_GT) {
			eattoken (w);
			break;
		}
		if (c == TOK_EQU) {
			eattoken (w);
			argp->val = cp;
			if (!(cp = parse_name (w, cp, ARGBUF-(cp-buf))))
				goto bad;
		}
	}
	if (i <= ARGMAX) {
		for (cp = args[0].arg; *cp; ++cp)
			*cp = tolower (*cp);
		if ((ftab = lookup_tag_fn (args[0].arg)))
			return (*ftab->fn) (w, i, args);
		return -1;
	}

	/*
	 * something went wrong: eat the rest of the tag
	 */
bad:
	while ((c = gettoken (w, 0)) != TOK_EOF && c != TOK_GT)
		eattoken (w);
	eattoken (w);
	return -1;
}

/*
 * skip leading whitspace in 'cp'
 */
static inline char *
skipspace (char *cp)
{
	while (*cp == TOK_SPACE)
		++cp;
	return cp;
}

/*
 * do a linebreak
 */
static inline void
dolinebreak (html_widget_t *w)
{
	w->cur_x = w->area_x + w->indent;
	w->cur_y += MAX (w->line_dist, w->line_ht);
	w->line_ht = 0;
}

/*
 * print out what is in the buffer and advance x and y position
 * accordingly.
 */
static void
doprint (html_widget_t *w, char *buf)
{
	int y, wd;

	if (!w->preformat && w->cur_x <= w->area_x + w->indent) {
		/*
		 * skip leading space if at beginning of line and
		 * not in <PRE> mode.
		 */
		buf = skipspace (buf);
	}
	if (*buf) {
		wd = w_strlen (w->fonts[w->cur_f], buf);
		y = w->cur_y + w->line_dist - w->fonts[w->cur_f]->height;
		/*
		 * dont print if its not inside the window
		 */
		if (w->mode == HTMLModeGrow || w->cur_y < w->w.h) {
			w_printstring (w->w.win, w->cur_x, y, buf);
		}
		if (w->cur_a) {
			anchor_add_rect (w, w->cur_x, w->cur_y, wd,
				w->line_dist, 0);
		}
		w->cur_x += wd;
	}
}

/*
 * parse and print out the html text.
 */
static int
parse_html (html_widget_t *w, char *code)
{
	char buf[100], *cp, *breakchar;
	int c, i, wd;

	if (code) {
		/*
		 * append 'code' to the current position.
		 */
		w->cp = w->cp2 = code;
	} else {
		/*
		 * reset to intial state and print out w->text.
		 */
		w->cp = w->cp2 = w->text;

		w->preformat = 0;
		w->italic = 0;
		w->bold = 0;
		w->underline = 0;
		w->tt = 0;

		setfont (w);
		settextstyle (w);
		anchor_free_all (w);

		w->last_char_was_space = 0;

		w->line_ht = 0;
		w->indent = 0;
		w->cur_x = w->area_x;
		w->cur_y = w->area_y;

		w->lsp = 0;

		if (w->area_w > 0 && w->w.y + w->w.h - w->area_y > 0) {
			/*
			 * clear the text output area
			 */
			w_setmode (w->w.win, M_CLEAR);
			w_pbox (w->w.win, w->area_x, w->area_y, w->area_w,
				w->w.y + w->w.h - w->area_y);
		}
		if (!w->text)
			return 0;
	}
	wd = 0;
	breakchar = cp = buf;
	while ((c = gettoken (w, w->preformat)) != TOK_EOF) {
		eattoken (w);
		if (c == TOK_LT) {
			/*
			 * a tag
			 */
			*cp = 0;
			doprint (w, buf);
			breakchar = cp = buf;
			wd = 0;
			parse_tag (w);
			continue;
		}
		switch ((c = convert_special (w, c))) {
		case TOK_TAB:
			/*
			 * horizontal tab. will only occure in <PRE> mode.
			 */
			*cp = 0;
			doprint (w, buf);
			breakchar = cp = buf;

			wd = 8*w->fonts[w->cur_f]->widths['M'];
			i = w->cur_x - w->area_x + wd;
			w->cur_x = w->area_x + i - i % wd;

			while (w->cur_x > w->area_x + w->area_w) {
				w->cur_x -= (w->area_w - w->indent);
				w->cur_y += MAX (w->line_dist, w->line_ht);
				w->line_ht = 0;
			}
			wd = 0;
			break;

		case TOK_NL:
			/*
			 * linefeed. will only occure in <PRE> mode.
			 */
			*cp = 0;
			doprint (w, buf);
			breakchar = cp = buf;
			wd = 0;
			dolinebreak (w);
			break;

		case TOK_SPACE:
			/*
			 * space.
			 */
			if (!w->preformat && w->last_char_was_space) {
				/*
				 * compress multiple spaces when not in <PRE>
				 * mode
				 */
				break;
			}
			w->last_char_was_space = 1;
			if (w->cur_x + wd <= w->area_x + w->area_w) {
				breakchar = cp;
			}
			*cp++ = TOK_SPACE;
			wd += w->fonts[w->cur_f]->widths[TOK_SPACE];
			break;

		default:
			/*
			 * nonspace character
			 */
			w->last_char_was_space = 0;
			*cp++ = c;
			wd += w->fonts[w->cur_f]->widths[(uchar)c];
			break;
		}
		if (w->cur_x + wd > w->area_x + w->area_w) {
			/*
			 * exceeded the right border. need a line-break.
			 */
			*cp = 0;
			if (breakchar > buf && breakchar < cp) {
				/*
				 * up to 'breakchar' everything in the
				 * buffer will fit into current line.
				 *
				 * 'breakchar' points to a character where
				 * we can nicely break the line (ie a space
				 * between two words).
				 */
				*breakchar = 0;
				doprint (w, buf);
				*breakchar = TOK_SPACE;
				if (!w->preformat) {
					breakchar = skipspace (breakchar);
				}
				wd = w_strlen (w->fonts[w->cur_f], breakchar);
				strncpy (buf, breakchar, cp-breakchar);
				cp = buf + (cp-breakchar);
				dolinebreak (w);
			} else if (w->cur_x > w->area_x + w->indent) {
				/*
				 * we have already something written to the
				 * current line and current word doesn't
				 * fit. break line and then print what we've
				 * got.
				 */
				dolinebreak (w);
				doprint (w, buf);
				breakchar = cp = buf;
				wd = 0;
			} else {
				/*
				 * the line is empty but the word is so long
				 * that it doesn't fit into one line. print
				 * it anyway and perform a linebreak
				 * afterwards.
				 */
				doprint (w, buf);
				breakchar = cp = buf;
				wd = 0;
				dolinebreak (w);
			}
		} else if (cp - buf >= sizeof (buf) - 2) {
			/*
			 * buffer full. flush it
			 */
			*cp = 0;
			doprint (w, buf);
			breakchar = cp = buf;
			wd = 0;
		}
	}
	*cp = 0;
	doprint (w, buf);
	return 0;
}

/************************ Tag Functions ********************************/

/*
 * process <h1>, ..., <h6>
 */
static long
tag_h (html_widget_t *w, int argc, arg_t *argv)
{
	int level = argv[0].arg[1];

	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist;

	switch (level) {
	case '1':
		if (++w->underline == 1) {
			settextstyle (w);
		}
	case '2':
		if (++w->bold == 1) {
			setfont (w);
		}
		break;
	case '3':
		if (++w->italic == 1) {
			setfont (w);
		}
		break;
	case '4':
	case '5':
	case '6':
		break;
	}
	return 0;
}

/*
 * process </h1>, ..., </h6>
 */
static long
tag_not_h (html_widget_t *w, int argc, arg_t *argv)
{
	char level = argv[0].arg[2];

	switch (level) {
	case '1':
		if (w->underline > 0 && --w->underline == 0) {
			settextstyle (w);
		}
	case '2':
		if (w->bold > 0 && --w->bold == 0) {
			setfont (w);
		}
		break;
	case '3':
		if (w->italic > 0 && --w->italic == 0) {
			setfont (w);
		}
		break;
	case '4':
	case '5':
	case '6':
		break;
	}
	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist/2;
	return 0;
}

static long
tag_hr (html_widget_t *w, int argc, arg_t *argv)
{
	int margin;

	margin = (w->area_w - w->indent)*5/100;
	if (margin < 0)
		margin = 0;

	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist;
	w_setmode (w->w.win, M_DRAW);
	w_hline (w->w.win, w->area_x + w->indent + margin,
		 w->cur_y, w->area_x + w->area_w - margin - 1);
	w->cur_y += w->line_dist;
	return 0;
}

static long
tag_br (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	return 0;
}

static long
tag_p (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist/2;
	return 0;
}

static long
tag_pre (html_widget_t *w, int argc, arg_t *argv)
{
	++w->preformat;
	return 0;
}

static long
tag_not_pre (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->preformat > 0)
		--w->preformat;
	return 0;
}

static long
tag_b (html_widget_t *w, int argc, arg_t *argv)
{
	if (++w->bold == 1)
		setfont (w);
	return 0;
}

static long
tag_not_b (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->bold > 0 && --w->bold == 0)
		setfont (w);
	return 0;
}

static long
tag_i (html_widget_t *w, int argc, arg_t *argv)
{
	if (++w->italic == 1)
		setfont (w);
	return 0;
}

static long
tag_not_i (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->italic > 0 && --w->italic == 0)
		setfont (w);
	return 0;
}

static long
tag_u (html_widget_t *w, int argc, arg_t *argv)
{
	if (++w->underline == 1)
		settextstyle (w);
	return 0;
}

static long
tag_not_u (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->underline > 0 && --w->underline == 0)
		settextstyle (w);
	return 0;
}

static long
tag_tt (html_widget_t *w, int argc, arg_t *argv)
{
	if (++w->tt == 1)
		setfont (w);
	return 0;
}

static long
tag_not_tt (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->tt > 0 && --w->tt == 0)
		setfont (w);
	return 0;
}

/*
 * set the area in which to print out text (HTML extension)
 */
static long
tag_area (html_widget_t *w, int argc, arg_t *argv)
{
	int i, j, changed = 0;

	for (i = 1; i < argc; ++i) {
		j = argv[i].val ? atol (argv[i].val) : 0;
		if (!strcasecmp ("x", argv[i].arg)) {
			w->area_x = j;
			changed = 1;
		} else if (!strcasecmp ("y", argv[i].arg)) {
			w->area_y = j;
			changed = 1;
		} else if (!strcasecmp ("width", argv[i].arg)) {
			w->area_w = j;
			changed = 1;
		}
	}
	if (changed) {
		w->cur_x = w->area_x + w->indent;
		w->cur_y = w->area_y;
	}
	return 0;
}

/*
 * set fonts (HTML extension)
 */
static long
tag_font (html_widget_t *w, int argc, arg_t *argv)
{
	WFONT *fp[4];
	int i, j;
	char *cp;

	for (i = 1; i < argc; ++i) {
		if (!strcasecmp ("teletype", argv[i].arg)) {
			j = FONT_C;
		} else if (!strcasecmp ("normal", argv[i].arg)) {
			j = FONT_N;
		} else {
			continue;
		}
		cp = (j == FONT_C) ? FONT_C_DEF : FONT_N_DEF;
		cp = argv[i].val ? argv[i].val : cp;

#if 1
		/*
		 * work around 8 fonts-per-client-limit :(
		 */
		unloadfonts (&w->fonts[j]);
		if (loadfonts (w, cp, fp)) {
			loadfonts (w, j == FONT_C ? FONT_C_DEF:FONT_N_DEF, fp);
			continue;
		}
#else
		if (loadfonts (w, cp, fp))
			continue;
		unloadfonts (&w->fonts[j]);
#endif
		w->fonts[j+0] = fp[0];
		w->fonts[j+1] = fp[1];
		w->fonts[j+2] = fp[2];
		w->fonts[j+3] = fp[3];
	}
	i = MAX (w->fonts[FONT_N]->height, w->fonts[FONT_C]->height);
	if (i < w->line_dist && w->cur_x > w->area_x + w->indent) {
		/*
		 * line distance is smaller for the new fonts
		 * and the current line is not empty.
		 */
		if (w->line_dist > w->line_ht)
			w->line_ht = w->line_dist;
	}
	w->line_dist = i;
	return 0;
}

/*
 * process <ul>, <ol>, <dl>, <menu> or <dir>.
 */
static long
tag_ul (html_widget_t *w, int argc, arg_t *argv)
{
	liststack_t *lsp;

	if (++w->lsp >= LIST_STACK_SZ)
		return 0;

	lsp = &w->lstack[w->lsp];

	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	if (!strcmp ("ol", argv[0].arg)) {
		lsp->indent = 5*w->fonts[w->cur_f]->widths['M'];
		lsp->n = 1;
	} else if (!strcmp ("dl", argv[0].arg)) {
		lsp->indent = w->area_w/4;
		lsp->n = 0;
		w->cur_x -= lsp->indent;
	} else {
		lsp->indent = 3*w->fonts[w->cur_f]->widths['M'];
		lsp->n = 0;
	}
	lsp->y = w->cur_y-1; /* must be != w->cur_y */

	w->indent += lsp->indent;
	w->cur_x  += lsp->indent;
	return 0;
}

/*
 * process </ul>, </ol>, </dl>, </menu> or </dir>.
 */
static long
tag_not_ul (html_widget_t *w, int argc, arg_t *argv)
{
	if (w->lsp <= 0)
		return 0;

	w->indent -= w->lstack[w->lsp].indent;
	w->cur_x  -= w->lstack[w->lsp].indent;

	if (w->cur_x > w->area_x + w->indent) {
		dolinebreak (w);
	}
	--w->lsp;
	return 0;
}

/*
 * process <li> inside <ol>, <ul>, <menu> or <dir>.
 */
static long
tag_li (html_widget_t *w, int argc, arg_t *argv)
{
	liststack_t *lsp;
	int x, y, r;

	if (w->lsp >= LIST_STACK_SZ)
		return 0;

	lsp = &w->lstack[w->lsp];

	if (w->cur_x > w->area_x + w->indent || lsp->y == w->cur_y) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist/4;
	lsp->y    = w->cur_y;

	if (lsp->n == 0) {
		/*
		 * <UL>, <MENU> or <DIR>
		 */
		x = w->cur_x - lsp->indent/2;
		y = w->cur_y + w->line_dist - w->fonts[w->cur_f]->height/2 - 1;
		r = MIN (lsp->indent, w->fonts[w->cur_f]->height)/4;

		w_setmode (w->w.win, M_DRAW);
		w_pcircle (w->w.win, x, y, r);
	} else {
		/*
		 * <OL>
		 */
		char num[10];
		sprintf (num, "%3d. ", lsp->n);

		x = w->cur_x - w_strlen (w->fonts[w->cur_f], num);
		y = w->cur_y + w->line_dist - w->fonts[w->cur_f]->height;

		w_printstring (w->w.win, x, y, num);
		++lsp->n;
	}
	return 0;
}

/*
 * process <dt> inside <dl>.
 */
static long
tag_dt (html_widget_t *w, int argc, arg_t *argv)
{
	liststack_t *lsp;
	
	if (w->lsp >= LIST_STACK_SZ)
		return 0;

	lsp = &w->lstack[w->lsp];

	if (w->cur_x > w->area_x + w->indent - lsp->indent ||
	    lsp->y == w->cur_y) {
		dolinebreak (w);
	}
	w->cur_y += w->line_dist/4;
	w->cur_x = w->area_x + w->indent - lsp->indent;

	lsp->n = w->cur_y-1; /* must be != w->cur_y */
	lsp->y = w->cur_y;
	return 0;
}

/*
 * process <dd> inside <dl>.
 */
static long
tag_dd (html_widget_t *w, int argc, arg_t *argv)
{
	liststack_t *lsp;

	if (w->lsp >= LIST_STACK_SZ)
		return 0;

	lsp = &w->lstack[w->lsp];

	if (w->cur_x > w->area_x + w->indent || lsp->n == w->cur_y) {
		dolinebreak (w);
	}
	w->cur_x = w->area_x + w->indent;
	lsp->n   = w->cur_y;
	return 0;
}

/*
 * ignores ALIGN (does always top-alignment).
 */
static long
tag_img (html_widget_t *w, int argc, arg_t *argv)
{
	char *alt = NULL, *src = NULL;
	int x, y, wd, ht, i, ismap = 0;
	long handle;

	if (!w->img_query_cb)
		return 0;
	for (i = 1; i < argc; ++i) {
		if (!strcasecmp ("src", argv[i].arg)) {
			src = argv[i].val;
		} else if (!strcasecmp ("alt", argv[i].arg)) {
			alt = argv[i].val;
		} else if (!strcasecmp ("ismap", argv[i].arg)) {
			ismap = 1;
		}
	}
	if (src) {
		/*
		 * ask user for the size of the picture
		 */
		handle = (*w->img_query_cb) ((widget_t *)w, src, &wd, &ht);
		if (!handle)
			return 0;
		/*
		 * break the line if picture does not fit and something is
		 * in the line already
		 */
		if (w->cur_x + wd + 2 > w->area_x + w->area_w &&
		    w->cur_x > w->area_x + w->indent) {
			dolinebreak (w);
		}
		x = w->cur_x + 1;
		y = w->cur_y + 1;

		if (w->cur_a) {
			anchor_add_rect (w, x, y, wd, ht, ismap);
		}
		/*
		 * draw an "outline" of the picture.
		 */
		wt_box3d (w->w.win, x, y, wd, ht);
		/*
		 * draw the ALT string (if present and fits) into the outline
		 */
		if (alt && w->fonts[FONT_N]->height <= ht-4 &&
		    w_strlen (w->fonts[FONT_N], alt) <= wd-4) {
			w_setfont (w->w.win, w->fonts[FONT_N]);
			w_settextstyle (w->w.win, 0);
			wt_text (w->w.win, w->fonts[FONT_N], alt,
				x, y, wd, ht, AlignCenter);
			setfont (w);
			settextstyle (w);
		}
		/*
		 * tell user where the picture is placed (he is responsible
		 * for getting the picture and putblock()ing it to the
		 * position we tell him).
		 */
		if (w->img_place_cb) {
			(*w->img_place_cb) ((widget_t *)w, handle, x, y);
		}

		w->last_char_was_space = 0;

		ht += 2;
		wd += 2;
		if (ht > w->line_ht)
			w->line_ht = ht;
		if ((w->cur_x += wd) >= w->area_x + w->area_w)
			dolinebreak (w);
	}
	return 0;
}

/*
 * an anchor. turn on underlining for HREF's.
 */
static long
tag_a (html_widget_t *w, int argc, arg_t *argv)
{
	char *url = NULL;
	int i, is_href = 0;

	for (i = 1; i < argc; ++i) {
		if (!strcasecmp ("name", argv[i].arg)) {
			url = argv[i].val;
			is_href = 0;
		} else if (!strcasecmp ("href", argv[i].arg)) {
			url = argv[i].val;
			is_href = 1;
		}
	}
	if (url) {
		anchor_add (w, url, is_href);
		if (is_href && ++w->underline == 1)
			settextstyle (w);
	}
	return 0;
}

static long
tag_not_a (html_widget_t *w, int argc, arg_t *argv)
{
	if (!w->cur_a) {
		/* Hmmm. Probably a missing <A ...> */
		return 0;
	}
	if (w->cur_a->is_href && w->underline > 0 && --w->underline == 0)
		settextstyle (w);
	w->cur_a = NULL;
	return 0;
}

/************************** Widget Code ******************************/


static long html_query_geometry (widget_t *, long *, long *, long *, long *);

static long
html_init (void)
{
	return 0;
}

static widget_t *
html_create (widget_class_t *cp)
{
	html_widget_t *wp;
	int i;

	wp = malloc (sizeof (html_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (html_widget_t));
	wp->w.class = wt_html_class;

	if (loadfonts (wp, FONT_N_DEF, &wp->fonts[FONT_N])) {
		free (wp);
		return NULL;
	}
	if (loadfonts (wp, FONT_C_DEF, &wp->fonts[FONT_C])) {
		unloadfonts (&wp->fonts[FONT_N]);
		free (wp);
		return NULL;
	}
	i = MAX (wp->fonts[FONT_N]->height, wp->fonts[FONT_C]->height);
	wp->line_dist = i;

	wp->mode = HTMLModeFixed;

	return (widget_t *)wp;
}

static long
html_delete (widget_t *_w)
{
	html_widget_t *w = (html_widget_t *)_w;

	if (w->is_realized)
		w_delete (w->w.win);
	unloadfonts (&w->fonts[FONT_N]);
	unloadfonts (&w->fonts[FONT_C]);
	free (w);
	return 0;
}

static long
html_close (widget_t *_w)
{
	html_widget_t *w = (html_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->w.win);
		w->is_open = 0;
	}
	return 0;
}

static long
html_open (widget_t *_w)
{
	html_widget_t *w = (html_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
html_addchild (widget_t *parent, widget_t *w)
{
	return -1;
}

static long
html_delchild (widget_t *parent, widget_t *w)
{
	return -1;
}

static long
html_realize (widget_t *_w, WWIN *parent)
{
	html_widget_t *w = (html_widget_t *)_w;
	long x, y, wd, ht;

	if (w->is_realized)
		return -1;

	html_query_geometry (_w, &x, &y, &wd, &ht);
	w->w.w = wd;
	w->w.h = ht;

	if (w->area_w <= 0)
		w->area_w = w->w.w;

	w->w.win = wt_create_window (parent, wd, ht,
					W_NOBORDER|W_MOVE|EV_MOUSE);
	if (!w->w.win)
		return -1;
	w->w.win->user_val = (long)w;

	setfont (w);
	parse_html (w, NULL);

	w->is_realized = 1;
	w->is_open = 1;
	w_open (w->w.win, w->w.x, w->w.y);
	return 0;
}

static long
html_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	html_widget_t *w = (html_widget_t *)_w;

	*xp = w->w.x;
	*yp = w->w.y;
	*wdp = w->w.w > 0 ? w->w.w : 100;
	*htp = w->w.h > 0 ? w->w.h : 100;
	return 0;
}

static long
html_setopt (widget_t *_w, long key, void *val)
{
	html_widget_t *w = (html_widget_t *)_w;
	short mask = 0;
	char *cp;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		if (w->w.w != *(long *)val) {
			w->w.w = *(long *)val;
			mask |= WT_CHANGED_SIZE;
			if (w->is_realized) {
				/*
				 * tell the user the positions of the inline
				 * images are becoming invalid.
				 */
				if (w->img_inval_cb)
					(*w->img_inval_cb) (_w, 0);
				parse_html (w, NULL);
			}
		}
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_LABEL:
		if (!(cp = strdup (val))) {
			return -1;
		}
		if (w->text) {
			/*
			 * tell the user the positions of the inline
			 * images are becoming invalid.
			 */
			if (w->is_realized && w->img_inval_cb)
				(*w->img_inval_cb) (_w, 1);
			free (w->text);
		}
		w->text = cp;
		if (w->is_realized) {
			parse_html (w, NULL);
		}
		break;

	case WT_APPEND:
		if (!w->text) {
			w->text = strdup ((char *)val);
			if (!w->text)
				return -1;
		} else {
			char *cp;
			int len1, len2;

			len1 = strlen (w->text);
			len2 = strlen ((char *)val);
			cp = realloc (w->text, len1 + len2 + 1);
			if (!cp)
				return -1;
			memcpy (cp+len1, val, len2);
			cp[len1+len2] = 0;
			w->text = cp;
		}
		if (w->is_realized) {
			parse_html (w, (char *)val);
		}
		break;

	case WT_MODE:
		w->mode = *(long *)val;
		break;

	case WT_QUERY_CB:
		w->img_query_cb = val;
		break;

	case WT_PLACE_CB:
		w->img_place_cb = val;
		break;

	case WT_INVAL_CB:
		w->img_inval_cb = val;
		break;

	case WT_ACTION_CB:
		w->anchor_cb = val;
		break;
	
	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	return 0;
}

static long
html_getopt (widget_t *_w, long key, void *val)
{
	html_widget_t *w = (html_widget_t *)_w;
	int x, y;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_MODE:
		*(long *)val = w->mode;
		break;

	case WT_POSITION:
		if (anchor_position (w, (char *)val, &x, &y)) {
			*(long *)val = y;
			break;
		}
		return -1;
	
	case WT_QUERY_CB:
		*(void **)val = w->img_query_cb;
		break;

	case WT_PLACE_CB:
		*(void **)val = w->img_place_cb;
		break;

	case WT_INVAL_CB:
		*(void **)val = w->img_inval_cb;
		break;

	case WT_ACTION_CB:
		*(void **)val = w->anchor_cb;
		break;

	case WT_LABEL:
		*(char **)val = w->text;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
html_event (widget_t *_w, WEVENT *ev)
{
	html_widget_t *w = (html_widget_t *)_w;
	rect_t *rp;

	switch (ev->type) {
	case EVENT_MPRESS:
		if (!(ev->key & BUTTON_LEFT))
			return -1;
		w->sel_a = anchor_find (w, ev->x, ev->y, &rp);
		if (w->sel_a) {
			if (rp->ismap) {
				w->sel_x = ev->x - rp->x;
				w->sel_y = ev->y - rp->y;
			} else {
				w->sel_x = -1;
				w->sel_y = -1;
			}
			anchor_highlight (w, w->sel_a);
			if (w->anchor_cb) {
				(*w->anchor_cb) (_w, w->sel_a->url,
					w->sel_x, w->sel_y, 1);
			}
		}
		break;

	case EVENT_MRELEASE:
		if (!(ev->key & BUTTON_LEFT))
			return -1;
		if (w->sel_a) {
			anchor_t *ap;

			ap = anchor_find (w, ev->x, ev->y, NULL);
			anchor_highlight (w, w->sel_a);
			if (w->anchor_cb) {
				/*
				 * if the user has left the anchor
				 * when releasing button he will get -1,
				 * and 0 if the mouse is still over the
				 * same anchor.
				 */
				(*w->anchor_cb) (_w, w->sel_a->url,
					w->sel_x, w->sel_y,
					w->sel_a == ap ? 0 : -1);
			}
			w->sel_a = NULL;
		}
		break;

	default:
		return -1;
	}
	return 0;
}

static long
html_changes (widget_t *w, widget_t *w2, short changes)
{
	return 0;
}

static widget_class_t _wt_html_class = {
	"html", 0,
	html_init,
	html_create,
	html_delete,
	html_close,
	html_open,
	html_addchild,
	html_delchild,
	html_realize,
	html_query_geometry,
	html_setopt,
	html_getopt,
	html_event,
	html_changes,
	html_changes
};

widget_class_t *wt_html_class = &_wt_html_class;
