/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * shell widget (toplevel window with title bar etc).
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

typedef struct {
	widget_t w;
	char title[64];
	short is_open;
	short is_realized;
	short is_iconified;
	short mode;
	WWIN *icon;
	WFONT *iconfont;
	short icon_x, icon_y;
	void (*close_cb) (widget_t *w);
} shell_widget_t;


static long shell_query_geometry (widget_t *, long *, long *, long *, long *);

static void
shell_iconify (shell_widget_t *w)
{
	short wd, ht, x, y;
	char *name;

	if (w->is_realized && !w->is_iconified) {
		w_querywindowpos (w->w.win, 1, &x, &y);
		w_close (w->w.win);
		w->w.x = x;
		w->w.y = y;

		if (w->icon_x == UNDEF || w->icon_y == UNDEF) {
			w->icon_x = x;
			w->icon_y = y;
		}

		name = *w->title ? w->title : "Icon";
		wd = w_strlen (w->iconfont, name) + w->iconfont->height/2;
		ht = w->iconfont->height;
		w->icon = w_create (wd, ht, W_MOVE|EV_MOUSE);
		if (!w->icon)
			return;
		w->icon->user_val = (long)w;

		w_setfont (w->icon, w->iconfont);
		wt_text (w->icon, w->iconfont, name, 0, 0, wd, ht, 0);
		w_open (w->icon, w->icon_x, w->icon_y);
		w->is_iconified = 1;
	}
}
	
static void
shell_deiconify (shell_widget_t *w)
{
	if (w->is_realized && w->is_iconified) {
		w_querywindowpos (w->icon, 1, &w->icon_x, &w->icon_y);
		w_delete (w->icon);
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_iconified = 0;
	}
}

static long
shell_init (void)
{
	return 0;
}

static widget_t *
shell_create (widget_class_t *cp)
{
	shell_widget_t *wp = malloc (sizeof (shell_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (shell_widget_t));
	wp->w.x = UNDEF;
	wp->w.y = UNDEF;
	wp->w.class = wt_shell_class;
	wp->iconfont = w_loadfont ("fixed13.wfnt");
	if (!wp->iconfont) {
		free (wp);
		return NULL;
	}
	wp->mode = ShellModeMain;
	wp->icon_x = UNDEF;
	wp->icon_y = UNDEF;
	return (widget_t *)wp;
}

static long
shell_delete (widget_t *_w)
{
	shell_widget_t *w = (shell_widget_t *)_w;
	widget_t *wp, *next;

	for (wp = w->w.childs; wp; wp = next) {
		next = wp->next;
		(*wp->class->delete) (wp);
	}
	if (w->is_realized)
		w_delete (w->w.win);
	w_unloadfont (w->iconfont);
	free (w);
	return 0;
}

static long
shell_close (widget_t *_w)
{
	shell_widget_t *w = (shell_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->w.win);
		w->is_open = 0;
	}
	return 0;
}

static long
shell_open (widget_t *_w)
{
	shell_widget_t *w = (shell_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->w.win, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
shell_addchild (widget_t *parent, widget_t *w)
{
	wt_add_before (parent, parent->childs, w);
	return 0;
}

static long
shell_delchild (widget_t *parent, widget_t *w)
{
	wt_remove (w);
	return 0;
}

static long
shell_realize (widget_t *_w, WWIN *parent)
{
	shell_widget_t *w = (shell_widget_t *)_w;
	widget_t *wp;
	long x, y, wd, ht;
	short flags;

	if (w->is_realized)
		return -1;

	shell_query_geometry (_w, &x, &y, &wd, &ht);
	w->w.w = wd;
	w->w.h = ht;

	flags = W_MOVE|W_CONTAINER;
	switch (w->mode) {
	case ShellModeMain:
		flags |= W_TITLE|W_CLOSE|W_ICON;
		break;

	case ShellModePopup:
		break;

	case ShellModeNoBorder:
		flags |= W_NOBORDER;
		break;
	}
	w->w.win = wt_create_window (parent, wd, ht, flags);
	if (!w->w.win)
		return -1;
	w->w.win->user_val = (long)w;
	for (wp = w->w.childs; wp; wp = wp->next) {
		if ((*wp->class->realize) (wp, w->w.win) < 0) {
			w_delete (w->w.win);
			return -1;
		}
	}
	w->is_realized = 1;
	w->is_open = 1;
	if (w->title[0])
		w_settitle (w->w.win, w->title);
	w_open (w->w.win, w->w.x, w->w.y);
	return 0;
}

static long
shell_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	shell_widget_t *w = (shell_widget_t *)_w;
	widget_t *wp;
	long x, y, wd, ht;

	*xp = w->w.x;
	*yp = w->w.y;
	if (w->w.w > 0 && w->w.h > 0) {
		*wdp = w->w.w;
		*htp = w->w.h;
		return 0;
	}
	*wdp = 0;
	*htp = 0;
	for (wp = w->w.childs; wp; wp = wp->next) {
		(*wp->class->query_geometry) (wp, &x, &y, &wd, &ht);

		if (x+wd > *wdp)
			*wdp = x+wd;
		if (y+ht > *htp)
			*htp = y+ht;
	}
	*wdp = w->w.w > 0 ? w->w.w : (*wdp > 0 ? *wdp : 100);
	*htp = w->w.h > 0 ? w->w.h : (*htp > 0 ? *htp : 100);
	return 0;
}

static long
shell_setopt (widget_t *_w, long key, void *val)
{
	shell_widget_t *w = (shell_widget_t *)_w;
	short mask = 0;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->w.win, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		w->w.w = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_LABEL:
		strncpy (w->title, (char *)val, sizeof (w->title));
		w->title[sizeof (w->title) - 1] = 0;
		if (w->is_realized) {
			w_settitle (w->w.win, w->title);
		}
		break;

	case WT_MODE:
		w->mode = *(long *)val;
		break;

	case WT_ACTION_CB:
		w->close_cb = val;
		break;

	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	return 0;
}

static long
shell_getopt (widget_t *_w, long key, void *val)
{
	shell_widget_t *w = (shell_widget_t *)_w;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_LABEL:
		strcpy ((char *)val, w->title);
		break;

	case WT_MODE:
		*(long *)val = w->mode;
		break;

	case WT_ACTION_CB:
		*(void **)val = w->close_cb;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
shell_event (widget_t *_w, WEVENT *ev)
{
	shell_widget_t *w = (shell_widget_t *)_w;

	if (ev->win == w->w.win) {
		switch (ev->type) {
		case EVENT_GADGET:
			switch (ev->key) {
			case GADGET_CLOSE:
				if (w->close_cb) {
					(*w->close_cb) (_w);
				} else {
					wt_break (-1);
				}
				break;

			case GADGET_ICON:
				shell_iconify (w);
				break;
			}
			break;
		}
	} else if (w->is_iconified && w->icon == ev->win) {
		switch (ev->type) {
		case EVENT_MPRESS:
			if (ev->key & BUTTON_LEFT)
				shell_deiconify (w);
			break;
		}
	}
	return 0;
}

static long
shell_changes (widget_t *w, widget_t *w2, short changes)
{
	return 0;
}

static widget_class_t _wt_shell_class = {
	"shell", 0,
	shell_init,
	shell_create,
	shell_delete,
	shell_close,
	shell_open,
	shell_addchild,
	shell_delchild,
	shell_realize,
	shell_query_geometry,
	shell_setopt,
	shell_getopt,
	shell_event,
	shell_changes,
	shell_changes
};

widget_class_t *wt_shell_class = &_wt_shell_class;
