/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

static input_t *glob_inputs;
static timeout_t *glob_timeouts;

static short run_end = 0;
static short screen_wd, screen_ht;

static fd_set glob_rfdset, glob_wfdset, glob_efdset;

/**
 **
 ** INPUT HANDLER STUFF
 **
 **/

static inline void
wt_fdset_set (fd_set *s1, fd_set *s2)
{
	char *s, *d;
	int i;

	s = (char *)s2;
	d = (char *)s1;
	for (i = 0; i < sizeof (fd_set); ++i)
		*d++ |= *s++;
}

static inline int
wt_fdset_isset (fd_set *s1, fd_set *s2)
{
	char *s, *d;
	int i;

	s = (char *)s2;
	d = (char *)s1;
	for (i = 0; i < sizeof (fd_set); ++i) {
		if (*d++ & *s++)
			return 1;
	}
	return 0;
}

long
wt_addinput (fd_set *r, fd_set *w, fd_set *e,
	void (*h) (long, fd_set *, fd_set *, fd_set *),
	long arg)
{
	static long theid = 0;
	input_t *ip;

	ip = malloc (sizeof (input_t));
	if (!ip)
		return -1;

	FD_ZERO (&ip->rd);
	FD_ZERO (&ip->wr);
	FD_ZERO (&ip->ex);

	if (r) {
		wt_fdset_set (&glob_rfdset, r);
		ip->rd = *r;
	}
	if (w) {
		wt_fdset_set (&glob_wfdset, w);
		ip->wr = *w;
	}
	if (e) {
		wt_fdset_set (&glob_efdset, e);
		ip->ex = *e;
	}
	ip->arg = arg;
	ip->handler = h;
	ip->id = theid;
	ip->next = glob_inputs;
	glob_inputs = ip;

	if (++theid < 0)
		theid = 0;

	return ip->id;
}

void
wt_delinput (long handle)
{
	input_t **prev, *this;

	prev = &glob_inputs;
	for (this = *prev; this; prev = &this->next, this = *prev) {
		if (this->id == handle)
			break;
	}
	if (this) {
		*prev = this->next;
		free (this);

		FD_ZERO (&glob_rfdset);
		FD_ZERO (&glob_wfdset);
		FD_ZERO (&glob_efdset);
		for (this = glob_inputs; this; this = this->next) {
			wt_fdset_set (&glob_rfdset, &this->rd);
			wt_fdset_set (&glob_wfdset, &this->wr);
			wt_fdset_set (&glob_efdset, &this->ex);
		}
	}
}

static void
wt_do_input (fd_set *r, fd_set *w, fd_set *e)
{
	input_t *ip;

	for (ip = glob_inputs; ip; ip = ip->next) {
		if (wt_fdset_isset (&ip->rd, r) ||
		    wt_fdset_isset (&ip->wr, w) ||
		    wt_fdset_isset (&ip->ex, e))
			(*ip->handler) (ip->arg, r, w, e);
	}
}

/**
 **
 ** TIMEOUT STUFF
 **
 **/

static long
wt_gettime (void)
{
#ifdef __MINT__
	return (clock () * 1000L / CLK_TCK);
#else
	struct timeval tv;
	gettimeofday (&tv, NULL);
	return (tv.tv_sec*1000L + tv.tv_usec/1000L);
#endif
}

static void
wt_update_timeouts (void)
{
	static long last = 0;
	static short init = 1;
	long diff, curr;

	curr = wt_gettime ();
	if (init || !glob_timeouts || curr - last < 0) {
		last = curr;
		init = 0;
	} else {
		diff = curr - last;
		glob_timeouts->delta -= diff;
		last = curr;
	}
}

static long
wt_can_sleep (void)
{
	wt_update_timeouts ();
	if (glob_timeouts)
		return MAX (glob_timeouts->delta, 1);
	return 10000;
}

static void
wt_do_timeouts (void)
{
	timeout_t *tp;

	wt_update_timeouts ();
	while ((tp = glob_timeouts) && tp->delta <= 0) {
		glob_timeouts = tp->next;
		if (glob_timeouts)
			glob_timeouts->delta += tp->delta;
		(*tp->handler) (tp->arg);
		free (tp);
		wt_update_timeouts ();
	}
}

long
wt_addtimeout (long delta, void (*f) (long arg), long arg)
{
	static long theid = 0;
	timeout_t **prev, *this, *tp;

	if (delta <= 0)
		return -1;

	tp = malloc (sizeof (timeout_t));
	if (!tp)
		return -1;

	tp->handler = f;
	tp->arg = arg;
	tp->id = theid;

	if (++theid < 0)
		theid = 0;

	wt_update_timeouts ();
	prev = &glob_timeouts;
	for (this = *prev; this; prev = &this->next, this = *prev) {
		if (this->delta <= delta)
			delta -= this->delta;
		else {
			this->delta -= delta;
			break;
		}
	}
	tp->delta = delta;
	tp->next = this;
	*prev = tp;
	return tp->id;
}

void
wt_deltimeout (long handle)
{
	timeout_t **prev, *this, *freeme;

	prev = &glob_timeouts;
	for (this = *prev; this; prev = &this->next, this = *prev) {
		if (this->id == handle)
			break;
	}
	if (this) {
		freeme = this;
		*prev = this->next;
		if (this->next)
			this->next->delta += this->delta;
		free (this);
	}
}

/**
 **
 ** WIDGET STUFF
 **
 **/

inline widget_t *
wt_win2widget (WWIN *win)
{
	return (widget_t *)win->user_val;
}

inline WWIN *
wt_widget2win (widget_t *w)
{
	return w->win;
}

widget_t *
wt_init (void)
{
	WSERVER *serv;
	widget_t *top;

	serv = w_init ();
	if (!serv)
		return NULL;
	screen_wd = serv->width;
	screen_ht = serv->height;

	if (!wt_top_class->initialized) {
		if ((*wt_top_class->init) () < 0)
			return NULL;
		wt_top_class->initialized = 1;
	}
	top = (*wt_top_class->create) (wt_top_class);
	if (!top)
		return NULL;

	return top;
}

widget_t *
wt_create (widget_class_t *class, widget_t *parent)
{
	widget_t *wp;

	if (!class->initialized) {
		if ((*class->init) () < 0)
			return NULL;
		class->initialized = 1;
	}
	wp = (*class->create) (class);
	if (!wp)
		return NULL;
	if (parent && (*parent->class->addchild) (parent, wp) < 0) {
		(*class->delete) (wp);
		return NULL;
	}
	return wp;
}

long
wt_delete (widget_t *w)
{
	if (w->parent)
		(*w->parent->class->delchild) (w->parent, w);
	(*w->class->delete) (w);
	return 0;
}

long
wt_open (widget_t *w)
{
	return (*w->class->open) (w);
}

long
wt_close (widget_t *w)
{
	return (*w->class->close) (w);
}

long
wt_realize (widget_t *top)
{
	return (*top->class->realize) (top, NULL);
}

long
wt_run (void)
{
	WEVENT *ev;
	widget_t *wp;
	fd_set rfdset, wfdset, efdset;

	run_end = 0;
	while (!run_end) {
		rfdset = glob_rfdset;
		wfdset = glob_wfdset;
		efdset = glob_efdset;
		ev = w_queryevent (&rfdset, &wfdset, &efdset, wt_can_sleep ());
		wt_do_input (&rfdset, &wfdset, &efdset);
		wt_do_timeouts ();
		if (!ev)
			continue;
		if (ev->type == EVENT_GADGET && ev->key == GADGET_EXIT)
			run_end = -1;
		wp = wt_win2widget (ev->win);
		if (wp) {
			(*wp->class->event) (wp, ev);
		}
	}
	return run_end;
}

long
wt_break (long r)
{
	run_end = r;
	return 0;
}

long
wt_getopt (widget_t *w, ...)
{
	va_list args;
	long key, i;
	void *arg;

	va_start (args, w);
	for (i=1; (key = va_arg (args, long)) != WT_EOL; ++i) {
		arg = va_arg (args, void *);
		switch (key) {
		case WT_USRVAL:
			*(long *)arg = w->usrval;
			break;

		default:
			if ((*w->class->getopt) (w, key, arg) < 0)
				return i;
			break;
		}
	}
	va_end (args);
	return 0;
}

long
wt_setopt (widget_t *w, ...)
{
	va_list args;
	long key, i;
	void *arg;

	va_start (args, w);
	for (i=1; (key = va_arg (args, long)) != WT_EOL; ++i) {
		arg = va_arg (args, void *);
		switch (key) {
		case WT_USRVAL:
			w->usrval = (long)arg;
			break;

		default:
			if ((*w->class->setopt) (w, key, arg) < 0)
				return i;
			break;
		}
	}
	va_end (args);
	return 0;
}

/*
 * remove a widget from its widget tree
 */
void
wt_remove (widget_t *w)
{
	if (w->prev) {
		w->prev->next = w->next;
	} else if (w->parent) {
		w->parent->childs = w->next;
	}
	if (w->next) {
		w->next->prev = w->prev;
	}
}

/*
 * add the new window `neww' into the childlist of window parent
 * right after window `w'.
 */
void
wt_add_after (widget_t *parent, widget_t *w, widget_t *neww)
{
	neww->prev = w;
	neww->next = NULL;
	if (w) {
		if (w->next) {
			neww->next = w->next;
			w->next->prev = neww;
		}
		w->next = neww;
	}
	neww->parent = parent;
	if (!neww->prev) {
		parent->childs = neww;
	}
}

/*
 * add the new window `neww' into the childlist of window parent
 * right before window `w'.
 */
void
wt_add_before (widget_t *parent, widget_t *w, widget_t *neww)
{
	neww->next = w;
	neww->prev = NULL;
	if (w) {
		if (w->prev) {
			neww->prev = w->prev;
			w->prev->next = neww;
		}
		w->prev = neww;
	}
	neww->parent = parent;
	if (!neww->prev) {
		parent->childs = neww;
	}
}

/*
 * notify parent and childs of a widget configuration change
 */
void
wt_change_notify (widget_t *w, short changes)
{
	widget_t *wp;

	if (w->parent)
		(*w->parent->class->child_change) (w->parent, w, changes);
	if ((changes &= ~WT_CHANGED_POS)) {
		for (wp = w->childs; wp; wp = wp->next)
			(*wp->class->parent_change) (wp, w, changes);
	}
}

/*
 * create a new W window. This is needed because TeSche has messed up
 * w_createChild() in w1r3 so it is no impossible to craete childs of
 * the root window using w_createChild()...
 */
WWIN *
wt_create_window (WWIN *parent, short wd, short ht, short flags)
{
	if (parent == WROOT) {
		return w_create (wd, ht, flags);
	} else {
		return w_createChild (parent, wd, ht, flags);
	}
}
