/*
 * this file is part of "The W Toolkit".
 *
 * (W) 1996, Kay Roemer.
 *
 * viewport widget.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <Wlib.h>
#include <Wt.h>
#include "toolkit.h"

/*
 * scrollbar parameter scale factor
 */
#define SLSCALE	1024

/*
 * changes mask
 */
#define VP_SIZE		0x01
#define VP_CSIZE	0x02
#define VP_CPOS		0x04

typedef struct {
	widget_t w;
	short is_open;
	short is_realized;
	short is_locked;
	short sb_min_wd;
	short sb_min_ht;
	short cx, cy, cw, ch;
	short life_scroll;
	widget_t *vscroll;
	widget_t *hscroll;
	widget_t *child;
	WWIN *mainwin;
} viewp_widget_t;


static long viewp_query_geometry (widget_t *w, long *, long *, long *, long *);

static void
hscroll_callback (widget_t *w, int pos, int pressed)
{
	viewp_widget_t *vp = (viewp_widget_t *)w->parent;
	long x;

	if (pressed && !vp->life_scroll)
		return;

	x = pos * MIN (0, vp->w.w - vp->sb_min_wd - vp->cw) / SLSCALE;
	if (x != vp->cx && vp->child) {
		vp->is_locked = 1;
		(*vp->child->class->setopt) (vp->child, WT_XPOS, &x);
		vp->cx = x;
		vp->is_locked = 0;
	}
}

static void
vscroll_callback (widget_t *w, int pos, int pressed)
{
	viewp_widget_t *vp = (viewp_widget_t *)w->parent;
	long y;

	if (pressed && !vp->life_scroll)
		return;

	y = pos * MIN (0, vp->w.h - vp->sb_min_wd - vp->ch) / SLSCALE;
	if (y != vp->cy && vp->child) {
		vp->is_locked = 1;
		(*vp->child->class->setopt) (vp->child, WT_YPOS, &y);
		vp->cy = y;
		vp->is_locked = 0;
	}
}

static void
viewp_config (viewp_widget_t *w, short changes)
{
	long x, y, wd, ht;
	widget_t *child;

	w->is_locked = 1;

	if (changes & VP_SIZE) {
		wd = w->w.w - w->sb_min_wd + 1;
		ht = w->w.h - w->sb_min_wd + 1;
		x  = 0;
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_XPOS, &wd);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_YPOS, &x);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_HEIGHT, &ht);

		(*wt_scrollbar_class->setopt) (w->hscroll, WT_XPOS, &x);
		(*wt_scrollbar_class->setopt) (w->hscroll, WT_YPOS, &ht);
		(*wt_scrollbar_class->setopt) (w->hscroll, WT_WIDTH, &wd);

		/*
		 * resize w->w.win and w->mainwin
		 */
	}
	if ((child = w->child)) {
		(*child->class->query_geometry) (child, &x, &y, &wd, &ht);
		if (x > 0) {
			x = 0;
			(*child->class->setopt) (child, WT_XPOS, &x);
		} else if (x < w->w.w - w->sb_min_wd - wd) {
			x = MIN (0, w->w.w - w->sb_min_wd - wd);
			(*child->class->setopt) (child, WT_XPOS, &x);
		}
		if (y > 0) {
			y = 0;
			(*child->class->setopt) (child, WT_YPOS, &y);
		} else if (y < w->w.h - w->sb_min_wd - ht) {
			y = MIN (0, w->w.h - w->sb_min_wd - ht);
			(*child->class->setopt) (child, WT_YPOS, &y);
		}
		w->cx = x;
		w->cy = y;
		w->cw = wd;
		w->ch = ht;
	} else {
		w->cx = 0;
		w->cy = 0;
		w->cw = w->w.w - w->sb_min_wd;
		w->ch = w->w.h - w->sb_min_wd;
	}

	wd = w->w.w - w->sb_min_wd;
	ht = w->w.h - w->sb_min_wd;

	if (changes & (VP_CSIZE|VP_SIZE)) {
		x = wd * SLSCALE / w->cw;
		y = ht * SLSCALE / w->ch;

		(*wt_scrollbar_class->setopt) (w->hscroll, WT_SIZE, &x);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_SIZE, &y);

		x = - wd * SLSCALE / MIN (-1, wd - w->cw);
		y = - ht * SLSCALE / MIN (-1, ht - w->ch);

		(*wt_scrollbar_class->setopt) (w->hscroll, WT_PAGE_INC, &x);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_PAGE_INC, &y);

		x /= 4;
		y /= 4;

		(*wt_scrollbar_class->setopt) (w->hscroll, WT_LINE_INC, &x);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_LINE_INC, &y);
	}
	if (changes & (VP_CPOS|VP_SIZE)) {
		x = - w->cx * SLSCALE / MIN (-1, wd - w->cw);
		y = - w->cy * SLSCALE / MIN (-1, ht - w->ch);

		(*wt_scrollbar_class->setopt) (w->hscroll, WT_POSITION, &x);
		(*wt_scrollbar_class->setopt) (w->vscroll, WT_POSITION, &y);
	}
	w->is_locked = 0;
}

static long
viewp_init (void)
{
	return 0;
}

static widget_t *
viewp_create (widget_class_t *cp)
{
	long i;

	viewp_widget_t *wp = malloc (sizeof (viewp_widget_t));
	if (!wp)
		return NULL;
	memset (wp, 0, sizeof (viewp_widget_t));
	wp->w.class = wt_viewport_class;

	wp->life_scroll = ViewpModeLazyScroll;

	wp->hscroll = wt_create (wt_scrollbar_class, (widget_t *)wp);
	if (!wp->hscroll) {
		free (wp);
		return NULL;
	}

	wp->vscroll = wt_create (wt_scrollbar_class, (widget_t *)wp);
	if (!wp->vscroll) {
		wt_delete (wp->hscroll);
		free (wp);
		return NULL;
	}

	i = OrientHorz;
	(*wt_scrollbar_class->setopt) (wp->hscroll, WT_ORIENTATION, &i);

	/*
	 * query minimum extents of the scrollbar
	 */
	(*wt_scrollbar_class->getopt) (wp->vscroll, WT_HEIGHT, &i);
	wp->sb_min_ht = i;

	(*wt_scrollbar_class->getopt) (wp->vscroll, WT_WIDTH, &i);
	wp->sb_min_wd = i+1;

	(*wt_scrollbar_class->setopt) (wp->vscroll, WT_ACTION_CB,
		vscroll_callback);
	(*wt_scrollbar_class->setopt) (wp->hscroll, WT_ACTION_CB,
		hscroll_callback);

	return (widget_t *)wp;
}

static long
viewp_delete (widget_t *_w)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;
	widget_t *wp, *next;

	for (wp = w->w.childs; wp; wp = next) {
		next = wp->next;
		(*wp->class->delete) (wp);
	}
	if (w->is_realized) {
		w_delete (w->w.win);
		w_delete (w->mainwin);
	}
	free (w);
	return 0;
}

static long
viewp_close (widget_t *_w)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;

	if (w->is_realized && w->is_open) {
		w_close (w->mainwin);
		w->is_open = 0;
	}
	return 0;
}

static long
viewp_open (widget_t *_w)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;

	if (w->is_realized && !w->is_open) {
		w_open (w->mainwin, w->w.x, w->w.y);
		w->is_open = 1;
	}
	return 0;
}

static long
viewp_addchild (widget_t *_parent, widget_t *w)
{
	viewp_widget_t *parent = (viewp_widget_t *)_parent;

	wt_add_before (_parent, _parent->childs, w);
	if (w != parent->hscroll && w != parent->vscroll) {
		parent->child = w;
		if (parent->is_realized)
			viewp_config (parent, VP_CSIZE|VP_CPOS);
	}
	return 0;
}

static long
viewp_delchild (widget_t *_parent, widget_t *w)
{
	viewp_widget_t *parent = (viewp_widget_t *)_parent;

	wt_remove (w);
	if (w == parent->child) {
		parent->child = NULL;
		if (parent->is_realized)
			viewp_config (parent, VP_CSIZE|VP_CPOS);
	}
	return 0;
}

static long
viewp_realize (widget_t *_w, WWIN *parent)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;
	long x, y, wd, ht;

	if (w->is_realized)
		return -1;

	viewp_query_geometry (_w, &x, &y, &wd, &ht);
	w->w.w = wd;
	w->w.h = ht;

	wd -= w->sb_min_wd;
	ht -= w->sb_min_wd;

	w->mainwin = wt_create_window (parent, w->w.w, w->w.h,
		W_MOVE|W_NOBORDER|W_CONTAINER);
	if (!w->mainwin)
		return -1;

	w->w.win = wt_create_window (w->mainwin, wd, ht,
		W_MOVE|W_NOBORDER|W_CONTAINER);
	if (!w->w.win) {
		w_delete (w->mainwin);
		return -1;
	}
	w->w.win->user_val = (long)w;

	viewp_config (w, VP_SIZE|VP_CSIZE|VP_CPOS);

	if (w->child) {
		(*w->child->class->realize) (w->child, w->w.win);
	}
	(*wt_scrollbar_class->realize) (w->hscroll, w->mainwin);
	(*wt_scrollbar_class->realize) (w->vscroll, w->mainwin);

	w->is_realized = 1;
	w->is_open = 1;
	w_open (w->w.win, 0, 0);
	w_open (w->mainwin, w->w.x, w->w.y);
	return 0;
}

static long
viewp_query_geometry (widget_t *_w, long *xp, long *yp, long *wdp, long *htp)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;
	long x, y, wd, ht;
	
	*xp = w->w.x;
	*yp = w->w.y;
	if (w->w.w > 0 && w->w.h > 0) {
		*wdp = w->w.w;
		*htp = w->w.h;
		return 0;
	}
	wd = ht = 100;
	if (w->child)
		(*w->child->class->query_geometry) (w->child, &x, &y, &wd, &ht);

	x = w->sb_min_wd + w->sb_min_ht;
	*wdp = w->w.w > 0 ? MAX (w->w.w, x) : MAX (wd+w->sb_min_wd, x);
	*htp = w->w.h > 0 ? MAX (w->w.h, x) : MAX (ht+w->sb_min_wd, x);
	return 0;
}

static long
viewp_setopt (widget_t *_w, long key, void *val)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;
	short mask = 0;

	switch (key) {
	case WT_XPOS:
		w->w.x = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->mainwin, w->w.x, w->w.y);
		}
		break;

	case WT_YPOS:
		w->w.y = *(long *)val;
		mask |= WT_CHANGED_POS;
		if (w->is_realized) {
			w_move (w->mainwin, w->w.x, w->w.y);
		}
		break;

	case WT_WIDTH:
		w->w.w = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_HEIGHT:
		w->w.h = *(long *)val;
		mask |= WT_CHANGED_SIZE;
		break;

	case WT_MODE:
		w->life_scroll = *(long *)val;
		break;

	default:
		return -1;
	}
	if (mask && w->is_realized)
		wt_change_notify (_w, mask);
	return 0;
}

static long
viewp_getopt (widget_t *_w, long key, void *val)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;

	switch (key) {
	case WT_XPOS:
		*(long *)val = w->w.x;
		break;

	case WT_YPOS:
		*(long *)val = w->w.y;
		break;

	case WT_WIDTH:
		*(long *)val = w->w.w;
		break;

	case WT_HEIGHT:
		*(long *)val = w->w.h;
		break;

	case WT_MODE:
		*(long *)val = w->life_scroll;
		break;

	default:
		return -1;
	}
	return 0;
}

static long
viewp_event (widget_t *_w, WEVENT *ev)
{
	return -1;
}

static long
viewp_child_change (widget_t *_w, widget_t *child, short changes)
{
	viewp_widget_t *w = (viewp_widget_t *)_w;
	short mask = 0;

	if (w->child && w->child == child && w->is_realized && !w->is_locked) {
		if (changes & WT_CHANGED_SIZE)
			mask |= VP_CSIZE;
		if (changes & WT_CHANGED_POS)
			mask |= VP_CPOS;
		viewp_config (w, mask);
	}
	return 0;
}

static long
viewp_parent_change (widget_t *_w, widget_t *parent, short changes)
{
	return 0;
}

static widget_class_t _wt_viewp_class = {
	"viewport", 0,
	viewp_init,
	viewp_create,
	viewp_delete,
	viewp_close,
	viewp_open,
	viewp_addchild,
	viewp_delchild,
	viewp_realize,
	viewp_query_geometry,
	viewp_setopt,
	viewp_getopt,
	viewp_event,
	viewp_child_change,
	viewp_parent_change
};

widget_class_t *wt_viewport_class = &_wt_viewp_class;
