/**
*  Class BlockCypherHash<p>
*  -----------------------------------------------------------------------<p>
* Routines to support a simple secure hash based on 3-Way, with MD
*	strengthening as per MD5.  The algorithm is the first from Table 18.1
*	of Schneier, H[i] = E{H[i-1]}(M[i]) ^ M[i]
*  <P>
*  Coded Mr. Tines &lt;tines@windsong.demon.co.uk&gt; 1998
*  and released into the public domain
*  <P>
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
* BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*  <P>
* @author Mr. Tines
* @version 1.0 23-Dec-1998
*/

package uk.co.demon.windsong.crypt.mda;
import uk.co.demon.windsong.crypt.cea.*;
import uk.co.demon.windsong.crypt.mda.MDA;

public class BlockCypherHash implements MDA
{
    /**
    * Block cypher with keysize = blocksize
    */
    CEA cypher = null;

    /**
    * working buffer for hash intermediates
    */
	byte[] buf = null;
    /**
    * bits hashed mod 2^64
    */
    long bits;
    /**
    * batched up input
    */
	byte[] in = null;
    /**
    * bytes batched up
    */
	int over;
    /**
    * intermediate buffer
    */
    byte[] work = null;

    /**
    * Magic numbers to start the hash - taken from Blowfish S-boxes
    */
    private static final byte[] magic = {
        (byte)0xd1,0x31,0x0b,(byte)0xa6,
        (byte)0x98,(byte)0xdf,(byte)0xb5,(byte)0xac,
	    0x2f,(byte)0xfd,0x72,(byte)0xdb,
        (byte)0xd0,0x1a,(byte)0xdf,(byte)0xb7,
        (byte)0xb8,(byte)0xe1,(byte)0xaf,(byte)0xed,
        0x6a,0x26,0x7e,(byte)0x96,
        (byte)0xba,0x7c,(byte)0x90,0x45,
        (byte)0xf1,0x2c,0x7f,(byte)0x99 }; //256 bits

    /**
    * Default constructor uses 3-Way
    */
    public BlockCypherHash()
    {
        this(Cypher.getInstance(Cypher.TWAY));
    }

    /**
    * Constructs the hash engine from the cypher
    * @param block block cypher used
    */
    public BlockCypherHash(CEA block)
    {
        if(block.getKeysize() != block.getBlocksize())
        {
            RuntimeException e = new RuntimeException
                ("Assert block.getKeysize() == block.getBlocksize()");
            e.printStackTrace();
            System.exit(0);
        }
        cypher = block;
        buf = new byte[cypher.getBlocksize()];
        in = new byte[cypher.getBlocksize()];
	    work = new byte[cypher.getBlocksize()];
        init();
    }

    /**
    * zero all counts and load buffer with magic numbers
    */
    private void init()
    {
        for(int i=0; i<buf.length; ++i)
        {
            buf[i] = magic[i%magic.length];
        }
	    /* initialise the count */
        bits = 0;
        over = 0;
    }

    /**
    * H[i] = E{H[i-1]}(M[i]) ^ M[i]
    * @param h the H value
    * @param oh offset into h where our values start
    * @param m the M value
    * @param om offset into m where our values start
    */
    private void hash(byte[] h, int oh, byte[] m, int om)
    {
	    cypher.init(h, oh, false);
	    cypher.ecb(true, m, om, work, 0);
	    cypher.destroy();

	    for(int i=0; i<work.length; i++) h[i+oh] = (byte)(work[i] ^ m[i+om]);
    }

   /**
    * Feeds a batch of bytes into the hash
    * @param data the byte values
    * @param offset the first byte index to take
    * @param length the number of bytes to take
    */
    public void update(byte[] data, int offset, int length)
    {
        bits += length<<3;

	    /* handle any left-over bytes in scratch space */
	    int t = over;
	    if(over>0)
	    {
		    t = cypher.getBlocksize() - t; /* space left */
		    /* if there's still not enough, exit */
		    if(length < t)
            {
                System.arraycopy(data, offset, in, over, length);
                return;
            }
            System.arraycopy(data, offset, in, over, t);
            hash(buf, 0, in, 0);

            offset += t;
            length -= t;
	    }

	    /* the body of the input data */
	    for(;length >= cypher.getBlocksize();
			offset+=cypher.getBlocksize(), length-=cypher.getBlocksize())
        {
            hash(buf, 0, data, offset);
        }
	    /* tuck left-overs away */
	    over = length;
	    if(length > 0)
        {
            System.arraycopy(data, offset, in, 0, length);
        }
    }

   /**
    * Feeds a  byte into the hash
    * @param data the byte value
    */
    public void update(byte data)
    {
        byte[] temp = {data};
        update(temp, 0, temp.length);
    }

    /**
    * consolidates the input, and reinitialises the hash
    * @return the hash value
    */
    public byte[] digest()
    {
	    int count = over;
	    //byte *p = md->in + count;

	    /* there is always a free byte by construction */
        in[count] = (byte)0x80;
	    count = cypher.getBlocksize() - (count+1); /* free bytes in buffer */

	    if(count<8) /* not enough space for the count */
	    {
            for(int i=0; i<count;++i) in[over+1+i] = 0;
		    hash(buf, 0, in, 0);  /*so hash and start again*/
            for(int j=0; j<cypher.getBlocksize()-8; ++j) in[j] = 0;
	    }
	    else for(int k=0; k<count-8;++k) in[over+1+k] = 0;

        int n = cypher.getBlocksize() - 8;
	    /* pack high byte first */
	    in[n]   = (byte)((bits >> 56) & 0xFF);
	    in[n+1] = (byte)((bits >> 48) & 0xFF);
        in[n+2] = (byte)((bits >> 40) & 0xFF);
	    in[n+3] = (byte)((bits >> 32) & 0xFF);

  	    in[n+4] = (byte)((bits >> 24) & 0xFF);
	    in[n+5] = (byte)((bits >> 16) & 0xFF);
	    in[n+6] = (byte)((bits >>  8) & 0xFF);
	    in[n+7] = (byte)((bits      ) & 0xFF);

	    hash(buf, 0, in, 0);
        byte[] out = new byte[cypher.getBlocksize()];
        System.arraycopy(buf, 0, out, 0, buf.length);
        init();
        return out;
    }

    /**
    * Drive standard set of test vectors and post to stdout
    */
    public static void main(String[] args)
    {
        test();
    }

    /**
    * Do standard set of test vectors and post to stdout
    */
    public static void test ()
    {
        CEA worker = new ThreeWay();
        BlockCypherHash hash = new BlockCypherHash(worker);

        byte[] result = null;

        byte[] in1 = {(byte)'a', (byte)'b', (byte)'c'};
        hash.update(in1, 0, in1.length);
        result = hash.digest();

        System.out.println("");
        System.out.println("abc");

        int i;
		for(i=0; i<result.length; i++)
		{
			System.out.print(" "+Integer.toHexString(result[i] & 0xFF));
		}
        System.out.println("");
        System.out.println(" 8d 19 d8 bd 5d d7 e4 7f 18 e0 b7 85  expected\n");

        String fox = "The quick brown fox jumps over the lazy dog.";
        byte[] in2 = new byte[fox.length()];
        for(i=0; i<fox.length();++i) in2[i] = (byte)fox.charAt(i);

        hash.update(in2, 0, in2.length);
        result = hash.digest();

        System.out.println("");
        System.out.println(fox);

		for(i=0; i<result.length; i++)
		{
			System.out.print(" "+Integer.toHexString(result[i] & 0xFF));
		}
        System.out.println("");
        System.out.println(" f1 c8 c3 67 4d 4c 4e a0 93 a5 38 75 expected\n");

        String fly = "Vext cwm fly zing! jabs Kurd qoph.";
        byte[] in3 = new byte[fly.length()];
        for(i=0; i<fly.length();++i) in3[i] = (byte)fly.charAt(i);

        hash.update(in3, 0, in3.length);
        result = hash.digest();

        System.out.println("");
        System.out.println(fly);

		for(i=0; i<result.length; i++)
		{
			System.out.print(" "+Integer.toHexString(result[i] & 0xFF));
		}
        System.out.println("");
        System.out.println(" 2f 73 0c 5b 49 94 12 18 20 54 21 51 expected\n");

        String abc = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
        byte[] in4 = new byte[abc.length()];
        for(i=0; i<abc.length();++i) in4[i] = (byte)abc.charAt(i);

        hash.update(in4, 0, in4.length);
        result = hash.digest();

        System.out.println("");
        System.out.println(abc);

		for(i=0; i<result.length; i++)
		{
			System.out.print(" "+Integer.toHexString(result[i] & 0xFF));
		}
        System.out.println("");
        System.out.println(" 21 a8 8a ea 2c a7 bf 3d 19 d9 fb 67 expected\n");

      	byte[] a = new byte[8192];
	    int count;
        int as = 0;

        for(i=0;i<8192;i++) a[i] = (byte)'a';

        System.out.println("");
        for(count=1000000; count > 0; count -=8192)
        {
            int l = (count > 8192) ? 8192 : (int) count;
            hash.update(a, 0, l);
            as += l;
            System.out.print(""+as+" \"a\"s\r");
        }
        System.out.println("1000000 \"a\"s");
        result = hash.digest();

		for(i=0; i<result.length; i++)
		{
			System.out.print(" "+Integer.toHexString(result[i] & 0xFF));
		}
        System.out.println("");
        System.out.println(" 3c 05 ae 8f b7 17 61 93 7c 04 34 8d expected\n");

        System.out.println("Now an exception traceback is expected");

        CEA failure = new CAST5();
        hash = new BlockCypherHash(failure);
	}
}



