/**
 * Abstract superclass for all message digests.
 * Message digests produce a fixed-length digest of
 * an arbitrary-length message. A cryptographically
 * strong digest will do this in a way that a message
 * with a given digest cannot be found easily.
 *
 * <p> Once instatiated, a digest accepts a series of
 * update() calls; the digest value can then be retrieved
 * by a getDigest() call.  Multiple calls to getDigest
 * all return the same value.
 *
 * <p> A digest may be re-used by calling the reset()
 * method, after which the digest will act exactly
 * as if it had just been instantiated.
 *
 * @version 0.2, 3.3.96
 * @author  Markus L. Noga <mlnoga@mail.hh.provi.de>
 * @author  Greg Noel <greg@qualcomm.com>
 */
public abstract class Digest
        implements Cloneable
{
        /**
         * Returns a string identifying the digest algorithm.
         *
         **/
        public abstract String algName();
        public abstract String toString();

        /**
         * Create a duplicate of the digest, including state.
         *
         **/
        public abstract Digest Clone();

        /**
         * Returns the digest length in bytes.
         *
         **/
        public abstract int outputLen();

        /**
         * Resets the digest. New data can then be processed.
         *
         **/
        public abstract void reset();

        /**
         * Updates the digest.
         * @param input the input to be processed
         *
         */
        public abstract void update(byte input);

        /**
         * Updates the digest.
         * @param input the input to be processed
         * @param offset where to start processing
         * @param len number of bytes to process
         *
         */
        public void update(byte[] input, int offset, int len)
        {
                for (int i = 0; i < len; ++i)
                        update(input[offset+i]);
        }

        /**
         * Updates the digest.
         * @param input the input to be processed
         *
         **/
        public final void update(byte[] input)
        {
                update(input, 0, input.length);
        }

        /**
         * Updates the digest.
         * @param t the value to be processed
         *
         **/
        public final void update(short t)
        {
                update((byte)(t>>8));
                update((byte)t);
        }

        /**
         * Updates the digest.
         * @param t the value to be processed
         *
         **/
        public final void update(int t)
        {
                update((short)(t>>16));
                update((short)t);
        }

        /**
         * Updates the digest.
         * @param t the value to be processed
         *
         **/
        public final void update(long t)
        {
                update((int)(t>>32));
                update((int)t);
        }

        /**
         * Updates the digest.
         * @param in the input stream to be processed
         *
         **/
        public final void update(java.io.InputStream in)
                throws java.io.IOException
        {
                byte[] buf = new byte[64];
                int n;
                try {
                        while ((n = in.read(buf)) >= 0)
                                update(buf, 0, n);
                } catch (java.io.EOFException e) {
                        //
                } catch (java.io.IOException e) {
                        throw e;
                }
        }

        /**
         * Insert internal padding as a marker between digest segments;
         * this is done by various MAC algorithms.  If the function isn't
         * meaningful for a particular digest calculation (i.e., it's not
         * a cryptographically strong hash), the default implementation
         * throws an exception.
         *
         **/
        public void insertPadding()             // overridden if meaningful
                throws NoSuchMethodException
        {
                throw new NoSuchMethodException();
        }

        /**
         * Return final digest.  The value may be retrieved more than once.
         * Further calls to update() will result in undefined behavior.
         * (Maybe an exception ;-)
         *
         * @param output where to write the digest
         * @param offset position to start writing
         *
         */
        public abstract void getDigest(byte[] digest, int offset);

        /**
         * Return final digest.  The value may be retrieved more than once.
         * Further calls to update() will result in undefined behavior.
         * (Maybe an exception ;-)
         *
         * @param output where to write the digest
         *
         */
        public final void getDigest(byte[] digest)
        {
                getDigest(digest, 0);
        }

        /**
         * Return final digest.  The value may be retrieved more than once.
         * Further calls to update() will result in undefined behavior.
         * (Maybe an exception ;-)
         *
         * @return digest
         *
         */
        public final byte[] getDigest()
        {
                byte[] t = new byte[outputLen()];
                getDigest(t, 0);
                return t;
        }
}

