/* Implementation of a progress meter, which will display the state
   of a lengthy operation.

   Copyright by Thomas Ravnholt 1993
   progress.c is shareware - read progress.txt for further information.
*/

#include <windows.h>
#include "progress.h"

/* CreateProgress takes 6 parameters :
   HWND      parent     The progress meter will be displayed in this window.
   HINSTANCE hInstance  The global instance obtained when parent was created.
   int       x          The horizontal position relative to the top left
                        corner of parent.
   int       y          The vertical position.
   int       cx         The lenght of the progress meter.
   int       cy         The height of the progress meter.

   This function creates the progress meter and makes it ready
   to display the state of an operation.

   Return value :       A window handle to the progress meter. 
*/
HWND CreateProgress(HWND parent,HINSTANCE hInstance, int x, int y, int cx, int cy)
{
  HWND NewWnd,tmpWnd;

  NewWnd = CreateWindow("STATIC",
                        "",
                        WS_CHILD,
                        x,
                        y,
                        cx,
                        cy,
                        parent,
                        NULL,
                        hInstance,
                        NULL);
  if(NewWnd==NULL)
    return NULL;

  ShowWindow(NewWnd, SW_SHOWNORMAL);
 
  tmpWnd = CreateWindow("STATIC",
                        "",
                        WS_CHILD | SS_BLACKFRAME,
                        0,
                        0,
                        cx,
                        cy,
                        NewWnd,
                        NULL,
                        hInstance,
                        NULL);
  ShowWindow(tmpWnd, SW_SHOWNORMAL);

  if(tmpWnd==NULL)
    return NULL;
 
  return NewWnd;
}

/* UpdateProgress takes 4 parameters :
   HWND   hwnd       The handle of the progress meter.
   HBRUSH hbrush     A brush which will be used to show the state.
   long   position   Current state of the operation.
   long   max        The highest state value.

   This function updates the progress meter, eg. showing
   the state of the current operation.

   Return value :    TRUE if updating succeded.
                     FALSE otherwise.
*/
BOOL UpdateProgress(HWND hwnd,HBRUSH hbrush,long position,long max)
{
  RECT   rect;
  HDC    hdc;
  HBRUSH old_brush;
  char   buf[5];
  int    length;
  static int previous=0;

  if(max<position)
    return FALSE;

  if(previous==(position*100)/max)
    return TRUE;

  GetClientRect(hwnd,&rect);
  hdc=GetDC(hwnd);
  old_brush=SelectObject(hdc,hbrush);
  
  length=(int)((((long)(rect.right-2))*position)/max);
  PatBlt(hdc,1,
             1,
             length,
             rect.bottom-2,PATCOPY);

  if(length+1+1<rect.right)
      PatBlt(hdc,length+1+1,
                1,
                rect.right-length-1-2,
                rect.bottom-2,WHITENESS);

  SelectObject(hdc,old_brush);
  SetBkMode(hdc,TRANSPARENT);
  previous=(int)((position*100)/max);
  wsprintf(buf,"%d%%",previous);
  SetTextColor(hdc,RGB(0,0,0));
  DrawText(hdc,buf,-1,&rect,DT_CENTER | DT_SINGLELINE | DT_VCENTER);

  ReleaseDC(hwnd,hdc);

  return TRUE;
}

/* ClearProgress takes 1 parameter:
   HWND hwnd      The handle og the progress meter.

   This function clears the progress meter, making
   it ready for the next operation.

   Return value : TRUE if clearing the progress meter was possible. 
*/
BOOL ClearProgress(HWND hwnd)
{
  RECT   rect;
  HDC    hdc;

  GetClientRect(hwnd,&rect);
  hdc=GetDC(hwnd);
  PatBlt(hdc,1,
             1,
             rect.right-2,
             rect.bottom-2,WHITENESS);
  ReleaseDC(hwnd,hdc);
  return TRUE;
}

/* DestroyProgress takes 1 parameter:
   HWND hwnd      The handle og the progress meter.

   This function destroys the progress meter.

   Return value : No return value. 
*/
void DestroyProgress(HWND hwnd)
{
  DestroyWindow(hwnd);
}


