/*	Copyright (c) 1990, 1991 UNIX System Laboratories, Inc.	*/
/*	Copyright (c) 1984, 1986, 1987, 1988, 1989, 1990 AT&T	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF     	*/
/*	UNIX System Laboratories, Inc.		     	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/

#ifndef _FS_SPECFS_SNODE_H	/* wrapper symbol for kernel use */
#define _FS_SPECFS_SNODE_H	/* subject to change without notice */

#ident	"@(#)uts-3b2:fs/specfs/snode.h	1.10"

#include <sys/types.h>
#include <sys/sema.h>

/*
 * The snode represents a special file in any filesystem.  There is
 * one snode for each active special file.  Filesystems that support
 * special files use specvp(vp, dev, type, cr) to convert a normal
 * vnode to a special vnode in the ops lookup() and create().
 *
 * To handle having multiple snodes that represent the same
 * underlying device vnode without cache aliasing problems,
 * the s_commonvp is used to point to the "common" vnode used for
 * caching data.  If an snode is created internally by the kernel,
 * then the s_realvp field is NULL and s_commonvp points to s_vnode.
 * The other snodes that are created as a result of a lookup of a
 * device in a file system have s_realvp pointing to the vp that
 * represents the device in the file system while the s_commonvp points
 * at the "common" vnode for the device of another snode.
 */

struct snode {
	struct	snode *s_next;		/* hash link - must be first */
	struct	vnode *s_vnode;		/* vnode associated with this snode */
	struct	vnode *s_realvp;	/* vnode for the fs entry (if any) */
	struct	vnode *s_commonvp;	/* common device vnode */
	ushort	s_flag;			/* flags, see below */
	cnt_t	s_count;		/* count of opened references */
	dev_t	s_dev;			/* device the snode represents */
	dev_t	s_fsid;			/* file system identifier */
	daddr_t	s_nextr;		/* next byte read offset (read-ahead) */
	long	s_size;			/* block device size in basic blocks */
	time_t  s_atime;		/* time of last access */
	time_t  s_mtime;		/* time of last modification */
	time_t  s_ctime;		/* time of last attributes change */
	long    s_mapcnt;		/* count of mappings of pages */
	short	s_lockpid;		/* pid of process owning s_lock */
	short	s_locktrips;		/* number of lock reacquisitions */
	sema_t	s_lock;			/* mutual exclusion lock */
	long	s_gen;			/* commonvp gen number */
};

/* flags */
#define SUPD		0x01		/* update device access time */
#define SACC		0x02		/* update device modification time */
#define SCHG		0x04		/* update device change time */
#define	SMOUNTED	0x08		/* block device is mounted */
#define	SWANTCLOSE	0x10		/* device successfully opened */

/*
 * Convert between vnode and snode
 */
#define	VTOS(vp)	((struct snode *)((vp)->v_data))
#define	STOV(sp)	((sp)->s_vnode)

#ifdef _KERNEL
/*
 * Lock and unlock snodes.
 */
#define SLOCK(sp)	slock(sp)
#define SUNLOCK(sp)	sunlock(sp)

void slock(struct snode *);
void sunlock(struct snode *);

/*
 * Hold and release open snodes references.
 */
#define	SHOLD(sp)	{ SLOCK(sp); (sp)->s_count++; SUNLOCK(sp); }
#define	SRELE(sp)	{ SLOCK(sp); (sp)->s_count--; SUNLOCK(sp); }

/*
 * Compare an snode with its commonvp to see if its generation # permits
 * usage
 */
#define CMP_SGEN(s,v)	((s)->s_gen == VTOS(v)->s_gen)

enum vtype;
struct cred;
struct vnode;

/*
 * Construct a spec vnode for a given device that shadows a particular
 * "real" vnode.
 */
extern struct vnode *specvp(struct vnode *vp, dev_t, enum vtype, struct cred *);

/*
 * Construct a spec vnode for a given device that shadows nothing.
 */
extern struct vnode *makespecvp(dev_t, enum vtype);

/*
 * Convert a device vnode pointer into a common device vnode pointer.
 */
extern struct vnode *common_specvp(struct vnode *);

/*
 * Bogus SVR4 reference-count query function exported to generic os.
 */
extern int stillreferenced(dev_t, enum vtype);

/*
 * Flag a device (no longer) containing a mounted filesystem.
 */
extern void spec_mountedflag(struct vnode *, int);

#define spec_mounted(vp)	spec_mountedflag(vp, 1)
#define spec_unmounted(vp)	spec_mountedflag(vp, 0)

/* #ifdef MERGE */
extern int spec_rdchk(struct vnode *, struct cred *, int *);
/* #endif MERGE */

/*
 * If driver does not have a size routine (e.g. old drivers), the size of the
 * device is assumed to be infinite.
 */
#define UNKNOWN_SIZE 	0x7fffffff

/*
 * Size of block device i/o is parameterized here.
 * Currently the system supports page-sized i/o.
 */
#define	BLKDEV_IOSHIFT		BPCSHIFT
#define	BLKDEV_IOSIZE		(1<<BLKDEV_IOSHIFT)

/* convert a byte offset into an offset into a logical block for a block dev */
#define	BLKDEV_OFF(off)		((off) & (BLKDEV_IOSIZE - 1))

/* convert a byte offset into a block device logical block # */
#define	BLKDEV_LBN(off)		((off) >> BLKDEV_IOSHIFT)

/* number of bb's per block device block */
#define	BLKDEV_BB		BTOBB(BLKDEV_IOSIZE)

/* convert logical block dev block to physical blocks (== bb's) */
#define	BLKDEV_LTOP(bn)		((bn) * BLKDEV_BB)

/*
 * Snode lookup stuff.
 * These routines maintain a table of snodes hashed by dev so
 * that the snode for a dev can be found if it already exists.
 * NOTE: SPECTBSIZE must be a power of 2 for SPECTBHASH to work!
 */

#define	SPECTBSIZE	128	/* XXX too small? */
#define	SPECTBHASH(dev)	((getmajor(dev) + getminor(dev)) & (SPECTBSIZE - 1))

extern struct vnodeops spec_vnodeops;

#endif	/* _KERNEL */

#endif	/* _FS_SPECFS_SNODE_H */
