/*
 * dmnt.c - SGI IRIX mount support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.1 97/02/24 07:45:24 abe Exp $";
#endif


#include "lsof.h"


/*
 * readmnt() - read mount table
 */

int
readmnt()
{
	char *dn = NULL;
	int err = 0;
	char *ln;
	FILE *mfp;
	struct mntent *mp;
	struct mounts *mtp;
	char *opt, *opte;
	struct stat sb;

#if	defined(HASPROCFS)
	int procfs = 0;
#endif

/*
 * Open access to the mount table.
 */
	if ((mfp = setmntent(MOUNTED, "r")) == NULL) {
		(void) fprintf(stderr, "%s: can't access %s\n", Pn, MOUNTED);
		return(0);
	}
/*
 * Read mount table entries.
 */
	while ((mp = getmntent(mfp)) != NULL) {

#if	IRIXV>=50101
	/*
	 * Skip loop-back and to-be-ignored entries.
	 */

# if	defined(MNTTYPE_LO)
		if (strcmp(mp->mnt_type, MNTTYPE_LO) == 0
		||  strcmp(mp->mnt_type, MNTTYPE_IGNORE) == 0)
# else	/* !defined(MNTTYPE_LO) */
		if (strcmp(mp->mnt_type, MNTTYPE_IGNORE) == 0)
# endif	/* defined(MNTTYPE_LO) */

			continue;
#endif	/* IRIXV>=50101 */

	/*
	 * Interpolate a possible symbolic directory link.
	 */
		if (dn)
			(void) free((FREE_P *)dn);
		if ((dn = (char *)malloc((MALLOC_S)(strlen(mp->mnt_dir) + 1)))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(dn, mp->mnt_dir);
		if ((ln = Readlink(dn)) == NULL) {
		    if (!Fwarn){
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    err = 2;
		    continue;
		}
		if (ln != dn) {
			(void) free((FREE_P *)dn);
			dn = ln;
		}
	/*
	 * Stat() the directory.
	 */
		if (statsafely(dn, &sb)) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: WARNING: can't stat() %s file system %s\n",
			    Pn, mp->mnt_type, mp->mnt_dir);
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
			}
		    err = 2;
		    if ((opt = hasmntopt(mp, MNTINFO_DEV)) != NULL) {
			memset((char *)&sb, 0, sizeof(sb));
			if ((opte = x2dev(opt + 4, &sb.st_dev)) != NULL) {
			    sb.st_mode = S_IFDIR | 0777;

#if     defined(HASFSTYPE)
			    (void) strncpy(sb.st_fstype, mp->mnt_type,
				sizeof(sb.st_fstype));
			    sb.st_fstype[sizeof(sb.st_fstype) - 1 ] = '\0';
#endif  /* defined(HASFSTYPE */

			    if (!Fwarn)
				(void) fprintf(stderr,
				    "      assuming \"%.*s\" from %s\n",
				    (opte - opt), opt, MOUNTED);
			} else
			    opt = NULL;
		    }
		    if (opt == NULL)
			continue;
		}
	/*
	 * Allocate and fill a local mount structure.
	 */
		if ((mtp = (struct mounts *)malloc(sizeof(struct mounts)))
		== NULL) {
			err = 1;
			break;
		}
		if ((mtp->fsname =
			(char *)malloc((MALLOC_S)(strlen(mp->mnt_fsname)+1)))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(mtp->fsname, mp->mnt_fsname);

#if	defined(HASFSTYPE)
		if ((mtp->fstype =
			(char *)malloc((MALLOC_S)(strlen(sb.st_fstype) + 1)))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(mtp->fstype, sb.st_fstype);
#endif	/* HASFSTYPE */

		mtp->dir = dn;
		dn = NULL;

#if	defined(HASPROCFS)
# if	defined(HASFSTYPE)
		if (strcmp(sb.st_fstype, HASPROCFS) == 0)
# else
		if (strcmp(mp->mnt_fsname, "/proc") == 0)
# endif	/* HASFSTYPE */

		{

		/*
		 * Save information on exactly one procfs file system.
		 */
			if (procfs)
				Mtprocfs = NULL;
			else {
				procfs = 1;
				Mtprocfs = mtp;
			}
		}
#endif	/* HASPROCFS */

		mtp->next = Mtab;
		mtp->dev = sb.st_dev;
		mtp->rdev = sb.st_rdev;
		mtp->inode = sb.st_ino;
		mtp->mode = sb.st_mode;
		Mtab = mtp;
	}
	(void) endmntent(mfp);
	if (dn)
		(void) free((FREE_P *)dn);
/*
 * Handle errors.
 */
	switch (err) {
	case 1:
		(void) fprintf(stderr, "%s: no space for mount at %s (%s)\n",
			Pn, mp->mnt_fsname, mp->mnt_dir);
		return(0);
	case 2:
		return(1);
	}
	return(1);
}


#if	IRIXV>=50101
/*
 * readvfs() - read vfs structure
 */

struct l_vfs *
readvfs(ka, la)
	struct vfs *ka;			/* vfs structure kernel address, if
					 * must be read from kernel */
	struct vfs *la;			/* local vfs structure address, non-
					 * NULL if already read from kernel */
{
	struct vfs *v, tv;
	struct l_vfs *vp;

	if (!ka && !la)
		return(NULL);
/*
 * Search the local cache.
 */
	for (vp = Lvfs; vp; vp = vp->next) {
		if (ka == vp->addr)
			return(vp);
	}
/*
 * Allocate a new cache entry.
 */
	if ((vp = (struct l_vfs *)malloc(sizeof(struct l_vfs))) == NULL) {
		(void) fprintf(stderr, "%s: PID %d, no space for vfs\n",
			Pn, Lp->pid);
		Exit(1);
	}
	vp->dir = NULL;
	vp->fsname = NULL;

#if	defined(HASFSINO)
	vp->fs_ino = 0;
#endif	/* defined(HASFSINO) */

/*
 * Fill in the new cache entry from kernel data.
 */
	if (la)
		v = la;
	else if (kread((KA_T)ka, (char *)&tv, sizeof(tv))) {
		(void) free((FREE_P *)vp);
		return(NULL);
	} else
		v = &tv;
	(void) completevfs(vp, (dev_t *)&v->vfs_dev);
	vp->next = Lvfs;
	vp->addr = ka;
	Lvfs = vp;
	return(vp);
}
#endif	/* IRIXV>=50101 */
