/*
 * This software is copyrighted as noted below.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */
/*
 * rletorla - A program which will convert Utah's "rle" images into either
 *            Wavefronts "rla" or "rlb" image formats.
 *
 * Author:      Wesley C. Barris
 *              AHPCRC
 *              Minnesota Supercomputer Center, Inc.
 * Date:        June 21, 1990
 *              Copyright @ 1990, Minnesota Supercomputer Center, Inc.
 *
 * RESTRICTED RIGHTS LEGEND
 *
 * Use, duplication, or disclosure of this software and its documentation
 * by the Government is subject to restrictions as set forth in subdivision
 * { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 * clause at 52.227-7013.
 */
static char rcsid[] = "$Header: /l/spencer/src/urt/cnv/RCS/rletorla.c,v 3.0.1.5 1992/04/30 14:00:52 spencer Exp $";
/*
rletorla()				Tag the file.
*/
/*-----------------------------------------------------------------------------
 * System includes.
 */
#include "rle.h"
#include <string.h>
#include "rla_header.h"
#include "rlb_header.h"

#define VPRINTF if (verbose || header) fprintf
#define GRAYSCALE   001	/* 8 bits, no colormap */
#define PSEUDOCOLOR 010	/* 8 bits, colormap */
#define TRUECOLOR   011	/* 24 bits, colormap */
#define DIRECTCOLOR 100	/* 24 bits, no colormap */
#define TRUE          1
#define FALSE         0

typedef unsigned char U_CHAR;

/*
 * Wavefront type declarations.
 */
union {
    RLA_HEADER	rla_head;
    RLB_HEADER	rlb_head;
} head;
/*
 * Utah type declarations.
 */
rle_hdr		hdr;
rle_map		*colormap;

#ifdef CRAY2CC
#define SHORTWRITE(var, fp) {craybuf = *var<<48;fwrite(&craybuf, 2, 1, fp);}
#define LONGWRITE(var, fp) {craybuf = *var<<32;fwrite(&craybuf, 4, 1, fp);}
#else
#define SHORTWRITE(var, fp) {fwrite(var, 2, 1, fp);}
#define LONGWRITE(var, fp) {fwrite(var, 4, 1, fp);}
#endif
/*
 * Other declarations.
 */
static FILE		*fpin, *fpout;
static char		rlaname[BUFSIZ], progname[30];
static int		visual, isalpha, maplen;
static int		verbose = 0, header = 0;
static int		rlb_flag = 0;
static int		i;
/*-----------------------------------------------------------------------------
 *                                                         Read the rle header.
 */
void read_rle_header(minx, maxx, miny, maxy)
int	*minx, *maxx, *miny, *maxy;
{
   hdr.rle_file = fpin;
   rle_get_setup(&hdr);
   *minx = hdr.xmin;
   *maxx = hdr.xmax;
   *miny = hdr.ymin;
   *maxy = hdr.ymax;
   VPRINTF(stderr, "Image size: %dx%d\n", *maxx-*minx+1, *maxy-*miny+1);
   if (hdr.ncolors == 1 && hdr.ncmap == 3) {
      visual = PSEUDOCOLOR;
      colormap = hdr.cmap;
      maplen = (1 << hdr.cmaplen);
      VPRINTF(stderr, "Mapped color image with a map of length %d.\n", maplen);
      }
   else if (hdr.ncolors == 3 && hdr.ncmap == 0) {
      visual = DIRECTCOLOR;
      VPRINTF(stderr, "24 bit color image, no colormap.\n");
      }
   else if (hdr.ncolors == 3 && hdr.ncmap == 3) {
      visual = TRUECOLOR;
      colormap = hdr.cmap;
      maplen = (1 << hdr.cmaplen);
      VPRINTF(stderr, "24 bit color image with color map of length %d\n" ,maplen);
      }
   else if (hdr.ncolors == 1 && hdr.ncmap == 0) {
      visual = GRAYSCALE;
      VPRINTF(stderr, "Grayscale image.\n");
      }
   else {
      fprintf(stderr,
              "ncolors = %d, ncmap = %d, I don't know how to handle this!\n",
              hdr.ncolors, hdr.ncmap);
      exit(-1);
      }
   if (hdr.alpha == 0) {
      isalpha = FALSE;
      VPRINTF(stderr, "No alpha channel.\n");
      }
   else if (hdr.alpha == 1) {
      isalpha = TRUE;
      VPRINTF(stderr, "Alpha channel exists!\n");
      }
   else {
      fprintf(stderr, "alpha = %d, I don't know how to handle this!\n",
              hdr.alpha);
      exit(-1);
      }
   switch (hdr.background) {
      case 0:
         VPRINTF(stderr, "Use all pixels, ignore background color.");
         break;
      case 1:
         VPRINTF(stderr,
                  "Use only non-background pixels, ignore background color.");
         break;
      case 2:
         VPRINTF(stderr,
        "Use only non-background pixels, clear to background color (default).");
         break;
      default:
         VPRINTF(stderr, "Unknown background flag!\n");
         break;
      }
   for (i = 0; i < hdr.ncolors; i++)
      VPRINTF(stderr, " %d", hdr.bg_color[i]);
   if (hdr.ncolors == 1 && hdr.ncmap == 3) {
      VPRINTF(stderr, " (%d %d %d)\n",
              hdr.cmap[hdr.bg_color[0]]>>8,
              hdr.cmap[hdr.bg_color[0]+256]>>8,
              hdr.cmap[hdr.bg_color[0]+512]>>8);
      }
   else if (hdr.ncolors == 3 && hdr.ncmap == 3) {
      VPRINTF(stderr, " (%d %d %d)\n",
              hdr.cmap[hdr.bg_color[0]]>>8,
              hdr.cmap[hdr.bg_color[1]+256]>>8,
              hdr.cmap[hdr.bg_color[2]+512]>>8);
      }
   else
      VPRINTF(stderr, "\n");
   if (hdr.comments)
      for (i = 0; hdr.comments[i] != NULL; i++)
         VPRINTF(stderr, "%s\n", hdr.comments[i]);
}
/*-----------------------------------------------------------------------------
 *                                                  Write the wavefront header.
 */
void write_rla_header(minx, maxx, miny, maxy, frame_number)
int	minx, maxx, miny, maxy, frame_number;
{
   char		*ctime();
   char		*d_str;
   long		second;
   long		time();
#ifdef CRAY2CC
   long		craybuf;
#endif

   bzero(&head, 740);
   head.rla_head.window.left   = minx;
   head.rla_head.window.right  = maxx;
   head.rla_head.window.top    = maxy;
   head.rla_head.window.bottom = miny;
   head.rla_head.active_window.left   = head.rla_head.window.left;
   head.rla_head.active_window.right  = head.rla_head.window.right;

   head.rla_head.active_window.top    = head.rla_head.window.top;
   head.rla_head.active_window.bottom = head.rla_head.window.bottom;
   head.rla_head.frame     = frame_number;
   head.rla_head.num_chan  = 3;
   head.rla_head.num_matte = 1;
   strcpy(head.rla_head.gamma, " 2.2000");
   strcpy(head.rla_head.red_pri, "0.670 0.330");
   strcpy(head.rla_head.green_pri, "0.210 0.710");
   strcpy(head.rla_head.blue_pri, "0.140 0.080");
   strcpy(head.rla_head.white_pt, "0.310 0.316");
   d_str = strrchr(rlaname, '/');
   if ( d_str == NULL )
       d_str = rlaname;
   strncpy(head.rla_head.name, d_str, 29);
   head.rla_head.name[29] = '\0';
   strcpy(head.rla_head.desc, "A Wavefront file converted from an rle file.");
   strcpy(head.rla_head.program, progname);
   gethostname(head.rla_head.machine, 32);
   strcpy(head.rla_head.user, getenv("USER"));
   second = time((long *)NULL);
   d_str = ctime(&second);
   strncpy(head.rla_head.date, &d_str[4], 12);
   strncpy(&head.rla_head.date[12], &d_str[19], 5);
   head.rla_head.date[17] = '\0';
   if ( rlb_flag ) {
      if ((maxx-minx+1) == 646 && (maxy-miny+1) == 485)
	 strcpy(head.rlb_head.aspect, "ntsc_4d");
      else if ((maxx-minx+1) == 720 && (maxy-miny+1) == 486)
	 strcpy(head.rlb_head.aspect, "qtl_ntsc");
      else if ((maxx-minx+1) == 636 && (maxy-miny+1) == 484)
	 strcpy(head.rlb_head.aspect, "iris_ntsc");
      else if ((maxx-minx+1) == 1024 && (maxy-miny+1) == 1024)
	 strcpy(head.rlb_head.aspect, "1k_square");
      else
	 strcpy(head.rlb_head.aspect, "user defined");
      sprintf(head.rlb_head.aspect_ratio, "%3.2f",
	      (float)(maxx-minx+1)/(maxy-miny+1));
      }
   else {
      if ((maxx-minx+1) == 646 && (maxy-miny+1) == 485)
	 strcpy(head.rla_head.aspect, "ntsc_4d");
      else if ((maxx-minx+1) == 720 && (maxy-miny+1) == 486)
	 strcpy(head.rla_head.aspect, "qtl_ntsc");
      else if ((maxx-minx+1) == 636 && (maxy-miny+1) == 484)
	 strcpy(head.rla_head.aspect, "iris_ntsc");
      else if ((maxx-minx+1) == 1024 && (maxy-miny+1) == 1024)
	 strcpy(head.rla_head.aspect, "1k_square");
      else
	 strcpy(head.rla_head.aspect, "ntsc_4d");
   }
   strcpy(head.rla_head.chan, "rgb");
/*
 * The following fields do not really need to be set because they should
 * all be zero and already are because bzero was used on the entire header.
 */
/*
   head.rla_head.storage_type = 0;
   head.rla_head.num_aux = 0;
   head.rla_head.aux_mask = 0;
   head.rla_head.job_num = 0;
   if ( rlb_flag ) {
      head.rlb_head.field = 0;
      head.rlb_head.filter_type = 0;
      head.rlb_head.magic_number = 0;
      head.rlb_head.lut_size = 0;
      head.rlb_head.user_space_size = 0;
      head.rlb_head.wf_space_size = 0;
      head.rlb_head.lut_type = 0;
      head.rlb_head.mix_type = 0;
      head.rlb_head.encode_type = 0;
      head.rlb_head.padding = 0;
      }
*/
#ifdef CRAY2CC
   SHORTWRITE(&head.rla_head.window.left, fpout);
   SHORTWRITE(&head.rla_head.window.right, fpout);
   SHORTWRITE(&head.rla_head.window.bottom, fpout);
   SHORTWRITE(&head.rla_head.window.top, fpout);
   SHORTWRITE(&head.rla_head.active_window.left, fpout);
   SHORTWRITE(&head.rla_head.active_window.right, fpout);
   SHORTWRITE(&head.rla_head.active_window.bottom, fpout);
   SHORTWRITE(&head.rla_head.active_window.top, fpout);
   SHORTWRITE(&head.rla_head.frame, fpout);
   SHORTWRITE(&head.rla_head.storage_type, fpout);
   SHORTWRITE(&head.rla_head.num_chan, fpout);
   SHORTWRITE(&head.rla_head.num_matte, fpout);
   SHORTWRITE(&head.rla_head.num_aux, fpout);
   SHORTWRITE(&head.rla_head.aux_mask, fpout);
   fwrite(&head.rla_head.gamma, 16, 1, fpout);
   fwrite(&head.rla_head.red_pri, 24, 1, fpout); /*red_pri*/
   fwrite(&head.rla_head.green_pri, 24, 1, fpout); /*green_pri*/
   fwrite(&head.rla_head.blue_pri, 24, 1, fpout); /*blue_pri*/
   fwrite(&head.rla_head.white_pt, 24, 1, fpout); /*white_pri*/
   LONGWRITE(&head.rla_head.job_num, fpout);
   fwrite(&head.rla_head.name, 128, 1, fpout);
   fwrite(&head.rla_head.desc, 128, 1, fpout);
   fwrite(&head.rla_head.program, 64, 1, fpout);
   fwrite(&head.rla_head.machine, 32, 1, fpout);
   fwrite(&head.rla_head.user, 32, 1, fpout);
   fwrite(&head.rla_head.date, 20, 1, fpout);
   if (rlb_flag) {
      fwrite(&head.rlb_head.aspect, 24, 1, fpout);
      fwrite(&head.rlb_head.aspect_ratio, 8, 1, fpout);
      fwrite(&head.rlb_head.chan, 32, 1, fpout);
      SHORTWRITE(&head.rlb_head.field, fpout);
      SHORTWRITE(&head.rlb_head.filter_type, fpout);
      LONGWRITE(&head.rlb_head.magic_number, fpout);
      LONGWRITE(&head.rlb_head.lut_size, fpout);
      LONGWRITE(&head.rlb_head.user_space_size, fpout);
      LONGWRITE(&head.rlb_head.wf_space_size, fpout);
      SHORTWRITE(&head.rlb_head.lut_type, fpout);
      SHORTWRITE(&head.rlb_head.mix_type, fpout);
      SHORTWRITE(&head.rlb_head.encode_type, fpout);
      SHORTWRITE(&head.rlb_head.padding, fpout);
      fwrite(&head.rlb_head.space, 100, 1, fpout);
      }
   else {
      fwrite(&head.rla_head.aspect, 32, 1, fpout);
      fwrite(&head.rla_head.chan, 32, 1, fpout);
      fwrite(&head.rla_head.space, 128, 1, fpout);
      }
#else
   fwrite(&head, 740, 1, fpout);
#endif
}
/*-----------------------------------------------------------------------------
 *                                              Encode the data before writing.
 *
 *  NOTE: There are some subtle but important differences between the
 *        encoding shown here and that which is documented in
 *        Wavefront's "Wavefront's Format for Stored Pictures" paper.
 *        First and most important, counts <= 127 signify encoded runs
 *        (instead of the documented value of 126).  Second, data is
 *        not encoded if the repeat count is TWO OR LESS.  This differs
 *        from the documented "LESS THAN TWO" statement.  I have
 *        determined these two subtle but important points by reading
 *        the data from Wavefront-generated rla files.  After making
 *        these changes, I have found that core dumps which happened
 *        five percent of the time went away.
 */
int encode(c_in, c_out, width)
U_CHAR  *c_in;
U_CHAR  *c_out;
int     width;
{
   int          len;
   int          ct;

   len = 0;
   while (width > 0) {
/*
 * Notice that we test for runs of three or more like pixels here.
 */
      if ((width > 2) && (c_in[0] == c_in[1]) && (c_in[0] == c_in[2])) {
/*
 * Cruise until we find a different pixel value.
 */
         for (ct = 2; ct < width; ct++) {
            if (c_in[ct] != c_in[ct-1])
               break;
            if (ct >= 128)
               break;
            }
/*
 * Write out the count (which will always be 127 or less).
 */
         *c_out++ = ct - 1;
         len++;
/*
 * Write out the value.
 */
         *c_out++ = *c_in;
         len++;
         width -= ct;
         c_in += ct;
         }
      else {
/*
 * Cruise until they can be encoded again.
 */
         for (ct = 1; ct < width; ct++) {
/*
 * Notice that we test for runs of three or more like pixels here.
 */
            if ((width - ct > 2) && (c_in[ct] == c_in[ct+1])
                                 && (c_in[ct] == c_in[ct+2]))
               break;
            if (ct >= 128)
               break;
            }
/*
 * Write out the count (which will always be 128 or greater).
 */
         *c_out++ = 256 - ct;
         len++;
/*
 * Copy string of pixels.
 */
         for (; ct-- > 0; len++, width--)
            *c_out++ = *c_in++;
         }
      }
      return len;
}
/*-----------------------------------------------------------------------------
 *                                          Write the data portion of the file.
 */
void write_rla_data()
{
   rle_pixel	**scanline;
   U_CHAR	*red, *green, *blue, *matte;
   U_CHAR	*buf;
   int		*offset;
   int		width, height;
   int		scan, x, y;
   short	len;
   long		offptr;
#ifdef CRAY2CC
   long		craybuf;
#endif
/*
 * Create a scanline offset table then write it to reserve space in the file.
 */
   width  = head.rla_head.window.right - head.rla_head.window.left + 1;
   height = head.rla_head.window.top - head.rla_head.window.bottom + 1;
   if (!(offset = (int *)malloc(sizeof(int) * height))) {
      fprintf(stderr, "Offset malloc failed!\n");
      exit(-3);
      }
   offptr = ftell(fpout);
   if (fwrite(offset, 4, height, fpout) != height) {
      fprintf(stderr, "Offset table write failed!\n");
      exit(-4);
      }
   if (!(buf = (U_CHAR *)malloc(width * 2))) {
      fprintf(stderr, "Buf malloc failed!\n");
      exit(-7);
      }
   if (!(red = (U_CHAR *)malloc(width * 4))) {
      fprintf(stderr, "Red scanline malloc failed!\n");
      exit(-8);
      }
   green = &red[width];
   blue  = &green[width];
   matte = &blue[width];
/*
 * Loop through those scan lines.
 */
   if (!(rle_row_alloc(&hdr, &scanline) == 0)) {
      fprintf(stderr, "rle row alloc failed!\n");
      exit(-1);
      }
   for (scan = 0; scan < height; scan++) {
      y = rle_getrow(&hdr, scanline);
      switch (visual) {
         case GRAYSCALE:	/* 8 bits without colormap */
               red   = scanline[0];
               green = scanline[0];
               blue  = scanline[0];
               for (x = 0; x < width; x++)
                  matte[x] = (red[x] || green[x] || blue[x] ? 255 : 0);
            break;
         case TRUECOLOR:	/* 24 bits with colormap */
            if (isalpha) {
               matte = scanline[-1];
               for (x = 0; x < width; x++) {
                  red[x]   = colormap[scanline[0][x]]>>8;
                  green[x] = colormap[scanline[1][x]+256]>>8;
                  blue[x]  = colormap[scanline[2][x]+512]>>8;
                  }
               }
            else
               for (x = 0; x < width; x++) {
                  if (colormap[scanline[0][x]]>>8 != scanline[0][x] ||
                      colormap[scanline[1][x]+256]>>8 != scanline[1][x] ||
                      colormap[scanline[2][x]+512]>>8 != scanline[2][x])
                     /*fprintf(stderr,"A truecolor image with colormap whose resulting values don't match!\n");*/
                  red[x]   = colormap[scanline[0][x]]>>8;
                  green[x] = colormap[scanline[1][x]+256]>>8;
                  blue[x]  = colormap[scanline[2][x]+512]>>8;
                  matte[x] = (red[x] || green[x] || blue[x] ? 255 : 0);
                  }
            break;
         case DIRECTCOLOR:	/* 24 bits without colormap */
            if (isalpha) {
               matte = scanline[-1];
               red   = scanline[0];
               green = scanline[1];
               blue  = scanline[2];
               }
            else {
               red   = scanline[0];
               green = scanline[1];
               blue  = scanline[2];
               for (x = 0; x < width; x++)
                  matte[x] = (red[x] || green[x] || blue[x] ? 255 : 0);
               }
            break;
         case PSEUDOCOLOR:	/* 8 bits with colormap */
               for (x = 0; x < width; x++) {
                  red[x]   = colormap[scanline[0][x]]>>8;
                  green[x] = colormap[scanline[0][x]+256]>>8;
                  blue[x]  = colormap[scanline[0][x]+512]>>8;
                  matte[x] = (red[x] || green[x] || blue[x] ? 255 : 0);
                  }
            break;
         default:
            break;
         }
/*
 * Record the location in the file.
 */
      offset[scan] = ftell(fpout);
/*
 * Write the red scan line.
 */
      len = encode(red, buf, width);
      SHORTWRITE(&len, fpout);
      fwrite(buf, 1, (int)len, fpout);
/*
 * Write the green scan line.
 */
      len = encode(green, buf, width);
      SHORTWRITE(&len, fpout);
      fwrite(buf, 1, (int)len, fpout);
/*
 * Write the blue scan line.
 */
      len = encode(blue, buf, width);
      SHORTWRITE(&len, fpout);
      fwrite(buf, 1, (int)len, fpout);
/*
 * Write the matte scan line.
 */
      len = encode(matte, buf, width);
      SHORTWRITE(&len, fpout);
      fwrite(buf, 1, (int)len, fpout);
      } /* end of for scan = 0 to height */
/*
 * Write that offset table again with correct values.
 */
   fseek(fpout, offptr, 0);
#ifdef CRAY2CC
   for (scan=0;scan<height;scan++) {
      craybuf = offset[scan] << 32; fwrite(&craybuf, 4, 1, fpout);
      }
#else
   if (fwrite(offset, sizeof(int), height, fpout) != height) {
      fprintf(stderr, "Offset table write failed at end!\n");
      exit(-4);
      }
#endif
/*
 * Free up some stuff.
 */
   free(offset);
   free(buf);
   /*free(red); for some reason this dumps the core on an iris */
}
/*-----------------------------------------------------------------------------
 *                                        Convert an rle file into an rla file.
 */
int
main(argc, argv)
int argc;
char **argv;
{
   char		*lperiodP, *lslashP, *fname = NULL, *outname = NULL;
   int		minx, maxx, miny, maxy, frame_number = 1;
   int		oflag = 0;
/*
 * Get those options.
 */
   if (!scanargs(argc,argv,
       "% b%- v%- h%- o%-outfile.rla!s infile%s",
       &rlb_flag,
       &verbose,
       &header,
       &oflag, &outname,
       &fname))
      exit(-1);
/*
 * Open the file.
 */
   fpin = rle_open_f( cmd_name( argv ), fname, "r" );
   rle_names( &hdr, cmd_name(argv), fname, 0 );
/*
 * If we're just listing the header info, don't open an output file.
 */
   if (!header) {
/*
 * Use the same basename for the "rla/rlb" file (unless we're using stdin).
 */
      if (oflag)
         strcpy(rlaname, outname);
      else {
         if (fname == NULL)
            strcpy(rlaname, "stdin");
         else
            strcpy(rlaname, fname);
         }
      lperiodP = strrchr(rlaname, (int)'.');
      if (lperiodP) {
         if (rlb_flag)
            strcpy(lperiodP, ".rlb");
         else
            strcpy(lperiodP, ".rla");
         }
      else {
         if (rlb_flag)
            strcat(rlaname, ".rlb");
         else
            strcat(rlaname, ".rla");
         }
/*
 * Strip off everything (the path) but the filename.
 */
      if (!oflag) {
	  lslashP = strrchr(rlaname, (int)'/');
	  if (lslashP)
	      strcpy(rlaname, ++lslashP);
      }
      if (!(fpout = fopen(rlaname, "w"))) {
         fprintf(stderr, "Cannot open %s for writing.\n", rlaname);
         exit(-1);
         }
      }
   strncpy(progname, cmd_name( argv ), 29);
   progname[29] = 0;
/*
 * Read the rle file header.
 */
   read_rle_header(&minx, &maxx, &miny, &maxy);
   if (header)
      exit(0);
/*
 * Write the rla file header.
 */
   write_rla_header(minx, maxx, miny, maxy, frame_number);
/*
 * Write the rest of the Wavefront (rla) file.
 */
   write_rla_data();
   fclose(fpin);
   fclose(fpout);

   return 0;
}
