/*
   File: discev.c
*/
#include <strings.h>
#include <stdio.h>
#include <tmc.h>
#include <cvr.h>
#include "discevconst.h"
#include "tmcode.h"
#include "utils.h"

/* command line flags */
static int showorig = TRUE;
static int symtabtr = FALSE;
static int stat = FALSE;

/* common variables */
#define infile stdin
#define outfile stdout
FILE *tracestream = stderr;

/*
   The definitions given will be split into several lists
*/
formcon deffc;
def_list atom_list;
def_list syn_list;
def_list atom_appl_list;
val output_val;

/*
   We will generate procedures for every signal
   Numbers for these procs will be given from a certain offset.
*/
#define proclowend 100
static int procnr = proclowend;

/*
   Table of debugging flags plus associated information.
   Table is ended by an entry with flagchar '\0'
*/
static dbflag flagtab[] =
	{{ 's', &stat, "statistics" },
	 { 't', &symtabtr, "symbol table tracing" },
	 { '\0', (int *)0, "" },
	};

#define nr_of_elts(arr) ((int) (sizeof (arr)/sizeof (arr[0])))
#define streq(s1,s2) (strcmp ((s1),(s2)) == 0)

/*
   Die with errormessage
*/
static void Die (s)
 char *s;	
	{ fprintf (stderr, "%s\n", s);
	  exit (1);
	};

/*
   Name from symbol.
   Because the macroexpander adds _<number> to every symbol
   we must remove that suffix when recognizing atoms or source symbols
*/
static char* name_from_symbol (s)
 symbol s;
	{ char Buf[80];
	  char *name = symbolstr (s);
	  int ix;
	  for (ix = 0; ix < strlen (name) && !(name[ix] == '_'); ix++)
	     Buf [ix] = name [ix];
	  Buf [ix] ='\0';
	  return (new_string (Buf));
	};

/*
   Given a symbol 's', search the context for a definition with
   that name, and return a pointer to it.
*/
static ctx_list context;

static int symbol_occurs_in (s, v)
 symbol s;
 val v;
	{ switch (v -> tag)
	     { case TAGVSym: return (v -> VSym.sym == s);
	       case TAGVList:
		  { val_list vl = v -> VList.l;
		    register int ix;
		    for (ix=0; ix < vl -> sz; ix++)
		       if (symbol_occurs_in (s, vl -> arr[ix])) return (1);
		    return (0);
		  };
	       default: badtag (v -> tag);
	     };
	};
		  
static def find_def (s)
 symbol s;
	{ register unsigned int cix;
	  register unsigned int dix;

	  for (cix = 0; cix < context -> sz; cix++)
	     { register def_list l = context -> arr[cix] -> defs;
	       for (dix = 0; dix < l -> sz; dix++)
		  { register def d = l -> arr[dix];
		    switch (d -> tag)
		       { case TAGDefAtom:
		            if (d -> DefAtom.atnm == s) return (d);
		            break;

		         case TAGDefBasetype:
		            if (d -> DefBasetype.basename == s) return (d);
		            break;

			 case TAGDefVal:
			    if (d -> DefVal.valnm == s) return (d);
		            break;

			 case TAGDefTyp:
		            if (d -> DefTyp.typnm == s) return (d);
		            break;

			 case TAGDefCon:
			    if (symbol_occurs_in (s, d -> DefCon.defcon))
			       return (d);
			    break;

			 default:
		    	    badtag (d -> tag);
	    	       };
		  };
	}
	return (defNIL);
	};

/* Print usage of this program */
static void usage (f)
 FILE *f;
 	{ fprintf (f, "Usage: discev [-d<debugging flags>]\n");
	  helpdbflags (f, flagtab);
	};

/* scan arguments and options */
static void scanargs (argc, argv)
 int argc;
 char *argv[];
	{ int op;
	  argv++;
	  argc--;
	  while (argc>0)
	     { if (argv[0][0] != '-')
		  { fprintf (stderr, "too many arguments\n");
		    usage (stderr);
		    exit (1);
		  };
	       op = argv[0][1];
	       switch (op)
		 { case 'd': setdbflags (&argv[0][2], flagtab, TRUE);
			     break;
		   case 'h':
		   case 'H': usage (stdout);
			     exit (0);

		   case 'o': showorig = FALSE;
			     break;

		   default: usage (stderr);
			    exit (1);
	         };
	       argc--;
	       argv++;
	     };
	};

/*
   Check that the datastructures read are of the correct format
   that is they are the output of uflat2
*/
#define IllegalFormat() Die("Illegal datastructures format")
static void checkformat_of_deflist (dl)
 def_list dl;
	{ int ix;
	  def d;
	  atom_list = new_def_list ();
	  for (ix=0; ix < dl -> sz - 1; ix++)
	     { d = dl -> arr [ix];
	       switch (d -> tag)
		  { case TAGDefBasetype:
		       { if (getprior (d -> DefBasetype.basename) == proclowend)
			    IllegalFormat ();
			 setprior (d -> DefBasetype.basename, proclowend);
		       };
		       break;

		    case TAGDefAtom:
		       { typ t = d -> DefAtom.atctyp;
			 if (t -> tag != TAGTypUni) IllegalFormat ();
			 app_def_list (atom_list, rdup_def (d));
		       };
		       break;

		    default:
		       IllegalFormat ();
		  };
	     };
	};

static void checkformat_of_lastdef (dl)
 def_list dl;
	{ val rval1, rval2, rval3;
	  def lastdef = dl -> arr [dl -> sz - 1];
	  if (lastdef -> tag != TAGDefVal) IllegalFormat ();
	  if (lastdef -> DefVal.valtyp -> tag != TAGTypUni) IllegalFormat ();
	  rval1 = lastdef -> DefVal.valas;
	  if (rval1 -> tag != TAGVLambda) IllegalFormat ();
	  deffc = rdup_formcon (rval1 -> VLambda.lpar);
	  rval2 = rval1 -> VLambda.lval;
	  if (rval2 -> tag != TAGVWhere) IllegalFormat ();
	  rval3 = rval2 -> VWhere.wval;
	  if (rval3 -> tag != TAGVWhere)
	     { /* format without original input names */
	       syn_list = def_listNIL;
	       atom_appl_list = rdup_def_list (rval2 -> VWhere.wdefs);
	       output_val = rdup_val (rval3);
	     }
	  else
	     { syn_list = rdup_def_list (rval2 -> VWhere.wdefs);
	       atom_appl_list = rdup_def_list (rval3 -> VWhere.wdefs);
	       output_val = rdup_val (rval3 -> VWhere.wval);
	     };
	};

/*
   Assign signal numbers to all of the left hand sides
*/
static void assign_signalnr_to_val (v)
 val v;
	{ switch (v -> tag)
	     { case TAGVSym:
		  { if (getprior (v -> VSym.sym) < proclowend)
		       { setprior (v -> VSym.sym, procnr);
			 procnr++;
		       };
		  };	
		  break;

	       case TAGVList:
		  { int ix;
		    val_list vl = v -> VList.l;
		    for (ix = 0; ix < vl -> sz; ix++)
		       assign_signalnr_to_val (vl -> arr[ix]);
		  };
		  break;

	       default: badtag (v -> tag);
	     };
	};

static void assign_signalnr_to_formcon (fc)
 formcon fc;
	{ switch (fc -> tag)
	     { case TAGFCSym:
		  { if (getprior (fc -> FCSym.sym) < proclowend)
		       { setprior (fc -> FCSym.sym, procnr);
			 procnr++;
		       };
		  };	
		  break;

	       case TAGFCList:
		  { int ix;
		    formcon_list fcl = fc -> FCList.l;
		    for (ix = 0; ix < fcl -> sz; ix++)
		       assign_signalnr_to_formcon (fcl -> arr[ix]);
		  };
		  break;

	       default: badtag (fc -> tag);
	     };
	};

static void assign_signalnrs_to_defs (dl)
 def_list dl;
	{ int ix;
	  for (ix = 0; ix < dl -> sz; ix++)
	     { def d = dl -> arr[ix];
	       if (d -> tag != TAGDefCon) IllegalFormat ();
	       assign_signalnr_to_val (d -> DefCon.defcon);
	     };
	};

static void assign_signalnrs ()
	{ if (syn_list == def_listNIL)
	     { assign_signalnr_to_formcon (deffc);
	     }
	  else
	     { assign_signalnrs_to_defs (syn_list);
	     };
	  assign_signalnrs_to_defs (atom_appl_list);
	};

/*
   prepare the definitions
*/
static void prepare (dl)
 def_list dl;
	{ fprintf (stderr, "discev: preparing...\n");
	  checkformat_of_deflist (dl);
	  checkformat_of_lastdef (dl);
	  assign_signalnrs ();
	};
/*
   coding
*/
static void code_zero (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ int lnr = getprior (lhs -> VSym.sym);
	  fprintf (f, "\t  enqueue (%d, 0, 0);\n", lnr - proclowend);
	};

static void code_one (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ int lnr = getprior (lhs -> VSym.sym);
	  fprintf (f, "\t  enqueue (%d, 0, 1);\n", lnr - proclowend);
	};

static void code_buf (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ int lnr = getprior (lhs -> VSym.sym);
	  fprintf (f, "\t  if (val == 0) { enqueue (%d, time+buf_tpll, 0); }\n",
			lnr - proclowend);
	  fprintf (f, "\t  else enqueue (%d, time+buf_tphh, 1);\n",
			lnr - proclowend); 
	};

static void code_not (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ int lnr = getprior (lhs -> VSym.sym);
	  fprintf (f, "\t  if (val == 0) { enqueue (%d, time+not_tplh, 1); }\n",
			lnr - proclowend);
	  fprintf (f, "\t  else enqueue (%d, time+not_tphl, 0);\n",
			lnr - proclowend); 
	};

static void code_nand (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ val_list vl = atcpar -> VList.l;
	  int lnr = getprior (lhs -> VSym.sym);
	  int ix;
	  int coded = 0;
	  fprintf (f, "\t  if (");
	  for (ix = 0; ix < vl -> sz; ix++)
	     { int nr2 = getprior (vl -> arr[ix] -> VSym.sym);
	       if (nr2 != nr)
		  { fprintf (f, "(Sigs[%d].cv == 1)", nr2 - proclowend);
		    coded++;
		    if (coded + 1 < vl -> sz)
		       { fprintf (f, " && ");
			 if (coded%4 == 3) fprintf (f, "\n\t      ");
		       }
		  };
	     };
	  fprintf (f, ")\n");
	  fprintf (f, "\t     { if (val == 1) ");
	  fprintf (f, "{ enqueue (%d, time+nand_tphl, 0); }\n",
			lnr - proclowend);
	  fprintf (f, "\t       else enqueue (%d, time+nand_tplh, 1);\n",
			lnr - proclowend);
	  fprintf (f, "\t     }\n");
	  fprintf (f, "\t  else check_enqueue (%d, time+nand_tplh, 1);\n",
			lnr - proclowend);
	};

static void code_and (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ val_list vl = atcpar -> VList.l;
	  int lnr = getprior (lhs -> VSym.sym);
	  int ix;
	  int coded = 0;
	  fprintf (f, "\t  if (");
	  for (ix = 0; ix < vl -> sz; ix++)
	     { int nr2 = getprior (vl -> arr[ix] -> VSym.sym);
	       if (nr2 != nr)
		  { fprintf (f, "(Sigs[%d].cv == 1)", nr2 - proclowend);
		    coded++;
		    if (coded + 1 < vl -> sz)
		       { fprintf (f, " && ");
			 if (coded%4 == 3) fprintf (f, "\n\t      ");
		       }
		  };
	     };
	  fprintf (f, ")\n");
	  fprintf (f, "\t     { if (val == 1) ");
	  fprintf (f, "{ enqueue (%d, time+and_tphh, 1); }\n",
			lnr - proclowend);
	  fprintf (f, "\t       else enqueue (%d, time+and_tpll, 0);\n",
			lnr - proclowend);
	  fprintf (f, "\t     }\n");
	  fprintf (f, "\t  else check_enqueue (%d, time+and_tpll, 0);\n",
			lnr - proclowend);
	};

static void code_nor (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ val_list vl = atcpar -> VList.l;
	  int lnr = getprior (lhs -> VSym.sym);
	  int ix;
	  int coded = 0;
	  fprintf (f, "\t  if (");
	  for (ix = 0; ix < vl -> sz; ix++)
	     { int nr2 = getprior (vl -> arr[ix] -> VSym.sym);
	       if (nr2 != nr)
		  { fprintf (f, "(Sigs[%d].cv == 0)", nr2 - proclowend);
		    coded++;
		    if (coded + 1 < vl -> sz)
		       { fprintf (f, " && ");
			 if (coded%4 == 3) fprintf (f, "\n\t      ");
		       }
		  };
	     };
	  fprintf (f, ")\n");
	  fprintf (f, "\t     { if (val == 0) ");
	  fprintf (f, "{ enqueue (%d, time+nor_tplh, 1); }\n",
			lnr - proclowend);
	  fprintf (f, "\t       else enqueue (%d, time+nor_tphl, 0);\n",
			lnr - proclowend);
	  fprintf (f, "\t     }\n");
	  fprintf (f, "\t  else check_enqueue (%d, time+nor_tphl, 0);\n",
			lnr - proclowend);
	};

static void code_or (f, lhs, atvpar, atcpar, nr)
 FILE *f;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ val_list vl = atcpar -> VList.l;
	  int lnr = getprior (lhs -> VSym.sym);
	  int ix;
	  int coded = 0;
	  fprintf (f, "\t  if (");
	  for (ix = 0; ix < vl -> sz; ix++)
	     { int nr2 = getprior (vl -> arr[ix] -> VSym.sym);
	       if (nr2 != nr)
		  { fprintf (f, "(Sigs[%d].cv == 0)", nr2 - proclowend);
		    coded++;
		    if (coded + 1 < vl -> sz)
		       { fprintf (f, " && ");
			 if (coded%4 == 3) fprintf (f, "\n\t      ");
		       }
		  };
	     };
	  fprintf (f, ")\n");
	  fprintf (f, "\t     { if (val == 0) ");
	  fprintf (f, "{ enqueue (%d, time+or_tpll, 0); }\n",
			lnr - proclowend);
	  fprintf (f, "\t       else enqueue (%d, time+or_tphh, 1);\n",
			lnr - proclowend);
	  fprintf (f, "\t     }\n");
	  fprintf (f, "\t  else check_enqueue (%d, time+or_tphh, 1);\n",
			lnr - proclowend);
	};

/*
   Recognizing atoms
*/
typedef struct
	{ char *atomname;
	  void (*atomcode) ();
	} init_atom;

init_atom known_atoms [] =
	{{ "zero",  code_zero},
	 { "one",   code_one},
	 { "buf",   code_buf},
	 { "not",   code_not},
	 { "nand",  code_nand },
	 { "nand2", code_nand },
	 { "nand3", code_nand },
	 { "nand4", code_nand },
	 { "nand5", code_nand },
	 { "nand6", code_nand },
	 { "nand7", code_nand },
	 { "nand8", code_nand },
	 { "and",  code_and },
	 { "and2", code_and },
	 { "and3", code_and },
	 { "and4", code_and },
	 { "and5", code_and },
	 { "and6", code_and },
	 { "and7", code_and },
	 { "and8", code_and },
	 { "nor",  code_nor },
	 { "nor2", code_nor },
	 { "nor3", code_nor },
	 { "nor4", code_nor },
	 { "nor5", code_nor },
	 { "nor6", code_nor },
	 { "nor7", code_nor },
	 { "nor8", code_nor },
	 { "or",  code_or },
	 { "or2", code_or },
	 { "or3", code_or },
	 { "or4", code_or },
	 { "or5", code_or },
	 { "or6", code_or },
	 { "or7", code_or },
	 { "or8", code_or },
/*	 { "xor",   code_generic_mport, "xor2" },
	 { "xor2",  code_generic_mport, "xor2" },
	 { "dff",   code_dff,           "dff" },
	 { "tff",   code_tff,           "tff" },
	 { "jkff",  code_jkff,          "jkff" }};
*/
	};
/*
   coding of atom applications
*/
static void code_atom_application (f, atnm, lhs, atvpar, atcpar, nr)
 FILE *f;
 symbol atnm;
 val lhs;
 parval_list atvpar;
 val atcpar;
 int nr;
	{ int ix;
	  char *atomname = name_from_symbol (atnm);
	  for (ix = 0; ix < nr_of_elts (known_atoms); ix++)
	     if (streq (known_atoms[ix].atomname, atomname))
		{ known_atoms[ix].atomcode (f, lhs, atvpar, atcpar, nr);
		  rfre_string (atomname);
		  return;
		};
	  fprintf (stderr, "Unknown atom: %s\n", atomname);
	  exit (1);
	};


/*
   code all the includes of the generated program
*/
static void code_include_header (f)
 FILE *f;
	{ fprintf (f, "#include <stdio.h>\n");
	  fprintf (f, "#include <Intrinsic.h>\n");
	  fprintf (f, "#include <StringDefs.h>\n");
	  fprintf (f, "#include <cursorfont.h>\n");
	  fprintf (f, "#include <Xaw/Form.h>\n");
	  fprintf (f, "#include <Shell.h>\n");
	  fprintf (f, "#include <Xaw/Paned.h>\n");
	  fprintf (f, "#include \"XtArgs.h\"\n");
	  fprintf (f, "#include \"Signalmgr.h\"\n");
	  fprintf (f, "#include \"Signal.h\"\n");
	  fprintf (f, "#include \"Command.h\"\n");
	  fprintf (f, "#include \"Toggle.h\"\n\n");
	  fprintf (f, "#define MaxTime 2000\n");
	  fprintf (f, "#define MaxSignals %d\n", procnr - proclowend);
	  fprintf (f, "#include \"DiscEvents.c\"\n");
	  fprintf (f, "#include \"DiscEvProp.h\"\n");
	  fprintf (f, "\n");
	};

/*
   generate the tail of a routine
*/
static code_routinetail (f)
 FILE *f;
	{ fprintf (f, "\t};\n\n");
	};

/*
   code signal procs
*/
static void code_signal_proc_header (f, nr)
 FILE *f;
 int nr;
	{ fprintf (f, "static void p%d (time, val)\n", nr);
	  fprintf (f, " int time,val;\n");
	  fprintf (f, "\t{\n");
	};

static int val_contains (v, nr)
 val v;
 int nr;
	{ switch (v -> tag)
	     { case TAGVSym: return (getprior (v -> VSym.sym) == nr);
	       case TAGVList:
		  { val_list vl = v -> VList.l;
		    register int ix;
		    for (ix = 0; ix < vl -> sz; ix++)
		       if (val_contains (vl -> arr[ix], nr)) return (1);
		    return (0);
		  };
	       default: badtag (v -> tag);
	     };
	};

static void code_signal_proc_body (f, nr)
 FILE *f;
 int nr;
	{ int ix;
	  for (ix = 0; ix < atom_appl_list -> sz; ix++)
	     { def d = atom_appl_list -> arr[ix];
	       val lhs = d -> DefCon.defcon;
	       val rhs = d -> DefCon.conas;
	       symbol atnm = rhs -> VAtom.atnm;
	       parval_list atvpar = rhs -> VAtom.atvpar;
	       val atcpar = rhs -> VAtom.atcpar;
	       if (val_contains (atcpar, nr))
		  code_atom_application (f, atnm, lhs, atvpar, atcpar, nr);
	     };
	};

static void code_signal_procs (f)
 FILE *f;
 	{ register int ix;
	  for (ix = proclowend; ix < procnr; ix++)
	     { code_signal_proc_header (f, ix);
	       code_signal_proc_body (f, ix);
	       code_routinetail (f);
	     };
	};

/*
   code recompute header
*/
static code_recompute_header (f)
 FILE *f;
	{ fprintf (f, "static void ReCompute ()\n");
	  fprintf (f, "\t{ int prevval, ix, len;\n");
	  fprintf (f, "\t  int *loc;\n");
	  fprintf (f, "\t  int max_deq = 1;\n\n");
	  fprintf (f, "\t  for (ix = 0; ix < MaxSignals; ix++) ");
	  fprintf (f, "enqueue (ix, 0, 0);\n");
	};

/*
   code the nullary atoms
*/
static void code_nullary_atoms (f)
 FILE *f;
	{ int ix;
	  for (ix = 0; ix < atom_appl_list -> sz; ix++)
	     { def d = atom_appl_list -> arr [ix];
	       val lhs = d -> DefCon.defcon;
	       val rhs = d -> DefCon.conas;
	       symbol atnm = rhs -> VAtom.atnm;
	       val atcpar = rhs -> VAtom.atcpar;
	       parval_list atvpar = rhs -> VAtom.atvpar;
	       if (atcpar -> tag != TAGVList) continue;
	       if (atcpar -> VList.l -> sz != 0) continue;
	       code_atom_application (f, atnm, lhs, atvpar, atcpar, 0);
	     };
	};

/*
   code fetching the data from the widget
*/
static void code_obtain_from_widget (f, nr)
 FILE *f;
 int nr;
	{ fprintf (f, "\t  StartArgs;\n");
	  fprintf (f, "\t  SetArg (XtNsample, &loc);\n");
	  fprintf (f, "\t  SetArg (XtNsampleLength, &len);\n");
	  fprintf (f, "\t  XtGetValues (Sigs[%d].ws, UseArgs);\n",
				nr - proclowend);
	  fprintf (f, "\t  if (len > max_deq) max_deq = len;\n");
	  fprintf (f, "\t  prevval = 0;\n");
	  fprintf (f, "\t  for (ix = 0; ix < len; ix++)\n");
	  fprintf (f, "\t     if (loc[ix] != prevval)\n");
	  fprintf (f, "\t\t{ enqueue (%d, ix, loc[ix]);\n", nr - proclowend);
	  fprintf (f, "\t\t  prevval = loc [ix];\n");
	  fprintf (f, "\t\t};\n");
	  fprintf (f, "\t  if (prevval != 0) enqueue (%d, len, 0);\n",
				nr - proclowend);
	};

static void code_obtain_from_val (f, v)
 FILE *f;
 val v;
	{ switch (v -> tag)
	     { case TAGVSym:
		  code_obtain_from_widget (f, getprior (v -> VSym.sym));
		  break;

	       case TAGVList:
	          { val_list vl = v -> VList.l;
		    register int ix;
		    for (ix = 0; ix < vl -> sz; ix++)
		       code_obtain_from_val (f, vl -> arr[ix]);
	          };
		  break;

	       default: badtag (v -> tag);
	     };
	};

static void code_obtain_from_formcon (f, fc)
 FILE *f;
 formcon fc;
	{ switch (fc -> tag)
	     { case TAGFCSym:
		  code_obtain_from_widget (f, getprior (fc -> FCSym.sym));
		  break;

	       case TAGFCList:
		  { formcon_list fcl = fc -> FCList.l;
		    register int ix;
		    for (ix = 0; ix < fcl -> sz; ix++)
			code_obtain_from_formcon (f, fcl -> arr [ix]);
		  };
		  break;

	       default: badtag (fc -> tag);
	     };
	};

static void code_obtain_from_defs (f, dl)
 FILE *f;
 def_list dl;
	{ register int ix;
	  for (ix = 0; ix < dl -> sz; ix++)
	     code_obtain_from_val (f, dl -> arr[ix] -> DefCon.defcon);
	};

static void code_obtain_from_widgets (f)
 FILE *f;
	{ if (syn_list == def_listNIL)
	     { code_obtain_from_formcon (f, deffc);
	     }
	  else
	     { code_obtain_from_defs (f, syn_list);
	     };
	};

/*
   code the main loop of recompute
*/
static void code_recompute_main_loop (f)
 FILE *f;
	{ fprintf (f, "\t  while (the_queue != empty_queue)\n");
	  fprintf (f, "\t     { int nr, time, val;\n");
	  fprintf (f, "\t       dequeue (&nr, &time, &val);\n");
	  fprintf (f, "\t       if (time >= MaxTime) break;\n");
	  fprintf (f, "\t       if (time > max_deq) max_deq = time;\n");
	  fprintf (f, "\t       update (nr, time, val);\n");
	  fprintf (f, "\t       Sigs [nr].event_proc (time, val);\n");
	  fprintf (f, "\t     };\n\n");
	};

/*
   code the updating of the window
*/
static void code_update_widget (f, nr)
 FILE *f;
 int nr;
	  { fprintf (f, "\t  update (%d, max_deq, Sigs[%d].cv);\n",
				nr - proclowend, nr - proclowend);
	    fprintf (f, "\t  StartArgs;\n");
	    fprintf (f, "\t  SetArg (XtNsampleLength, max_deq);\n");
	    fprintf (f, "\t  SetArg (XtNsample, Sigs[%d].sig);\n",
				nr - proclowend);
	    fprintf (f, "\t  XtSetValues (Sigs[%d].ws, UseArgs);\n",
				nr - proclowend);
	  };

static void code_update_from_val (f, v)
 FILE *f;
 val v;
	{ switch (v -> tag)
	     { case TAGVSym:
		  code_update_widget (f, getprior (v -> VSym.sym));
		  break;

	       case TAGVList:
		  { val_list vl = v -> VList.l;
		    register int ix;
		    for (ix = 0; ix < vl -> sz; ix++)
		       code_update_from_val (f, vl -> arr[ix]);
		  };
		  break;

	       default: badtag (v -> tag);
	     };
	};

static void code_update_widgets (f)
 FILE *f;
	{ code_update_from_val (f, output_val);
	};

static void code_recompute (f)
 FILE *f;
	{ code_recompute_header (f);
	  code_nullary_atoms (f);
	  code_obtain_from_widgets (f);
	  code_recompute_main_loop (f);
	  code_update_widgets (f);
	  code_routinetail (f);
	};

/*
   code the main program
*/
/*
   code signal administration initialisation
*/
static void code_init_sigadm (f)
 FILE *f;
	{ register int ix;
	  fprintf (f, "\t  for (ix=0; ix<MaxSignals; ix++)\n");
	  fprintf (f, "\t     { Sigs[ix].time = 0;\n");
	  fprintf (f, "\t       Sigs[ix].cv = 0;\n");
	  fprintf (f, "\t       Sigs[ix].ws = NULL;\n");
	  fprintf (f, "\t     };\n\n");
	  for (ix = proclowend; ix < procnr; ix++)
	     fprintf (f, "\t  Sigs[%d].event_proc = p%d;\n",
			ix - proclowend, ix);
	};

/*
   code voor de signal widget initialisatie
*/
static void code_init_widget (f, name, nr, father, ed)
 FILE *f;
 char *name;
 int nr;
 char *father;
 int ed;
	  { fprintf (f,"\t  StartArgs;\n");
	    fprintf (f,"\t  SetArg (XtNradioGroup, button);\n");
	    fprintf (f,"\t  button = XtCreateManagedWidget (\"%s\",\n", name);
	    fprintf (f,"\t\ttoggleWidgetClass, %s, UseArgs);\n", father);
	    fprintf (f,"\t  StartArgs;\n");
	    fprintf (f,"\t  SetArg (XtNlevels, %d);\n", 2);
	    fprintf (f,"\t  SetArg (XtNtimeScale, 3);\n");
	    fprintf (f,"\t  SetArg (XtNeditable, %d);\n", ed);
	    fprintf (f,"\t  SetArg (XtNsample, init_sig);\n");
	    fprintf (f,"\t  SetArg (XtNmaxSampleLength, MaxTime);\n");
	    fprintf (f,"\t  SetArg (XtNsampleLength, XtNumber (init_sig));\n");
	    fprintf (f,"\t  Sigs[%d].ws = XtCreateManagedWidget (\"signal\",\n",
				nr - proclowend);
	    fprintf (f,"\t\tsignalWidgetClass, %s, UseArgs);\n", father);
	  };

static void code_init_widget_from_formcon (f, fc)
 FILE *f;
 formcon fc;
	{ switch (fc -> tag)
	     { case TAGFCSym:
		  code_init_widget (f, symbolstr (fc -> FCSym.sym),
				getprior (fc -> FCSym.sym),
				"inputs", 1);
		  break;

	       case TAGFCList:
		  { formcon_list fcl = fc -> FCList.l;
		    register int ix;
		    for (ix = 0; ix < fcl -> sz; ix++)
			code_init_widget_from_formcon (f, fcl -> arr [ix]);
		  };
		  break;

	       default: badtag (fc -> tag);
	     };
	};

static void code_init_widget_from_val (f, name, father, v, ed)
 FILE *f;
 char *name;
 char *father;
 val v;
 int ed;
	{ switch (v -> tag)
	     { case TAGVSym:
		  code_init_widget (f, name, getprior (v -> VSym.sym),
					father, ed);
		  break;

	       case TAGVList:
		  { val_list vl = v -> VList.l;
		    register int ix;
		    for (ix=0; ix < vl -> sz; ix++)
		       { char Buf[40];
			 sprintf (Buf, "%s.%d", name, ix);
			 code_init_widget_from_val (f, Buf, father,
					vl -> arr[ix], ed);
		       };
		  };
		  break;

	       default: badtag (v -> tag);
	     };
	};

static void code_init_widgets_from_defs (f, dl)
 FILE *f;
 def_list dl;
	{ register int ix;
	  for (ix = 0; ix < dl -> sz; ix++)
	     { val lhs, rhs;
	       char *srcname;
	       def d = dl -> arr[ix];
	       lhs = d -> DefCon.defcon;
	       rhs = d -> DefCon.conas;
	       if (rhs -> tag != TAGVSym) IllegalFormat ();
	       srcname = name_from_symbol (rhs -> VSym.sym);
	       code_init_widget_from_val (f, srcname, "inputs", lhs, 1);
	       rfre_string (srcname);
	     };
	};

static void code_init_widgets (f)
 FILE *f;
	{ if (syn_list == def_listNIL)
	     { code_init_widget_from_formcon (f, deffc);
	     }
	  else
	     { code_init_widgets_from_defs (f, syn_list);
	     };
	  code_init_widget_from_val (f, "out", "outputs", output_val, 0);
	};

/*
   the actual main part is included
*/
static void code_mainprogram (f)
 FILE *f;
	{ fprintf (f,"#include \"DiscEvMain.c\"\n");
	  code_init_sigadm (f);
	  code_init_widgets (f);
	  fprintf (f,"\t  XtRealizeWidget (top);\n");
	  fprintf (f,"\t  XtMainLoop ();\n");
	  code_routinetail (f);
	};

/*
   code the program
*/
static void code (f)
 FILE *f;
	{ fprintf (stderr, "discev: coding...\n");
	  code_include_header (f);
	  code_signal_procs (f);
	  code_recompute (f);
	  code_mainprogram (f);
	};

/*
   Load all the definitions
*/
static void load (f, dl)
 FILE *f;
 def_list *dl;
	{ if (fscan_def_list (f, dl))
	     { fprintf (stderr, "Read error: (%d): %s\n", tmlineno, tmerrmsg);
               exit (1);
	     };
	};

main (argc, argv)
 int argc;
 char *argv [];
	{ def_list all_defs;
	  initsymbol ();
	  scanargs (argc, argv);
	  tmlineno = 1;
	  load (infile, &all_defs);
	  prepare (all_defs);
	  rfre_def_list (all_defs);
	  code (outfile);
	  if (stat)
	     { flushsymbol ();
	       stat_ds (stderr);
	       stat_string (stderr);
	     };
	}
