-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: dmain.p
-- Author: Andy Lowry
-- SCCS Info: @(#)dmain.p	1.5 3/13/90

-- This process presents the rmain interface to application programs,
-- and also establishes network services and can exchange resource
-- manager capabilities with other interpreters.  The first command
-- line argument should be the interpreter invocation number for this
-- interpreter on the current machine.  Following can be any number
-- (including zero) of link specifications which cause this
-- interpreter to import resource manager capabilities from other
-- interpreters.  Each link specification takes the form:
--
--		-link hostname n
--
-- where hostname is the name of the machine on which the target
-- interpreter is running, and n is the interpreter invocation number
-- of that interpreter.  A chained resource manager is created
-- containing a local resource manager and all the resource linked
-- managers, in the order specified.  

-- All following command line arguments are interpreted as command
-- lines to be invoked via the rMain or sysRMain interface.
-- Semicolons appear between commands.  If a command begins with the
-- word "-sys", then the remainder of the command is invoked via the
-- sysRMain interface, else via the rMain interface.  All the commands
-- are invoked in sequence.

dmain: using (rManager, sysRManager, chainedRM, userRM, rmain,
  distributed, main, common, string, linkServer, stdio)
linking (rManager, chainedSysRM, userRM, atoi, linkServer, ac_none)

process (Q: main_q)

declare
  args: main_intf;
  localRM: sysRManager;
  RMList: sysRMList;
  chainedRM: sysRManager;
  rm: rManager;
  argv: charStringList;
  arg: charString;
  prefix: charString;
  userName: charString;
  interpNo: integer;
  atoi: string2IntFn;
  startNet: startNetworkFn;
  setLink: setLinkingPortFn;
  getLink: getLinkFn;
  remoteRM: sysRManager;
begin
  receive args from Q;

  atoi <- procedure of process atoi;

  unwrap startNet from args.CLoader("start network") {init};
  unwrap setLink from args.CLoader("set linking port") {init};
  unwrap getLink from args.CLoader("get link") {init};
  -- strip name of this program (dmain) from argument vector
  argv := every of a in args.argv where (position of a <> 1);
  remove prefix from argv[0];

  -- create local resource manager and install it in our RM list
  localRM <- rManagerInitFn#(create of process rManager)();
  new RMList;
  insert copy of localRM into RMList;

  -- get our interpreter invocation number and all the linking
  -- specifications
  remove arg from argv[0];
  interpNo <- atoi(arg);
  call startNet(interpNo);
  
  block begin
      while (argv[0] = "-link") repeat
	  block declare
	      hostname: charString;
	      rInterpNo: integer;
	      linkFn: genericFn;
	      linkObj: polymorph;
	      empty: empty;
	    begin
	      remove arg from argv[0];	-- this is "-link"
	      remove hostname from argv[0];
	      remove arg from argv[0];
	      rInterpNo <- atoi(arg);
	      linkFn <- getLink(hostName, rInterpNo);
	      wrap empty as linkObj;
	      call linkFn(linkObj);
	      unwrap remoteRM from linkObj {
		  init, init(post), init(insert), init(get), init(remove), 
		  init(delete)};
	      insert remoteRM into RMList;
	    end block;
	end while;
    on (NotFound)
    end block;
  
  -- Create the chained resource manager, which will handle all the
  -- managers we just created
  chainedRM <- chainedSysRMInitFn#(create of process chainedSysRM)(RMList);

  -- Fire up a process that will give our resource manager out to
  -- those who ask for it
  block declare
    linkServer: linkServerFn;
  begin
    linkServer <- create of process linkServer;
    call linkServer(chainedRM, setLink);
  end block;
    
  -- Install basic system resources into the local resource manager
  block declare
    resource: polymorph;
  begin
    wrap copy of args.environ as resource;
    call localRM.post("system", "environ",
      resource, create of process ac_none);
    wrap copy of args.CLoader as resource;
    call localRM.post("system", "CLoader", resource, 
      create of process ac_none);
    wrap copy of args.unix as resource;
    call localRM.post("system", "unix", resource, create of process ac_none);
    
    -- add all the stdenv functions as resources
    wrap copy of args.std.load as resource;
    call localRM.post("system", "load", resource, create of process ac_none);
    wrap copy of args.std.pathLoad as resource;
    call localRM.post("system", "pathLoad", resource, 
      create of process ac_none);
    wrap copy of args.std.readObj as resource;
    call localRM.post("system", "readObj", resource, 
      create of process ac_none);
    wrap copy of args.std.pathReadObj as resource;
    call localRM.post("system", "pathReadObj", resource, 
      create of process ac_none);
    wrap copy of args.std.writeObj as resource;
    call localRM.post("system", "writeObj", resource, 
      create of process ac_none);
    wrap copy of args.std.libWriteObj as resource;
    call localRM.post("system", "libWriteObj", resource, 
      create of process ac_none);
    wrap copy of args.std.store as resource;
    call localRM.post("system", "store", resource, create of process ac_none);
    wrap copy of args.std.libStore as resource;
    call localRM.post("system", "libStore", resource, 
      create of process ac_none);
    wrap copy of args.std.getCwd as resource;
    call localRM.post("system", "getCwd", resource, create of process ac_none);
    wrap copy of args.std.setCwd as resource;
    call localRM.post("system", "setCwd", resource, create of process ac_none);
    wrap copy of args.std.terminal as resource;
    call localRM.post("system", "terminal", resource, 
      create of process ac_none);
    block declare
      stdin: stdin;
      stdout: stdout;
    begin
      new stdin;
      stdin.getChar := args.std.terminal.getChar;
      stdin.getString := args.std.terminal.getString;
      new stdout;
      stdout.putChar := args.std.terminal.putChar;
      stdout.putString := args.std.terminal.putString;
      stdout.putLine := args.std.terminal.putLine;
      wrap stdin as resource;
      call localRM.post("system", "stdin", resource, 
	create of process ac_none);
      wrap stdout as resource;
      call localRM.post("system", "stdout", resource,
	create of process ac_none);
    end block;
  end block;

  -- get the user name for running user-level processes
  block begin
    inspect entry in args.environ["USER"] begin
      userName := entry.value;
    end inspect;
  on (NotFound)
    block begin
      inspect entry in args.environ["LOGNAME"] begin
	userName := entry.value;
      end inspect;
    on (NotFound)
      userName := "anonymous";
    end block;
  end block;
  
  -- now invoke specified commands
  while size of argv <> 0 repeat
    block declare
      semiPos: integer;
      thisArgv: charStringList;
      programName: charString;
      sys: boolean;
      resource: polymorph;
      rMain: rMainFn;
      sysRMain: sysRMainFn;
    begin
      block begin
	inspect a in argv where (a = ";") begin
	  semiPos <- position of a;
	end inspect;
      on (notFound)
	semiPos <- size of argv;
      end block;
      extract thisArgv from a in argv where (position of a < semiPos);
      if size of argv <> 0 then
	remove arg from argv[0];-- get rid of semicolon
      end if;
      if size of thisArgv = 0 then
	exit continue;
      end if;
      -- check which interface to use for this program
      if thisArgv[0] = "-sys" then
	sys <- 'true';
	remove arg from thisArgv[0];
	if size of thisArgv = 0 then
	  exit continue;
	end if;
      else
	sys <- 'false';
      end if;
      -- extract program name for this program, and add prefix to arg
      -- vector
      programName <- thisArgv[0];
      insert copy of prefix into thisArgv at 0;
      wrap thisArgv as resource;
      call chainedRM.post("system", "argv", resource, 
	create of process ac_none);
      if sys then
	-- invoke program using system level resource manager... each
	-- much retrieve and copy its argv resource before returning
	sysRMain <- create of args.std.pathLoad(programName);
	call sysRMain(chainedRM);
      else
	rm <- userRMFn#(create of process userRM)(userName, chainedRM);
	rMain <- create of args.std.pathload(programName);
	call rMain(rm);
      end if;
      call chainedRM.delete("system", "argv");
    on exit(continue)
    end block;
  end while;
  
  return args;

end process
