-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: acpipe.p
-- Author: Andy Lowry
-- SCCS Info: @(#)acpipe.p	1.2 3/13/90

-- This process serves as an access control function for pipe end
-- resources for processes launched by the plumber.  The pipe is
-- created when the first request for a pipe end comes in.  The pipe
-- type is taken from the user parameter in the request.  Currently,
-- the pipe type is used to construct a process name which is used to
-- dynamically load the code that will create the pipe.  For example,
-- a pipe of type "foo" would be created by a process named
-- "makefoopipe".

-- When a request for the second end of the pipe arrives, the type
-- specified therein must match the type in the first request, else
-- the second request and all subsequent requests for either pipe end
-- are denied.  In any case, all but the first request for either pipe
-- end are denied.

-- The initialization message must provide a process loading
-- capability of type load!load_func.  Two access capabilities, for
-- accessing the two ends of the pipe, are returned.  The resource
-- stored in the resource manager is not used... this access function
-- creates the real resources and hands them out.

acPipe: using (acPipe, makePipe, rmanager, common, load)

process (Q: acPipeQ)

declare
  args: acPipe;
  load: load_func;
  maker: makePipeFn;
  sourceQ: accessQ;
  sinkQ: accessQ;
  pipeReq: access;
  requestor: pipeEnd;
  type: charString;
  sourceDone: boolean;
  sinkDone: boolean;
  output: polymorph;
  input: polymorph;
  empty: empty;
begin
  receive args from Q;

  -- save the load capability and create access ports
  load := args.load;
  new sourceQ;
  connect args.sourceAC to sourceQ;
  new sinkQ;
  connect args.sinkAC to sinkQ;
  return args;
  discard Q;

  -- neither end has been given out so far
  sourceDone <- 'false';
  sinkDone <- 'false';

  -- wait for first request for a pipe end
  select
  event sourceQ
    receive pipeReq from sourceQ;
    requestor <- 'source';
  event sinkQ
    receive pipeReq from sinkQ;
    requestor <- 'sink';
  otherwise
    exit cantHappen;
  end select;

  -- create the pipe according to the type specified in the first request
  type := pipeReq.parameter;
  maker <- makePipeFn#(create of program#(load("make" | type | "pipe")));
  call maker(output, input);

  -- fill the first request
  if requestor = 'source' then
    pipeReq.resource <- output;
    sourceDone <- 'true';
    wrap empty as output;	-- so typestate won't discard this in
				-- the other case
  else
    pipeReq.resource <- input;
    sinkDone <- 'true';
    wrap empty as input;	-- ditto
  end if;
  return pipeReq;

  -- Deny any future requests for the end we just returned, but accept
  -- a conforming request for the other end and then deny all
  -- subsequent requests for that end as well
  while 'true' repeat
    select
    event sourceQ and where (sourceDone and sinkDone)
      -- deny second source request
      receive pipeReq from sourceQ;
      return pipeReq exception accessDenied;

    event sourceQ and where (not sourceDone)
      -- accept first source request if its type matches that of the
      -- sink request
      receive pipeReq from sourceQ;
      if pipeReq.parameter = type then
	pipeReq.resource <- output;
	wrap empty as output;	-- again, for typestate's sake...
	return pipeReq;
	sourceDone <- 'true';	-- deny all following requests
      else
	return pipeReq exception accessDenied; -- pipe type mismatch
      end if;

    event sinkQ and where (sinkDone and sourceDone)
      -- deny second sink request
      receive pipeReq from sinkQ;
      return pipeReq exception accessDenied;

    event sinkQ and where (not sinkDone)
      -- accept first sink request if its type matches that of the
      -- source request
      receive pipeReq from sinkQ;
      if pipeReq.parameter = type then
	pipeReq.resource <- input;
	wrap empty as input;	-- for typestate
	return pipeReq;
	sinkDone <- 'true';
      else
	return pipeReq exception accessDenied; -- pipe type mismatch
      end if;

    otherwise
    end select;
  end while;

on exit(cantHappen)
  
end process
  
