-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: makestdiopipe.p
-- Author: Andy Lowry
-- SCCS Info: @(#)makestdiopipe.p	1.2 3/13/90

-- This process creates stdio!stdin and stdio!stdout capability
-- records and then handles requests from those capabilities.  The
-- data from output requests are used to fill input requests.

makeStdioPipe: using (stdio, terminalIO, common, makePipe)

process (Q: makePipeQ)
  
declare
  args: makePipe;
  stdin: stdin;
  stdout: stdout;
  getCharQ: getCharQ;
  getStringQ: getStringQ;
  putCharQ: putCharQ;
  putStringQ: putStringQ;
  putLineQ: putStringQ;
begin
  receive args from Q;
  
  -- allocate all the ports and build the stdin and stdout records
  new getCharQ;
  new getStringQ;
  new stdin;
  connect stdin.getChar to getCharQ;
  connect stdin.getString to getStringQ;
  
  new putCharQ;
  new putStringQ;
  new putLineQ;
  new stdout;
  connect stdout.putChar to putCharQ;
  connect stdout.putString to putStringQ;
  connect stdout.putLine to putLineQ;
  
  -- wrap up the pipe ends and return them
  wrap stdin as args.input;
  wrap stdout as args.output;
  return args;
  discard Q;
  
  -- Now set up to server the requests
  block declare
    data: charStringList;	-- pending data not yet sent to sink
				-- end of pipe... guaranteed no
				-- empty strings here
    putCharData: charString;	-- pending data received via putChar
				-- requests since last putString or putLine
    nextData: charString;	-- next string to be sent on input requests
    haveData: boolean;		-- 'true' means either data or nextData
				-- putCharData is nonempty
    endOfInput: boolean;	-- true when all our stdout ports are
				-- disconnected
    getData: charString;	-- data for a partially filled
				-- getString request (no newline found yet)
    gcArgs: getCharIntf;
    gsArgs: getStringIntf;
    pcArgs: putCharIntf;
    psArgs: putStringIntf;
  begin
    new data;
    nextData <- "";
    putCharData <- "";
    endOfInput <- 'false';
    getData <- "";

    while 'true' repeat
      if nextData = "" then
	if size of data <> 0 then
	  remove nextData from data[0];
	else
	  nextData <- putCharData;
	  putCharData <- "";
	end if;
      end if;
      haveData <- nextData <> "";

      block begin
	select
	event getCharQ and where (haveData)
	  receive gcArgs from getCharQ;
	  remove gcArgs.char from nextData[0];
	  return gcArgs;

	event getCharQ and where (endOfInput and not haveData)
	  receive gcArgs from getCharQ;
	  return gcArgs exception endOfInput;

	event getStringQ and where (haveData)
	  -- Transfer chars from next data string to getData until we
	  -- hit a newline, then field the pending getString request.
	  -- If no newline present, don't dequeue the request and
	  -- loop, waiting for more data
	  block declare
	    c: char;
	  begin
	    remove c from nextData[0];
	    while (c <> 'NL') repeat
	      insert c into getData;
	      remove c from nextData[0];
	    end while;
	    receive gsArgs from getStringQ;
	    gsArgs.string <- getData;
	    return gsArgs;
	    getData <- "";
	  on (notFound)
	    -- no newline... wait for more data
	  end block;

	event getStringQ and where (endOfInput and not haveData)
	  receive gsArgs from getStringQ;
	  return gsArgs exception endOfInput;

	event putCharQ
	  receive pcArgs from putCharQ;
	  insert copy of pcArgs.char into putCharData;
	  return pcArgs;

	event putStringQ
	  receive psArgs from putStringQ;
	  if psArgs.string <> "" then
	    if putCharData <> "" then
	      insert putCharData into data;
	      putCharData <- "";
	    end if;
	    insert copy of psArgs.string into data;
	  end if;
	  return psArgs;

	event putLineQ
	  receive psArgs from putLineQ;
	  if psArgs.string <> "" then
	    if putCharData <> "" then
	      insert putCharData into data;
	      putCharData <- "";
	    end if;
	    insert copy of psArgs.string into data;
	  end if;
	  return psArgs;
	  insert 'NL' into putCharData;

	otherwise
	  exit cantHappen;

	end select;
	
      on (disconnected)
	-- First time this happens it must mean all the putXXX ports
	-- have dropped, so we should signal end of input for all
	-- future getXXX requests.  If it happens again, it means the
	-- getXXX ports have also dropped, so we should really just go
	-- away.
	if endOfInput then
	  exit done;
	else
	  endOfInput <- 'true';
	end if;
	
      end block;
    end while;
  end block;
  
on exit(done)
  -- both ends of pipe have been dropped
on exit(cantHappen)
end process
