-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: parsepipe.p
-- Author: Andy Lowry
-- SCCS Info: @(#)parsepipe.p	1.2 3/13/90

-- Parses a pipe specification from a pshell command line.  A pipe
-- specification consists of everything from the vertical bar through
-- the optional end names.  This process assumes the vertical bar has
-- already been removed from the tokens list.

parsePipe: using (pshell, tokenize)

process (Q: parsePipeQ)
  
declare
  args: parsePipe;
  sourceName: charString;
  destName: charString;
  state: pipeState;
  tok: token;
  delim: char;
begin
  receive args from Q;
  
  -- set defaults for all return values and set initial state
  sourceName <- "stdout";
  destName <- "stdin";
  state <- 'start';
  
  -- get first token... if none, we avoid entering the state machine
  -- and return the defaults
  block begin
    block begin
      remove tok from args.toks[0];
    on (notFound)
      exit doneState;
    end block;

    while state <> 'done' repeat
      -- retrieve delimiter char now to simplify select guards below
      if case of tok = 'delimiter' then
	reveal tok.delimiter;
	delim := tok.delimiter;
      else
	delim <- ' ';
      end if;
      
      select state
      where ('start')
	-- If next token is not a left bracket we put it back to be
	-- rescanned and terminate using defaults.  Otherwise we scan
	-- for parms
	if case of tok = 'delimiter' and delim = '[' then
	  -- advance token and state
	  block begin
	    remove tok from args.toks[0];
	    state <- 'name1';
	  on (notFound)
	    exit syntaxError;
	  end block;
	else
	  -- No pipe parms... push this token back for rescanning and finish
	  insert copy of tok into args.toks at 0;
	  state <- 'done';
	end if;

      where ('name1')
	-- Just saw a left bracket... optional word token is source end name
	if case of tok = 'word' then
	  reveal tok.word;
	  sourceName := tok.word;
	  -- advance token, syntax error if can't
	  block begin
	    remove tok from args.toks[0];
	  on (notFound)
	    exit syntaxError;
	  end block;
	end if;
	-- now go try to get a comma
	state <- 'comma';

      where ('comma')
	-- we just parsed the optional source pipe name, now looking for
	-- optional destination name preceded by a comma
	if case of tok = 'delimiter' and delim = ',' then
	  state <- 'name2';
	  -- advance token, error if unable
	  block begin
	    remove tok from args.toks[0];
	  on (notFound)
	    exit syntaxError;
	  end block;
	else
	  -- no comma, so skip the a destination name
	  state <- 'rBracket';
	end if;

      where ('name2')
	-- got a comma to introduce optional destination pipe name,
	-- which must be a word
	if case of tok = 'word' then
	  reveal tok.word;
	  destName := tok.word;
	  -- advance token, syntax error if unable
	  block begin
	    remove tok from args.toks[0];
	  on (notFound)
	    exit syntaxError;
	  end block;
	end if;
	-- Now go get closing bracket
	state <- 'rBracket';

      where ('rBracket')
	-- Here to close off the pipe names specification... must have
	-- right bracket next.  If so, we're all finished.
	if case of tok = 'delimiter' and delim = ']' then
	  state <- 'done';
	else
	  exit syntaxError;
	end if;

      otherwise
	exit cantHappen;
      end select;
    end while;
  
  on exit(doneState)
    -- Here to avoid entering the while loop in 'done' state with tok
    -- uninit, which causes typestate problems
  end block;

  -- Here when we've finished parsing the pipe spec... wrap it up in a
  -- pipeInfo structure with bogus source and destination procID's and
  -- return it
  new args.pipe;
  args.pipe.id <- unique;
  args.pipe.source <- unique;
  args.pipe.sourceName <- sourceName;
  args.pipe.dest <- unique;
  args.pipe.destName <- destName;

  return args;
  
on exit (syntaxError)
  return args exception syntaxError;

end process
