-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: pshell.p
-- Author: Andy Lowry
-- SCCS Info: @(#)pshell.p	1.3 3/13/90

-- This module implements a shell that is capable of launching
-- generalized pipelines of processes according to a "plumbing" syntax
-- as described below.  The pipelines are more flexible than those
-- supported by Unix shells in that pipes can be of types other than
-- "char" and the connection network can be non-linear.  The shell
-- normally waits for a pipeline to finish before prompting for a new
-- command.  If a command is terminated with an ampersand, however,
-- the shell will start the pipeline and then immediately reissue its
-- prompt and accept additional commands.

-- If there are command line arguments available, they are
-- concatenated with intervening blanks to form a command line to be
-- parsed and executed.  In this case, no prompt is issued and the
-- shell terminates after executing the single command.

-- The shell prompt is taken from the HPROMPT environment variable if
-- it exists; it defaults to "pshell".  The following commands are
-- implemented directly:
--	cwd - change the current working directory if a setCwd
--		resource is available
--	pwd - print the current working directory if a getCwd resource
--		is available
--	exit - return from the shell to its caller

-- The plumbing syntax is as follows:

--    command := network | network '&'
--    network := pipeline | pipeline `;' network
--    pipeline := joint | joint pipe pipeline
--    named-process := process | label `:' | label `:' process
--    joint := named-process | named-process `<' pipe joint `>'
--             | `<' joint pipe `>' named-process
--    pipe := `|' | `|' `:' opt-typename opt-endnames
--    opt-typename := empty | typename
--    opt-endnames := empty | `[' opt-endname `]'
--		      | `[' opt-endname `,' opt-endname `]'
--    opt-endname := empty | endname
--    process := name args
--    args := empty | arg args
--    arg := WORD | QUOTED-STRING | PUNCTUATION
--    name | endname | typename | label := WORD
--    empty :=

-- A "pipeline" is a linear network of "processes", with each process
-- receiving input from its predecessor and sending output to its
-- successor, via "pipes".  A process is specified via a "name" (which
-- is used to load the program to be run in the process) and a list of
-- command-line "args".  Quote marks can be placed around an "arg" to
-- cause enclosed characters to lose their syntactic significance.
-- The quotation marks are stripped when the arguments for process are
-- collected.  The args are available to a process as a charstringList
-- via the named resource, "argv".

-- Each pipe is introduced by the character "|".  Optional pipe end
-- names can be specified after the bar by enclosing them in square
-- brackets.  The pipe end names are resource names by which the
-- processes on either end of a pipe retrieve the ports by which they
-- will communicate over the pipe.  The first end name is the name
-- used by the predecessor, and the second is for the successor.
-- Either or both may be omitted, but if the first is omitted and not
-- the second, the comma must be present (e.g. "[,succName]").  If
-- either endname is omitted it defaults to "stdout" or "stdin" for
-- the predecessor or successor, respectively.  The pipe type must be
-- declared by both processes at the ends of the pipe when they make
-- their resource manager calls to obtain the pipe ends.  The types
-- specified at both ends must agree for a successful connection.

-- Normally, each pipe is attached to the two neighboring processes.
-- However, angle brackets can be used to enclose a pipeline and cause
-- it to be transparent with regard to the linking of processes
-- specified around it.  For example:
--   foo | bar |[x,y] frotz
--	links foo.stdout to bar.stdin and bar.x to frotz.y
--   foo <| bar> |[x,y] frotz
--	links foo.stdout to bar.stdin and foo.x to frotz.y
--   foo | < bar |[x,y]> frotz
--	links foo.stdout to frotz.stdin and bar.x to frotz.y

-- Processes in a pipeline can be named by preceding the process
-- specification with a label and a colon.  Elsewhere in the pipeline
-- (or more generally in the containing network), the same process
-- (not another invocation of it) can be referred to again by giving
-- the label and the colon, without a following process specification.
-- In addition, multiple pipelines can be combined to form a more
-- complex network by separating the pipelines with semicolons.  This
-- can be used, along with the label mechanism, to produce arbitrary
-- networks of processes.  For example:

--   A:foo | bar | A:
--      links foo.stdout to bar.stdin and bar.stdout to foo.stdin,
--      thereby creating a circular process network
--   A:foo <| bar | B:frotz> |[x,y] B: |[z,w] A:
--      links foo.stdout to bar.stdin, bar.stdout to frotz.stdin,
--      foo.x to frotz.y, and frotz.z to foo.w.

-- Any process that is not explicitly supplied with a "stdin" or
-- "stdout" pipe will inherit the resources (looking like stdio pipe
-- ends) from the pshell, assuming the pshell has those resources.
-- Normally, the resources would give access to the user "console" (a
-- terminal or window or whatever).

-- If a pipe network cannot be constructed due to syntax error or
-- unsuccessful search for necessary programs, an error message will
-- be issued and no portion of the network will be invoked.

pshell:
using (
  rmain, rmanager, personalRM, plumbing, plumber, pshell, common,
  terminalIO, stdio, tokenize, root, load, cwd, chainedRm
)
linking (
  ac_none, tokenize, parseProcNet, plumber, chainedRm, 
  personalRM
)

process (Q: rmainQ)
  
declare
  args: rmain;
  resource: polymorph;
  stdin: stdin;
  stdout: stdout;
  chainedRm: rManager;
  prompt: charString;
  setCwd: setCwdFn;
  getCwd: getCwdFn;
  tokenize: tokenizeFn;
  parser: parseProcNetFn;
  plumber: plumberFn;
  done: boolean;
  firstCmd: boolean;
begin
  receive args from Q;
  
  -- retrieve capabilities to read and write to the console terminal
  block declare
    term: terminalFunctions;
  begin
    block begin
      unwrap stdin from args.rm.get("stdin", "stdio") 
	  {init, init(getChar), init(getString)};
    on (getResource.notFound)
      -- If "stdin" resource doesn't exist, we create one out of the
      -- "terminal" resource
      unwrap term from args.rm.get("terminal", "") {
	init, init(getChar), init(getString), 
	init(putChar), init(putString), init(putLine) };
      new stdin;
      stdin.getChar := term.getChar;
      stdin.getString := term.getString;
    end block;
    
    block begin
      unwrap stdout from args.rm.get("stdout", "stdio") 
	  {init, init(putChar), init(putString), init(putLine)};
      
    on (getResource.notFound)
      -- Create and post a stdout if none supplied, similarly to
      -- treatment for stdin above
      unwrap term from args.rm.get("terminal", "") {
	init, init(putString), init(putLine), init(getString),
	init(putChar), init(getChar) };
      new stdout;
      stdout.putChar := term.putChar;
      stdout.putString := term.putString;
      stdout.putLine := term.putLine;
    end block;
  on (getResource.notFound, getResource.accessDenied)
    print charstring#
	"Standard I/O capabilities not available... shell cannot continue";
    exit done;
  end block;

  -- Install copies of our stdin and stdout into a chained resource
  -- manager so our progeny can use them
  block declare
    chainedRmInit: chainedRmInitFn;
    personalRM: personalRMFn;
    rmList: rmList;
    ac_none: accessFn;
  begin
    personalRM <- create of process personalRM;
    chainedRmInit <- create of process chainedRm;
    ac_none <- create of process ac_none;
    new rmList;
    insert personalRM("anonymous") into rmList;
    insert copy of args.rm into rmList;
    chainedRm <- chainedRmInit(rmList);
    wrap copy of stdin as resource;
    call chainedRm.post("stdin", resource, ac_none);
    wrap copy of stdout as resource;
    call chainedRm.post("stdout", resource, ac_none);
  end block;

  -- get a command line tokenizer
  block declare 
    tokInit: tokenizeInitFn;
    whiteChars: charString;
    wordChars: charString;
    quoteChars: charString;
    bracketChars: charString;
  begin
    whiteChars <- " ";
    insert 'HT' into whiteChars;
    wordChars <- "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
	| "0123456789-_./~*?";
    quoteChars <- """'";
    bracketChars <- "";
    tokInit<- create of process tokenize;
    tokenize <- tokInit(wordChars, whiteChars, quoteChars, bracketChars);
  end block;
  
  -- get a procedure for parsing a token list and generating a process
  -- network specification
  parser <- procedure of process parseProcNet;

  -- Get access to the plumber.
  plumber <- procedure of process plumber;
  
  -- get cwd and pwd capabilities, or rig up disconnected capabilities
  -- (and continue with non-functioning cwd and pwd commands) if not
  -- available
  block declare
    setQ: setCwdQ;
  begin
    unwrap setCwd from args.rm.get("setCwd", "") {init};
  on (getResource.notFound, getResource.accessDenied)
    new setQ;
    connect setCwd to setQ;
    discard setQ;
  end block;
  
  block declare
    getQ: getCwdQ;
  begin
    unwrap getCwd from args.rm.get("getCwd", "") {init};
  on (getResource.notFound, getResource.accessDenied)
    new getQ;
    connect getCwd to getQ;
    discard getQ;
  end block;

  -- get preferred prompt from environment variable if present
  block declare
    env: root!environ;
  begin
    unwrap env from args.rm.get("environ", "") {init};
    inspect entry in env["HPROMPT"] begin
      prompt := entry.value;
    end inspect;
  on (notFound, getResource.notFound)
    prompt <- "pshell: ";
  end block;
  
  -- Now loop accepting commands until we get an "exit" command or end
  -- of file or finished executing single command from our own command
  -- line args
  done <- 'false';
  firstCmd <- 'true';
  while not done repeat
    block declare
      cmd: charString;
      tokens: tokenList;
      firstToken: token;
      procs: procSet;
      pipes: pipeSet;
      asynchronous: boolean;
      failer: charString;
    begin

      -- first time through loop we check for command line args of our
      -- own, to form a single command to be executed.  Thereafter we
      -- always just prompt for a command line
      if firstCmd then
	block declare
	  argv: charStringList;
	begin
	  block begin
	    unwrap argv from args.rm.get("argv", "") {init};
	    argv <- every of arg in argv where (position of arg >= 2);
	  on (getResource.notFound, getResource.accessDenied)
	    new argv;
	  end block;
	  if size of argv <> 0 then
	    -- nonempty command line args... use it, and avoid trying to
	    -- execute another command
	    new cmd;
	    for arg in argv[] inspect
	      merge arg | " " into cmd;
	    end for;
	    done <- 'true';
	  else
	    -- prompt for first command, and avoid all this for
	    -- subsequent commands
	    call stdout.putString(prompt);
	    cmd <- stdin.getString();
	    firstCmd <- 'false';
	  end if;
	end block;
      else
	-- 2nd and subsequent commands are obtained here
	call stdout.putString(prompt);
	cmd <- stdin.getString();
      end if;
      
      tokens <- tokenize(cmd);
      if size of tokens = 0 then
	exit continue;
      end if;
      
      -- handle built-in commands if first token is a word naming one
      -- of the commands
      firstToken := tokens[0];
      if case of firstToken = 'word' then
	reveal firstToken.word;
	select firstToken.word

	where ("cd")
	  block declare
	    secondToken: token;
	  begin
	    if size of tokens <> 2 then
	      exit cwdSyntax;
	    end if;
	    secondToken := tokens[1];
	    if case of secondToken <> 'word' then
	      exit cwdSyntax;
	    end if;
	    reveal secondToken.word;
	    call setCwd(secondToken.word);
	  on exit (cwdSyntax)
	    call stdout.putLine("Usage: cwd directory");
	  on (disconnected)
	    call stdout.putLine("cwd command not available");
	  end block;
	  exit continue;

	where ("pwd")
	  block begin
	    if size of tokens <> 1 then
	      exit pwdSyntax;
	    end if;
	    call stdout.putLine(getCwd());
	  on exit(pwdSyntax)
	    call stdout.putLine("Usage: pwd");
	  on (disconnected)
	    call stdout.putLine("pwd command not available");
	  end block;
	  exit continue;
	    
	where ("exit")
	  if size of tokens <> 1 then
	    call stdout.putLine("Usage: exit");
	    exit continue;
	  else
	    exit done;
	  end if;
	    
	otherwise
	  -- Not a built-in command... parse it as a network
	end select;
      end if;
	
      -- Parse the command as a process network and then call the
      -- plumber to set it up and start it running and (if the command
      -- was terminated by an ampersand), wait for all the invoked
      -- processes to return.  Install our stdin and stdout in the
      -- base resource manager so the application processes can get at
      -- them. 
      call parser(args.rm, tokens, procs, pipes, asynchronous, failer);
      call plumber(args.rm, procs, pipes, chainedRm, asynchronous);

    on (getStringIntf.endOfInput)
      -- end of input on terminal... print ^D like Unix shell and quit
      call stdout.putLine("^D");
      exit done;
      
    on (tokenize.illFormed, parseProcNet.syntaxError)
      call stdout.putLine("Command syntax error -- nothing executed");

    on (parseProcNet.procFailure)
      call stdout.putLine("Unable to create process '" | failer | "'");
      
    on (parseProcNet.pipeFailure)
      call stdout.putLine("Unable to create pipe '" | failer | "'");

    on exit(continue)
      -- get another command

    on exit(done)
      -- propogate this through the others handler on this block
      exit done;
      
    on (others)
      call stdout.putLine("Unexpected error executing command");
      
    end block;
  end while;

  -- drop out of loop only after executing single command built from
  -- our own command line args
  exit done;

on exit(done)
  -- time to leave...
  return args;
end process
