-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- SCCS info: @(#)infer.p	1.5 1/25/92 

-- This file contains Infer, the 'master' inference process that calls the 
-- the type inference functions:
-- sameas
-- casetypeof
-- matchinginportof
-- messagetypeof
-- elementtypeof

-- The type inference functions have signature: 
--   typename X definitions -> typename.  

infer: USING(type, type_inference, errors, inferredType, positions)
LINKING (sameas, casetypeof, matchinginportof,messagetypeof,elementtypeof)

PROCESS(SetupQ: SetupInferQueue)

DECLARE
  SetupM: SetupInferMessage;
  InferQ: InferQueue;
  InferC: InferCapa;
  sameasport: InferRuleCapa;
  casetypeofport: InferRuleCapa;
  matchinginportofport: InferRuleCapa;
  messagetypeofport: InferRuleCapa;
  elementtypeofport: InferRuleCapa;
  findType: findTypeCapa;
  findDef: findDefCapa;
  program: proc;
  defs: definitions_modules;
  blankError: error;
  error: error;
  errObj: errorObject;
  inference: InferredDefinition;
  bp: backpatchRecord;
  bpInfer: backpatchInfer;

BEGIN
  NEW InferQ;
  CONNECT InferC TO InferQ;

  RECEIVE SetupM FROM SetupQ;

  -- load the individual inference rule processes
  sameasport <- InferRuleCapa#(procedure of program#(process sameas));
  casetypeofport <- InferRuleCapa#(procedure of program#(process casetypeof));
  matchinginportofport <- 
      InferRuleCapa#(procedure of program#(process matchinginportof));
  messagetypeofport <- 
      InferRuleCapa#(procedure of program#(process messagetypeof));
  elementtypeofport <- 
      InferRuleCapa#(procedure of program#(process elementtypeof));
  -- keep a copy of findType and findDef ports, and the absprog
  findType := SetupM.findType;
  program := SetupM.program;
  defs := SetupM.definitions;

  -- provide an output port to calling environment
  SetupM.infer_func <- InferC;  
  RETURN SetupM;
  discard SetupQ;

  -- make a blank error object template so we can build error objects
  -- more easily when needed
  new blankError;
  blankError.code <- errorcode#'type inference mismatch';
  new blankError.objects;
  
  -- now loop, handling requests
  while boolean#'true' repeat
    block declare
      InferM: inferMessage;
      argType: typename;
      infType: typename;
      knownType: typename;
    begin
      receive InferM from InferQ;

      -- get the type of the argument operand
      block begin
	argType <- typename#(findType(InferM.argument, 
	    program.executable_part.scopes, defs, InferM.inferred));
      on (findTypeMessage.unknown_type)
	-- argument object type not yet known... add it to the
	-- backpatch table so we can try again later
	new bp;
	bp.triggerObj := InferM.argument.root;
	new bpInfer;
	bpInfer.function := InferM.function;
	bpInfer.resultObj := InferM.object;
	bpInfer.argumentObj := InferM.argument;
	bpInfer.position := InferM.position;
	unite bp.info.infer from bpInfer;
	insert bp into InferM.backpatch;
	exit backpatched;
      end block;

      -- get the type inferred from this rule
      select InferM.function
      where (type_inference_function#'sameas')
	infType <- typename#(sameasport(argType, defs));
      where (type_inference_function#'casetypeof')
	infType <- typename#(casetypeofport(argType, defs));
      where (type_inference_function#'matchinginportof')
	infType <- typename#(matchinginportofport(argType, defs));
      where (type_inference_function#'messagetypeof')
	infType <- typename#(messagetypeofport(argType, defs));
      where (type_inference_function#'elementtypeof')
	infType <- typename#(elementtypeofport(argType, defs));
      otherwise
	exit cantHappen;
      end select;  

      -- look up this type from declarations and current inferences
      block begin
	knownType <- typename#(findType(InferM.object, 
	    program.executable_part.scopes, defs, InferM.inferred));
	if boolean#(knownType <> infType) then
	  -- types don't match... issue an error
	  error := blankError;
	  unite error.position.apos from aposition#(copy of InferM.position);
	  -- error objects are the object, its previously known type,
	  -- the argument object, and the inferred type
	  unite errObj.objectname from objectname#(copy of InferM.object);
	  insert errObj into error.objects;
	  unite errObj.typename from knownType;
	  insert errObj into error.objects;
	  unite errObj.objectname from objectname#(copy of InferM.argument);
	  insert errObj into error.objects;
	  unite errObj.typename from infType;
	  insert errObj into error.objects;
	  insert error into InferM.errors;
	end if;
	
      on (findTypeMessage.unknown_type)
	-- type was previously unknown, so we infer it now if it's a
	-- root object, or backpatch it (maybe for the second time,
	-- but definitely the last) otherwise
	if boolean#(integer#(size of InferM.object.components) = integer#0) then
	  new inference;
	  inference.root := InferM.object.root;
	  inference.type <- infType;
	  insert inference into InferM.inferred;
	  block begin
	    insert rootname#(copy of InferM.object.root)
		into InferM.newlyInferred;
	  on (DuplicateKey)
	  end block;
	else
	  new bp;
	  bp.triggerObj := InferM.object.root;
	  new bpInfer;
	  bpInfer.function := InferM.function;
	  bpInfer.resultObj := InferM.object;
	  bpInfer.argumentObj := InferM.argument;
	  bpInfer.position := InferM.position;
	  unite bp.info.infer from bpInfer;
	  insert bp into InferM.backpatch;
	  exit backpatched;
	end if;
      end block;
	
      return InferM;

    on (InferRuleMessage.WrongArgType)
      -- argument object was of wrong type... there will already be a
      -- message about it from a class rule, so we just let it go
      return InferM;
      
    on exit(backpatched)
      -- here when we added a backpatch entry... just return without
      -- actually invoking the rule
      return InferM;
      
    on exit(cantHappen)
      discard InferM;

    end block;
  end while;

ON (disconnected)
  -- this always happens when the type checker is finished
END process
