-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: tsapost.p
-- Author: Rob Strom
-- SCCS Info: @(#)tsapost.p	1.2 8/2/89

 -- ApplyPostcondition:                                              
 -- Given a set of adds and drops, modify the typestate,             
 -- checking that the resulting typestate is still possible,         
 -- and complaining (and refusing) if it is not possible.            
 -- Assumptions:                                                     
 -- To save time and effort, we eliminate checks for impossible      
 -- typestates which could only be generated if DeterminePostcondition
 -- gives nonsensical output.  For instance, we don't check that     
 -- an operation which adds init(CM) also puts the components        
 -- of the CM in at least the minimum typestate, since there is      
 -- no way that a user error can cause this to happen. We do check   
 -- that we don't drop components of a CM below the minimum typestate
 -- since the user could code DISCARD CM.X where X is required to    
 -- remain INIT.                                                     
 -- In exchange for this convenience, it means that this module      
 -- may have to be changed if new operations are added to the language
 -- which break the assumptions.                                     
 -- Algorithm:                                                       
 -- 1. Check that the additions are OK:                              
 -- 1.1. If attr(CM.A) or attr(R.A) is added, check that             
 --      INIT(CM) or INIT(R) is either already present or implied or to be added
 -- 1.2. If attr(V.A) is added, check that CASE(V, V.A) is           
 --      either already present or implied or to be added                       
 -- If prerequisite is already present or implied, no further check necessary.
 -- If prerequisite is "to be added", recursively call to add it.
 -- If it is legal to add the attribute, and the attribute doesn't   
 -- already exist and isn't implied, then add it.  Otherwise, do nothing.   
 -- If it is illegal to add the attribute, complain and do nothing.  
 -- 2. Check that deletions are OK:                                  
 -- 2.1. Determine prerequisite and co-requisite drops               
 -- 2.2. Check that each prerequisite is to be dropped; drop it.     
 -- 2.3. Check that each corequisite is to be dropped; then drop all.
TSAPOST : USING(Predefined, tscheck, Positions, Errors  ) PROCESS ( ApplyPostconditionInit : ApplyPostconditionInport )
  DECLARE
    FP: ApplyPostconditionCall ;
    AddIfPossible: AddDropOutport;
    DropIfPossible: AddDropOutport;
    AddedAttribute: Attribute; -- attribute on Adds list
    DroppedAttribute: Attribute; -- attribute on Drops list
  BEGIN
    AddIfPossible <- AddDropOutport # (PROCEDURE OF program # (PROCESS  (AddIfPossibleInit: AddDropInport)
      DECLARE
        FP: AddDropCall;
      BEGIN
        RECEIVE FP FROM AddIfPossibleInit;
        /* Add if not already added */ BLOCK
          BEGIN
            /* If in form Parent.Comp, check prerequisites */ BLOCK
              DECLARE
                ParentObject: Objectname; -- Parent
                PrerequisiteAttribute: Attribute; -- attribute which must be present first
                Nothing: Empty; -- qualifier for coercions
                LastComponent: ComponentId; -- A in CM.A
                Type: TypeName; -- A's type
                AddIfPossible: AddDropOutport;
              BEGIN
		ParentObject <- ObjectName # (Object IN FP.Attribute.Objects WHERE(boolean # (integer # (POSITION OF Object) = integer # (0))));
		REMOVE LastComponent FROM LC IN ParentObject.Components WHERE(boolean # (integer # (POSITION OF LC) = integer # (integer # (SIZE OF ParentObject.Components) - integer # (1))));
		Type <- TypeName # (FP.Services.TypeOf(FP.Services, FP.Declarations, FP.Definitions, FP.Context, ParentObject));
		INSPECT Module IN FP.Definitions WHERE(boolean # (Module.Id = Type.ModuleId))
                  BEGIN
                    INSPECT Definition IN Module.Type_Definitions WHERE(boolean # (Definition.Id = Type.TypeId))
                      BEGIN
                        NEW PrerequisiteAttribute;
			NEW PrerequisiteAttribute.Objects;
                        SELECT
                          WHERE (boolean # (boolean # (Primitive_Types # (CASE OF Definition.Specification) = Primitive_Types # ('recordtype')) OR boolean # (Primitive_Types # (CASE OF Definition.Specification) = Primitive_Types # ('callmessagetype'))))
			    UNITE PrerequisiteAttribute.Name.Init FROM Nothing;
			    INSERT ParentObject INTO PrerequisiteAttribute.Objects;
			  OTHERWISE /* variant */
			    UNITE PrerequisiteAttribute.Name.Case FROM Nothing;
			    INSERT ObjectName # (COPY OF ParentObject) INTO PrerequisiteAttribute.Objects;
			    INSERT LastComponent INTO ParentObject.Components;
			    INSERT ParentObject INTO PrerequisiteAttribute.Objects;
                          END SELECT;
                      END INSPECT;
                  END INSPECT;
                IF boolean # (EXISTS OF Attr IN FP.CurrentTS WHERE(boolean # (Attr = PrerequisiteAttribute)))
                  THEN
                  ELSE
                    IF boolean # (EXISTS OF Attr IN FP.PostCondition.Adds WHERE(boolean # (Attr = PrerequisiteAttribute)))
                      THEN
                        BLOCK
                          BEGIN
                            AddIfPossible <- AddDropOutport # (PROCEDURE OF Program # (CURRENTPROGRAM));
                            CALL AddIfPossible (FP.Services, FP.Declarations, FP.Definitions,  FP.Position, FP.Context, FP.PostCondition, FP.CurrentTS, PrerequisiteAttribute);
                          ON (AddDropCall.Illegal)
                            EXIT IllegalToAdd;
                          END BLOCK;
                      ELSE
                        IF FP.Services.ImpliedAttribute(FP.Services, FP.Declarations, FP.Definitions, FP.Context, PrerequisiteAttribute, FP.CurrentTS)
                          THEN
                          ELSE
			    /* error: can't add FP.Attribute */
			    INSERT (EVALUATE ErrorMessage3: Error FROM
			      NEW ErrorMessage3;
			      UNITE ErrorMessage3.Position.APos FROM COPY OF FP.Position;
			      ErrorMessage3.Code <- ErrorCode # 'CannotAdd';
			      NEW ErrorMessage3.Objects;
			      INSERT (EVALUATE ErrorObject: ErrorObject
				FROM
				  UNITE ErrorObject.Attribute FROM Attribute # (COPY OF FP.Attribute);
				END) INTO ErrorMessage3.Objects;
			      END) INTO FP.Context.ErrorMessages;
			    EXIT IllegalToAdd;
			  END IF;   
                      END IF;
                  END IF;
              END BLOCK;
            IF FP.Services.ImpliedAttribute(FP.Services, FP.Declarations, FP.Definitions, FP.Context, FP.Attribute, FP.CurrentTS)
              THEN
              ELSE
		/* Insert if not already present */ BLOCK
		  BEGIN
		    INSERT Attribute # (COPY OF FP.Attribute) INTO FP.CurrentTS;
		  ON (DuplicateKey)
		  END BLOCK;
              END IF;
            RETURN FP;
	  ON (NotFound) -- Object not in the form Parent.Component; no check needed. 
	    -- We assume that an object with no parent can't be implied.
	    -- For performance reasons, we elide the test for implied.
	    -- If the rule for implied changes, this code will have to change
	    /* Insert if not already present */ BLOCK
	      BEGIN
		INSERT Attribute # (COPY OF FP.Attribute) INTO FP.CurrentTS;
	      ON (DuplicateKey)
	      END BLOCK;
            RETURN FP;
          ON EXIT (IllegalToAdd)
            RETURN FP EXCEPTION Illegal;
          END /* Add if not already added */ BLOCK;
      END PROCESS));
      
    DropIfPossible <- AddDropOutport # (PROCEDURE OF program # (PROCESS (DropIfPossibleInit: AddDropInport)
    -- Algorithm:
    -- 1. Get prerequisites and corequisites for dropping the attribute
    -- 2. For each prerequisite, call recursively to drop the attribute if possible
    -- 3. If all prerequisites were dropped, and all corequisites droppable,
    --    drop the corequisites from the typestate
    -- 4. In any case, drop the corequisites from the drops list
      DECLARE
        FP: AddDropCall;
        Coercion: Statement; -- the statement which must be executed to drop some attribute
        Prerequisites: Typestate; -- attributes which must be dropped before coercion can be applied
        Corequisites: Typestate; -- attributes which are dropped when coercion is applied
        Legal: Boolean; -- true if all prerequisites were dropped and corequisites droppable
        DropIfPossible: AddDropOutport;
	DeletedAttributes: Typestate;
      BEGIN
        BLOCK
          BEGIN
            RECEIVE FP FROM DropIfPossibleInit;
	    BLOCK
	      BEGIN
		CALL FP.Services.AttemptToCoerce(FP.Services, FP.Declarations, FP.Context, FP.Definitions, FP.Attribute, FP.CurrentTS, Coercion, Prerequisites, Corequisites);
	      ON (AttemptToCoerceCall.NoCoercion)
	      END BLOCK;
	    DropIfPossible <- AddDropOutport # (PROCEDURE OF Program # (CURRENTPROGRAM));
	    Legal <- FORALL OF PrerequisiteAttribute IN Prerequisites WHERE(EXISTS OF Drop IN FP.Postcondition.Drops WHERE(Drop = PrerequisiteAttribute));
	    FOR PrerequisiteAttribute in Prerequisites WHERE(boolean # ('true'))
	      INSPECT
		IF boolean # (EXISTS OF Drop IN FP.Postcondition.Drops WHERE(boolean # (Drop = PrerequisiteAttribute)))
		  THEN
		    BLOCK
		      BEGIN
			CALL DropIfPossible(FP.Services, FP.Declarations, FP.Definitions, FP.Position, FP.Context, FP.Postcondition, FP.CurrentTS, PrerequisiteAttribute);
		      ON (AddDropCall.Illegal)
			Legal <- boolean # ('false');
		      END BLOCK;
		  END IF;
	      END FOR;
	    IF Legal
	      THEN
		IF boolean # (FORALL OF Corequisite in Corequisites WHERE(boolean # (boolean # (Corequisite = FP.Attribute) OR boolean # (EXISTS OF Drop IN FP.Postcondition.Drops WHERE(boolean # (Drop = Corequisite))))))
		  THEN
		    -- drop all corequisites from TS
                      EXTRACT DeletedAttributes FROM Attr IN FP.CurrentTS WHERE(EXISTS OF Corequisite in Corequisites WHERE(Attr = Corequisite));
		  ELSE
		    -- illegal: corequisites not droppable
		    Legal <- 'false';
		    INSERT (EVALUATE ErrorMessage: Error FROM
		      NEW ErrorMessage;
		      UNITE ErrorMessage.Position.APos FROM COPY OF FP.Position;
		      ErrorMessage.Code <- ErrorCode # 'CannotDrop';
		      NEW ErrorMessage.Objects;
		      INSERT (EVALUATE ErrorObject: ErrorObject
			FROM
			  UNITE ErrorObject.Attribute FROM Attribute # (COPY OF FP.Attribute);
			END) INTO ErrorMessage.Objects;
		      END) INTO FP.Context.ErrorMessages;
		  END IF;
	      ELSE
		-- illegal: prerequisites not dropped
		INSERT (EVALUATE ErrorMessage2: Error FROM
		  NEW ErrorMessage2;
		  UNITE ErrorMessage2.Position.APos FROM COPY OF FP.Position;
		  ErrorMessage2.Code <- ErrorCode # 'CannotDrop';
		  NEW ErrorMessage2.Objects;
		  INSERT (EVALUATE ErrorObject: ErrorObject
		    FROM
		      UNITE ErrorObject.Attribute FROM Attribute # (COPY OF FP.Attribute);
		    END) INTO ErrorMessage2.Objects;
		  END) INTO FP.Context.ErrorMessages;
	      END IF;
	    -- don't try to drop these corequisites again
	    EXTRACT DeletedAttributes FROM Attr IN FP.Postcondition.Drops WHERE(EXISTS OF Corequisite in Corequisites WHERE(Attr = Corequisite));
-- end extract
	    IF Legal
	      THEN
                RETURN FP;
              ELSE
                RETURN FP EXCEPTION Illegal;
              END IF;
          END BLOCK;
        
      END PROCESS));
      
    RECEIVE FP FROM ApplyPostconditionInit;
    WHILE boolean # (integer # (SIZE OF FP.PostCondition.Adds) > integer # (0))
      REPEAT
        REMOVE AddedAttribute FROM Attr IN FP.PostCondition.Adds WHERE(boolean # ('true'));
        BLOCK
          BEGIN
            CALL AddIfPossible(FP.Services, FP.Declarations, FP.Definitions, FP.Position, FP.Context, FP.Postcondition, FP.CurrentTS, AddedAttribute);
          ON (AddDropCall.Illegal) -- add as many attributes as are legal
          ON (Others)
          END BLOCK;
      END WHILE;
    WHILE boolean # (integer # (SIZE OF FP.PostCondition.Drops) > integer # (0))
      REPEAT
        REMOVE DroppedAttribute FROM Attr IN FP.PostCondition.Drops WHERE(boolean # ('true'));
        BLOCK
          BEGIN
            CALL DropIfPossible(FP.Services, FP.Declarations, FP.Definitions, FP.Position, FP.Context, FP.Postcondition, FP.CurrentTS, DroppedAttribute);
          ON (AddDropCall.Illegal) -- drop as many attributes as are legal
          ON (Others)
          END BLOCK;
      END WHILE;
    RETURN FP;
  END PROCESS
