#|
			DEMO-OTHELLO

|#

(in-package "DEMO-OTHELLO" :nicknames '("DOTH") :use '("KR" "LISP"))

(export '(Do-Go Do-Stop Start-Game Stop-Game Set-Score))

(defmacro half (n)
  `(round (/ ,n 2.0)))

(defvar *game-started* NIL)

(defparameter *max-size*	16)	;; no more than 16 x 16 board!
(defparameter *rect-count*	(* *max-size* *max-size*))
(defparameter *rectangles*	NIL)	;; array of the rectangles
(defparameter *squares*		NIL)	;; aggregate of the rectangles

(defvar board-array NIL)		;; 2-d array of the rectangles

(defparameter *window-left*	100)
(defparameter *window-top*	 50)
(defparameter *window-width*	800)
(defparameter *window-height*	606)
(defparameter *window-center*	(half *window-width*))
(defparameter *title-width-offset*	10)
(defparameter *title-height-offset*	5)
(defparameter *title-line-x-offset*	19)
(defparameter *title-line-y-offset*	20)

(defparameter *rightmost-box-point*	190)
(defparameter *score-box-piece-width*    20)
(defparameter *score-box-x-indent*	  5)
(defparameter *score-box-y-indent*	  5)

(defvar *score-box-bottom*  10)		;; This will be set in Make-Score-Box

(defvar *inter-box-y-offset* 10)	;;; This will be set in Make-Scroll-Bar

(defvar *test-debug* NIL)
(defvar scroll-bar NIL)
(defvar scroll-bar-font NIL)

(defparameter *scroll-bar-x-offset* 20)
(defparameter *scroll-bar-height*  190)
(defparameter *scroll-bar-width*    21)

(defparameter *message-box-height*   75)
(defparameter *messages-indentation* 10)

(defparameter *panel-x-indent*	 20)
(defparameter *panel-y-indent*	 20)

(defparameter *board-x-offset*	10)
(defparameter *board-offset*	10)

(defparameter *mat-left*	(+ *rightmost-box-point* *board-x-offset*))
(defparameter *mat-right*	(- *window-width* (* 2 *title-line-x-offset*)))

(defparameter *board-left*	(+ *mat-left* *board-offset*))
(defparameter *board-right*	(- *mat-right* *board-offset*))
(defparameter *board-width*	(- *board-right* *board-left*))

(defvar *topmost-box-point* NIL)
(defvar *bottommost-box-point* NIL)
(defvar *mat-top* NIL)
(defvar *mat-bottom* NIL)
(defvar *board-top* NIL)
(defvar *board-height* NIL)
(defvar *board-bottom* NIL)

(defun Preliminaries ()
  (setq *game-started* NIL)
  (create-instance 'title-font opal:font-from-file
	(:font-name "timbi24"))
  (create-instance 'standard-font opal:font-from-file
	(:font-name "timb18"))
  (create-instance 'squares opal:aggregate)
  (setq board-array (make-array (list *max-size* *max-size*)))
  (setq *rectangles* (make-array *rect-count*))
  (dotimes (index *rect-count*)
    (let ((rect (create-instance NIL opal:rectangle
				(:select-outline-only NIL))))
      (setf (aref *rectangles* index) rect)
      (opal:add-component squares rect)))
  (create-instance 'game-piece-1 opal:circle
	(:line-style NIL)
	(:filling-style opal::black-fill)
	)
  (create-instance 'game-piece-2 opal:circle
	(:line-style opal:default-line-style)
	(:filling-style NIL)
	)
)

(defun Make-Window-And-Title ()
  (create-instance 'w inter:interactor-window
				  (:left   *window-left*)
				  (:top    *window-top*)
				  (:width  *window-width*)
				  (:height *window-height*)
				  (:title "Demo-Othello")
				  (:icon-title "Othello"))
  (s-value w :aggregate (create-instance 'top-agg opal:aggregate))
  (opal:update w)
  (create-instance 'title opal:text
	(:string "Garnet Othello")
	(:font title-font))
  (opal:add-component top-agg title)
  (let* ((title-height (g-value title :height))
	 (title-width  (g-value title :width))
	 (title-left   (- *window-center* (half title-width)))
	 (title-right  (+ title-left title-width))
	 (title-frame-left       (- title-left *title-width-offset*))
	 (title-frame-right      (+ title-right *title-width-offset*))
	 (title-frame-mid-height (+ *title-height-offset* (half title-height)))
	 (title-line-right	 (- *window-width* *title-line-x-offset*))
	 (title-line-bottom	 (- *window-height* *title-line-y-offset*)))
	(setq *topmost-box-point* (+ title-height (* 2 *title-height-offset*)))
	(setq *bottommost-box-point*	(- *window-height*
					   (* 2 *title-line-y-offset*)))
	(setq *mat-top*	*topmost-box-point*)
	(setq *board-top*	(+ *mat-top* *board-offset*))
	(setq *mat-bottom*	*bottommost-box-point*)
	(setq *board-bottom*	(- *mat-bottom* *board-offset*))
	(setq *board-height*	(- *board-bottom* *board-top*))
	(s-value title :left title-left)
	(s-value title :top *title-height-offset*)
	(create-instance 'title-line opal:polyline
	  (:point-list
	    (list
		title-frame-left	title-frame-mid-height
		*title-line-x-offset*	title-frame-mid-height
		*title-line-x-offset*	title-line-bottom
		title-line-right	title-line-bottom
		title-line-right	title-frame-mid-height
		title-frame-right	title-frame-mid-height
	    )))
	(opal:add-component top-agg title-line))
)

(defun Make-Score-Box ()
  (let* ((score-box-top  *topmost-box-point*)
 	 (score-box-left (* 2 *title-line-x-offset*))
	 (player-text-left (+ score-box-left
			      *score-box-piece-width*
			      (* 2 *score-box-x-indent*)))
	 (player1-top    (+ score-box-top *score-box-y-indent*))
	 player2-top)
  	(create-instance 'player1-text opal:text
		(:string "Player 1")
		(:font standard-font)
		(:left player-text-left)
		(:top  player1-top))
	(opal:add-component top-agg player1-text)
  	(create-instance 'player2-text opal:text
		(:string "Player 2")
		(:font standard-font)
		(:left player-text-left)
		(:top (setq player2-top (+ player1-top
					   (g-value player1-text :height)
					   *score-box-y-indent*))))
  	(opal:add-component top-agg player2-text)
	(create-instance 'score-box opal:rectangle
		(:top    score-box-top)
		(:left   score-box-left)
		(:width  (- *rightmost-box-point* score-box-left))
		(:height (- (+ player2-top
			       (g-value player2-text :height)
			       *score-box-y-indent*)
			    score-box-top)))
  	(opal:add-component top-agg score-box)
	(create-instance 'player1-score opal:text
		(:string "0")
		(:font standard-font)
		(:left (o-formula (- *rightmost-box-point*
				     *score-box-x-indent*
				     (gvl :width))))
		(:top player1-top))
	(create-instance 'player2-score opal:text
		(:string "0")
		(:font standard-font)
		(:left (o-formula (- *rightmost-box-point*
				     *score-box-x-indent*
				     (gvl :width))))
		(:top player2-top))
	(opal:add-components top-agg player1-score player2-score)
	(setq *score-box-bottom* (+ score-box-top (g-value Score-Box :height)))
  )
)

(defun Make-Control-Panel ()
  (let* ((panel-bottom *bottommost-box-point*)
 	 (panel-left   (* 2 *title-line-x-offset*))
	 (panel-width  (- *rightmost-box-point* panel-left))
	 (panel-middle (+ panel-left (half panel-width))))
	(opal:add-component top-agg
		(create-instance 'Control-Panel-Agg opal:aggregate))

(format t "***** MAKING THE FAKE CONTROL PANEL ****~%")
	(create-instance 'Quit-Button opal:text
		(:string "Quit-Button")
		(:left (o-formula (- panel-middle (half (gvl :width)))))
		(:top  (o-formula (- panel-bottom
				     *panel-y-indent*
				     (gvl :height)))))
	(create-instance 'Cheat-Button opal:text
		(:string "Cheat-Button")
		(:left (o-formula (- panel-middle (half (gvl :width)))))
		(:top  (o-formula (- (gv Quit-Button :top)
				     *panel-y-indent*
				     (gvl :height)))))
	(create-instance 'Start-Button opal:text
		(:string "Start-Button")
		(:left (o-formula (- panel-middle (half (gvl :width)))))
		(:top  (o-formula (- (gv Cheat-Button :top)
				     *panel-y-indent*
				     (gvl :height)))))
	(opal:add-components Control-Panel-Agg
		Quit-Button Cheat-Button Start-Button)

;;;; Done making the fake control panel
	(create-instance 'control-panel opal:rectangle
		(:top (o-formula (- (gv Control-Panel-Agg :top)
				    *panel-y-indent*)))
		(:left panel-left)
		(:width panel-width)
		(:height (o-formula (- panel-bottom (gvl :top)))))
	(opal:add-component Top-Agg Control-Panel)
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;  SCROLL BAR SECTION  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This is essentially DEMO-MAC, stolen, and changed as little as
;;; possible to work in DEMO-OTHELLO.
;;; Designed and implemented by Brad Vander Zanden

;;;; create functions called by this demo

;;; The Clip-and-Map procedure works as follows:
;;;    (Clip-and-Map val val-min val-max target-min target-max) takes val,
;;;    clips it to be in the range val-min..val-max, and then scales and
;;;    translates the value (using linear-interpolation) to be between
;;;    target-min and target-max

(defun Clip-and-Map (val val-min val-max target-min target-max)
    (cond ((< val val-min) target-min)
	  ((> val val-max) target-max)
	  (t (+ target-min (floor (*  (- val val-min)
				      (- target-max target-min))
				  (- val-max val-min))))))

;;; function to create lines for the arrows. 

(defun create-arrow-line (obj init-x1 init-y1 init-x2 init-y2 x-offset y-offset)
    (create-instance NIL opal:line
	(:x1 (formula `(gv ',obj :x2) init-x1))
	(:y1 (formula `(gv ',obj :y2) init-y1))
	(:x2 (formula `(+ (gvl :x1) ,x-offset) init-x2))
	(:y2 (formula `(+ (gvl :y1) ,y-offset) init-y2))))

;;;---------------------------------
;;; function to create a scroll bar
;;;---------------------------------

(defun create-scroll-bar (viewport top-agg scroll-bar-name
			  left top width height)

  (let (slider trill-box-incr trill-box-decr slider-shell indicator)

;;; create the scroll bar and add it to top-agg

    (opal:add-component top-agg
	     (setf scroll-bar-name (create-instance NIL opal:aggregate
		    (:overlapping NIL))))
				

;;; create all the objects I need for a slider
    
    (setf slider (create-instance NIL opal:aggregate))
    (setf slider-shell (create-instance NIL opal:rectangle))
    (setf indicator (create-instance NIL opal:rectangle))

;;; create the slot definitions for the scroll box

    (s-value scroll-bar-name :left left)
    (s-value scroll-bar-name :top top)
    (s-value scroll-bar-name :width width)
    (s-value scroll-bar-name :height height)

;;; create the slot definitions for the slider

    (s-value slider :overlapping t)
    (s-value slider :indicator indicator)

;;; create the slot definitions for the slider shell

    (s-value slider-shell :left (formula `(gv ',scroll-bar-name :left)))
    (s-value slider-shell :top (formula `(gv ',scroll-bar-name :top)))
    (s-value slider-shell :width (formula `(gv ',scroll-bar-name :width)))
    (s-value slider-shell :height (formula `(gv ',scroll-bar-name :height)))
    (s-value slider-shell :bottom (formula '(+ (gvl :top) (gvl :height))))
    (s-value slider-shell :filling-style (opal:halftone 50))

;;; create the slot definitions for the indicator

    (s-value indicator :box `(0 ,(+ top (half height)) 10 10)) ; set by mouse interactor
    (s-value indicator :percent (formula `(Clip-and-Map (second (gvl :box))
						       (gv ',slider-shell :top)
						       (- (gv ',slider-shell :bottom)
							  (gvl :height) 2)
						       0 100) 50))
    (s-value indicator :left (formula `(+ 1 (gv ',slider-shell :left))))
    (s-value indicator :top (formula `(1+ (Clip-and-Map (gvl :percent)
						   0 100
						   (gv ',slider-shell :top)
						   (- (gv ',slider-shell :bottom)
						      (gvl :height) 2)))))
    (s-value indicator :filling-style (opal:halftone 0))
    (s-value indicator :width (formula `(- (gv ',slider-shell :width) 2)))
    (s-value indicator :height (formula `(gvl :width)))

;;; add the components to the slider

    (opal:add-component slider slider-shell)
    (opal:add-component slider indicator)

;;; create the increment and decrement trill boxes

    (setf trill-box-incr (make-incr-box viewport 'trill-box-incr1 indicator
			   (formula `(gv ',scroll-bar-name :left))
			   (formula `(- (gv ',scroll-bar-name :top)
					    (gvl :height)))
			   (formula `(gv ',scroll-bar-name :width))
			   20))

    (setf trill-box-decr (make-decr-box viewport 'trill-box-decr1 indicator
			   (formula `(gv ',scroll-bar-name :left))
			   (formula `(+ (gv ',scroll-bar-name :top)
					    (gv ',scroll-bar-name :height)))
			   (formula `(gv ',scroll-bar-name :width))
			   20))

;;; add the slider and the trill boxes to the scroll bar

    (opal:add-components scroll-bar-name slider trill-box-incr trill-box-decr)
    (s-value scroll-bar-name :slider slider)

;;; create the scroll bar interactor

    (create-instance NIL inter:Move-Grow-Interactor
		       (:window viewport)
		       (:continuous t)
		       (:start-event :leftdown)
		       (:start-where (list :in-box indicator))
		       (:running-where (list :in-box slider-shell))
		       (:outside :last)
		       (:obj-to-be-moved indicator)
		       (:feedback-obj NIL)
		       (:attach-point :center)
		       (:grow-p NIL)))

  scroll-bar-name)

;;;---------------------------------------------
;;; function to create the increment trill box
;;;---------------------------------------------

(defun make-incr-box (viewport trill-box-incr indicator left top width height)
  (let (incr-box up-arrow up-arrow-line1 up-arrow-line2 up-arrow-line3
		   up-arrow-line4 up-arrow-line5 up-arrow-line6 up-arrow-line7)
 
;;; create the objects I need to make an increment trill box

    (setf trill-box-incr (create-instance NIL opal:aggregate))
    (setf incr-box (create-instance NIL opal:rectangle))
    (setf up-arrow (create-instance NIL opal:aggregate))
    (setf up-arrow-line1 (create-instance NIL opal:line))

;;; create the slot definitions for the trill box

    (s-value trill-box-incr :left left)
    (s-value trill-box-incr :top top)
    (s-value trill-box-incr :width width)
    (s-value trill-box-incr :height height)

;;; create the slot definitions for the increment box

    (s-value incr-box :left (formula `(gv ',trill-box-incr :left)))
    (s-value incr-box :top (formula `(gv ',trill-box-incr :top)))
    (s-value incr-box :width (formula `(gv ',trill-box-incr :width)))
    (s-value incr-box :height (formula `(gv ',trill-box-incr :height)))

;;; create the slot definitions for the arrow lines

    (s-value up-arrow-line1 :x1 (formula `(+ (gv ',incr-box :left) 6)))
    (s-value up-arrow-line1 :y1 (formula `(+ (gv ',incr-box :top) 17)))
    (s-value up-arrow-line1 :x2 (formula '(+ (gvl :x1) 8)))
    (s-value up-arrow-line1 :y2 (formula '(gvl :y1)))

;;; create the other six lines used in the up arrow

    (setf up-arrow-line2 (create-arrow-line up-arrow-line1 114 165 114 159 0 -6))
    (setf up-arrow-line3 (create-arrow-line up-arrow-line2 114 159 119 159 5 0))
    (setf up-arrow-line4 (create-arrow-line up-arrow-line3 119 159 110 150 -9 -9))
    (setf up-arrow-line5 (create-arrow-line up-arrow-line4 110 150 101 159 -9 9))
    (setf up-arrow-line6 (create-arrow-line up-arrow-line5 101 159 106 159 5 0))
    (setf up-arrow-line7 (create-arrow-line up-arrow-line6 106 159 106 165 0 6))

;;; add the arrow lines to the up arrow aggregate

    (opal:add-component up-arrow up-arrow-line1)
    (opal:add-component up-arrow up-arrow-line2)
    (opal:add-component up-arrow up-arrow-line3)
    (opal:add-component up-arrow up-arrow-line4)
    (opal:add-component up-arrow up-arrow-line5)
    (opal:add-component up-arrow up-arrow-line6)
    (opal:add-component up-arrow up-arrow-line7)

;;; add the increment box and up arrow aggregate to the trill box aggregate

    (opal:add-component trill-box-incr incr-box)
    (opal:add-component trill-box-incr up-arrow)

;;; create the interactor for the increment trill box

    (create-instance NIL inter:Button-Interactor
		       (:window viewport)
		       (:continuous t)
		       (:feedback-obj NIL)
		       (:start-where (list :element-of trill-box-incr))
		       (:exception NIL)  ; no exceptions
		       (:start-event :leftdown)
		       (:stop-event :leftup)
		       (:stop-action #'(lambda (interactor final-obj-over)
					 (declare (ignore interactor final-obj-over))
					 (let ((n (g-value scroll-bar :n)))
					 (unless (= n 2)
					     (s-value indicator :percent
						(round (* (- (/ n 2) 2)
							  14.2857)))
					  ))))))

  trill-box-incr)


;;;---------------------------------------------
;;; function to create the decrement trill box
;;;---------------------------------------------

(defun make-decr-box (viewport trill-box-decr indicator left top width height)
  (let (decr-box down-arrow down-arrow-line1 down-arrow-line2 down-arrow-line3
		   down-arrow-line4 down-arrow-line5 down-arrow-line6 
		   down-arrow-line7)
 
;;; create the objects I need to make a decrement trill box

    (setf trill-box-decr (create-instance NIL opal:aggregate))
    (setf decr-box (create-instance NIL opal:rectangle))
    (setf down-arrow (create-instance NIL opal:aggregate))
    (setf down-arrow-line1 (create-instance NIL opal:line))

;;; create the slot definitions for the trill box

    (s-value trill-box-decr :left left)
    (s-value trill-box-decr :top top)
    (s-value trill-box-decr :width width)
    (s-value trill-box-decr :height height)

;;; create the slot definitions for the decrement box

    (s-value decr-box :left (formula `(gv ',trill-box-decr :left)))
    (s-value decr-box :top (formula `(gv ',trill-box-decr :top)))
    (s-value decr-box :width (formula `(gv ',trill-box-decr :width)))
    (s-value decr-box :height (formula `(gv ',trill-box-decr :height)))

;;; create the slot definitions for the first down arrow line

    (s-value down-arrow-line1 :x1 (formula `(+ (gv ',decr-box :left) 6) 142))
    (s-value down-arrow-line1 :y1 (formula `(+ (gv ',decr-box :top) 2) 150))
    (s-value down-arrow-line1 :x2 (formula `(+ (gvl :x1) 8) 150))
    (s-value down-arrow-line1 :y2 (formula `(gvl :y1) 150))

;;; create the other six lines used in the down arrow

    (setf down-arrow-line2 (create-arrow-line down-arrow-line1 150 150 150 156 0 6))
    (setf down-arrow-line3 (create-arrow-line down-arrow-line2 150 156 155 156 5 0))
    (setf down-arrow-line4 (create-arrow-line down-arrow-line3 155 156 146 165 -9 9))
    (setf down-arrow-line5 (create-arrow-line down-arrow-line4 146 165 137 156 -9 -9))
    (setf down-arrow-line6 (create-arrow-line down-arrow-line5 137 156 142 156 5 0))
    (setf down-arrow-line7 (create-arrow-line down-arrow-line6 142 156 142 150 0 -6))

;;; add the arrow lines to the down arrow aggregate

    (opal:add-component down-arrow down-arrow-line1)
    (opal:add-component down-arrow down-arrow-line2)
    (opal:add-component down-arrow down-arrow-line3)
    (opal:add-component down-arrow down-arrow-line4)
    (opal:add-component down-arrow down-arrow-line5)
    (opal:add-component down-arrow down-arrow-line6)
    (opal:add-component down-arrow down-arrow-line7)

;;; add the decrement box and down arrow aggregate to the trill box aggregate

    (opal:add-component trill-box-decr decr-box)
    (opal:add-component trill-box-decr down-arrow)

;;; create the interactor decrement trill box

    (create-instance NIL inter:button-interactor
		       (:window viewport)
		       (:feedback-obj NIL)
		       (:continuous t)
		       (:start-where (list :element-of trill-box-decr))
		       (:exception NIL)  ; no exceptions
		       (:start-event :leftdown)
		       (:stop-event :leftup)
		       (:stop-action #'(lambda (interactor final-obj-over)
					 (declare (ignore interactor final-obj-over))
					 (let ((n (g-value scroll-bar :n)))
					 (unless (= n 16)
					     (s-value indicator :percent
						(round (* (/ n 2)
							  14.2857)))
					  ))))))

  trill-box-decr)

;;;--------------------------------------
;;; function to create text that displays
;;; current value of the scroll bar
;;;--------------------------------------

(defun make-meter-text (top-agg meter-name scroll-bar-name left top)

;;; create the meter and add it to top-agg

    (opal:add-component top-agg
	     (setf meter-name (create-instance NIL opal:text)))

;;; create the slot definitions for the meter

  (s-value meter-name :left left)
  (s-value meter-name :top top)
  (s-value top-agg :n
	(formula
	   `(* (1+ (round (* .07
		             (gv ',scroll-bar-name :slider
				  :indicator :percent))))
	       2)))
  (s-value meter-name :string (formula `(prin1-to-string (gv ',top-agg :n))))
  (s-value meter-name :font scroll-bar-font)
  meter-name)

;;;---------------------------------
;;; Do-Go
;;;---------------------------------

(defparameter scroll-bar1 NIL)
(defparameter meter1 NIL)

(defun Make-Scroll-Bar ()

 (create-instance 'scroll-bar-font opal:font-from-file
	(:font-name "timr12"))

 (setq *inter-box-y-offset*
   (round (/ (- (g-value Control-Panel :top)
	        *score-box-bottom*
		*scroll-bar-height*
		*message-box-height*)
	     3.0)))

 (let* ((scroll-bar-top  (- (g-value Control-Panel :top)
			    *inter-box-y-offset*
			    *scroll-bar-height*))
	(scroll-bar-left (+ (g-value Control-Panel :left)
			    *scroll-bar-x-offset*))
	(scroll-text-top  (+ scroll-bar-top
			     (round (/ *scroll-bar-height* 3.0))))
	(scroll-text-left (+ scroll-bar-left
			     *scroll-bar-width*
			     *scroll-bar-x-offset*)))
  (opal:add-component top-agg (create-instance 'Scroll-Bar opal:aggregate))

  (setq scroll-bar1 (create-scroll-bar w scroll-bar 'scroll-bar1
			scroll-bar-left scroll-bar-top
			*scroll-bar-width* *scroll-bar-height*))
  (setq meter1 (make-meter-text scroll-bar 'meter1 scroll-bar1
		 (formula `(- (+ (gv ',scroll-bar1 :slider :indicator :left)
				     (floor (gv ',scroll-bar1 :slider
					         :indicator :width) 2))
				  (floor (gvl :width) 2)))

		 (formula `(- (+ (gv ',scroll-bar1 :slider :indicator :top)
				     (floor (gv ',scroll-bar1 :slider
						 :indicator :height) 2))
				  (floor (gvl :height) 2)))
 
		 ))
  (create-instance 'scroll-text1 opal:text
	(:string "Next Game:")
	(:font  scroll-bar-font)
	(:left  scroll-text-left)
	(:top   scroll-text-top))
  (create-instance 'scroll-text2 opal:text
	(:string
	  (o-formula
	    (let ((nstring (princ-to-string (gv scroll-bar :n))))
		(concatenate 'string nstring " x " nstring))))
	(:font scroll-bar-font)
	(:left
	  (o-formula
	    (+ (gv scroll-text1 :left)
	       (half (- (gv scroll-text1 :width)
			(gvl :width))))))
	(:top
	  (o-formula
	    (+ (gv scroll-text1 :top)
	       (gv scroll-text1 :height)
	       8))))
  (opal:add-components scroll-bar scroll-text1 scroll-text2)
 )
)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;  END OF SCROLL BAR SECTION  ;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun Make-Message-Box ()
  (let* ((message-box-top   (+ *score-box-bottom* *inter-box-y-offset*))
	 (message-box-left  (g-value score-box :left))
	 (message-box-width (- *rightmost-box-point* message-box-left))
	 (messages-left  (+ message-box-left *messages-indentation*))
	 messages-top
	 (messages-bottom (+ message-box-top *message-box-height*)))
    (create-instance 'Message-Box opal:rectangle
	(:top message-box-top)
	(:left message-box-left)
	(:width message-box-width)
	(:height *message-box-height*))
    (create-instance 'messages-header opal:text
	(:string "Comments:")
	(:left messages-left)
	(:top (+ message-box-top 10)))
    (opal:add-components top-agg message-box messages-header)
    (setq messages-top (+ (g-value messages-header :top)
			  (g-value messages-header :height)
			  3))
    (create-instance 'messages-underline opal:line
	(:x1 messages-left)
	(:x2 (+ messages-left (g-value messages-header :width)))
	(:y1 messages-top)
	(:y2 messages-top))
    (opal:add-component top-agg messages-underline)
    (create-instance 'message opal:text
	(:string "Player 1's move")
	(:top (o-formula
		(+ messages-top (half (- messages-bottom
					 messages-top
					 (gvl :height))))))
	(:left (o-formula
		(+ message-box-left (half (- message-box-width
					     (gvl :width)))))))
    (opal:add-component top-agg message)))

(defvar board NIL)
(defvar board-inter NIL)
(defvar *current-player* 1)	;;; This is either 1 or 2
(defvar *other-player* 2)	;;; This is either 1 or 2
(defvar *scores* NIL)		;;; array of the two folks' scores

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; THE ACTUAL GAME... ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun Set-Scores ()
  (set-score 1 (aref *scores* 1))
  (set-score 2 (aref *scores* 2))
)

(defun Flip (square)
 (let* ((old-player (g-value square :player))
	(new-player (- 3 old-player)))
  (incf (aref *scores* new-player) 1)
  (decf (aref *scores* old-player) 1)
  (s-value square :player new-player)
  (opal:update w)
))

(defun Make-Game-Piece-Pair (row column player)
  (let* ((square (aref board-array row column))
	 (piece1 (create-instance NIL game-piece-1
	 		(:left   (+ (g-value square :left) 2))
	 		(:top    (+ (g-value square :top)  2))
	 		(:width  (- (g-value square :width) 4))
	 		(:height (- (g-value square :height) 4))
			(:visible (o-formula (eq (gvl :square :player) 1)))
	 		(:square square)))
	 (piece2 (create-instance NIL game-piece-2
	 		(:left   (+ (g-value square :left) 2))
	 		(:top    (+ (g-value square :top)  2))
	 		(:width  (- (g-value square :width) 4))
	 		(:height (- (g-value square :height) 4))
			(:visible (o-formula (eq (gvl :square :player) 2)))
	 		(:square square))))
	(opal::add-components board piece1 piece2)
	(s-value square :player player)))

(defmacro Out-Of-Bounds (row col n)
  `(or (>= ,row ,n) (>= ,col ,n) (< ,row 0) (< ,col 0)))

(defun Legal-Move? (x-off y-off base-row base-col n flip-p)
  (let ((row (+ base-row x-off))
	(col (+ base-col y-off))
	square)
    (if (or (out-of-bounds row col n)
	    (not (eq (g-value (setq square (aref board-array row col)) :player)
		     *other-player*)))
	(return-from Legal-Move? NIL))
    (if flip-p (flip square))
    (loop
	(incf row x-off)
	(incf col y-off)
	(if (out-of-bounds row col n) (return-from Legal-Move? NIL))
	(if (eq (g-value (setq square (aref board-array row col)) :player)
		*current-player*)
	    (return-from Legal-Move? T)
	    (if flip-p (flip square))))))

(defun Process-Button-Press (interactor square-selected)
  (declare (ignore interactor))
  (let ((illegal-move T)
	(base-row (g-value square-selected :row))
	(base-col (g-value square-selected :column))
	(n (g-value board :n))
	temp)
    (unless (g-value square-selected :player)
	(dolist (y-off '(-1 0 1))
	  (dolist (x-off '(-1 0 1))
	    (unless (and (zerop y-off) (zerop x-off))
	      (when (legal-move? x-off y-off base-row base-col n NIL)
		(legal-move? x-off y-off base-row base-col n T)
		(setq illegal-move NIL)))))
	)
    (if illegal-move
      (progn
	(s-value message :string
	  (if (eq *current-player* 1)
		"Try Again, #1"
		"Try Again, #2"))
	#+cmu(ext::beep)
	#+cmu(ext::beep)
	)
      (progn
	(s-value square-selected :player *current-player*)
	(Make-Game-Piece-Pair base-row base-col *current-player*)
	(opal:update w)
	(incf (aref *scores* *current-player*))
	(setq temp *current-player*)
	(setq *current-player* *other-player*)
	(setq *other-player* temp)
	(set-scores)
	(opal:update w)
	(s-value message :string
		(if (eq *current-player* 1) "Player 1's move"
					    "Player 2's move"))
	))))

(defun Stop-Game ()
  (when *game-started*
    (opal:remove-component top-agg board)
    (opal:remove-component top-agg score-box-game-pieces)
    (opal:update w)
    (if (member squares (get-local-values board :components))
	(opal:remove-component board squares))
    (opal:destroy board)
    (opal:destroy board-inter)
    (opal:destroy score-box-game-pieces)
    (setq *game-started* NIL)
  )
)

(defun Start-Game ()
  (let ((n (g-value scroll-bar :n))
	(rect-index 0)
	(row-left *board-left*)
	(row-end 0)
	(old-row-end 0)
        (row-width 0)
        col-top col-end old-col-end temp
	active-list
	(piece1-size (min *score-box-piece-width*
			  (g-value player1-text :height)))
	(piece2-size (min *score-box-piece-width*
			  (g-value player2-text :height)))
	(pieces-base-left (+ (* 2 *title-line-x-offset*)
			     *score-box-x-indent*))
	)
    (if *game-started* (Stop-Game))
    (setq *game-started* T)
    (create-instance 'score-box-game-pieces opal:aggregate)
    (create-instance 'score-box-piece-1 game-piece-1
		(:top (g-value player1-text :top))
		(:left (+ pieces-base-left (half (- *score-box-piece-width*
						    piece1-size))))
		(:width  piece1-size)
		(:height piece1-size))
    (create-instance 'score-box-piece-2 game-piece-2
		(:top (g-value player2-text :top))
		(:left (+ pieces-base-left (half (- *score-box-piece-width*
						    piece2-size))))
		(:width  piece2-size)
		(:height piece2-size))
    (opal:add-components score-box-game-pieces score-box-piece-1
					       score-box-piece-2)

    (create-instance 'board opal:aggregate)

		;; These are the 4 rectangles which compose the outside
		;; dark gray region around the board...

    (dolist (rect (list (list *mat-top* *mat-bottom* *mat-left* *board-left*)
			(list *mat-top* *mat-bottom* *board-right* *mat-right*)
			(list *mat-top* *board-top* *board-left* *board-right*)
			(list *board-bottom* *mat-bottom*
					 *board-left* *board-right*)))
	(let ((top  (first rect))
	      (left (third rect)))
	  (opal:add-component board (create-instance NIL opal:rectangle
				(:line-style NIL)
				(:filling-style opal:dark-gray-fill)
				(:left left)
				(:top top)
				(:width (- (fourth rect) left))
				(:height (- (second rect) top))))))
    (create-instance 'outer-square opal:rectangle
				(:left *mat-left*)
				(:top  *mat-top*)
				(:width (- *mat-right* *mat-left*))
				(:height (- *mat-bottom* *mat-top*)))
    (opal:add-component board outer-square)
    (s-value board :n n)

		;; Now we must create the N x N squares...
    (dotimes (row n)
      (incf row-left row-width)
      (setq old-row-end row-end)
      (setq row-end (floor (* (/ (1+ row) n) *board-width*)))
      (setq row-width (- row-end old-row-end))
      (setq col-top *board-top*)
      (setq old-col-end (setq col-end 0))
      (dotimes (col n)
        (incf col-top (- col-end old-col-end))
	(setq old-col-end col-end)
        (setq col-end (floor (* (/ (1+ col) n) *board-height*)))
	(setq temp (aref *rectangles* rect-index))
	(push temp active-list)
	(incf rect-index 1)
	(s-value temp :left row-left)
	(s-value temp :width row-width)
	(s-value temp :top col-top)
	(s-value temp :height (- col-end old-col-end))
	(s-value temp :row row)
	(s-value temp :column col)
	(s-value temp :player NIL)
	(s-value temp :visible T)
	(setf (aref board-array row col) temp)))
    (s-value squares :active-list active-list)

		;; Set all the remaining rectangles to be invisible
    (do ((x rect-index (1+ x)))
	((>= x *rect-count*))
	(s-value (aref *rectangles* x) :visible NIL))

    (opal:add-component board squares :front)

    (let* ((hi-square (/ n 2))
	   (lo-square (1- hi-square)))
	(Make-Game-Piece-Pair lo-square lo-square 1)
	(Make-Game-Piece-Pair hi-square hi-square 1)
	(Make-Game-Piece-Pair lo-square hi-square 2)
	(Make-Game-Piece-Pair hi-square lo-square 2))

    (s-value message :string "Player 1's move")
    (setq *current-player* 1)
    (setq *other-player* 2)
    (setq *scores* (make-array 3))
    (setf (aref *scores* 1) 2)
    (setf (aref *scores* 2) 2)
    (set-scores)
    (opal:add-components top-agg board score-box-game-pieces)

    (create-instance 'board-inter inter:button-interactor
		(:start-where `(:list-element-of ,squares :active-list))
		(:window w)
		(:stop-action #'process-button-press))

    ;; For some reason, if we don't include this, the update takes forever!
    (opal:update w T)
  )
)

(defun Do-Go ()
  (Preliminaries)
  (Make-Window-And-Title)
  (Make-Score-Box)
  (Make-Control-Panel)
  (Make-Scroll-Bar)
  (Make-Message-Box)
  (Start-Game)
)

(defun Set-Score (player score)
  (unless (numberp score) (setq score 0))
  (let  ((score-string (princ-to-string score)))
	(if (eq player 1)
		(s-value player1-score :string score-string)
		(s-value player2-score :string score-string))))

(defun Do-Stop  ()
;  (opal:destroy w)
  (format t "~% quit Othello ~%")
)
