(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: LoopStmt.m3                                           *)
(* Last modified on Mon Mar  2 11:10:54 PST 1992 by kalsow     *)
(*      modified on Mon Oct 16 14:12:01 1989 by muller         *)

MODULE LoopStmt;

IMPORT M3, Emit, Scanner, Stmt, StmtRep, Marker, Token;

TYPE
  P = Stmt.T OBJECT
        body    : Stmt.T;
      OVERRIDES
        check    := Check;
	compile  := Compile;
        outcomes := GetOutcome;
      END;

PROCEDURE Parse (READONLY fail: Token.Set): Stmt.T =
  TYPE TK = Token.T;
  VAR p: P;
  BEGIN
    p := NEW (P);
    StmtRep.Init (p);
    Scanner.Match (TK.tLOOP, fail, Token.Set {TK.tEND});
    p.body := Stmt.Parse (fail + Token.Set {TK.tEND});
    Scanner.Match1 (TK.tEND, fail);
    RETURN p;
  END Parse;

PROCEDURE Check (p: P;  VAR cs: Stmt.CheckState) =
  BEGIN
    Marker.PushExit (0);
    Stmt.TypeCheck (p.body, cs);
    Marker.Pop ();
  END Check;

PROCEDURE Compile (p: P): Stmt.Outcomes =
  VAR label: INTEGER;  oc: Stmt.Outcomes;
  BEGIN
    label := M3.NextLabel;  INC (M3.NextLabel, 2);
    Marker.PushExit (label+1);
    Emit.OpL ("@:;\n\001", label);
    oc := Stmt.Compile (p.body);
    Marker.Pop ();
    IF (Stmt.Outcome.FallThrough IN oc) THEN
      Emit.OpL ("goto @;\n", label);
      oc := oc - Stmt.Outcomes {Stmt.Outcome.FallThrough};
    END;
    Emit.Op ("\002");
    IF (Stmt.Outcome.Exits IN oc) THEN
      Emit.OpL ("@:;\n", label+1);
      oc := oc + Stmt.Outcomes {Stmt.Outcome.FallThrough}
               - Stmt.Outcomes {Stmt.Outcome.Exits};
    END;
    RETURN oc;
  END Compile;

PROCEDURE GetOutcome (p: P): Stmt.Outcomes =
  VAR oc := Stmt.GetOutcome (p.body);
  BEGIN
    oc := oc - Stmt.Outcomes {Stmt.Outcome.FallThrough};
    IF (Stmt.Outcome.Exits IN oc) THEN
      oc := oc + Stmt.Outcomes {Stmt.Outcome.FallThrough}
               - Stmt.Outcomes {Stmt.Outcome.Exits};
    END;
    RETURN oc;
  END GetOutcome;

BEGIN
END LoopStmt.
