(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created 1990-01-17 by Jorge Stolfi                          *)
(*      inspired on an original novel by John DeTreville       *)
(* Last modified on Tue Feb 11 17:13:34 PST 1992 by muller     *)
(*      modified on Mon Jun 24 18:39:44 1991 by kalsow     *)
(*      modified on Fri Mar 29 15:17:02 PST 1991 by stolfi     *)

MODULE LongRealTime;

IMPORT Time, Thread;

PROCEDURE ToTime(rt: T): Time.T =
  VAR it: Time.T;
  BEGIN
    rt := rt + 0.5d0 * Microsecond;
    it.seconds := TRUNC(rt / Second);
    it.microseconds := MIN(999999, MAX(0,
      TRUNC((rt - FLOAT(it.seconds, LONGREAL) * Second) / Microsecond)
    ));
    RETURN it;
  END ToTime;

PROCEDURE FromTime(it: Time.T): T =
  BEGIN
    RETURN FLOAT(it.seconds, LONGREAL) * Second 
         + FLOAT(it.microseconds, LONGREAL) * Microsecond
  END FromTime;

PROCEDURE Now(): T =
  VAR it: Time.T;
  BEGIN
    it := Time.Now();
    RETURN FLOAT(it.seconds, LONGREAL) * Second 
         + FLOAT(it.microseconds, LONGREAL) * Microsecond
  END Now;

PROCEDURE Pause(rt: T) RAISES {Thread.Alerted} =
  CONST
    MaxShortPause = FLOAT(LAST(CARDINAL) - 1, LONGREAL) * Microsecond;
    MaxLongPause = FLOAT(LAST(CARDINAL) - 1, LONGREAL) * Second;
  BEGIN
    IF rt <= 0.0d0 THEN RETURN  END;
    IF rt < MaxShortPause THEN
      Time.Pause(TRUNC(rt / Microsecond))
    ELSIF rt < MaxLongPause THEN
      Time.LongPause(TRUNC(rt / Second))
    ELSE
      Time.LongPause(LAST(CARDINAL))
    END
  END Pause;

PROCEDURE PauseUntil(rt: T) RAISES {Thread.Alerted} =
  CONST MaxUntil = FLOAT(LAST(Time.Seconds) - 1, LONGREAL) * Second;
  VAR it: Time.T;
  BEGIN
    IF rt < 0.0d0 THEN RETURN  END;
    rt := rt + 0.5d0 * Microsecond;
    IF rt >= MaxUntil THEN
      it.seconds := LAST(Time.Seconds);
      it.microseconds := 0
    ELSE
      it.seconds := TRUNC(rt / Second);
      it.microseconds := MIN(999999, MAX(0,
        TRUNC((rt - FLOAT(it.seconds, LONGREAL) * Second) / Microsecond)
      ));
    END;
    Time.PauseUntil(it)
  END PauseUntil;

BEGIN

END LongRealTime.

