	/* continuations */

#include "fools.h"
#include "codegen.h"
#include "cont.h"

#ifndef lint
static char SccsId[] = "@(#)cont.c	1.4 1/16/90";
#endif /* lint */

extern char *realloc();

Obj curCont;		/* current continuation */
contState_t *ccState;	/* state of current continuation */
Mem stateAlloc;		/* continuation state allocator */

static list_t contList;	/* list of continuations on the stack */

static void contPush(), contPop();

/* callback function to restore the continuation of the previous codeExec */
static void contRestore(prev)
     Obj prev;
{
    if (prev) {
	ASSERT(DATA(prev, rc, basicInst) > 0);
	--DATA(prev, rc, basicInst);
    }
    contResume(prev);
}

/* make a top-level continuation for the current codeExec */
void codeBegin(cb, cvec, frame)
     Callback_t *cb;
     Obj cvec, frame;
{
    if (curCont) objLink(curCont);
    cb->arg = (Ptr)curCont;
    cb->func = contRestore;

    contPush(newCont(gcNew, cvec, frame, (Obj)NULL));
    errorPushCB(cb);
}

/* restore the continuation of the previous codeExec */
void codeEnd(cb)
     Callback_t *cb;
{
    contRestore((Obj)cb->arg);
    errorPopCB();
}

/* resume continuation cont */
void contResume(cont)
     Obj cont;
{
#ifdef DEBUG
    if (debugCont)
	objfPrintf(stderr, "resuming %O\n", cont);
#endif /* DEBUG */

    ASSERT(cont != curCont);
    if (cont) {
	objLink(cont);
	if (listFind(&contList, (F_BOOLEAN)NULL, cont)) {
	    while (cont != curCont) contPop();
	    if (DATA(cont, rc, basicInst) > 2) {
		/* resume a duplicate of the current continuation */
		contPop();
		contPush(cont); /* resumes a duplicate by side-effect */
	    }
	}
	else {
	    while (!listEmpty(&contList)) contPop();
	    contPush(cont);
	}
	objUnlink(cont);
    }
    else while (!listEmpty(&contList)) contPop();
}    

/* resume the parent continuation of the current continuation
 *
 * When the current continuation completes the parent continuation is
 * resumed with the result of the current continuation.
 * If the current continuation is a top-level continuation, then
 * its result is returned (otherwise, NULL is returned). */
Obj contContinue()
{
    Obj par, res;

    res = stackPop(argStack);
    if (par = ccState->parent) {
	contResume(par);
	stackPush(res, argStack);
#ifdef DEBUG
	if (debugCont)
	    objfPrintf(stderr, "continuing %O\n", curCont);
#endif /* DEBUG */
	return (Obj)NULL;
    }
    return  res;
}

/* start a child continuation of the current continuation
 *
 * the new continuation will initially have the current continuation
 * pushed on its stack */
void contStart()
{
    Obj parent, new;
    stackFrame_t *sf;

    ASSERT(curCont != (Obj)NULL);
    objLink(parent = curCont);

    new = newCont(gcNew, ccVec, (Obj)NULL, parent);
    sf = NEW_FRAME();
    sf->proc = (Obj)NULL;
    sf->frame = (Obj)NULL;
    sf->vec = DATA(ccVec, vec, codevecInst);
    listPush((Ptr)sf, DATA(new, state, contInst)->alist);
    contPush(new);
    stackPush(parent, argStack);

#ifdef DEBUG
    if (debugCont)
	objfPrintf(stderr, "starting %O\n", curCont);
#endif /* DEBUG */
}

/* If the continuation is currently executing a primitive print out an
 * an activation for it */
static void primActivationDump(cont)
     Obj cont;
{
    Obj *argv;
    int argc;
    contState_t *state;

    state = DATA(cont, state, contInst);
    if (state->prim) {
	argc = state->argc;
	argv = checkCond(cont, ZAP) ?
	    state->base + state->size - argc : stackAddr(argStack, argc);

	objfPrintf(stderr, "(%O", state->prim);
	while (--argc >= 0)
	    objfPrintf(stderr, " %O", *argv++);
	fputs(")\n", stderr);
    }
}

/* print the activition for stack frame sf */
static void activationDump(sf)
     stackFrame_t *sf;
{
    Obj *fixed, formals;
    int nints, nargs;

    if (sf->proc == (Obj)NULL) return ;

    nargs = DATA(sf->proc, numargs, procInst);
    if (checkCond(sf->proc, OPTARG)) ++nargs;
    fixed = DATA(sf->frame, fixed, frameInst);
    formals = DATA(sf->frame, formals, frameInst);
    nints = DATA(sf->frame, numfixed, frameInst) - nargs;

    objfPrintf(stderr, "(%O", sf->proc);
    while (--nargs >= 0) {
	ASSERT(CLASS(formals) == Pair);
	objfPrintf(stderr, " %O=%O", objCar(formals), *fixed++);
	formals = objCdr(formals);
    }
    (void)fprintf(stderr, ")%s\n", checkCond(sf->frame, ZAP) ? "*" : "");
    if (nints > 0) {
	fputs("* internal definitions\n", stderr);
	while (--nints >= 0) {
	    ASSERT(CLASS(formals) == Pair);
	    objfPrintf(stderr, "%O=%O\n", objCar(formals), *fixed++);
	    formals = objCdr(formals);
	}
	fputs("*\n", stderr);
    }
}

/* print the activation list of cont */
static void contDump(cont)
     Obj cont;
{
    List alist;
    contState_t *state;

    if (cont == (Obj)NULL) return ;

    state = DATA(cont, state, contInst);
    alist = state->alist;
    if (!listEmpty(alist)) {
	primActivationDump(cont);
	listApply1(alist, activationDump);
    }
}

/* display the stacks of the current continuation and its parents */
void stackDump()
{
    Obj cc;

    if (curCont) {
	fputs("*** call stack\n", stderr);
	for (cc = curCont; cc; cc = DATA(cc, state, contInst)->parent)
	    contDump(cc);
	fputs("*** end\n", stderr);
    }
}

/* copy the activation list */
static void copyActivationList(alist, copy)
     List alist, copy;
{
    stackFrame_t *sf;
    Obj obj;

    ASSERT(listEmpty(copy));
    LIST_FOR_EACH(alist, item, {
	sf = NEW_FRAME();
	*sf = *(stackFrame_t *)item;

	if (obj = sf->frame) {
	    ASSERT(objIsClass(obj, Frame) && checkCond(obj, ZAP));
	    objLink(obj);
	}
	if (obj = sf->proc) {
	    ASSERT(objIsClass(obj, Proc));
	    objLink(obj);
	}
	listPush((Ptr)sf, copy);
    });
    listReverse(copy);
}

/* copy the continuation cont
 *
 * The stack of the copy is built on the argStack. */
static Obj contCopy(cont)
     Obj cont;
{
    contState_t *state;
    Obj *ptr, obj, new;
    int size;

    ASSERT(checkCond(cont, ZAP));
    state = DATA(cont, state, contInst);
    new = newCont(gcNew, state->cvec, (Obj)NULL, state->parent);

#ifdef DEBUG
    if (debugCont)
	objfPrintf(stderr, "copying %O -> %O\n", cont, new);
#endif /* DEBUG */

    size = state->size;
    ptr = state->base;

    while (--size >= 0) {
	ASSERT(*ptr != (Obj)NULL);
	objLink(obj = *ptr++);
	stackPush(obj, argStack);
    }
    copyActivationList(state->alist, DATA(new, state, contInst)->alist);

    return new;
}	

/* remove the stack of the continuation from argStack to the heap */
static void contSaveStack(state)
     contState_t *state; /* state pointer of the continuation */
{
    List alist;
    Obj *ptr, *base_ptr, *base, *lim, frame;
    int size;

    alist = state->alist;
    ptr = state->base;

    size = state->size;
    stackAdj(argStack, size);
    base_ptr = base = NEWVEC(Obj, size);

    listReverse(alist); /* go through activation list backwards */
    LIST_FOR_EACH(alist, item, {
	/* remove locals from the stack */

	if (frame = ((stackFrame_t *)item)->frame) {
	    ASSERT(objIsClass(frame, Frame));
	    if (!checkCond(frame, ZAP)) {
		int numfixed = DATA(frame, numfixed, frameInst);

		/* copy args below the fixed vector to the stack */
		if (numfixed > 0) {
		    lim = DATA(frame, fixed, frameInst);
		    while (ptr < lim) {
			ASSERT(*ptr != (Obj)NULL);
			*base_ptr++ = *ptr++;
		    }
		    ptr += numfixed;
		}
		saveFrame(frame, FALSE);
	    }
	}
    });

    /* copy remaining args to the stack */
    lim = state->base + size;
    while (ptr < lim) {
	ASSERT(*ptr != (Obj)NULL);
	*base_ptr++ = *ptr++;
    }
    listReverse(alist); /* put alist back in order */

    if ((state->size = base_ptr - base) == 0) {
	state->base = (Obj *)NULL;
	(void)free((char *)base);
    }
    else state->base = base;

#ifdef DEBUG
	if (debugCont)
	    objfPrintf(stderr, "copied stack of %O\n", curCont);
#endif /* DEBUG */
}
    
/* remove the top continuation from the stack replacing it with the
 * continuation below it */
static void contPop()
{

#ifdef DEBUG
    if (debugCont)
	objfPrintf(stderr, "popping %O\n", curCont);
#endif /* DEBUG */

    ccState->size = stackPtr(argStack) - ccState->base;
    if (DATA(curCont, rc, basicInst) > 1) {
	/* save arg stack of continuation */
	setCond(curCont, ZAP);
	contSaveStack(ccState);
    }
    (void)listPop(&contList);
    objUnlink(curCont);

    curCont = (Obj)listPeek(&contList);
    ccState = curCont ? DATA(curCont, state, contInst) : (contState_t *)NULL;
}

/* make cont the current continuation */
static void contPush(cont)
     Obj cont;
{
    Obj *save, *ptr;
    int size;
    contState_t *state;

#ifdef DEBUG
    if (debugCont)
	objfPrintf(stderr, "pushing %O\n", cont);
#endif /* DEBUG */

    ASSERT(curCont == (Obj)NULL || !checkCond(curCont, ZAP));
    if (curCont)
	ccState->size = stackPtr(argStack) - ccState->base;

    if (DATA(cont, rc, basicInst) > 0)
	cont = contCopy(cont);
    else if (checkCond(cont, ZAP)) {
	clearCond(cont, ZAP);
	state = DATA(cont, state, contInst);
	ptr = save = state->base;
	state->base = stackPtr(argStack);
	if ((size = state->size) > 0) {
	    while (--size >= 0)
		stackPush(*ptr++, argStack);

	    ASSERT((long)save < (long)argStack->min
		   || (long)save >= (long)argStack->max);
	    (void)free((char *)save);
	}
    }

    ASSERT(listFind(&contList, (F_BOOLEAN)NULL, (Ptr)cont) == (Ptr)NULL);
    objLink(cont);
    listPush((Ptr)cont, &contList);
    curCont = cont;
    ccState = DATA(cont, state, contInst);
}

	/* continuation instance */

/* unlink and pop frame from the stack */
static void contSaveFrame(frame)
     Obj frame;
{
    Obj *fixed;
    int numfixed;

    if (!checkCond(frame, ZAP)) {
	fixed = DATA(frame, fixed, frameInst);
	numfixed = DATA(frame, numfixed, frameInst);
	if (DATA(frame, rc, basicInst) > 1)
	    saveFrame(frame, FALSE); /* save fixed vector */
	objUnlink(frame);
	while (--numfixed >= 0)
	    /* zero frame on the stack to avoid a double unlink */
	    *fixed++ = (Obj)NULL;
    }
    else objUnlink(frame);
}

/* destructor */
static void contDestroy(cont)
     Obj cont;
{
    List alist;
    Obj obj, *ptr;
    int size;
    contState_t *state;
    stackFrame_t *sf;

    ASSERT(listFind(&contList, (F_BOOLEAN)NULL, (Ptr)cont) == (Ptr)NULL);

    state = DATA(cont, state, contInst);
    objUnlink(state->cvec);
    if (obj = state->parent) objUnlink(obj);
    if (obj = state->prim) objUnlink(obj);

    /* unlink activation list */
    alist = state->alist;
    while (sf = (stackFrame_t *)listPop(alist)) {
	/* unlink each stack frame in the activation list */

	if (obj = sf->frame) {
	    ASSERT(objIsClass(obj, Frame));
	    contSaveFrame(obj);
	    if (obj = sf->proc) {
		ASSERT(objIsClass(obj, Proc));
		objUnlink(obj);
	    }
	}
	FREE_FRAME(sf);
    }
    listFree(alist, (F_VOID)NULL);

    if (checkCond(cont, ZAP)) {
	if ((size = state->size) > 0) {
	    ptr = state->base;
	    ASSERT((long)ptr < (long)argStack->min
		   || (long)ptr >= (long)argStack->max);
	    while (--size >= 0)
		if (obj = *ptr++) objUnlink(obj);
	    (void)free((char *)state->base);
	}
    }
    else {
	size = stackPtr(argStack) - state->base;
	while (--size >= 0)
	    if (obj = stackPop(argStack)) objUnlink(obj);
	ASSERT(stackPtr(argStack) == state->base);
    }

    memFreeBlock((Ptr)state, stateAlloc);
}

/* create a new continuation
 *
 * If frame is not NULL then an initial stack frame is added to the
 * activation list. */
Obj newCont(alloc, cvec, frame, par)
     F_OBJ alloc;
     Obj cvec, frame, par;
{
    contState_t *state;
    stackFrame_t *sf;
    Obj new;

    new = (*alloc)(Continuation);
    DATA(new, numargs, procInst) = 1;
    DATA(new, state, contInst) = state = (contState_t *)memBlock(stateAlloc);
    if (state->parent = par) objLink(par);
    objLink(state->cvec = cvec);
    state->prim = (Obj)NULL;
    state->base = stackPtr(argStack);
    state->size = 0;
    state->alist = listNew();

    if (frame) {
	sf = NEW_FRAME();
	listPush((Ptr)sf, state->alist);
	objLink(sf->frame = frame);
	sf->proc = (Obj)NULL;
	sf->vec = DATA(cvec, vec, codevecInst);
    }

    return new;
}

/* continuation class variable */
basicClass_t protoCont =
    DEFBASIC(Proc, contInst_t, (F_VOID)NULL, contDestroy, "continuation");
