/* error handling routines */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include <errno.h>
#include <signal.h>

/* include proper varargs header */
#ifdef __STDC__
/* ANSI varargs */
#include "stdarg.h"
#else
#include <varargs.h>
#endif /* defined(__STDC__) */

#ifndef lint
static char SccsId[] = "@(#)error.c	1.12 3/2/90";
#endif

/* NOTE: the printing routines have STDC and CC versions */

extern int errno, sys_nerr;
extern char *sys_errlist[];

/* global error code */
error_code errorFlag;

/* list of objects currently held by C functions */
List gcStack, gcList;

/* callback stack and the error context stack */    
static List callbackStack, errorStack;

/* don't output error messages for these */
#define noErrMessage(c)\
    ((c) == NoInput || (c) == SigInt)

/* don't dump the stack for these errors */
#define noStackDump(c)\
    ((c) == StackOverflow || (c) == NoInput)

/* Push a callback on the callback stack.  */
void errorPushCB(cb_ptr)
     Callback_t *cb_ptr;
{
    listPush((Ptr)cb_ptr, callbackStack);
}

/* Pop a callback from the callback stack. */
void errorPopCB()
{
    (void)listPop(callbackStack);
}

/* Call the callbacks left on the stack. */
static void errorCallCB()
{
    Callback_t *cb_ptr;

    while (cb_ptr = (Callback_t *)listPop(callbackStack))
	(*cb_ptr->func)(cb_ptr->arg);
}

/* Set an error level.  The jmp_buf is registered so that all following
 * errors can return to the calling function's setjmp.
 * Only callbacks registered after the current level will be called until
 * this error level is removed. */
void errorSetLevel(buf)
     jmp_buf buf;
{
    listPush((Ptr)gcStack, errorStack);
    listPush((Ptr)callbackStack, errorStack);
    listPush((Ptr)buf, errorStack); /* leave on top */

    gcList = (List)NULL;
    gcStack = listNew();

    callbackStack = listNew();
}
     
/* Remove the topmost error level.  Pops the jmp_buf and restores the
 * callback and garbage stacks. */
void errorClearLevel()
{
    /* remove jmp_buf */
    ASSERT(!listEmpty(errorStack));
    (void)listPop(errorStack);

    /* restore callbackStack */
    if (!listEmpty(callbackStack))
	errorCallCB();
    listFree(callbackStack, (F_VOID)NULL);
    callbackStack = (List)listPop(errorStack);

    /* restore garbage stack */
    if (!listEmpty(gcStack)) {
	while (!listEmpty(gcStack))
	    listFree((List)listPop(gcStack), _objUnlink);
    }
    listFree(gcStack, (F_VOID)NULL);
    gcStack = (List)listPop(errorStack);
    gcList = (gcStack != (List)NULL ? (List)listPeek(gcStack) : (List)NULL);
}

/* exit */
void errorExit(ret_code)
     int ret_code;
{
    while (errorStack && !listEmpty(errorStack))
	errorClearLevel();
    exit(ret_code);
}

/* message corresponding to error_code code */
char *errorMesg(code)
     error_code code;
{
    char *mesg;

    switch (code) {
    case NoInput:
    case Abort:
    case SigInt:
	mesg = (char *)NULL;
	break ;
    case StackOverflow:
	mesg = "Stack overflow";
	break ;
    case SigFPE:
	mesg = "Floating point exception";
	break ;
    case BadParen:
	return "Unmatched right parenthesis";
    case BadBrace:
	return "Unmatched right brace";
    case BadDot:
	return "Incorrect dot notation";
    case BadFormals:
	return "Incorrect formal parameter";
    case BadArgs:
	return "Incorrect number of arguments";
    case BadSyntax:
	return "Bad list structure";
    case BadChar:
	return "Incorrect character";
    case BadSymbol:
	return "Undefined symbol";
    case BadProc:
	return "Application of non-procedure object";
    case BadClass:
	return "Type mismatch:";
    case BadFile:
	mesg = (errno >= sys_nerr || errno <= 0)
	    ? "while opening file" : sys_errlist[ errno ];
	errno = 0;
	return mesg;
    case BadParse:
	mesg = "EOF while parsing";
	break ;
    case BadRead:
    case BadWrite:
	mesg = (errno >= sys_nerr || errno <= 0)
	    ? (code == BadRead ? "read error" : "write error")
		: sys_errlist[ errno ];
	errno = 0;
	break ;
    case BadVal:
	return "Value out of bounds";
    case BadSplice:
	return "Invalid context for splice";
    case Other:
	return (char *)NULL;
    default:
	return "Unknown error";
    }
    return mesg;
}

/* variable argument object printer */
void VobjPrintf(f, fmt, ap)
     FILE *f;
     char *fmt;
     va_list *ap;
{
    char c;
    Obj arg;

    for (; *fmt; fmt++) {
	switch (c = *fmt) {
	  case '%':
	    switch (c = *(++fmt)) {
	      case '\0':
		return ;
	      case 'O': /* object */
		if (arg = va_arg(*ap, Obj))
		    objPrint(arg, f);
		else fputs("#<undef>", f);
		break ;
	      case 'd':
		(void)fprintf(f, "%d", va_arg(*ap, int));
		break ;
	      case '%':
		fputc('%', f);
		break ;
	      case 's':
		fputs(va_arg(*ap, char *), f);
		break ;
	    case 'c':
		putc(va_arg(*ap, int), f);
		break ;
	      default:
		fputc(c, f);
		break ;
	    }
	    break ;
	  default:
	    fputc(c, f);
	    break ;
	}
    }
}

#ifdef lint

/*ARGSUSED*/
/*VARARGS2*/
void objfPrintf(fp, fmt)
     FILE *fp;
     char *fmt;
{
}

/*ARGSUSED*/
/*VARARGS1*/
void objPrintf(fmt)
     char *fmt;
{
}

#else

#ifdef __STDC__

void objfPrintf(FILE *fp, char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    VobjPrintf(fp, fmt, &args);
    va_end(args);
}

void objPrintf(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    VobjPrintf(stdout, fmt, &args);
    va_end(args);
}

#else

/*VARARGS*/
void objfPrintf(va_alist)
     va_dcl
{
    va_list args;
    FILE *fp;
    char *fmt;

    va_start(args);
    fp = va_arg(args, FILE *);
    fmt = va_arg(args, char *);
    VobjPrintf(fp, fmt, &args);
    va_end(args);
}

/*VARARGS*/
void objPrintf(va_alist)
     va_dcl
{
    va_list args;
    char *fmt;

    va_start(args);
    fmt = va_arg(args, char *);
    VobjPrintf(stdout, fmt, &args);
    va_end(args);
}

#endif /* defined(__STDC__) */

#endif /* defined(lint) */

/* Call callbacks and longjmp back to the current error level. */
static void errorJump()
{
    extern void stackDump();
    ASSERT(!listEmpty(errorStack));

    if (!noStackDump(errorFlag)) stackDump();
    errorCallCB();
    while (!listEmpty(gcStack))
	listFree((List)listPop(gcStack), _objUnlink);
    longjmp(/* (jmp_buf) */ listPeek(errorStack), 1);
}

/* print an error message and reset to the top jmp_buf in ErrorList */

#ifdef lint

/*ARGSUSED*/
/*VARARGS2*/
void errorPrint(code, fmt)
     error_code code;
     char *fmt;
{
}

#else

#ifdef __STDC__

void errorPrint(error_code code, char *fmt, ...)
{
    va_list args;
    char *mesg;

    errorFlag = code;
    fflush(stdout);

    intDisable();

    if (!noErrMessage(errorFlag)) {
	if (mesg = errorMesg(code)) {
	    (void)fputs(mesg, stderr);
	    putc(' ', stderr);
	}
	if (fmt) {
	    va_start(args, fmt);
	    VobjPrintf(stderr, fmt, &args);
	    va_end(args);
	}
	putc('\n', stderr);
    }

    errorJump();
}

#else

/*VARARGS*/
void errorPrint(va_alist)
     va_dcl
{
    va_list args;
    error_code code;
    char *fmt;
    char *mesg;

    va_start(args);
    code = va_arg(args, error_code);
    fmt = va_arg(args, char *);

    intDisable();
    errorFlag = code;
    fflush(stdout);

    if (!noErrMessage(errorFlag)) {
	if (mesg = errorMesg(code)) {
	    (void)fputs(mesg, stderr);
	    putc(' ', stderr);
	}
	if (fmt) {
	    VobjPrintf(stderr, fmt, &args);
	    va_end(args);
	}
	putc('\n', stderr);
    }

    errorJump();
}

#endif /* defined(__STDC__) */

#endif /* defined(lint) */

/* misc signal handlers */

static volatile Boolean intHit = FALSE;
static Boolean intOkay = FALSE;

/*ARGSUSED*/
static int intHandler(sig, code, scp)
     int sig, code;
     struct sigcontext *scp;
{
    if (!intOkay) {
	intHit = TRUE;
	return ;
    }

    fputs("\nQuit!\n", stderr);
    errorPrint(Abort, (char *)NULL);
    /*NOTREACHED*/
}

/* check for a SIGINT */
void intCheck()
{
    if (intHit) {
	intHit = FALSE;

	fputs("\nQuit!\n", stderr);
	errorPrint(SigInt, (char *)NULL);
    }
}

/* enable SIGINT trapping */
void intEnable()
{
    intOkay = TRUE;

    intCheck();
}

/* ignore SIGINTs */
void intDisable()
{
    intOkay = FALSE;
}

/*ARGSUSED*/
static int fpeHandler(sig, code, scp)
     int sig, code;
     struct sigcontext *scp;
{
    errorPrint(SigFPE, (char *)NULL);
    /*NOTREACHED*/
}

void errorInit()
{
    errorStack = listNew();	/* error context stack */
    callbackStack = gcStack = gcList = (List)NULL;

    signal(SIGINT, intHandler);
    signal(SIGFPE, fpeHandler);

    intDisable();
}
