/* primitives */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include "prim.h"
#include "refine.h"
#include "parser.h"
#include "utils.h"
#include "codegen.h"
#include <sys/file.h>
#include <sys/stat.h>
#include <ctype.h>

#ifndef lint
static char SccsId[] = "@(#)prim.c	1.17 2/23/90";
#endif

	/* pairs */

DEFINE(primCons)
{
    ASSERT(argc == 2);
    return newPair(gcNew, argv[0], argv[1]);
}

DEFINE(primCar)
{
    Obj pair = argv[0];

    ASSERT(argc == 1);
    typeStrict(pair, Pair);
    return objCar(pair);
}

DEFINE(primCdr)
{
    Obj pair = argv[0];

    ASSERT(argc == 1);
    typeStrict(pair, Pair);
    return objCdr(pair);
}

DEFINE(primList)
{
    Obj list;

    list = NilSymb;
    argv += argc;
    while (--argc >= 0)
	list = newPair(gcNew, *(--argv), list);
    return list;
}

DEFINE(primListStar)
{
    Obj list;

    ASSERT(argc >= 1);
    argv += argc;
    list = *(--argv);
    while (--argc > 0)
	list = newPair(gcNew, *(--argv), list);
    return list;
}

DEFINE(primAppend)
{
    Obj base = argv[0], rest = argv[1], res, next;

    ASSERT(argc == 2);

    if (base == NilSymb) return rest;
    typeStrict(base, Pair);

    for (next = (Obj)NULL; CLASS(base) == Pair; base = objCdr(base)) {
	Obj temp;

	intCheck();

	temp = newPair(gcNew, objCar(base), NilSymb);
	if (next == (Obj)NULL) res = next = temp;
	else {
	    objSetCdr(next, temp);
	    next = temp;
	}
    }
    if (base != NilSymb) errorPrint(BadClass, "%O is not a pair", base);
    objSetCdr(next, rest);

    return res;
}

DEFINE(primSetCar)
{
    ASSERT(argc == 2);

    typeStrict(argv[0], Pair);
    objSetCar(argv[0], argv[1]);
    return argv[1];
}

DEFINE(primSetCdr)
{
    ASSERT(argc == 2);

    typeStrict(argv[0], Pair);
    objSetCdr(argv[0], argv[1]);
    return argv[1];
}

	/* strings */
    
DEFINE(primMakeStr)
{
    int size;
    Obj new;
    char *str, fill = ' ';

    ASSERT(argc >= 1);
    
    typeCheck(argv[0], Integer);
    if (argc > 1) {
	typeCheck(argv[1], Character);
	fill = (char)objInteger(argv[1]);
    }
    if ((size = (int)objNum(argv[0])) < 0)
	errorPrint(BadVal, "(negative value for string size)");
    new = newString1(gcNew, size);
    str = objString(new);

    if (size > 0) {
	while (--size >= 0) *(str++) = fill;
	*str = '\0';
    }

    return new;
}

DEFINE(primStrCat)
{
    Obj nstr;
    char *str;
    int i, len;

    ASSERT(argc >= 2);

    for (i = len = 0; i < argc; i++) {
	typeCheck(argv[i], String);
	len += strlen(objString(argv[i]));
    }

    nstr = newString1(gcNew, len);
    if (len > 0) {
	str = objString(nstr);
	(void)strcpy(str, objString(*argv++));
	while (--argc > 0)
	    (void)strcat(str, objString(*argv++));
    }

    return nstr;
}

DEFINE(primStrLen)
{
    ASSERT(argc == 1);

    typeCheck(argv[0], String);
    return newInteger(gcNew, (long)strlen(DATA(argv[0], string, stringInst)));
}

DEFINE(primStrRef)
{
    int ref;
    char *str;

    ASSERT(argc == 2);

    typeCheck(argv[0], String);
    typeCheck(argv[1], Number);

    ref = (int)objNum(argv[1]);
    str = DATA(argv[0], string, stringInst);
    if (ref < 0 || ref >= DATA(argv[0], maxsize, stringInst))
	errorPrint(BadVal, "(index %d into size %d string)", ref,
		   DATA(argv[0], maxsize, stringInst));

    return newChar(gcNew, str[ref]);
}

DEFINE(primStrSet)
{
    int ref, val;
    char *str;

    ASSERT(argc == 3);

    typeStrict(argv[0], String); /* don't allow symbol modification */
    typeCheck(argv[1], Integer);
    typeCheck(argv[2], Character);

    val = objInteger(argv[2]);
    ref = (int)objInteger(argv[1]);
    str = DATA(argv[0], string, stringInst);
    if (ref < 0 || ref >= DATA(argv[0], maxsize, stringInst))
	errorPrint(BadVal, "(index %d into size %d string)", ref,
		   DATA(argv[0], maxsize, stringInst));

    str[ref] = val;
    return argv[0];
}

DEFINE(primStrCpy)
{
    typeCheck(argv[0], String);
    return newString(gcNew, objString(argv[0]));
}

DEFINE(primSubStr)
{
    int start, end, len;
    Obj str;
    char *s, *s1;

    typeCheck(argv[0], String);
    typeCheck(argv[1], Integer);
    typeCheck(argv[2], Integer);

    s1 = objString(argv[0]);
    start = objInteger(argv[1]);
    end = objInteger(argv[2]);
    if (end < 0 || start < 0 || start > end || end >= strlen(s1))
	errorPrint(Other, "Index values %d and %d are not valid for %O",
		   start, end, argv[0]);
    len = end - start + 1;

    str = newString1(gcNew, len);
    s = objString(str);
    strncpy(s, s1, len);
    s[len] = '\0';
    return str;
}

DEFINE(primStrGT)
{
    ASSERT(argc == 2);

    typeCheck(argv[0], String);
    typeCheck(argv[1], String);
    return strcmp(objString(argv[0]), objString(argv[1])) > 0
 	? TrueSymb : NilSymb;
}

DEFINE(primStrGE)
{
    ASSERT(argc == 2);

    typeCheck(argv[0], String);
    typeCheck(argv[1], String);
    return strcmp(objString(argv[0]), objString(argv[1])) >= 0
	? TrueSymb : NilSymb;
}

DEFINE(primStrLT)
{
    ASSERT(argc == 2);

    typeCheck(argv[0], String);
    typeCheck(argv[1], String);
    return strcmp(objString(argv[0]), objString(argv[1])) < 0
	? TrueSymb : NilSymb;
}

DEFINE(primStrLE)
{
    ASSERT(argc == 2);

    typeCheck(argv[0], String);
    typeCheck(argv[1], String);
    return strcmp(objString(argv[0]), objString(argv[1])) <= 0
	? TrueSymb : NilSymb;
}

	/* vectors */

DEFINE(primVector)
{
    Obj vec;

    vec = newVector(gcNew, argc);
    argv += argc;
    while (--argc >= 0)
	objVectorSet(vec, argc, *(--argv));
    return vec;
}

DEFINE(primMakeVec)
{
    Obj vec;
    int size;
    ASSERT(argc >= 1);

    typeCheck(argv[0], Integer);
    size = (int)objInteger(argv[0]);
    if (size < 0)
	errorPrint(BadVal, "(negative value for vector size)");
    vec = newVector(gcNew, size);
    if (argc > 1) {
	while (--size >= 0)
	    objVectorSet(vec, size, argv[1]);
    }
    return vec;
}

DEFINE(primVecRef)
{
    Obj vec = argv[0];
    int pos;

    ASSERT(argc == 2);

    typeCheck(vec, Vector);
    typeCheck(argv[1], Integer);

    pos = objInteger(argv[1]);
    if (pos < 0 || pos >= DATA(vec, size, vectorInst))
	errorPrint(BadVal, "(index %d into size %d vector)", pos,
		   DATA(vec, size, vectorInst));

    return objVectorRef(vec, pos);
}

DEFINE(primVecSet)
{
    Obj vec = argv[0];
    int pos;

    ASSERT(argc == 3);

    typeStrict(vec, Vector);
    typeCheck(argv[1], Integer);

    pos = objInteger(argv[1]);
    if (pos < 0 || pos >= DATA(vec, size, vectorInst))
	errorPrint(BadVal, "(index %d into size %d vector)", pos,
		   DATA(vec, size, vectorInst));

    objVectorSet(vec, pos, argv[2]);

    return vec;
}

DEFINE(primVecSize)
{
    Obj vec = argv[0];

    ASSERT(argc == 1);

    typeCheck(vec, Vector);
    return newInteger(gcNew, (long)objVectorSize(vec));
}

	/* boxes */

DEFINE(primMakeBox)
{
    ASSERT(argc == 1);
    return newBox(gcNew, argv[0]);
}

DEFINE(primUnbox)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Box);
    return DATA(argv[0], ref, boxInst);
}

DEFINE(primSetBox)
{
    ASSERT(argc == 2);
    typeCheck(argv[0], Box);
    objBoxSet(argv[0], argv[1]);
    return argv[1];
}

	/* files */

DEFINE(primOpenFile)
{
    Obj filename = argv[0], mode = argv[1];
    FILE *file;

    ASSERT(argc == 2);

    typeCheck(filename, String);
    typeCheck(mode, String);

    file = fopen(expandFilename(objString(filename)), objString(mode));
    if (file == (FILE *)NULL)
	errorPrint(BadFile, "%O (mode %O)", filename, mode);
    return newFile(gcNew, file, strcmp(objString(mode), "w") == 0);
}

DEFINE(primCloseFile)
{
    Obj file = argv[0];

    ASSERT(argc == 1);

    typeCheck(file, File);
    fileClose(file);

    return file;
}

DEFINE(primFlushFile)
{
    FILE *file;

    ASSERT(argc == 1);
    typeCheck(argv[0], File);
    if (file = DATA(argv[0], file, fileInst))
	return fflush(file) == EOF ? FalseSymb : TrueSymb;
    return FalseSymb;
}
	
DEFINE(primAccess)
{
    Obj fn = argv[0];
    char *ptr;
    int mode = F_OK;

    ASSERT(argc == 2);
    typeCheck(fn, String);
    typeCheck(argv[1], String);
    ptr = objString(argv[1]);
    while (*ptr) {
	switch (*(ptr++)) {
	case 'r':
	    mode |= R_OK;
	    break ;
	case 'w':
	    mode |= W_OK;
	    break ;
	case 'x':
	    mode |= X_OK;
	    break ;
	case 'f':
	    mode |= F_OK;
	    break ;
	default:
	    errorPrint(Other, "valid modes for access are r, w, x, or f");
	}
    }
	
    return (access(expandFilename(objString(fn)), mode) == 0) ?
	TrueSymb : FalseSymb;
}

DEFINE(primFileTell)
{
    FILE *file;

    ASSERT(argc == 1);
    typeCheck(argv[0], File);
    if (file = DATA(argv[0], file, fileInst))
	return newInteger(gcNew, ftell(file));
    return FalseSymb;
}

DEFINE(primFileSeek)
{
    FILE *file;
    int type = 0;

    ASSERT(argc == 3);
    typeCheck(argv[0], File);
    typeCheck(argv[1], Integer);
    typeCheck(argv[2], Character); /* #\b == 0, #\c == 1, #\e == 2 */

    switch (objInteger(argv[2])) {
    case 'b':
    case 'B':
	type = 0;
	break ;
    case 'c':
    case 'C':
	type = 1;
	break ;
    case 'e':
    case 'E':
	type = 2;
	break ;
    default:
	errorPrint(Other, "file-seek:  seek type is #\\b, #\\c, or #\\e");
    }

    if ((file = DATA(argv[0], file, fileInst)) == (FILE *)NULL)
	return FalseSymb;

    return fseek(file, objInteger(argv[1]), type) == 0 ? TrueSymb : FalseSymb;
}

DEFINE(primFileString)
{
    struct stat sbuf;
    char *fn, *buf;
    Obj str;
    int len, fd, cnt;

    typeCheck(argv[0], String);

    fn = expandFilename(objString(argv[0]));
    if (stat(fn, &sbuf) < 0)
	errorPrint(BadRead, "%s", fn);
    len = (int)sbuf.st_size;

    if ((fd = open(fn, 0, O_RDONLY)) < 0)
	errorPrint(BadRead, "%s", fn);

    (void)printf(";;; opening length %d string for %s\n", len, fn);
    str = gcNew(String);
    DATA(str, string, stringInst) = buf = NEWVEC(char, len + 1);
    DATA(str, maxsize, stringInst) = len;

    while ((cnt = read(fd, buf, len)) > 0) {
	if ((len -= cnt) <= 0) break ;
	buf += cnt;
    }
    close(fd);
    if (cnt < 0) {
	objDestroy(str);
	errorPrint(BadRead, "%s", fn);
    }
    return str;
}
    
	/* type converters */

DEFINE(primVec2List)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Vector);
    return vectorToList(argv[0]);
}

DEFINE(primList2Vec)
{
    Obj vec;

    ASSERT(argc == 1);
    typeStrict(argv[0], Pair);
    if ((vec = listToVector(argv[0])) == (Obj)NULL)
	errorPrint(BadSyntax, "in %O; can't convert to vector", argv[0]);
    return vec;
}

struct bpair_s {
    List list;
    Obj frame;
};

static void bindingAdd(binding, arg)
     Obj binding;
     struct bpair_s *arg;
{
    Obj val;

    if (val = objUnbind(binding, arg->frame))
	listPush((Ptr)newPair(gcNew, objCar(binding), val), arg->list);
}

static Boolean bindingCmp(b1, b2)
     Obj b1, b2;
{
    return strcmp(objString(objCar(b1)), objString(objCar(b2))) > 0;
}

DEFINE(primEnv2List)
{
    Obj *fixed, fp, lst;
    Tree local;
    struct bpair_s arg;

    ASSERT(argc == 1);
    typeCheck(argv[0], Frame);

    arg.frame = argv[0];
    arg.list = listNew();

    if ((local = DATA(arg.frame, local, frameInst)) != (Tree)NULL)
	treeInOrder(local, bindingAdd, (Ptr)&arg);
    if ((fixed = DATA(arg.frame, fixed, frameInst)) != (Obj *)NULL) {
	fp = DATA(arg.frame, formals, frameInst);
	while (fp != NilSymb) {
	    listPush((Ptr)newPair(gcNew, objCar(fp), *(fixed++)), arg.list);
	    fp = objCdr(fp);
	}
    }
    listSort(arg.list, bindingCmp);
    lst = NilSymb;
    while (fp = (Obj)listPop(arg.list))
	lst = newPair(gcNew, fp, lst);
    listFree(arg.list, (F_VOID)NULL);
    return lst;
}

DEFINE(primStr2Int)
{
    ASSERT(argc == 2);
    typeCheck(argv[0], String);
    return parseInteger(objString(argv[0]));
}

DEFINE(primInt2Str)
{
    int base = 10;

    ASSERT(argc >= 1);
    typeCheck(argv[0], Number);
    if (argc > 1) {
	typeCheck(argv[1], Character);

	switch (objInteger(argv[1])) {
	case 'b': base = 2; break ;
	case 'o': base = 8; break ;
	case 'd': base = 10; break ;
	case 'x': base = 16; break ;
	default:
	    errorPrint(Other,
		       "integer->string:  allowed bases are b, o, d, or x");
	}
    }
    return newString(gcNew, num2str(objInteger(argv[0]), base));
}

DEFINE(primNum2Int)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Number);
    return newInteger(gcNew, (long)objNum(argv[0]));
}

DEFINE(primInt2Num)
{
    Obj new;

    ASSERT(argc == 1);
    typeCheck(argv[0], Integer);
    new = newNumber(gcNew, (double)objInteger(argv[0]));
    if (checkCond(argv[0], EXACT)) setCond(new, EXACT);
    return new;
}

DEFINE(primChar2Int)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Character);
    return newInteger(gcNew, objInteger(argv[0]));
}

DEFINE(primInt2Char)
{
    long num;

    ASSERT(argc == 1);
    typeCheck(argv[0], Integer);
    num = objInteger(argv[0]);
    if (num < 0 || num > 255)
	errorPrint(BadVal, "(%d is not in the character domain)", num);
    return newChar(gcNew, (char)num);
}    

DEFINE(primStr2Sym)
{
    ASSERT(argc == 1);

    typeStrict(argv[0], String);
    return objIntern(DATA(argv[0], string, stringInst), 0);
}

DEFINE(primStr2Uninterned)
{
    Obj new;
    char *str;

    ASSERT(argc == 1);
    typeCheck(argv[0], String);
    new = gcNew(Symbol);
    str = objString(argv[0]);
    DATA(new, string, stringInst) = strcopy(str);
    DATA(new, maxsize, stringInst) = strlen(str);

    return new;
}

DEFINE(primList2Str)
{
    int len;
    Obj ptr, str;
    char *s;

    for (len = 0, ptr = argv[0]; CLASS(ptr) == Pair; ++len, ptr = objCdr(ptr))
	typeStrict(objCar(ptr), Character);
    if (ptr != NilSymb)
	errorPrint(BadClass, "%O is not a proper list", argv[0]);
    str = newString1(gcNew, len);
    if (len > 0) {
	s = objString(str);
	for (ptr = argv[0]; --len >= 0; ptr = objCdr(ptr))
	    *s++ = (char)DATA(objCar(ptr), integer, intInst);
	*s = '\0';
    }
    return str;
}

DEFINE(primStr2List)
{
    int len;
    char *s;
    Obj list;

    len = strlen(s = objString(argv[0]));
    s += len;
    list = NilSymb;
    while (--len >= 0)
	list = newPair(gcNew, newChar(gcNew, *--s), list);
    return list;
}

	/* type check primitives */

#define BOOL(x) ((x) ? TrueSymb : FalseSymb)

DEFINE(primNullp)
{
    return BOOL(argv[0] == NilSymb);
}

DEFINE(primBooleanp)
{
    return BOOL(argv[0] == FalseSymb || argv[0] == TrueSymb);
}

DEFINE(primPairp)
{
    return BOOL(CLASS(argv[0]) == Pair);
}

DEFINE(primNumberp)
{
    return BOOL(objIsClass(argv[0], Number));
}

DEFINE(primIntegerp)
{
    return BOOL(objIsClass(argv[0], Integer));
}

DEFINE(primCharp)
{
    return BOOL(objIsClass(argv[0], Character));
}

DEFINE(primVectorp)
{
    return BOOL(objIsClass(argv[0], Vector));
}

DEFINE(primStringp)
{
    return BOOL(objIsClass(argv[0], String));
}

DEFINE(primSymbolp)
{
    return BOOL(objIsClass(argv[0], Symbol));
}

DEFINE(primProcedurep)
{
    return BOOL(objIsClass(argv[0], Proc));
}

DEFINE(primEOFp)
{
    return BOOL(argv[0] == EOFSymb);
}

DEFINE(primBoxp)
{
    return BOOL(objIsClass(argv[0], Box));
}

DEFINE(primObjType)
{
    Obj obj = argv[0];

    ASSERT(argc == 1);
    return objType(obj);
}

	/* equivalence predicates */

DEFINE(primEqp)
{
    ASSERT(argc == 2);

    return BOOL(argv[0] == argv[1]);
}

DEFINE(primEqvp)
{
    Class class;
    Obj a = argv[0], b = argv[1];

    ASSERT(argc == 2);
    if (a == b) return TrueSymb;

    if ((class = CLASS(a)) == CLASS(b)) {
	if (class == Number)
	    return objNum(a) == objNum(b) ? TrueSymb : FalseSymb;
	else if (class == Integer || class == Character)
	    return objInteger(a) == objInteger(b) ? TrueSymb : FalseSymb;
	else if (class == String)
	    return strcmp(objString(a), objString(b)) == 0 ?
		TrueSymb : FalseSymb;
	else if (class == Package)
	    return DATA(a, sym, packageInst) == DATA(b, sym, packageInst) &&
		DATA(a, package, packageInst) == DATA(b, package, packageInst)
		    ? TrueSymb : FalseSymb;
    }
    return FalseSymb;
}

DEFINE(primNot)
{
    ASSERT(argc == 1);

    return BOOL(argv[0] == FalseSymb);
}

	/* I/O */

DEFINE(primWrite)
{
    FILE *file;

    if (argc > 1) {
	typeStrict(argv[1], File);
	if ((file = objFile(argv[1])) == (FILE *)NULL)
	    errorPrint(BadWrite, "%O is closed", argv[1]);
    }
    else file = stdout;
      
    objPrint(argv[0], file);
    return TrueSymb;
}

DEFINE(primRead)
{
    Obj in, res;
    FILE *fp = (FILE *)NULL;

    if (argc > 0) {
	typeStrict(in = argv[0], File);
	if (argc > 1) {
	    typeStrict(argv[1], File);
	    if ((fp = objFile(argv[1])) == (FILE *)NULL)
		errorPrint(BadWrite, "%O is closed", argv[1]);
	}
    }
    else in = InFile;

    if (objFile(in) == (FILE *)NULL)
	errorPrint(BadRead, "%O is closed", in);

    gcBegin();
    intEnable();
    if ((res = parseObj(in, fp)) == (Obj)NULL) res = EOFSymb;
    objLink(res);
    intDisable();
    gcEnd();    /* garbage collect the read */

    --DATA(res, rc, basicInst);
    ASSERT(DATA(res, rc, basicInst) >= 0);
    return res;
}

DEFINE(primReadChar)
{
    FILE *infile;
    int c;

    if (argc > 0) {
	typeStrict(argv[0], File);
	if ((infile = objFile(argv[0])) == (FILE *)NULL)
	    errorPrint(BadRead, "%O is closed", argv[0]);
    }
    else infile = stdin;

    intEnable();
    c = getc(infile);
    intDisable();
    return c == EOF ? EOFSymb : newChar(gcNew, (char)c);
}

DEFINE(primWriteChar)
{
    FILE *outfile;
    Obj num = argv[0];
    int chr;

    if (argc > 1) {
	typeStrict(argv[1], File);
	if ((outfile = objFile(argv[1])) == (FILE *)NULL)
	    errorPrint(BadWrite, "%O is closed", argv[1]);
    }
    else outfile = stdout;

    typeCheck(num, Character);
    chr = objInteger(num);
    putc(chr, outfile);

    return num;
}

DEFINE(primUnreadChar)
{
    FILE *infile;
    Obj num = argv[0];
    int chr;

    if (argc > 1) {
	typeStrict(argv[1], File);
	if ((infile = objFile(argv[1])) == (FILE *)NULL)
	    errorPrint(BadRead, "%O is closed", argv[1]);
    }
    else infile = stdin;

    typeCheck(num, Character);
    chr = objInteger(num);
    ungetc(chr, infile);

    return TrueSymb;
}

DEFINE(primLoadNoisily)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], String);
    loadFile(objString(argv[0]), FALSE, TRUE, FALSE);
    return errorFlag == NoInput ? TrueSymb : FalseSymb;
}

DEFINE(primLoad)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], String);
    loadFile(objString(argv[0]), FALSE, FALSE, FALSE);
    return errorFlag == NoInput ? TrueSymb : FalseSymb;
}

DEFINE(primDisplay)
{
    FILE *port = stdout;

    if (argc == 2) {
	typeCheck(argv[1], File);
	if ((port = objFile(argv[1])) == (FILE *)NULL)
	    errorPrint(BadWrite, "%O is closed", argv[1]);
    }
    else if (argc > 2)
	errorPrint(BadClass, "display expects one or two arguments\n");

    objDisplay(argv[0], port);

    return TrueSymb;
}

DEFINE(primMDisplay)
{
    while (--argc >= 0)
	objDisplay(*(argv++), stdout);

    return TrueSymb;
}

DEFINE(primFDisplay)
{
    FILE *file;

    typeCheck(argv[0], File);
    if ((file = objFile(argv[0])) == (FILE *)NULL)
	errorPrint(BadWrite, "%O is closed", argv[0]);

    while (--argc > 0)
	objDisplay(*(++argv), file);

    return TrueSymb;
}

DEFINE(primPrintDepth)
{
    if (argc == 0)
	return ListDepth < 0 ? FalseSymb : newNumber(gcNew, (double)ListDepth);
    else if (argc > 1)
	errorPrint(BadArgs, "to print-depth (expects one or none)");
    if (argv[0] == FalseSymb)
	ListDepth = -1;
    else {
	typeCheck(argv[0], Number);
	ListDepth = (int)objNum(argv[0]);
    }
    return argv[0];
}
    
DEFINE(primPrintBreadth)
{
    if (argc == 0)
	return ListBreadth < 0 ?
	    FalseSymb : newNumber(gcNew, (double)ListBreadth);
    else if (argc > 1)
	errorPrint(BadArgs, "to print-breadth (expects one or none)");
    if (argv[0] == FalseSymb)
	ListBreadth = -1;
    else {
	typeCheck(argv[0], Number);
	ListBreadth = (int)objNum(argv[0]);
    }
    return argv[0];
}

DEFINE(primNumberFormat)
{
    if (argc == 0)
	return newString(gcNew, numberFormat);
    else if (argc > 1)
	errorPrint(BadArgs, "to print-number-format (expects one or none)");
    typeCheck(argv[0], String);
    (void)strcpy(numberFormat, objString(argv[0]));
    return argv[0];
}

DEFINE(primIntegerFormat)
{
    if (argc == 0)
	return newString(gcNew, integerFormat);
    else if (argc > 1)
	errorPrint(BadArgs, "to print-integer-format (expects one or none)");
    typeCheck(argv[0], String);
    (void)strcpy(integerFormat, objString(argv[0]));
    return argv[0];
}

DEFINE(primInPackage)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Symbol);
    CurrentPackage = objCdr(packageByName(argv[0]));
    return TrueSymb;
}

DEFINE(primPackageEnv)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Symbol);
    return objCdr(packageByName(argv[0]));
}

	/* utilities */

DEFINE(primEval)
{
    Obj expr;

    ASSERT(argc == 2);
    typeCheck(argv[1], Frame);
    gcBegin();
    expr = objEval(argv[0], argv[1]);
    objLink(expr);
    gcEnd(); /* garbage collect the eval */
    --DATA(expr, rc, basicInst);
    ASSERT(DATA(expr, rc, basicInst) >= 0);
    return expr;
}

DEFINE(primMacroExpand)
{
    ASSERT(argc == 1);
    return macroExpand(argv[0]);
}

/*ARGSUSED*/
DEFINE(primAbort)
{
    ASSERT(argc == 0);
    errorPrint(Abort, (char *)NULL);
    /*NOTREACHED*/
}

/*ARGSUSED*/
DEFINE(primExit)
{
    ASSERT(argc == 0);
    errorExit(0);
    /*NOTREACHED*/
}

DEFINE(primTraceEntry)
{
    register Obj proc = argv[0];

    ASSERT(argc == 1);
    typeCheck(proc, Proc);
    setCond(proc, TRACE_ENTRY);
    return proc;
}

DEFINE(primTraceExit)
{
    register Obj proc = argv[0];

    ASSERT(argc == 1);
    typeCheck(proc, Proc);
    setCond(proc, TRACE_EXIT);
    return proc;
}

DEFINE(primUnTraceEntry)
{
    register Obj proc = argv[0];

    ASSERT(argc == 1);
    typeCheck(proc, Proc);
    clearCond(proc, TRACE_ENTRY);
    return proc;
}

DEFINE(primUnTraceExit)
{
    register Obj proc = argv[0];

    ASSERT(argc == 1);
    typeCheck(proc, Proc);
    clearCond(proc, TRACE_EXIT);
    return proc;
}

	/* procedure functions */

DEFINE(primMacro)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], Symbol);
    if (checkCond(argv[0], MACRO))
	return objMacro(argv[0]);
    errorPrint(Other, "%O does not have a macro", argv[0]);
    /*NOTREACHED*/
}

DEFINE(primCodeVec)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], User);
    codevecPrint(objCode(argv[0]), 0);
    return TrueSymb;
}

DEFINE(primCodeBody)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], User);
#ifdef SAVE_LAMBDA_BODIES
    return newPair(gcNew, LambdaSymb,
		   DATA(objCode(argv[0]), expr, codevecInst));
#else
    return argv[0];
#endif /* SAVE_LAMBDA_BODIES */
}

DEFINE(primCodeFrame)
{
    typeCheck(argv[0], User);
    return DATA(argv[0], frame, userInst);
}

	/* system functions */

DEFINE(primChdir)
{
    ASSERT(argc == 1);
    typeCheck(argv[0], String);
    if (chdir(expandFilename(objString(argv[0]))) != 0)
	errorPrint(BadFile, "%s", objString(argv[0]));
    return argv[0];
}

	/* DEBUGGING ROUTINES */

#ifdef DEBUG

DEFINE(primDebugTrace)
{
    ASSERT(argc == 1);

    debugTrace = !(argv[0] == FalseSymb);

    return argv[0];
}

DEFINE(primDebugGC)
{
    ASSERT(argc == 1);

    debugGC = !(argv[0] == FalseSymb);
    return argv[0];
}

DEFINE(primDebugDestroy)
{
    ASSERT(argc == 1);

    debugDestroy = !(argv[0] == FalseSymb);
    return argv[0];
}

DEFINE(primDebugMacro)
{
    ASSERT(argc == 1);

    debugMacro = !(argv[0] == FalseSymb);
    return argv[0];
}

DEFINE(primDebugCode)
{
    ASSERT(argc == 1);

    debugCode = !(argv[0] == FalseSymb);
    return argv[0];
}

DEFINE(primDebugCont)
{
    ASSERT(argc == 1);

    debugCont = !(argv[0] == FalseSymb);
    return argv[0];
}

#endif /* defined(DEBUG) */

	/* misc */

DEFINE(primToStr)
{
    char *str;
    Obj new;

    ASSERT(argc == 2);
    str = objAsString(argv[0],
		      (argv[1] == FalseSymb ?
		       (F_VOID)objDisplay : (F_VOID)objPrint));
    if (str == (char *)NULL)
	errorPrint(Other, "->string:  object has a cycle");
    new = gcNew(String);
    DATA(new, string, stringInst) = str;
    DATA(new, maxsize, stringInst) = strlen(str);
    
    return new;
}

DEFINE(primCyclep)
{
    ASSERT(argc == 1);

    return objHasCycle(argv[0]) ? TrueSymb : FalseSymb;
}

struct prim_s {
    char *name;
    F_OBJ func;
    int options, args;
} primitives[] = {
    /* name		func			options	args */

    { "->string",	primToStr,		0,	2 },
    { "cycle?",		primCyclep,		0,	1 },

    { "cons",		primCons,		0,	2 },
    { "list",		primList,		OPTARG,	0 },
    { "list*",		primListStar,		OPTARG,	1 },
    { "append",		primAppend,		0,	2 },
    { "car",		primCar,		0,	1 },
    { "cdr",		primCdr,		0,	1 },
    { "set-car!",	primSetCar,		0,	2 },
    { "set-cdr!",	primSetCdr,		0,	2 },

    { "make-string",	primMakeStr,		OPTARG,	1 },
    { "string-append",	primStrCat,		OPTARG,	2 },
    { "string-length",	primStrLen,		0,	1 },
    { "string-ref",	primStrRef,		0,	2 },
    { "string-set!",	primStrSet,		0,	3 },
    { "string-copy",	primStrCpy,		0,	1 },
    { "string>?",	primStrGT,		0,	2 },
    { "string>=?",	primStrGE,		0,	2 },
    { "string<?",	primStrLT,		0,	2 },
    { "string<=?",	primStrLE,		0,	2 },
    { "substring",	primSubStr,		0,	3 },

    { "make-vector",	primMakeVec,		OPTARG,	1 },
    { "vector",		primVector,		OPTARG,	0 },
    { "vector-ref",	primVecRef,		0,	2 },
    { "vector-set!",	primVecSet,		0,	3 },
    { "vector-length",	primVecSize,		0,	1 },

    { "box",		primMakeBox,		0,	1 },
    { "unbox",		primUnbox,		0,	1 },
    { "set-box!",	primSetBox,		0,	2 },

    { "file-open",	primOpenFile,		0,	2 },
    { "file-close",	primCloseFile,		0,	1 },
    { "file-flush",	primFlushFile,		0,	1 },
    { "file-access",	primAccess,		0,	2 },
    { "file-seek",	primFileSeek,		0,	3 },
    { "file-tell",	primFileTell,		0,	1 },

    { "vector->list",	primVec2List,		0,	1 },
    { "list->vector",	primList2Vec,		0,	1 },
    { "environment->list", primEnv2List,	0,	1 },
    { "string->integer", primStr2Int,		0,	1 },
    { "integer->string", primInt2Str,		OPTARG,	1 },
    { "symbol->string",	primStrCpy,		0,	1 },
    { "string->symbol",	primStr2Sym,		0,	1 },
    { "integer->number", primInt2Num,		0,	1 },
    { "number->integer", primNum2Int,		0,	1 },
    { "char->integer",	primChar2Int,		0,	1 },
    { "integer->char",	primInt2Char,		0,	1 },
    { "string->uninterned-symbol", primStr2Uninterned, 0, 1 },
    { "file->string",	primFileString,		0,	1 },
    { "string->list",	primStr2List,		0,	1 },
    { "list->string",	primList2Str,		0,	1 },

    { "object-type",	primObjType,		0,	1 },
    { "null?",		primNullp,		0,	1 },
    { "boolean?",	primBooleanp,		0,	1 },
    { "pair?",		primPairp,		0,	1 },
    { "string?",	primStringp,		0,	1 },
    { "symbol?",	primSymbolp,		0,	1 },
    { "vector?",	primVectorp,		0,	1 },
    { "number?",	primNumberp,		0,	1 },
    { "integer?",	primIntegerp,		0,	1 },
    { "char?",		primCharp,		0,	1 },
    { "procedure?",	primProcedurep,		0,	1 },
    { "box?",		primBoxp,		0,	1 },
    { "eof-object?",	primEOFp,		0,	1 },

    { "eq?",		primEqp,		0,	2 },
    { "eqv?",		primEqvp,		0,	2 },
    { "not",		primNot,		0,	1 },

    { "write",		primWrite,		OPTARG,	1 },
    { "write-char",	primWriteChar,		OPTARG,	1 },
    { "read",		primRead,		OPTARG,	0 },
    { "read-char",	primReadChar,		OPTARG,	0 },
    { "unread-char",	primUnreadChar,		OPTARG,	1 },
    { "display",	primDisplay,		OPTARG,	1 },
    { "mdisplay",	primMDisplay,		OPTARG,	1 },
    { "fdisplay",	primFDisplay,		OPTARG,	2 },
    { "load-noisily",	primLoadNoisily,	0,	1 },
    { "load",		primLoad,		0,	1 },

    { "print-depth",	primPrintDepth,		OPTARG,	0 },
    { "print-breadth",	primPrintBreadth,	OPTARG,	0 },
    { "print-number-format", primNumberFormat,	OPTARG,	0 },
    { "print-integer-format", primIntegerFormat, OPTARG, 0 },

    { "in-package",	primInPackage,		0,	1 },
    { "package-environment", primPackageEnv,	0,	1 },

    { "eval",		primEval,		0,	2 },
    { "abort",		primAbort,		0,	0 },
    { "exit",		primExit,		0,	0 },
    { "macro-expand",	primMacroExpand,	0,	1 },
    { "trace-entry",	primTraceEntry,		0,	1 },
    { "trace-exit",	primTraceExit,		0,	1 },
    { "untrace-entry",	primUnTraceEntry,	0,	1 },
    { "untrace-exit",	primUnTraceExit,	0,	1 },

    { "macro",		primMacro,		0,	1 },
    { "code-vector",	primCodeVec,		0,	1 },
    { "code-body",	primCodeBody,		0,	1 },
    { "code-frame",	primCodeFrame,		0,	1 },

    { "chdir",		primChdir,		0,	1 },

#ifdef DEBUG
    { "debug-trace",	primDebugTrace,		0,	1 },
    { "debug-macro",	primDebugMacro,		0,	1 },
    { "debug-gc",	primDebugGC,		0,	1 },
    { "debug-destroy",	primDebugDestroy,	0,	1 },
    { "debug-code",	primDebugCode,		0,	1 },
    { "debug-cont",	primDebugCont,		0,	1 },
#endif
};

void primInit()
{
    int i;

    for (i = 0; i < sizeof (primitives) / sizeof (struct prim_s); i++)
	(void)newPrim(gcNew, primitives[i].name, primitives[i].func, GlobalEnv,
		      primitives[i].options, primitives[i].args);
}
