/* $Header: set.c,v 3.0 88/04/13 16:17:06 jos Exp $ */
/*
 *  This file is part of the Amsterdam SGML Parser.
 *
 *  Copyright: Faculteit Wiskunde en Informatica
 *             Department of Mathematics and Computer Science
 *             Vrije Universiteit Amsterdam
 *             The Netherlands
 *
 *  Authors:   Sylvia van Egmond
 *             Jos Warmer
 */
#include "types.h"
#include "set.h"

#define nr_ints(set) ( ( (set->size-1) div (8*sizeof(int)))+1 )

static int Total = 0;
static int Total_sets = 0;

typedef struct set_struct {
    int         size;
    int        *elems;
} Set;

typedef struct setiter_struct {
    P_Set       set;
    int         current;
} Set_iterator;

/*  Creates a new set of length 'size'.
 *  Members of this set are:  0 .. size-1
 */
P_Set   new_set(size)
int size;
{
    int     l;
    int     i;
    P_Set   set;

    set        = (P_Set)MALLOC( (unsigned)sizeof(Set));
    Total_sets += 1;
    set->size  = size;
    l          = nr_ints(set);
    set->elems = (int*)CALLOC(l, (unsigned)sizeof(int));
    Total += (l * sizeof(int));
    for( i=0; i<l; i++){
        set->elems[i] = 0;
    }
    return set;
}

void set_clear(set)
P_Set   set;
{
    int     l;
    int     i;
    
    l = nr_ints(set);
    for( i=0; i<l; i++){
        set->elems[i] = 0;
    }
}

void set_assign(dest, src)
P_Set   dest;
P_Set   src;
{
    assert( dest->size >= src->size );
    set_clear(dest);
    set_union(dest, src);
}

P_Set set_copy(set)
P_Set   set;
{
    P_Set s;

    s = new_set(set->size);
    set_union(s, set);     /* copies all members of set to s */
    return s;
}

void    delete_set(set)
P_Set   set;
{
    if( set != 0 ){
	FREE(set->elems, nr_ints(set));
	FREE(set, sizeof(Set));
    }
}

void    set_add(set, elem)
P_Set    set;
int      elem;
{
    int  offset;
    int  index;
    assert( (elem >= 0) and (elem < set->size));

    index  = elem div (8*sizeof(int));
    offset = elem mod (8*sizeof(int));

    set->elems[index] |= (1<<offset);
}

void    set_delete(set, elem)
P_Set    set;
int      elem;
{
    int  offset;
    int  index;
    assert( (elem >= 0) and (elem < set->size));

    index  = elem div (8*sizeof(int));
    offset = elem mod (8*sizeof(int));

    set->elems[index] &= (~(1<<offset));
}

void    set_union(set1, set2)
P_Set  set1;
P_Set  set2;
{
    int i;
    assert( set1->size >= set2->size );

    for(i=0; i< nr_ints(set2); i++){
        set1->elems[i] |= set2->elems[i];
    }
}

void    set_intersect(set1, set2)
P_Set  set1;
P_Set  set2;
{
    int i;
    assert( set1->size >= set2->size );

    for(i=0; i<nr_ints(set2); i++){
        set1->elems[i] &= set2->elems[i];
    }
}


Bool    set_member(set, elem)
P_Set    set;
int      elem;
{
    int  offset;
    int  index;
    assert( (elem >= 0) and (elem < set->size));

    index  = elem div (8*sizeof(int));
    offset = elem mod (8*sizeof(int));

    return ( ((1<<offset) & set->elems[index])?(TRUE):(FALSE) );
}

Bool set_empty(set)
P_Set   set;
{
    int i;

    for(i=0; i<nr_ints(set); i++){
        if( set->elems[i] != 0 ){ return FALSE; }
    }
    return TRUE;
}

P_Set_iterator  set_iterator(set)
P_Set   set;
{
    P_Set_iterator it;

    it = (P_Set_iterator)MALLOC( sizeof(struct setiter_struct));
    it->set     = set;
    it->current = set->size;
    return it;
} 

int set_next(it)
P_Set_iterator it;
{
    it->current--;
    while( it->current and (not set_member(it->set, it->current)) ){
	it->current--;
    }
    if( it->current ){
	return it->current;
    } else {
	FREE(it, sizeof(Set_iterator));
	return 0;
    }
}

void  set_generator(set, func)
P_Set            set;
VoidFunctionInt  func;
{
    int   i;
    
    if( set == 0 ){ return; }

    for(i=0; i<set->size; i++){
        if( set_member(set, i) ){
            (*func)(i);
        }
    }
}

void    set_minus(set1, set2)
P_Set  set1;
P_Set  set2;
{
    int i;
    assert( set1->size == set2->size );

    for(i=0; i<set1->size; i++){
	if( set_member(set2, i) ){
	    set_delete(set1, i);
	}
    }
}
