/* 
 * mxSelect.c --
 *
 *	This file contains routines used by Mx and Tx for manipulating
 *	the selection.
 *
 * Copyright (C) 1986, 1987, 1988 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /project/tcl/src/mxedit/RCS/mxSelect.c,v 2.0 1993/05/13 00:35:46 welch Exp $ SPRITE (Berkeley)";
#endif not lint


#include <X11/Xlib.h>
#include <stdio.h>
#include <strings.h>
#include <X11/Xatom.h>

#include "mxWidget.h"

/*
 * Information about the selection.  Since there's only one selection
 * at a time, this can be global rather than window-specific.
 */

MxFileInfo *MxSelectedFile = NULL;	/* File containing selection.  NULL
					 * means no selection here. */
MxHighlight *MxSelection;		/* Describes selection. */

/*
 * Forward references to procedures defined later in this file:
 */

static void		MxSelChange();
static int		MxSelFetch();

/*
 *----------------------------------------------------------------------
 *
 * MxSelectionSet --
 *
 *	Make the selection be a given range in a given window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection is changed to correspond to the range between
 *	first and last (inclusive) in mxwPtr.
 *
 *----------------------------------------------------------------------
 */

void
MxSelectionSet(mxwPtr, first, last)
    MxWidget *mxwPtr;		/* Window in which to make selection. */
    Mx_Position first, last;	/* Range for new selection. */
{
    if (mxwPtr->fileInfoPtr != MxSelectedFile) {
	Tk_CreateSelHandler(mxwPtr->tkwin, XA_STRING, MxSelFetch,
		    (ClientData)mxwPtr, XA_STRING);
	Tk_OwnSelection(mxwPtr->tkwin, MxSelectionClear, (ClientData)mxwPtr);
	MxSelectedFile = mxwPtr->fileInfoPtr;
	MxSelection = MxHighlightCreate(mxwPtr, first, last, MX_REVERSE);
    } else {
	MxHighlightSetRange(MxSelection, first, last);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MxSelectionClear --
 *
 *	If the selection belongs to mxwPtr, then clear it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection may get cleared.
 *
 *----------------------------------------------------------------------
 */

void
MxSelectionClear(mxwPtr)
    MxWidget *mxwPtr;		/* Window in which to clear selection.  If
				 * NULL, then clear selection regardless
				 * of which window it's in (assuming that
				 * we've got it, of course). */
{
    if (MxSelection == NULL) {
	return;
    }
    if ((mxwPtr != NULL) && (mxwPtr->fileInfoPtr != MxSelectedFile)) {
	return;
    }
    MxHighlightDelete(MxSelection);
    MxSelection = NULL;
    MxSelectedFile = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * MxSelFetch --
 *
 *	Called by the Tk selection package when someone wants to know
 *	what's selected.
 *
 * Results:
 *	See the documentation for Tk_CreateSelHandler.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

    /* ARGSUSED */
static int
MxSelFetch(clientData, firstByte, selectionPtr, numBytes)
    ClientData clientData;		/* mxwPtr */
    int firstByte;			/* Index of first desired byte. */
    char *selectionPtr;			/* Store bytes of selection here. */
    int numBytes;			/* Max no. of bytes to return. */
{
    Mx_Position first, last;
    Mx_File file;
    char *line;
    int count, bytesThisLine, lineLength;

    if (MxSelectedFile == NULL) {
	return -1;
    }
    file = MxSelectedFile->file;
    MxHighlightGetRange(MxSelection, &first, &last);
    if (MX_POS_LESS(last, first)) {
	return -1;
    }
    for (count = 0, first = Mx_Offset(file, first, firstByte);
	 (count < numBytes) && MX_POS_LEQ(first, last); ) {
	     line = Mx_GetLine(file, first.lineIndex, &lineLength);
	     if (first.lineIndex == last.lineIndex) {
		 bytesThisLine = last.charIndex + 1 - first.charIndex;
	     } else {
		 bytesThisLine = lineLength - first.charIndex;
	     }
	     if (count + bytesThisLine > numBytes) {
		 bytesThisLine = numBytes - count;
	     }
	     strncpy(selectionPtr, &line[first.charIndex], bytesThisLine);
	     selectionPtr += bytesThisLine;
	     count += bytesThisLine;
	     first.lineIndex++;
	     first.charIndex = 0;
    }
    return count;
}

/*
 *----------------------------------------------------------------------
 *
 * MxSelChange --
 *
 *	This procedure is called by the Sx selection module whenever
 *	the selection changes out from underneath us.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection highlight is deleted and we remember that we
 *	no longer have the selection.
 *
 *----------------------------------------------------------------------
 */

static void
MxSelChange()
{
    MxSelectedFile = NULL;
    if (MxSelection != NULL) {
	MxHighlightDelete(MxSelection);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MxGetSelRange --
 *
 *	If the selection is in this window's file, return its range.
 *	Otherwise return an error.
 *
 * Results:
 *	TCL_OK is returned if the selection is in mxwPtr's file.  In
 *	addition, *firstPtr and *lastPtr get filled in with the selection's
 *	range.  Otherwise, TCL_ERROR is returned and a pointer to an
 *	error message is left in mxwPtr->interp->result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
MxGetSelRange(mxwPtr, firstPtr, lastPtr)
    register MxWidget *mxwPtr;		/* Window of interest. */
    Mx_Position *firstPtr;		/* To be filled in with location of
					 * first selected character. */
    Mx_Position *lastPtr;		/* To be filled in with location of
					 * last selected character. */
{
    if ((MxSelectedFile != mxwPtr->fileInfoPtr)) {
	goto noSelection;
    }
    MxHighlightGetRange(MxSelection, firstPtr, lastPtr);
    if (MX_POS_LESS(*lastPtr, *firstPtr)) {
	goto noSelection;
    }
    return TCL_OK;

    noSelection:
    mxwPtr->interp->result = "nothing is selected in this file";
    return TCL_ERROR;
}
