/*
 * tkIsing.c --  Huffi 1994
 *
 *	This module implements "ising" widgets.  A "ising" is
 *	a widget that displays a field in which ising model
 *	data are displayed.
 *
 */


#define HAVE_UNISTD_H
#include <tk/tkConfig.h>
#include <tk.h>
#include "mcmodel.h"
#include "IsingColor.h"

/*
 * global variables
 * 
 * 
 */
static int sWidth, sHeight;


/*
 * A data structure of the following type is kept for each ising
 * widget managed by this file:
 */

typedef struct {
    struct modeltype *pmodelstruct; /* Pointer to a struct which holds data
				 * needed for InitModel and ModelData */
    Tk_Window tkwin;		/* Window that embodies the ising.  NULL
				 * means window has been deleted but
				 * widget record hasn't been cleaned up yet. */
    Display *display;		/* X's token for the window's display. */
    Tcl_Interp *interp;		/* Interpreter associated with widget. */
    GC gc;			/* Graphics context for copying from
				 * off-screen pixmap onto screen. */
    int updatePending;		/* Non-zero means a call to IsingDisplay
				 * has already been scheduled. */
    Pixmap pm;			/* Screen buffer */ 

    /* 
     * model specific variables:
     */
    int xSize, ySize;		/* Width and height of ising in pixels. */
    int xRes, yRes;		/* real model resolution */
    int xZoomRes;		/* x model resolution in current view */
    int yZoomRes;		/* y model resolution in current view */
    int xOffset, yOffset;	/* offset in model data */
    int xScroll, yScroll;	/* values used when scrolling; 0 means
    				 * no scrolling */
    int sweeps;			/* Frame display interval */
    double beta1, beta2, beta3,
		beta4, beta5;	/* correlation parameters */
    int subLattices;		/* number of sublatics */

} Ising;

/*
 * Information used for argv parsing.
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_INT, "-xsize", "xsize", "XSize",
        "512", Tk_Offset(Ising, xSize), 0 },

    {TK_CONFIG_INT, "-ysize", "ysize", "YSize",
        "512", Tk_Offset(Ising, ySize), 0 },

    {TK_CONFIG_INT, "-xres", "xres", "XRes",
        "32", Tk_Offset(Ising, xRes), 0 },

    {TK_CONFIG_INT, "-yres", "yres", "YRes",
        "32", Tk_Offset(Ising, yRes), 0 },

    {TK_CONFIG_INT, "-xzoomres", "xzoomres", "XZoomRes",
        "32", Tk_Offset(Ising, xZoomRes), 0 },

    {TK_CONFIG_INT, "-yzoomres", "yzoomres", "YZoomRes",
        "32", Tk_Offset(Ising, yZoomRes), 0 },

    {TK_CONFIG_INT, "-xoffset", "xoffset", "XOffset",
        "0", Tk_Offset(Ising, xOffset), 0 },

    {TK_CONFIG_INT, "-yoffset", "yoffset", "YOffset",
        "0", Tk_Offset(Ising, yOffset), 0 },

    {TK_CONFIG_INT, "-xscroll", "xscroll", "XScroll",
        "0", Tk_Offset(Ising, xScroll), 0 },

    {TK_CONFIG_INT, "-yscroll", "yscroll", "YScroll",
        "0", Tk_Offset(Ising, yScroll), 0 },

    {TK_CONFIG_DOUBLE, "-beta1", "beta1", "Beta1",
        "1.33333", Tk_Offset(Ising, beta1), 0 },

    {TK_CONFIG_DOUBLE, "-beta2", "beta2", "Beta2",
        "0.0", Tk_Offset(Ising, beta2), 0 },

    {TK_CONFIG_DOUBLE, "-beta3", "beta3", "Beta3",
        "0.0", Tk_Offset(Ising, beta3), 0 },

	{TK_CONFIG_DOUBLE, "-beta4", "beta4", "Beta4",
		"0.0", Tk_Offset(Ising, beta4), 0 },

	{TK_CONFIG_DOUBLE, "-beta5", "beta5", "Beta5",
		"0.0", Tk_Offset(Ising, beta5), 0 },

    {TK_CONFIG_INT, "-sweeps", "sweeps", "Sweeps",
        "1", Tk_Offset(Ising, sweeps), 0 },
    {TK_CONFIG_SYNONYM, "-sw", "sweeps", (char *) NULL,
	(char *) NULL, 0, 0},

    {TK_CONFIG_INT, "-sublatices", "sublatices", "SubLatices",
        "2", Tk_Offset(Ising, subLattices), 0 },
    {TK_CONFIG_SYNONYM, "-lat", "sublatices", (char *) NULL,
	(char *) NULL, 0, 0},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		IsingConfigure _ANSI_ARGS_((Tcl_Interp *interp,
			    Ising *isingPtr, int argc, char **argv,
			    int flags));
static int		IsingInitModel _ANSI_ARGS_((Tcl_Interp *interp,
			    Ising *isingPtr, int argc, char **argv,
			    int flags));
static void		IsingDestroy _ANSI_ARGS_((ClientData clientData));
static void		IsingFillPixmap _ANSI_ARGS_((ClientData clientData));
static void		IsingDisplay _ANSI_ARGS_((ClientData clientData));
static void		IsingEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static int		IsingWidgetCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *, int argc, char **argv));
static int		IsingPrintPSFile _ANSI_ARGS_((ClientData clientData, 
			    Tcl_Interp*, int color, char *argv));
static int		IsingPrintPSDevice _ANSI_ARGS_((ClientData clientData, 
			    Tcl_Interp*, int color, char *argv));

/*
 *--------------------------------------------------------------
 *
 * IsingCmd --
 *
 *	This procedure is invoked to process the "ising" Tcl
 *	command.  It creates a new "ising" widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */

int
IsingCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window main = (Tk_Window) clientData;
    Ising *isingPtr;
    Tk_Window tkwin;
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL) {
	return TCL_ERROR;
    }
    Tk_SetClass(tkwin, "Ising");

    /*
     * Allocate and initialize the widget record.
     */

    isingPtr = (Ising *) ckalloc(sizeof(Ising));
    isingPtr->pmodelstruct = (struct modeltype *) ckalloc(sizeof(struct modeltype));
    isingPtr->pmodelstruct->prandstruct = NULL;
    isingPtr->pmodelstruct->n1 = 0;
    isingPtr->pmodelstruct->n2 = 0;
    isingPtr->pmodelstruct->isp = NULL;
    isingPtr->pmodelstruct->nxtx = NULL;
    isingPtr->pmodelstruct->nxty = NULL;
    isingPtr->pmodelstruct->nprx = NULL;
    isingPtr->pmodelstruct->npry = NULL;

    isingPtr->tkwin = tkwin;
    isingPtr->display = Tk_Display(tkwin);
    isingPtr->interp = interp;
    isingPtr->gc = None;
    isingPtr->updatePending = 0;
    isingPtr->pm = None;

    isingPtr->xSize = 0;
    isingPtr->ySize = 0;
    isingPtr->xRes = 0;
    isingPtr->yRes = 0;
    isingPtr->xZoomRes = 0;
    isingPtr->yZoomRes = 0;
    isingPtr->xOffset = 0;
    isingPtr->yOffset = 0;
    isingPtr->xScroll = 0;
    isingPtr->yScroll = 0;
    isingPtr->beta1 = 0.0;
    isingPtr->beta2 = 0.0;
    isingPtr->beta3 = 0.0;
	isingPtr->beta4 = 0.0;
	isingPtr->beta5 = 0.0; 
 	isingPtr->sweeps = 0;
    isingPtr->subLattices = 0;

    Tk_CreateEventHandler(isingPtr->tkwin, ExposureMask|StructureNotifyMask,
	    IsingEventProc, (ClientData) isingPtr);
    Tcl_CreateCommand(interp, Tk_PathName(isingPtr->tkwin), IsingWidgetCmd,
	    (ClientData) isingPtr, (void (*)()) NULL);
    if (IsingConfigure(interp, isingPtr, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(isingPtr->tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(isingPtr->tkwin);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * IsingWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
IsingWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Information about ising widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Ising *isingPtr = (Ising *) clientData;
    int result = TCL_OK;
    int length;
    char c;
    int color = 0;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tk_Preserve((ClientData) isingPtr);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, isingPtr->tkwin, configSpecs,
		    (char *) isingPtr, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, isingPtr->tkwin, configSpecs,
		    (char *) isingPtr, argv[2], 0);
	} else {
	    result = IsingConfigure(interp, isingPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    } else if ((c=='r') && (strncmp( argv[1], "run", length) == 0)) {
        if (!isingPtr->updatePending) {  
	   IsingFillPixmap( (ClientData) isingPtr );
	} 
    } else if ((c=='i') && (strncmp( argv[1], "init", length) == 0)) {
	if (argc >= 4) {
	    result = IsingInitModel( interp, isingPtr, argc-2, argv+2,
	   	    TK_CONFIG_ARGV_ONLY);
	}
    } else if ((c=='p') && (strncmp( argv[1],  "print",  length) == 0)){
	if (argc == 6){
	    c = argv[2][1];
	    length = strlen(argv[2]);
	    if(strncmp(argv[2], "-color", length)==0){
		color = atoi(argv[3]); /* I should really do some error checking */
	    }
	if (strncmp(argv[4], "-file", length)==0){
	    result = IsingPrintPSFile((ClientData) isingPtr, interp,
			color, argv[5]);
	    if(result != 0){
		Tcl_AppendResult(interp, "Couldn't open file", (char*) NULL);
		return TCL_ERROR;
	    }
	}
	if(strncmp(argv[4], "-device", length)==0){
	    result = IsingPrintPSDevice((ClientData) isingPtr, interp, color,  argv[5]);
	    if(result != 0){
		Tcl_AppendResult(interp, "Couldn't print to device", (char*) NULL);
		return TCL_ERROR;
	    }
	}
	}
	else{
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " -color int option string", (char *) NULL);
		return TCL_ERROR;
	}
    }

    if (!(isingPtr->updatePending)) {
 	Tk_DoWhenIdle(IsingDisplay, (ClientData) isingPtr);
 	isingPtr->updatePending = 1; 
    }
    Tk_Release((ClientData) isingPtr);
    return result;

}

/*
 *----------------------------------------------------------------------
 *
 * IsingConfigure --
 *
 *	This procedure is called to process an argv/argc list in
 *	conjunction with the Tk option database to configure (or
 *	reconfigure) a ising widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for isingPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
IsingConfigure(interp, isingPtr, argc, argv, flags)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Ising *isingPtr;			/* Information about widget. */
    int argc;				/* Number of valid entries in argv. */
    char **argv;			/* Arguments. */
    int flags;				/* Flags to pass to
					 * Tk_ConfigureWidget. */
{
    if (Tk_ConfigureWidget(interp, isingPtr->tkwin, configSpecs,
	    argc, argv, (char *) isingPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Set the background for the window and create a graphics context
     * for use during redisplay.
     */

   if (isingPtr->gc == None) {
	XGCValues gcValues;
	gcValues.function = GXcopy;
	gcValues.graphics_exposures = False;
	isingPtr->gc = Tk_GetGC(isingPtr->tkwin,
		GCFunction|GCGraphicsExposures, &gcValues);
    }


    /*
     * Calculate spin dimensions and widgetsize
     */
/*
 * moet hier de maximale schermsize opgevraagd worden? Uit TCl of XLib?
 */
    sWidth  = isingPtr->xSize / isingPtr->xZoomRes;
    sHeight = isingPtr->ySize / isingPtr->yZoomRes;
    isingPtr->xSize = sWidth  * isingPtr->xZoomRes;
    isingPtr->ySize = sHeight * isingPtr->yZoomRes;

    /*
     * Register the desired geometry for the window.  Then arrange for
     * the window to be redisplayed.
     */
    Tk_GeometryRequest(isingPtr->tkwin,isingPtr->xSize,isingPtr->ySize);
 
    if (!isingPtr->updatePending) {
	Tk_DoWhenIdle(IsingDisplay, (ClientData) isingPtr);
	isingPtr->updatePending = 1;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * IsingInitModel  --
 *
 *----------------------------------------------------------------------
 */

static int
IsingInitModel(interp, isingPtr, argc, argv, flags)
    Tcl_Interp *interp;			/* Used for error reporting. */
    Ising *isingPtr;			/* Information about widget. */
    int argc;				/* Number of valid entries in argv. */
    char **argv;			/* Arguments. */
    int flags;				/* Flags to pass to
					 * Tk_ConfigureWidget. */
{
    if (Tk_ConfigureWidget(interp, isingPtr->tkwin, configSpecs,
	    argc, argv, (char *) isingPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }


    /*
     * Initialize model dimensions
     */
    InitModel( isingPtr->pmodelstruct, isingPtr->xRes, isingPtr->yRes,
	        isingPtr->beta1, isingPtr->beta2, isingPtr->beta3,
			isingPtr->beta4, isingPtr->beta5 ); 
    
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * IsingEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on isings.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
IsingEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Ising *isingPtr = (Ising *) clientData;

    if (eventPtr->type == Expose) {
	if (!isingPtr->updatePending) {
	    Tk_DoWhenIdle(IsingDisplay, (ClientData) isingPtr);
	    isingPtr->updatePending = 1;
	}
    } else if (eventPtr->type == ConfigureNotify) {
	if (!isingPtr->updatePending) {
	    Tk_DoWhenIdle(IsingDisplay, (ClientData) isingPtr);
	    isingPtr->updatePending = 1;
	}
    } else if (eventPtr->type == DestroyNotify) {
	Tcl_DeleteCommand(isingPtr->interp, Tk_PathName(isingPtr->tkwin));
	isingPtr->tkwin = NULL;
	if (isingPtr->updatePending) {
	    Tk_CancelIdleCall(IsingDisplay, (ClientData) isingPtr);
	}
	Tk_EventuallyFree((ClientData) isingPtr, IsingDestroy);
    }
}

/*
 *--------------------------------------------------------------
 *
 * IsingFillPixmap --
 *
 *	This procedure calculates which colors are to be used
 *	in the ising window and draws a new frame into the
 *	widget's pixmap.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget's pixmap is updated.
 *
 *--------------------------------------------------------------
 */

static void
IsingFillPixmap( ClientData clientData )
{
    Ising *isingPtr = (Ising *) clientData;
    Tk_Window tkwin = isingPtr->tkwin;
    Pixmap pm = isingPtr->pm;
    Drawable d;
    int x, y; 
    int * data;
    unsigned long value_mask; 
    XGCValues values;
    XGCValues * values_return;
    Colormap cmap;
    XColor color_ret, dummy;
    int Spin; 
    static unsigned long *color = NULL;
    int xmax = isingPtr->xOffset + isingPtr->xZoomRes;
    int ymax = isingPtr->yOffset + isingPtr->yZoomRes;
    int i;
    int xScr,  yScr;	/* used for scrolling */
    
    XRectangle xrect;
    int r, g, b;
    double hue;

    if(color!=NULL)
	free(color);
    color = (unsigned long*) malloc(sizeof(unsigned long)
	* (isingPtr->subLattices+1));
    

    if (!(data = ModelData( isingPtr->pmodelstruct,
				 isingPtr->sweeps, isingPtr->beta1, 
			    isingPtr->beta2, isingPtr->beta3, isingPtr->beta4,
				isingPtr->beta5 )))
       exit(1);

 
    /*
     * Create a pixmap for buffering.
     */
 
    if (isingPtr->pm != None) {
	XFreePixmap( Tk_Display(tkwin), isingPtr->pm );
	isingPtr->pm = None; 
    }
    pm = XCreatePixmap( Tk_Display(tkwin), Tk_WindowId(tkwin),
			Tk_Width(tkwin), Tk_Height(tkwin),
    DefaultDepthOfScreen(Tk_Screen(tkwin)));
    d = pm;  /* Draw into the pixmap */

    /*
     * Draw the ising into the pixmap.
     */
    cmap = DefaultColormap( Tk_Display( tkwin ), Tk_ScreenNumber( tkwin ));
    value_mask = GCForeground;

/*    XAllocNamedColor( Tk_Display( tkwin ), cmap, COLOR1, &color_ret, &dummy);
    color[1] = color_ret.pixel;
    XAllocNamedColor( Tk_Display( tkwin ), cmap, COLOR2, &color_ret, &dummy);
    color[2] = color_ret.pixel;
    XAllocNamedColor( Tk_Display( tkwin ), cmap, COLOR3, &color_ret, &dummy);
    color[3] = color_ret.pixel;
    XAllocNamedColor( Tk_Display( tkwin ), cmap, OTHER, &color_ret, &dummy);
    color[0] = color_ret.pixel;
*/

    for( i = 1; i <= isingPtr->subLattices; i++ )
    {
	hue = (360.0 / isingPtr->subLattices)*i;
	hsv2rgb( hue, 1.0, 1.0, &r, &g, &b );
	color_ret.red = r;
	color_ret.green = g;
	color_ret.blue = b;
	XAllocColor( Tk_Display( tkwin ), cmap, &color_ret );
	*(color+i) = color_ret.pixel;
    }

    hsv2rgb( 0.0, 0.0, 0.0, &r, &g, &b );
    color_ret.red = r;
    color_ret.green = g;
    color_ret.blue = b;
    XAllocColor( Tk_Display( tkwin ), cmap, &color_ret );
    *(color) = color_ret.pixel;
    
    XSetForeground( Tk_Display(tkwin), isingPtr->gc, color[0] );
    XFillRectangle( Tk_Display(tkwin), d, isingPtr->gc,
				0,
			        0,
				isingPtr->xSize, isingPtr->ySize );  

    for ( i=1; i<=isingPtr->subLattices; i++)
    {
	XSetForeground( Tk_Display(tkwin), isingPtr->gc,
	    color[i] );
	for ( y=isingPtr->yOffset; y<ymax; y++ )
	{
	    for ( x=isingPtr->xOffset; x<xmax; x++ )
	    {
		xScr = (x + isingPtr->xScroll) % isingPtr->xRes;
		if(xScr<0) xScr=isingPtr->xRes+xScr;
		yScr = (y + isingPtr->yScroll) % isingPtr->yRes;
		if(yScr<0) yScr=isingPtr->yRes+yScr;
		Spin = *(data + xScr + yScr*isingPtr->xRes); 

		if ( Spin == 1 )
		{
		    if ( IsingColor(xScr, yScr, isingPtr->subLattices) == i)
		    {
			XFillRectangle( Tk_Display(tkwin), d, isingPtr->gc, 
			    (x - isingPtr->xOffset) * sWidth, 
			    (y - isingPtr->yOffset) * sHeight,
			    sWidth, sHeight);
		    }
		}
	    }
	}
    }
    isingPtr->pm = pm;
    
}
  

/*
 *--------------------------------------------------------------
 *
 * IsingPrintFilePS --
 *
 *	This procedure writes a PostScript-file to disk, name
 *	filename
 *
 * Results:
 *	PS-file with graphical spindata on disk.
 *
 *--------------------------------------------------------------
 */
  
static int
IsingPrintPSFile(ClientData clientData, Tcl_Interp* interp,
	 int color, char* filename)
{
    int i;
    int x, y;
    int boxsizex, boxsizey;
    int picsize;
    int tx, ty;
    int papersizex = 555;  /* +/- A4 - 7 mm marge*/
    int papersizey = 740;  /* in 1/72 inch */
    int paperxoffset = 20;
    int paperyoffset = 20;
    int c;
    Ising *isingPtr = (Ising*) clientData;
    int *data;  /* pointer to spindata */
    int spin;
    FILE *fpr;
    FILE *fpw;
    double colorscale;
    int result;  
 
    if(!(data = GetSpinData(isingPtr->pmodelstruct)))
	exit(1);  /* something is wrong, quit! */

    fpr = fopen( "isingheader.ps",  "r" );
    if(fpr==NULL)
	{
		Tcl_AppendResult(interp, "error opening 'isingheader.ps'",
 			(char*) NULL);
		return TCL_ERROR;
	}
	fpw = fopen( filename, "w" );
    if(fpw==NULL)
	{
		Tcl_AppendResult(interp, "error opening'", filename,
		"'", (char*) NULL);
		return TCL_ERROR;
	}
    while(( c = getc( fpr )) != EOF )
	{
		if( putc( c, fpw ) == 0 )  /* write header to filename */
		{
			Tcl_AppendResult(interp, "error writing", filename, (char*) NULL);
			return TCL_ERROR;
		}
	}	
    
    boxsizex = (papersizex / isingPtr->xZoomRes);
    boxsizey = (papersizey / isingPtr->yZoomRes);
 
    tx = (papersizex - boxsizex* isingPtr->xZoomRes) / 2 + paperxoffset;
    ty = (papersizey - boxsizey * isingPtr->yZoomRes) / 2 + paperyoffset;
    fprintf( fpw, "%d %d translate\n", tx, ty);
    fprintf( fpw, "%d %d scale\n", boxsizex, boxsizey );

	colorscale = 1.0 / isingPtr->subLattices;
for(i=1; i<=isingPtr->subLattices; i++)
{
	fprintf( fpw, "%.1f g\n", colorscale*(i-1) );
    for ( y=isingPtr->yOffset; y<(isingPtr->yOffset + isingPtr->yZoomRes); y++ )
    {
        for ( x=isingPtr->xOffset; x<(isingPtr->xOffset + isingPtr->xZoomRes); x++ )
        {

	    spin= *(data + x + y*isingPtr->xRes);

 	    if (spin == 1) {
		if(IsingColor(x, y, isingPtr->subLattices) == i)
		{ 
	    	fprintf( fpw, "%d %d f\n", x, (isingPtr->yOffset 
				 + isingPtr->yZoomRes) - y );
	    	/* Origin at the bottom in PS */
		}
  		}
		}
    }
}
    result = fprintf( fpw, "showpage\n" );
    if(result==0)
    {
	Tcl_AppendResult(interp, "error writing '", filename,  "'",
 	    (char*) NULL);
	return TCL_ERROR;
    }
    
    fclose(fpr);
    fclose(fpw);
    return 0;
}

static int
IsingPrintPSDevice(ClientData clientData, Tcl_Interp* interp, 
    int color, char* devicename)
{
    int i;
    int x, y;
    int boxsizex, boxsizey;
    int picsize;
    int tx, ty;
    int papersizex = 555;  /* +/- A4 - 7 mm marge*/
    int papersizey = 740;  /* in 1/72 inch */
    int paperxoffset = 20;
    int paperyoffset = 20;
    int c;
    Ising *isingPtr = (Ising*) clientData;
    int *data;  /* pointer to spindata */
    int spin;
    FILE *fpr;
    FILE *fpw;
    double colorscale;
    int result;  
 
    if(!(data = GetSpinData(isingPtr->pmodelstruct)))
	exit(1);  /* something is wrong, quit! */

    fpr = fopen( "isingheader.ps",  "r" );
    if(fpr==NULL)
	{
		Tcl_AppendResult(interp, "error opening 'isingheader.ps'",
 			(char*) NULL);
		return TCL_ERROR;
	}
	fpw = popen( devicename, "w" );
    if(fpw==NULL)
	{
		Tcl_AppendResult(interp, "error opening", devicename,
		    (char*) NULL);
		return TCL_ERROR;
	}
    while(( c = getc( fpr )) != EOF )
	{
		if( putc( c, fpw ) == 0 )  /* write header to devicename */
		{
			Tcl_AppendResult(interp, "error writing", devicename, 
			    (char*) NULL);
			return TCL_ERROR;
		}
	}	
    
    boxsizex = (papersizex / isingPtr->xZoomRes);
    boxsizey = (papersizey / isingPtr->yZoomRes);
 
    tx = (papersizex - boxsizex* isingPtr->xZoomRes) / 2 + paperxoffset;
    ty = (papersizey - boxsizey * isingPtr->yZoomRes) / 2 + paperyoffset;
    fprintf( fpw, "%d %d translate\n", tx, ty);
    fprintf( fpw, "%d %d scale\n", boxsizex, boxsizey );

	colorscale = 1.0 / isingPtr->subLattices;
for(i=1; i<=isingPtr->subLattices; i++)
{
	fprintf( fpw, "%.1f g\n", colorscale*(i-1) );
    for ( y=isingPtr->yOffset; y<(isingPtr->yOffset + isingPtr->yZoomRes); y++ )
    {
        for ( x=isingPtr->xOffset; x<(isingPtr->xOffset + isingPtr->xZoomRes); x++ )
        {

	    spin= *(data + x + y*isingPtr->xRes);

 	    if (spin == 1) {
		if(IsingColor(x, y, isingPtr->subLattices) == i)
		{ 
	    	fprintf( fpw, "%d %d f\n", x, (isingPtr->yOffset 
				 + isingPtr->yZoomRes) - y );
	    	/* Origin at the bottom in PS */
		}
  		}
		}
    }
}
    result = fprintf( fpw, "showpage\n" );

    if(result==0)
    {
	Tcl_AppendResult(interp, "error writing to '", devicename, "'", 
 	    (char*) NULL);
	return TCL_ERROR;
    }

    fflush(fpw);    
    fclose(fpr);
    pclose(fpw);


    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * IsingDisplay --
 *
 *	This procedure redraws the contents of a ising window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
IsingDisplay(ClientData clientData)
{

    Ising *isingPtr = (Ising *) clientData;
    Tk_Window tkwin = isingPtr->tkwin;
    isingPtr->updatePending = 0;
    if (!Tk_IsMapped(tkwin)) 
	return;
    
    /*
     *  Copy pixmap to current window
     */
    if (isingPtr->pm != None) {


        XCopyArea( Tk_Display(tkwin), isingPtr->pm, Tk_WindowId(tkwin), isingPtr->gc,
	           0, 0, Tk_Width(tkwin), Tk_Height(tkwin), 0, 0 );

   }
}

/*
 *----------------------------------------------------------------------
 *
 * IsingDestroy --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a ising at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the ising is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
IsingDestroy(clientData)
    ClientData clientData;	/* Info about ising widget. */
{
    Ising *isingPtr = (Ising *) clientData;

    Tk_FreeOptions(configSpecs, (char *) isingPtr, isingPtr->display, 0);
    if (isingPtr->gc != None) {
	Tk_FreeGC(isingPtr->display, isingPtr->gc);
    }
    ckfree((struct modeltype*) isingPtr->pmodelstruct);
    ckfree((char *) isingPtr);
}


