/*
 * window.c: Handles the Main Window stuff for irc.  This includes proper
 * scrolling, saving of screen memory, refreshing, clearing, etc. 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */
#include <stdio.h>
/* #include <values.h> */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <ctype.h>
#include "irc.h"
#include "menu.h"
#include "window.h"
#include "vars.h"
#include "server.h"
#include "list.h"
#include "term.h"
#include "names.h"
#include "ircaux.h"
#include "input.h"
#include "log.h"
#include "hook.h"
#include "dcc.h"
#include "translat.h"

/* value for underline mode, is 0 when on!  -lynx */
static	char	underline = 1;

/*
 * The following should synthesize MAXINT on any machine with an 8 bit
 * word.
 */
#define	MAXINT (-1&~(1<<(sizeof(int)*8-1)))

Window	*to_window;
Screen	*current_screen;
Screen	*main_screen;
Screen	*last_input_screen;
static	Window	*invisible_list = null(Window *);
						/* list of hidden windows */
static	char	*who_from = null(char *);	/* nick of person who's message
						 * is being displayed */
static	int	who_level = LOG_CRAP;/* Log level of message being displayed */
static	char	*save_from = null(char *);
static	int	save_level = LOG_CRAP;

static	in_window_command = 0;	/* set to true if we are in window().  This
				 * is used if a put_it() is called within the
				 * window() command.  We make sure all
				 * windows are fully updated before doing the
				 * put_it(). */

extern	int	in_help;	/* Used to suppress holding of help text */
extern	char	*redirect_nick;

/*
 * window_display: this controls the display, 1 being ON, 0 being OFF.  The
 * DISPLAY var sets this. 
 */
unsigned int	window_display = 1;

/*
 * status_update_flag: if 1, the status is updated as normal.  If 0, then all
 * status updating is suppressed 
 */
int	status_update_flag = 1;

/*
 * Declare this here because we use it if SCROLL is set to OFF when we're
 * at the bottom of the window.
 */
static	void	scroll_window();

/*
 * And this because we need to be able to display the lastlog lines when
 * redrawing the window.
 */
static	void	display_lastlog_lines();

static  void    add_to_invisible_list();
extern	void	close_all_dcc();
extern	void	close_all_exec();
extern	void	close_all_server();

/* Our list of screens */
Screen	*screen_list = null(Screen *);

/*
 * create_new_screen creates a new screen structure. with the help of
 * this structure we maintain ircII windows that cross screen window
 * boundaries.
 */
Screen	*create_new_screen()
{
	Screen	*new = null(Screen *),
		**list;
	static	int	refnumber = 0;

	for (list = &screen_list; *list; list = &((*list)->next))
	{
		if (!(*list)->alive)
		{
			new = *list;
			break;
		}
	}
	if (!new)
	{
		new = (Screen *) malloc(sizeof(Screen));
		new->screennum = ++refnumber;
		new->next = screen_list;
		if (screen_list)
			screen_list->prev = new;
		screen_list = new;
	}
	new->last_window_refnum = 1;
	new->window_list = null(Window *);
	new->window_list_end = null(Window *);
	new->cursor_window = null(Window *);
	new->current_window = null(Window *);
	new->visible_windows = 0;
	new->window_stack = null(WindowStack *);
	new->meta1_hit = new->meta2_hit = new->meta3_hit = new->meta4_hit = 0;
	new->quote_hit = new->digraph_hit = new->inside_menu = 0;
	new->buffer_pos = new->buffer_min_pos = 0;
	new->input_buffer[0] = '\0';
	new->fdout = 1;
	new->fpout = stdout;
	new->fdin = 0;
	new->fpin = stdin;
	new->alive = 1;
	new->promptlist = null(WaitPrompt *);
	new->redirect_name = null(char *);
	new->redirect_token = null(char *);
	new->redirect_server = -1;
	last_input_screen = new;
	return new;
}

/* 
 * add_wait_prompt:  Given a prompt string, a function to call when
 * the prompt is entered.. some other data to pass to the function,
 * and the type of prompt..  either for a line, or a key, we add 
 * this to the prompt_list for the current screen..  and set the
 * input prompt accordingly.
 */
void	add_wait_prompt(prompt, func, data, type)
char	*prompt;
void	(*func)();
char	*data;
int	type;
{
	WaitPrompt **AddLoc,
		   *New;

	New = (WaitPrompt *) new_malloc(sizeof(WaitPrompt));
	New->prompt = null(char *);
	malloc_strcpy(&New->prompt, prompt);
	New->data = null(char *);
	malloc_strcpy(&New->data, data);
	New->type = type;
	New->func = func;
	New->next = null(WaitPrompt *);
	for (AddLoc = &current_screen->promptlist; *AddLoc;
			AddLoc = &(*AddLoc)->next);
	*AddLoc = New;
	if (AddLoc == &current_screen->promptlist)
		change_input_prompt(1);
}

void	set_current_screen(screen)
Screen	*screen;
{
	if (screen->alive)
		current_screen = screen;
	else
		current_screen = screen_list;
}

/*
 * window_redirect: Setting who to non null will cause IRCII to echo all
 * commands and output from that server (including output generated by IRCII)
 * to who.  Setting who to null disables this 
 */
void	window_redirect(who, server)
char	*who;
int	server;
{
	char	buf[BIG_BUFFER_SIZE];

	sprintf(buf, "%s%04d", who, server);
	malloc_strcpy(&current_screen->redirect_token, buf);
	malloc_strcpy(&current_screen->redirect_name, who);
	current_screen->redirect_server = server;
}

extern	int	check_screen_redirect(nick)
char	*nick;
{
	Screen	*screen,
		*tmp_screen;
	
	for (screen = screen_list; screen; screen = screen->next)
	{
		if (!screen->redirect_token)
			continue;
		if (!strcmp(nick, screen->redirect_token))
		{
			tmp_screen = current_screen;
			current_screen = screen;
			window_redirect(null(char *), from_server);
			current_screen = tmp_screen;
			return 1;
		}
	}
	return 0;
}

/* Old screens never die. They just fade away. */

#ifdef WINDOW_CREATE
void	kill_screen(screen)
Screen	*screen;
{
	Window	*window;

	if (main_screen == screen)
	{
		say("You may not kill the main screen");
		return;
	}
	if (screen->fdin)
	{
		close(screen->fdout);
		close(screen->fdin);
	}
	for (;window = screen->window_list;)
	{
		screen->window_list = window->next;
		add_to_invisible_list(window);
	}

	if (last_input_screen == screen)
		last_input_screen = screen_list;
	screen->alive = 0;
}

int	is_main_screen(screen)
Screen	*screen;
{
	return (screen == main_screen);
}

#else

int	is_main_screen(screen)
Screen	*screen;
{
	return 1;
}

#endif /* WINDOW_CREATE */

/*
 * traverse_all_windows: This will do as the name implies, traverse every
 * window (visible and invisible) and return a pointer to each window on
 * subsequent calls.  If flag points to a non-zero value, then the traversal
 * in started from the beginning again, and flag is set to point to 0.  This
 * returns all visible windows first, then all invisible windows.  It returns
 * null after all windows have been returned.  It should generally be used as
 * follows: 
 *
 * flag = 1; while(tmp = traverse_all_windows(&flag)) { code here } 
 *
 * Major revamp by phone (phone@coombs.anu.edu.au), December 1992.
 */
Window	*traverse_all_windows(flag)
int	*flag;
{
	static	Window	*which;
	static	Screen	*screen;
	static	char	visible = 1;
	int	foo = 1;

	/* First call, return the current window basically */
	if (*flag)
	{
		*flag = 0;
		visible = 1;
		if (!screen_list)
			return null(Window *);
		screen = screen_list;
		which = screen->window_list;
		if (which)
			return (which);
		else
			foo = 0;
	}

	/*
	 * foo is used to indicate the the current screen has no windows.
	 * This happens when we create a new screen..  either way, we want
	 * to go on to the next screen, so if foo isn't set, then if which
	 * is already null, return it again (this should never happen, if
	 * traverse_all_windows()'s is called properly), else move on to
	 * the next window
	 */
	if (foo)
		if (!which)
			return null(Window *);
		else
			which = which->next;

	if (!which)
	{
		while (screen)
		{
			screen = screen->next;
			if (screen && screen->alive)
				break;
		}
		if (screen)
			which = screen->window_list;
	}

	if (which)
		return (which);
	/* 
	 * Got to the end of the visible list..  so we do the invisible list..
	 * Should also mean, that we've got to the end of all the visible
	 * screen..
	 */
	if (visible)
	{
		visible = 0;
		which = invisible_list;
		return (which);
	}
	return (null(Window *));
}

#if 0
/*
 * New version of traverse_all_windows that doesn't require that you don't
 * use it non-recusively.  Pass it a NULL pointer initially, and it will 
 * return 0 when it has finished, and also the address of an int, that is
 * initally 1
 */
static	int	new_window_traverse(window, visible)
Window	**window;
int	**visible;
{
	Screen	*screen,
		*old_screen;

	if (!*window)
	{
		if (!(screen = screen_list))
		{
			*visible = 0;
			if (!invisible_list)
				return 0;
			*window = invisible_list;
			return 1;
		}
		for (;screen && !screen->alive; screen = screen->next)
			;
		if (!screen)
		{
			*visible = 0;
			if (!invisible_list)
				return 0;
			*window = invisible_list;
			return 1;
		}
	}
	else
	{
		if ((*window)->next)
		{
			*window = (*window)->next;
			return 1;
		}
		if (!*visible)
			return 0;
		for (old_screen = screen = (*window)->screen;
				screen && !screen->alive; screen = screen->next)
			;
		if (!screen)
		{
			*visible = 0;
			if (!invisible_list)
				return 0;
			*window = invisible_list;
			return 1;
		}
		if (screen == old_screen)
			return 0;
	}
	*window = screen->current_window;
	return (*window) ? 1 : 0;
}
#endif

/* display_highlight: turns off and on the display highlight.  */
static	char	display_highlight(flag)
int	flag;
{
	static	int	highlight = OFF;

	fflush(current_screen->fpout);
	if (flag == highlight)
		return (flag);
	switch (flag)
	{
	case ON:
		highlight = ON;
		if (get_int_var(INVERSE_VIDEO_VAR))
			term_standout_on();
		return (OFF);
	case OFF:
		highlight = OFF;
		if (get_int_var(INVERSE_VIDEO_VAR))
			term_standout_off();
		return (ON);
	case TOGGLE:
		if (highlight == ON)
		{
			highlight = OFF;
			if (get_int_var(INVERSE_VIDEO_VAR))
				term_standout_off();
			return (ON);
		}
		else
		{
			highlight = ON;
			if (get_int_var(INVERSE_VIDEO_VAR))
				term_standout_on();
			return (OFF);
		}
	}
	return flag;
}

/* display_bold: turns off and on the display bolding.  */
static	char	display_bold(flag)
int	flag;
{
	static	int	bold = OFF;

	fflush(current_screen->fpout);
	if (flag == bold)
		return (flag);
	switch (flag)
	{
	case ON:
		bold = ON;
		if (get_int_var(BOLD_VIDEO_VAR))
			term_bold_on();
		return (OFF);
	case OFF:
		bold = OFF;
		if (get_int_var(BOLD_VIDEO_VAR))
			term_bold_off();
		return (ON);
	case TOGGLE:
		if (bold == ON)
		{
			bold = OFF;
			if (get_int_var(BOLD_VIDEO_VAR))
				term_bold_off();
			return (ON);
		}
		else
		{
			bold = ON;
			if (get_int_var(BOLD_VIDEO_VAR))
				term_bold_on();
			return (OFF);
		}
	}
}

/*
 * set_scroll_lines: called by /SET SCROLL_LINES to check the scroll lines
 * value 
 */
void	set_scroll_lines(size)
int	size;
{
	if (size == 0)
	{
		set_var_value(SCROLL_VAR, var_settings[0]);
		if (curr_scr_win)
			curr_scr_win->scroll = 0;
	}
	else if (size > curr_scr_win->display_size)
	{
		say("Maximum lines that may be scrolled is %d", 
		    curr_scr_win->display_size);
		set_int_var(SCROLL_LINES_VAR, curr_scr_win->display_size);
	}
}

/*
 * set_scroll: called by /SET SCROLL to make sure the SCROLL_LINES variable
 * is set correctly 
 */
void	set_scroll(value)
int	value;
{
	int	old_value;

	if (value && (get_int_var(SCROLL_LINES_VAR) == 0))
	{
		put_it("You must set SCROLL_LINES to a positive value first!");
		if (curr_scr_win)
			curr_scr_win->scroll = 0;
	}
	else
	{
		if (curr_scr_win)
		{
			old_value = curr_scr_win->scroll;
			curr_scr_win->scroll = value;
			if (old_value && !value)
				scroll_window(curr_scr_win);
		}
	}
}

/*
 * reset_line_cnt: called by /SET HOLD_MODE to reset the line counter so we
 * always get a held screen after the proper number of lines 
 */
void	reset_line_cnt(value)
int	value;
{
	curr_scr_win->hold_mode = value;
	curr_scr_win->hold_on_next_rite = 0;
	curr_scr_win->line_cnt = 0;
}

/*
 * set_continued_line: checks the value of CONTINUED_LINE for validity,
 * altering it if its no good 
 */
void	set_continued_line(value)
char	*value;
{
	if (value && (strlen(value) > (CO / 4)))
		value[CO / 4] = null(char);
}

/*
 * scroll_window: Given a pointer to a window, this determines if that window
 * should be scrolled, or the cursor moved to the top of the screen again, or
 * if it should be left alone. 
 */
static	void	scroll_window(window)
Window	*window;
{
	if (dumb)
		return;
	if (window->cursor == window->display_size)
	{
		if (window->scroll)
		{
			int	scroll,
			i;

			if ((scroll = get_int_var(SCROLL_LINES_VAR)) <= 0)
				scroll = 1;

			for (i = 0; i < scroll; i++)
			{
				new_free (&window->top_of_display->line);
				window->top_of_display =
					window->top_of_display->next;
			}
			if (window->visible)
			{
				if (term_scroll(window->top +
					window->menu.lines, window->top +
					window->menu.lines + window->cursor
					- 1, scroll))
				{
					if(current_screen->visible_windows == 1)
					{
						int	i;
						Window	*tmp;

			/*
			 * this method of sim-u-scroll seems to work fairly
			 * well. The penalty is that you can only have one
			 * window, and of course you can't use the scrollback
			 * buffer either. Or menus. Actually, this method
			 * really doesn't work very well at all anymore.
			 */
						tmp =
						  current_screen->cursor_window;
						term_move_cursor(0, LI - 2);
						if (term_clear_to_eol())
							term_space_erase(0);
						term_cr();
						term_newline();
						for (i = 0; i < scroll; i++)
						{
							term_cr();
							term_newline();
						}
					/* window->update |= UPDATE_STATUS; */
						update_window_status(window, 1);
						update_input(UPDATE_ALL);
						current_screen->cursor_window
							= tmp;
					}
					else
						redraw_window(window, 1);
				}
				window->cursor -= scroll;
				term_move_cursor(0, window->cursor +
					window->top + window->menu.lines);
			}
			else
				window->cursor -= scroll;
		}
		else
		{
			window->cursor = 0;
			if (window->visible)
				term_move_cursor(0, window->top +
					window->menu.lines);
		}
	}
	else if (window->visible && current_screen->cursor_window == window)
	{
		term_cr();
		term_newline();
	}
	if (window->visible && current_screen->cursor_window)
	{
		if (term_clear_to_eol())
		{
			term_space_erase(0);
			term_cr();
		}
		term_flush();
	}
}

/*
 * output_line prints the given string at the current screen position,
 * performing adjustments for ^B, ^V, and ^O
 */
int	output_line(str, result, startpos)
char	*str;
char	**result;
{
	static	int	high = OFF, bold = OFF;
	int	rev_tog, und_tog, bld_tog, both_off;
	char	*ptr;
	int	len;
	int	written = 0;
	char	c;
	char	*original;

	original=str;
	ptr = str;
	display_highlight(high);
	display_bold(bold);
	/* do processing on the string, handle inverse and bells */
	while (*ptr)
	{
		switch (*ptr)
		{
		case REV_TOG:
		case UND_TOG:
		case BOLD_TOG:
		case BOTH_OFF:
			len=(ptr-str);
			written += len;
			if (startpos)
			{
				if (ptr-original>startpos)
				{
					str+=len-(ptr-original-startpos);
					len=ptr-original-startpos;
					startpos=0;
				}
			}
			if (written > CO)
				len = len - (written - CO);
			if (!startpos)
			{
				fwrite(str, len, 1, current_screen->fpout);
			}
			rev_tog=und_tog=bld_tog=both_off=0;
			do
			{
				switch(*ptr)

				{
				case REV_TOG:
					rev_tog=(!rev_tog);
					break;
				case UND_TOG:
					und_tog=(!und_tog);
					break;
				case BOLD_TOG:
					bld_tog=(!bld_tog);
					break;
				case BOTH_OFF:
					both_off=1;
					und_tog=rev_tog=bld_tog=0;
					break;
				}
			} while ((ptr[1]==REV_TOG || ptr[1]==UND_TOG ||
			    ptr[1]==BOLD_TOG || ptr[1]==BOTH_OFF) && ptr++);
			if (both_off)
			{
				if (underline == 0)
				{
					term_underline_off();
					underline = 1;
				}
				display_highlight(OFF);
				display_bold(OFF);
				high = 0;
				bold = 0;
			}
			if (und_tog && get_int_var(UNDERLINE_VIDEO_VAR))
			{
				if (underline = 1 - underline)
					term_underline_off();
				else
					term_underline_on();
			}
			if (rev_tog)
			{
				high = display_highlight(TOGGLE);
				high = 1 - high;
			}
			if (bld_tog)
			{
				display_bold(TOGGLE);
				bold = 1 - bold;
			}
			str = ++ptr;
			break;
		case '\007':
		/*
		 * same as above, except after we display everything
		 * so far, we beep the terminal 
		 */
			c = *ptr;
			*ptr = null(char);
			len = strlen(str);
			written += len;
			if (startpos)
			{
				if (ptr-original>startpos)
				{
					str+=len-(ptr-original-startpos);
					len=ptr-original-startpos;
					startpos=0;
				}
			}
			if (written > CO)
				len = len - (written - CO);
			if (!startpos)
			{
				fwrite(str, len, 1, current_screen->fpout);
			}
			term_beep();
			*ptr = c;
			str = ++ptr;
			break;
		default:
			ptr++;
			break;
		}
	}
	if (result)
		*result=str;
	return written;
}

/*
 * rite: this routine displays a line to the screen adding bold facing when
 * specified by ^Bs, etc.  It also does handles scrolling and paging, if
 * SCROLL is on, and HOLD_MODE is on, etc.  This routine assumes that str
 * already fits on one screen line.  If show is true, str is displayed
 * regardless of the hold mode state.  If redraw is true, it is assumed we a
 * redrawing the screen from the display_ip list, and we should not add what
 * we are displaying back to the display_ip list again. 
 *
 * Note that rite sets display_highlight() to what it was at then end of the
 * last rite().  Also, before returning, it sets display_highlight() to OFF.
 * This way, between susequent rites(), you can be assured that the state of
 * bold face will remain the same and the it won't interfere with anything
 * else (i.e. status line, input line). 
 */
int	rite(window, str, show, redraw, backscroll, logged)
Window	*window;
char	*str;
int	show,
	redraw,
	backscroll,
	logged;
{
	static	int	high = OFF;
	int	written = 0,
		len;
	Screen	*old_current_screen = current_screen;

	if (window->hold_mode && window->hold_on_next_rite &&
			!redraw && !backscroll)
	{
		/* stops output from going to the window */
		window->hold_on_next_rite = 0;
		hold_mode(window, ON, 1);
		if (show)
			return (1);
	}
	/*
	 * Don't need to worry about the current_screen if the window isn't
	 * visible, as hidden windows aren't attached to a screen anyway
	 */
	if (window->visible)
	{
		old_current_screen = current_screen;
		set_current_screen(window->screen);
	}
	if (!show && (hold_output(window) || hold_queue(window)) && !in_help &&
	    !redraw && !backscroll)
		/* sends window output to the hold list for that window */
		add_to_hold_list(window, str, logged);
	else
	{
		if (!redraw && !backscroll)
		{
		/*
		 * This isn't a screen refresh, so add the line to the display
		 * list for the window 
		 */
			if (window->scroll)
				scroll_window(window);
			malloc_strcpy(&(window->display_ip->line), str);
			window->display_ip->linetype = logged;
			window->display_ip = window->display_ip->next;
			if (!window->scroll)
				new_free(&window->display_ip->line);
		}
		if (window->visible)
		{
			/* make sure the cursor is in the appropriate window */
			if (current_screen->cursor_window != window &&
					!redraw && !backscroll)
			{
				current_screen->cursor_window = window;
				term_move_cursor(0, window->cursor +
					window->top + window->menu.lines);
			}
			written = output_line(str, &str, 0);
			len = strlen(str);
			written += len;
			if (written > CO)
				len = len - (written - CO);
			fwrite(str, len, 1, current_screen->fpout);
			if (term_clear_to_eol())
				term_space_erase(written);
		}
		else if (!(window->miscflags & WINDOW_NOTIFIED))
		{
			window->miscflags |= WINDOW_NOTIFIED;
			if (window->miscflags & WINDOW_NOTIFY)
			{
				Window	*old_to_window;
				int	lastlog_level;

				lastlog_level = set_lastlog_msg_level(LOG_CRAP);
				old_to_window = to_window;
				to_window = curr_scr_win;
				say("Activity in window %d", window->refnum);
				to_window = old_to_window;
				set_lastlog_msg_level(lastlog_level);
			}
			update_all_status();
		}
		if (!redraw && !backscroll)
		{
			window->cursor++;
			window->line_cnt++;
			if (window->scroll)
			{
				if (window->line_cnt >= window->display_size)
				{
					window->hold_on_next_rite = 1;
					window->line_cnt = 0;
				}
			}
			else
			{
				scroll_window(window);
				if (window->cursor == (window->display_size -1))
					window->hold_on_next_rite = 1;
			}
		}
		else if (window->visible)
		{
			term_cr();
			term_newline();
		}
		if (window->visible)
		{
			high = display_highlight(OFF);
			term_flush();
		}
	}
	if (window->visible)
		set_current_screen(old_current_screen);
	return (0);
}

/*
 * cursor_not_in_display: This forces the cursor out of the display by
 * setting the cursor window to null.  This doesn't actually change the
 * physical position of the cursor, but it will force rite() to reset the
 * cursor upon its next call 
 */
void	cursor_not_in_display()
{
	current_screen->cursor_window = null(Window *);
}

/*
 * cursor_in_display: this forces the cursor_window to be the
 * current_screen->current_window. 
 * It is actually only used in hold.c to trick the system into thinking the
 * cursor is in a window, thus letting the input updating routines move the
 * cursor down to the input line.  Dumb dumb dumb 
 */
void	cursor_in_display()
{
	current_screen->cursor_window = curr_scr_win;
}

/*
 * is_cursor_in_display: returns true if the cursor is in one of the windows
 * (cursor_window is not null), false otherwise 
 */
int	is_cursor_in_display()
{
	if  (current_screen->cursor_window)
		return (1);
	else
		return (0);
}

/*
 * free_hold: This frees all the data and structures associated with the hold
 * list for the given window 
 */
static	void	free_hold(window)
Window	*window;
{
	Hold *tmp,
	    *next;

	for (tmp = window->hold_head; tmp; tmp = next)
	{
		next = tmp->next;
		new_free(&(tmp->str));
		new_free(&tmp);
	}
}

/*
 * free_lastlog: This frees all data and structures associated with the
 * lastlog of the given window 
 */
static	void	free_lastlog(window)
Window	*window;
{
	Lastlog *tmp,
	    *next;

	for (tmp = window->lastlog_head; tmp; tmp = next)
	{
		next = tmp->next;
		new_free(&(tmp->msg));
		new_free(&tmp);
	}
}

/*
 * free_display: This frees all memory for the display list for a given
 * window.  It resets all of the structures related to the display list
 * appropriately as well 
 */
static	void	free_display(window)
Window	*window;
{
	Display *tmp,
	    *next;
	int	i;

	if (window == null(Window *))
		window = curr_scr_win;
	for (tmp = window->top_of_display, i = 0; i < window->display_size; i++, tmp = next)
	{
		next = tmp->next;
		new_free(&(tmp->line));
		new_free(&tmp);
	}
	window->top_of_display = null(Display *);
	window->display_ip = null(Display *);
	window->display_size = 0;
}

static	void	free_nicks(window)
Window	*window;
{
	NickList *tmp,
	    *next;

	for (tmp = window->nicks; tmp; tmp = next)
	{
		next = tmp->next;
		new_free(&(tmp->nick));
		new_free(&tmp);
	}
}

/*
 * erase_display: This effectively causes all members of the display list for
 * a window to be set to empty strings, thus "clearing" a window.  It sets
 * the cursor to the top of the window, and the display insertion point to
 * the top of the display. Note, this doesn't actually refresh the screen,
 * just cleans out the display list 
 */
void	erase_display(window)
Window	*window;
{
	int	i;
	Display *tmp;

	if (dumb)
		return;
	if (window == null(Window *))
		window = curr_scr_win;
	for (tmp = window->top_of_display, i = 0; i < window->display_size;
			i++, tmp = tmp->next)
		new_free(&(tmp->line));
	window->cursor = 0;
	window->line_cnt = 0;
	window->hold_on_next_rite = 0;
	window->display_ip = window->top_of_display;
}

void	redraw_resized(window, Info, AnchorTop)
Window	*window;
ShrinkInfo Info;
int	AnchorTop;
{
	if (!AnchorTop)
	{
		if (Info.bottom < 0)
			term_scroll(window->top+window->menu.lines+Info.bottom,
				window->top + window->menu.lines +
				window->display_size - 1,
				Info.bottom);
		else if (Info.bottom)
			term_scroll(window->top+window->menu.lines,
				window->top + window->menu.lines +
				window->display_size -1, Info.bottom);
	}
}

/*
 * resize_display: After determining that the screen has changed sizes, this
 * resizes all the internal stuff.  If the screen grew, this will add extra
 * empty display entries to the end of the display list.  If the screen
 * shrank, this will remove entries from the end of the display list.  By
 * doing this, we try to maintain as much of the display as possible. 
 *
 * This has now been improved so that it returns enough information for
 * redraw_resized to redisplay the contents of the window without having
 * to redraw too much.
 */
ShrinkInfo resize_display(window)
Window	*window;
{
	int	cnt,
		i;
	Display *tmp, *pre_ip;
	int	Wrapped = 0;
	ShrinkInfo Result;

	Result.top = Result.bottom = 0;
	Result.position = window->cursor;
	if (dumb)
	{
		return Result;
	}
	if (window->top_of_display == null(Display *))
	{
		window->top_of_display = (Display *)new_malloc(sizeof(Display));
		window->top_of_display->line = null(char *);
		window->top_of_display->linetype = LT_UNLOGGED;
		window->top_of_display->next = window->top_of_display;
		window->display_ip = window->top_of_display;
		window->old_size = 1;
	}
	/* cnt = size - window->display_size; */
	cnt = window->display_size - window->old_size;
	if (cnt > 0)
	{
		Display *new = null(Display *);

	/*
	 * screen got bigger: got to last display entry and link in new
	 * entries 
	 */
		for (tmp = window->top_of_display, i = 0;
		    i < (window->old_size - 1);
		    i++, tmp = tmp->next);
		for (i = 0; i < cnt; i++)
		{
			new = (Display *) new_malloc(sizeof(Display));
			new->line = null(char *);
			new->linetype = LT_UNLOGGED;
			new->next = tmp->next;
			tmp->next = new;
		}
		if (window->display_ip == window->top_of_display &&
		    window->top_of_display->line)
			window->display_ip = new;
		Result.top = 0;
		Result.bottom = cnt;
		Result.position = 0;
	}
	else if (cnt < 0)

	{
		Display *ptr;

	/*
	 * screen shrank: find last display entry we want to keep, and remove
	 * all after that point 
	 */
		cnt = -cnt;
		for (pre_ip = window->top_of_display;
		    pre_ip->next != window->display_ip;
		    pre_ip = pre_ip->next);
		for (tmp = pre_ip->next, i =0; i < cnt; i++, tmp = ptr)
		{
			ptr = tmp->next;
			if (tmp == window->top_of_display)

			{
				if (tmp->line)
					Wrapped = 1;
				window->top_of_display = ptr;
			}
			if (Wrapped)
				Result.top--;
			else
				Result.bottom--;
			new_free(&(tmp->line));
			new_free(&tmp);
		}
		window->display_ip = pre_ip->next = tmp;
		window->cursor += Result.top;
		if (!window->scroll)
		{
			if (window->cursor == window->display_size)
				window->cursor = 0;
			new_free(&window->display_ip->line);
		}
	}
	window->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
	window->old_size = window->display_size;
	return Result;
}

/*
 * recalculate_windows: this is called when the terminal size changes (as
 * when an xterm window size is changed).  It recalculates the sized and
 * positions of all the windows.  Currently, all windows are rebalanced and
 * window size proportionality is lost 
 */
void	recalculate_windows()
{
	int	base_size,
	size,
	top,
	extra;
	Window	*tmp;

	if (dumb)
		return;
	base_size = ((LI - 1) / current_screen->visible_windows) - 1;
	extra = (LI - 1) - ((base_size + 1) * current_screen->visible_windows);
	top = 0;
	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
	{
		tmp->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
		if (extra)
		{
			extra--;
			size = base_size + 1;
		}
		else
			size = base_size;
#ifdef SCROLL_AFTER_DISPLAY
		tmp->display_size = size - tmp->menu.lines - 1;
#else
		tmp->display_size = size - tmp->menu.lines;
#endif /* SCROLL_AFTER_DISPLAY */
		if (tmp->display_size<=0)
			tmp->display_size = 1;
		tmp->top = top;
		tmp->bottom = top + size;
		top += size + 1;
	}
}

/*
 * clear_window: This clears the display list for the given window, or
 * current window if null is given.  
 */
static	void	clear_window(window)
Window	*window;
{
	int	i,
		cnt;

	if (dumb)
		return;
	if (window == null(Window *))
		window = curr_scr_win;
	erase_display(window);
	term_move_cursor(0, window->top + window->menu.lines);
	cnt = window->bottom - window->top - window->menu.lines;
	for (i = 0; i < cnt; i++)
	{
		if (term_clear_to_eol())
			term_space_erase(0);
		term_newline();
	}
	term_flush();
}

/* clear_all_windows: This clears all *visible* windows */
void	clear_all_windows()
{
	Window	*tmp;

	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
		clear_window(tmp);
}

/*
 * redraw_window: This redraws the display list for the given window. Some
 * special considerations are made if you are just redrawing one window as
 * opposed to using this routine to redraw the whole screen one window at a
 * time 
 *
 * A negative value in just_one indicates not all of the window needs to
 * be redrawn.
 */
void	redraw_window(window, just_one)
Window	*window;
int	just_one;
{
	Display *tmp;
	int	i;
	int	StartPoint;
	int	yScr;

	if (dumb || !window->visible)
		return;
	if (window == null(Window *))
		window = curr_scr_win;
	if (just_one < 0)

	{
	/* This part of the window is scrolling into view */
		StartPoint = -just_one;
		just_one = 0;
	}
	else
	{
		StartPoint = 0;
		if (window->scrolled_lines)
			display_lastlog_lines(window->scrolled_lines-window->display_size + 1, window->scrolled_lines, window);
	}
	if (window->menu.menu)
		ShowMenuByWindow(window, just_one ? SMF_ERASE : 0);
	if (window->scrolled_lines + StartPoint < window->display_size)
		yScr = window->scrolled_lines + StartPoint;
	else
		yScr = 0;
	term_move_cursor(0, window->top+window->menu.lines+yScr);
	/*
	 * if (term_clear_to_eol())
	 *	{ term_space_erase(0); term_cr(); } 
	 */
	for (tmp = window->top_of_display, i = 0;
			i < window->display_size-window->scrolled_lines;
			i++, tmp = tmp->next)
	{
		if (i < StartPoint)
			continue;
		if (tmp->line)
			rite(window, tmp->line, 1, 1, 0, 0);
		else
		{
			if (just_one)
			{
				if (term_clear_to_eol())
					term_space_erase(0);
			}
			term_newline();
		}
	}
	term_flush();
}

/*
 * recalculate_window_positions: This runs through the window list and
 * re-adjusts the top and bottom fields of the windows according to their
 * current positions in the window list.  This doesn't change any sizes of
 * the windows 
 */
static	void	recalculate_window_positions()
{
	Window	*tmp;
	int	top;

	top = 0;
	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
	{
		tmp->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
		tmp->top = top;
		tmp->bottom = top + tmp->display_size + tmp->menu.lines;
		top += tmp->display_size + tmp->menu.lines + 1;
	}
}

static	void	remove_from_invisible_list(window)
Window	*window;
{
	window->visible = 1;
	window->screen = current_screen;
	window->miscflags &= ~WINDOW_NOTIFIED;
	if (window->prev)
		window->prev->next = window->next;
	else
		invisible_list = window->next;
	if (window->next)
		window->next->prev = window->prev;
}

static	void	add_to_invisible_list(window)
Window	*window;
{
	if (window->next = invisible_list)
		invisible_list->prev = window;
	invisible_list = window;
	window->prev = null(Window *);
	window->visible = 0;
	window->screen = null(Screen *);
}

/*
 * swap_window: This swaps the given window with the current window.  The
 * window passed must be invisible.  Swapping retains the positions of both
 * windows in their respective window lists, and retains the dimensions of
 * the windows as well 
 */
static	void	swap_window(v_window, window)
Window	*v_window;
Window	*window;
{
	Window tmp, *prev, *next;
	int	top, bottom, size;

	if (window->visible || !v_window->visible)
	{
		say("You can only SWAP a hidden window with a visible window.");
		return;
	}
	prev = v_window->prev;
	next = v_window->next;

	current_screen->last_window_refnum = v_window->refnum;

	current_screen->last_window_refnum = v_window->refnum;

	remove_from_invisible_list(window);

	tmp = *v_window;
	*v_window = *window;
	v_window->top = tmp.top;
	v_window->bottom = tmp.bottom;
	v_window->display_size = tmp.display_size + tmp.menu.lines -
#ifdef SCROLL_AFTER_DISPLAY
		v_window->menu.lines - 1;
#else
		v_window->menu.lines;
#endif /* SCROLL_AFTER_DISPLAY */
	v_window->prev = prev;
	v_window->next = next;

	top = window->top;
	bottom = window->bottom;
	size = window->display_size;
	*window = tmp;
	window->top = top;
	window->bottom = bottom;
#ifdef SCROLL_AFTER_DISPLAY
	window->display_size = size - 1;
#else
	window->display_size = size;
#endif /* SCROLL_AFTER_DISPLAY */

	add_to_invisible_list(window);

	v_window->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
	window->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
}

/*
 * move_window: This moves a window offset positions in the window list. This
 * means, of course, that the window will move on the screen as well 
 */
static	void	move_window(window, offset)
Window	*window;
int	offset;
{
	Window	*tmp,
	    *last;
	int	win_pos,
	pos;

	if (offset == 0)
		return;
	last = null(Window *);
	for (win_pos = 0, tmp = current_screen->window_list; tmp;
	    tmp = tmp->next, win_pos++)
	{
		if (window == tmp)
			break;
		last = tmp;
	}
	if (tmp == null(Window *))
		return;
	if (last == null(Window *))
		current_screen->window_list = tmp->next;
	else
		last->next = tmp->next;
	if (tmp->next)
		tmp->next->prev = last;
	else
		current_screen->window_list_end = last;
	if (offset < 0)
		win_pos = (current_screen->visible_windows + offset + win_pos) %
		    current_screen->visible_windows;
	else
		win_pos = (offset + win_pos) % current_screen->visible_windows;
	last = null(Window *);
	for (pos = 0, tmp = current_screen->window_list;
	    pos != win_pos; tmp = tmp->next, pos++)
		last = tmp;
	if (last == null(Window *))
		current_screen->window_list = window;
	else
		last->next = window;
	if (tmp)
		tmp->prev = window;
	else
		current_screen->window_list_end = window;
	window->prev = last;
	window->next = tmp;
	recalculate_window_positions();
}

/*
 * grow_window: This will increase or descrease the size of the given window
 * by offset lines (positive offset increases, negative decreases).
 * Obviously, with a fixed terminal size, this means that some other window
 * is going to have to change size as well.  Normally, this is the next
 * window in the window list (the window below the one being changed) unless
 * the window is the last in the window list, then the previous window is
 * changed as well 
 */
static	void	grow_window(window, offset)
Window	*window;
int	offset;
{
	Window	*other,
	    *tmp;
	int	after,
	window_size,
	other_size;

	if (window == null(Window *))
		window = curr_scr_win;
	if (!window->visible)
	{
		say("You cannot change the size of hidden windows!");
		return;
	}
	if (window->next)
	{
		other = window->next;
		after = 1;
	}
	else
	{
		other = null(Window *);
		for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
		{
			if (tmp == window)
				break;
			other = tmp;
		}
		if (other == null(Window *))
		{
			say("Can't change the size of this window!");
			return;
		}
		after = 0;
	}
	window_size = window->display_size + offset;
	other_size = other->display_size - offset;
	if ((window_size < 3) ||
	    (other_size < 3))
	{
		say("Not enough room to resize this window!");
		return;
	}
	if (after)
	{
		window->bottom += offset;
		other->top += offset;
	}
	else
	{
		window->top -= offset;
		other->bottom -= offset;
	}
#ifdef SCROLL_AFTER_DISPLAY
	window->display_size = window_size - 1;
	other->display_size = other_size - 1;
#else
	window->display_size = window_size;
	other->display_size = other_size;
#endif /* SCROLL_AFTER_DISPLAY */
	window->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
	other->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
	term_flush();
}

/*
 * redraw_all_windows: This basically clears and redraws the entire display
 * portion of the screen.  All windows and status lines are draws.  This does
 * nothing for the input line of the screen.  Only visible windows are drawn 
 */
void	redraw_all_windows()
{
	Window	*tmp;

	if (dumb)
		return;
	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
		tmp->update |= REDRAW_STATUS | REDRAW_DISPLAY_FAST;
}

#define	MAXIMUM_SPLITS	40
char	**split_up_line(str)
char	*str;
{
	static	char	*output[MAXIMUM_SPLITS] =
	{ 
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *)
	};
	char	buffer[BIG_BUFFER_SIZE + 1];
	unsigned char *ptr;
	char	*cont_ptr,
		*cont = null(char *),
		*temp = null(char *),
		c;
	int	pos = 0,
		col = 0,
		nd_cnt = 0,
		word_break = 0,
		start = 0,
		i,
		len,
		indent = 0,
		beep_cnt = 0,
		beep_max,
		tab_cnt = 0,
		tab_max,
		line = 0;

	for (i = 0; i < BIG_BUFFER_SIZE + 1; ++i) buffer[i] = '\0';
	for (i = 0; i < MAXIMUM_SPLITS; i++)
		new_free(&output[i]);
	if (*str == null(char))
		str = " ";	/* special case to make blank lines show up */

	beep_max = get_int_var(BEEP_VAR) ? get_int_var(BEEP_MAX_VAR) : -1;
	tab_max = get_int_var(TAB_VAR) ? get_int_var(TAB_MAX_VAR) : -1;
	for (ptr = (unsigned char *)str; (*ptr && (pos < BIG_BUFFER_SIZE - 8));
			ptr++)
	{
		if (translation)
			*ptr = transToClient[*ptr];
		if (*ptr <= 32)
		{
			switch (*ptr)
			{
			case '\007':	/* bell */
				if (beep_max == -1)
				{
					buffer[pos++] = REV_TOG;
					buffer[pos++] = (*ptr & 127) | 64;
					buffer[pos++] = REV_TOG;
					nd_cnt += 2;
					col++;
				}
				else if (!beep_max || (++beep_cnt <= beep_max))
				{
					buffer[pos++] = *ptr;
					nd_cnt++;
					col++;
				}
				break;
			case '\011':	/* tab */
				if (tab_max && (++tab_cnt > tab_max))
				{
					buffer[pos++] = REV_TOG;
					buffer[pos++] = (*ptr & 127) | 64;
					buffer[pos++] = REV_TOG;
					nd_cnt += 2;
					col++;
				}
				else
				{
					if (indent == 0)
						indent = -1;
					len = 8 - (col % 8);
					word_break = pos;
					for (i = 0; i < len; i++)
						buffer[pos++] = ' ';
					col += len;
				}
				break;
			case ' ':	/* word break */
				if (indent == 0)
					indent = -1;
				word_break = pos;
				buffer[pos++] = *ptr;
				col++;
				break;
			case UND_TOG:
			case BOTH_OFF:
			case REV_TOG:
			case BOLD_TOG:
				buffer[pos++] = *ptr;
				nd_cnt++;
				break;
			default:	/* Anything else, make it displayable */
				if (indent == -1)
					indent = pos - nd_cnt;
				buffer[pos++] = REV_TOG;
				buffer[pos++] = (*ptr & 127) | 64;
				buffer[pos++] = REV_TOG;
				nd_cnt += 2;
				col++;
				break;
			}
		}
		else
		{
			if (indent == -1)
				indent = pos - nd_cnt;
			buffer[pos++] = *ptr;
			col++;
		}
		if (pos == BIG_BUFFER_SIZE)
			*ptr = null(char);
		if (col >= CO)
		{
			/* one big long line, no word breaks */
			if (word_break == 0)
				word_break = pos - (col - CO);
			c = buffer[word_break];
			buffer[word_break] = null(char);
			if (cont)
			{
				malloc_strcpy(&temp, cont);
				malloc_strcat(&temp, &(buffer[start]));
			}
			else
				malloc_strcpy(&temp, &(buffer[start]));
			malloc_strcpy(&output[line++], temp);
			buffer[word_break] = c;
			start = word_break;
			word_break = 0;
			while (buffer[start] == ' ')
				start++;
			if (start > pos)
				start = pos;
			if ((cont_ptr = get_string_var(CONTINUED_LINE_VAR))
					== null(char *))
				cont_ptr = empty_string;
			if (get_int_var(INDENT_VAR) && (indent < CO / 3))
			{
		/*
		 * INDENT thanks to Carlo "lynx" v. Loesch
		 * - hehe, nice to see this is still here... -lynx 91
		 */
				if (cont == null(char *))
				{
					if ((len = strlen(cont_ptr)) > indent)
					{
						cont = (char *) new_malloc(len
							+ 1);
						strcpy(cont, cont_ptr);
					}
					else
					{
						cont = (char *)
							new_malloc(indent + 1);
						strcpy(cont, cont_ptr);
						for (i = len; i < indent; i++)
							cont[i] = ' ';
						cont[indent] = null(char);
					}
				}
			}
			else
				malloc_strcpy(&cont, cont_ptr);
			col = strlen(cont) + (pos - start);
		}
	}
	buffer[pos] = null(char);
	if (buffer[start])
	{
		if (cont)
		{
			malloc_strcpy(&temp, cont);
			malloc_strcat(&temp, &(buffer[start]));
		}
		else
			malloc_strcpy(&temp, &(buffer[start]));
		malloc_strcpy(&output[line++], temp);
	}
	new_free(&cont);
	new_free(&temp);
	return output;
}

/*
 * add_to_window: adds the given string to the display.  No kidding. This
 * routine handles the whole ball of wax.  It keeps track of what's on the
 * screen, does the scrolling, everything... well, not quite everything...
 * The CONTINUED_LINE idea thanks to Jan L. Peterson (jlp@hamblin.byu.edu)  
 *
 * At least it used to. Now most of this is done by split_up_line, and this
 * function just dumps it onto the screen. This is because the scrollback
 * functions need to be able to figure out how to split things up too.
 */
static	void	add_to_window(window, str)
Window	*window;
char	*str;
{
	int flag;

	add_to_log(window->log_fp, str);
	flag = do_hook(WINDOW_LIST, "%u %s", window->refnum, str);
	add_to_lastlog(window, str);

	if (flag)
	{
		char	**lines;
		int	logged;

		display_highlight(OFF);
		display_bold(OFF);
		strmcat(str, "\017", BIG_BUFFER_SIZE);
		logged = islogged(window);
		for (lines = split_up_line(str); *lines; lines++)
		{
			rite(window, *lines, 0, 0, 0, logged);
			if (logged == 1)
				logged = 2;
		}
		term_flush();
	}
}

/*
 * save_message_from: this is used to save (for later restoration) the
 * who_from variable.  This comes in handy very often when a routine might
 * call another routine that might change who_from. 
 */
void	save_message_from()
{
	malloc_strcpy(&save_from, who_from);
	save_level = who_level;
}

/* restore_message_from: restores a previously saved who_from variable */
void	restore_message_from()
{
	malloc_strcpy(&who_from, save_from);
	who_level = save_level;
}

/*
 * message_from: With this you can the who_from variable and the who_level
 * variable, used by the display routines to decide which window messages
 * should go to.  
 */
void	message_from(who, level)
char	*who;
int	level;
{
	malloc_strcpy(&who_from, who);
	who_level = level;
}

/*
 * message_from_level: Like set_lastlog_msg_level, except for message_from.
 * this is needed by XECHO, because we could want to output things in more
 * than one level.
 */
int	message_from_level(level)
int	level;
{
	int	temp;

	temp = who_level;
	who_level = level;
	return temp;
}

/*
 * get_window_by_refnum: Given a reference number to a window, this returns a
 * pointer to that window if a window exists with that refnum, null is
 * returned otherwise.  The "safe" way to reference a window is throught the
 * refnum, since a window might be delete behind your back and and Window
 * pointers might become invalid.  
 */
Window	*get_window_by_refnum(refnum)
unsigned int refnum;
{
	Window	*tmp;
	int	flag = 1;

	if (refnum)
	{
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->refnum == refnum)
				return (tmp);
		}
	}
	else
		return (curr_scr_win);
	return (null(Window *));
}

/*
 * clear_window_by_refnum: just like clear_window(), but it uses a refnum. If
 * the refnum is invalid, the current window is cleared. 
 */
void	clear_window_by_refnum(refnum)
unsigned int refnum;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	clear_window(tmp);
}

/*
 * revamp_window_levels: Given a level setting for the current window, this
 * makes sure that that level setting is unused by any other window. Thus
 * only one window in the system can be set to a given level.  This only
 * revamps levels for windows with servers matching the given window 
 */
static	void	revamp_window_levels(window)
Window	*window;
{
	Window	*tmp;
	int	flag = 1;

	while (tmp = traverse_all_windows(&flag))
	{
		if ((tmp != window) && (window->server == tmp->server))
			tmp->window_level ^= (tmp->window_level &
				window->window_level);
	}
}

/*
 * set_level_by_refnum: This sets the window level given a refnum.  It
 * revamps the windows levels as well using revamp_window_levels() 
 */
void	set_level_by_refnum(refnum, level)
unsigned int refnum;
int	level;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	tmp->window_level = level;
	revamp_window_levels(tmp);
}

/*
 * set_prompt_by_refnum: changes the prompt for the given window.  A window
 * prompt will be used as the target in place of the query user or current
 * channel if it is set 
 */
void	set_prompt_by_refnum(refnum, prompt)
unsigned int refnum;
char	*prompt;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	malloc_strcpy(&(tmp->prompt), prompt);
}

/*
 * message_to: This allows you to specify a window (by refnum) as a
 * destination for messages.  Used by EXEC routines quite nicely 
 */
void	message_to(refnum)
unsigned int refnum;
{
	if (refnum)
		to_window = get_window_by_refnum(refnum);
	else
		to_window = null(Window *);
}

/*
 * add_to_screen: This adds the given null terminated buffer to the screen.
 * That is, it determines which window the information should go to, which
 * lastlog the information should be added to, which log the information
 * should be sent to, etc 
 */
void	add_to_screen(buffer)
char	*buffer;
{
	Window	*tmp;
	int	flag;

	/* Handles output redirection first */
	if (current_screen->redirect_name &&
			(from_server == current_screen->redirect_server))
	{
		int	i;

		if (*current_screen->redirect_name == '%')
		{	/* redirection to a process */
			if (is_number(current_screen->redirect_name + 1))
				i = atoi(current_screen->redirect_name + 1);
			else
			    i = logical_to_index(current_screen->redirect_name
				+ 1);
			if (is_process_running(i))
				text_to_process(i, buffer);
		}
#if 1	/* doesn't work yet */
		else if (*current_screen->redirect_name == '=')
			dcc_message_transmit(current_screen->redirect_name + 1,
				buffer, DCC_CHAT, 0);
#endif
		/*
		 * shouldn't this be a NOTICE?  -lynx
		 * agreed - phone, jan 1993
		 */
		else
			send_to_server("PRIVMSG %s %s",
				current_screen->redirect_name, buffer);
	}
	if (dumb)
	{
		add_to_lastlog(curr_scr_win, buffer);
		if (do_hook(WINDOW_LIST, "%u %s", curr_scr_win->refnum, buffer))
			puts(buffer);
		fflush(current_screen->fpout);
		return;
	}
	if (in_window_command)
		update_all_windows();
	if ((who_level == LOG_CURRENT) && (curr_scr_win->server == from_server))
	{
		add_to_window(curr_scr_win, buffer);
		return;
	}
	if (to_window)
	{
		add_to_window(to_window, buffer);
		return;
	}
	if (who_from)
	{
		flag = 1;
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->current_channel &&
				!stricmp(who_from, tmp->current_channel))
			{
				if (tmp->server == from_server)
				{
					add_to_window(tmp, buffer);
					return;
				}
			}
			if (tmp->query_nick &&
			    (((who_level==LOG_MSG || who_level==LOG_NOTICE) &&
			    stricmp(who_from, tmp->query_nick) == 0 &&
			    from_server == tmp->server) ||
			    (who_level==LOG_DCC &&
			    (*tmp->query_nick=='=' || *tmp->query_nick=='@') &&
			    stricmp(who_from, tmp->query_nick+1) == 0)))
			{
				add_to_window(tmp, buffer);
				return;
			}
		}
		flag = 1;
		while (tmp = traverse_all_windows(&flag))
		{
			if (from_server == tmp->server)
			{
				if (find_in_list(&(tmp->nicks), who_from,
					!USE_WILDCARDS))
				{
					add_to_window(tmp, buffer);
					return;
				}
			}
		}
	}
	flag = 1;
	while (tmp = traverse_all_windows(&flag))
	{
		if (((from_server == tmp->server) || (from_server == -1)) &&
		    (who_level & tmp->window_level))
		{
			add_to_window(tmp, buffer);
			return;
		}
	}
	if (from_server == curr_scr_win->server)
		tmp = curr_scr_win;
	else
	{
		flag = 1;
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->server == from_server)
				break;
		}
		if (tmp == null(Window *))
			tmp = curr_scr_win;
	}
	add_to_window(tmp, buffer);
}

/*
 * update_all_windows: This goes through each visible window and draws the
 * necessary portions according the the update field of the window. 
 */
void	update_all_windows()
{
	Window	*tmp;
	int	fast_window,
		full_window,
		r_status,
		u_status;

	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
	{
		if (tmp->display_size != tmp->old_size)
			resize_display(tmp);
		if (tmp->update)
		{
			fast_window = tmp->update & REDRAW_DISPLAY_FAST;
			full_window = tmp->update & REDRAW_DISPLAY_FULL;
			r_status = tmp->update & REDRAW_STATUS;
			u_status = tmp->update & UPDATE_STATUS;
			if (full_window)
				redraw_window(tmp, 1);
			else if (fast_window)
				redraw_window(tmp, 0);
			if (r_status)
				update_window_status(tmp, 1);
			else if (u_status)
				update_window_status(tmp, 0);
		}
		tmp->update = 0;
	}
	for (tmp = invisible_list; tmp; tmp = tmp->next)
	{
		if (tmp->display_size != tmp->old_size)
			resize_display(tmp);
		tmp->update = 0;
	}
	update_input(UPDATE_JUST_CURSOR);
	term_flush();
}

/*
 * get_next_window: This returns a pointer to the next *visible* window in
 * the window list.  It automatically wraps at the end of the list back to
 * the beginning of the list 
 */
static	Window	*get_next_window()
{
	if (curr_scr_win && curr_scr_win->next)
		return (curr_scr_win->next);
	else
		return (current_screen->window_list);
}

/*
 * get_previous_window: this returns the previous *visible* window in the
 * window list.  This automatically wraps to the last window in the window
 * list 
 */
static	Window	*get_previous_window()
{
	if (curr_scr_win && curr_scr_win->prev)
		return (curr_scr_win->prev);
	else
		return (current_screen->window_list_end);
}

/*
 * set_current_window: This sets the "current" window to window.  It also
 * keeps track of the last_current_screen->current_window by setting it to the
 * previous current window.  This assures you that the new current window is
 * visible.
 * If not, a new current window is chosen from the window list 
 */
void	set_current_window(window)
Window	*window;
{
	Window	*tmp;
	unsigned int	refnum;

	refnum = current_screen->last_window_refnum;
	if (curr_scr_win)

	{
		curr_scr_win->update |= UPDATE_STATUS;
		current_screen->last_window_refnum = curr_scr_win->refnum;
	}
	if ((window == null(Window *)) || (!window->visible))

	{
		if ((tmp = get_window_by_refnum(refnum)) && (tmp->visible))
			curr_scr_win = tmp;
		else
			curr_scr_win = get_next_window();
	}
	else
		curr_scr_win = window;
	curr_scr_win->update |= UPDATE_STATUS;
}

/*
 * swap_last_window:  This swaps the current window with the last window
 * that was hidden.
 */

void	swap_last_window()
{
	if (invisible_list == null(Window *))
	{
		/* say("There are no hidden windows"); */
		/* Not sure if we need to warn   - phone. */
		return;
	}
	swap_window(curr_scr_win, invisible_list);
	message_from(null(char *), LOG_CRAP);
	update_all_windows();
	cursor_to_input();
}

/*
 * next_window: This switches the current window to the next visible window 
 */
void	next_window()
{
	if (current_screen->visible_windows == 1)
		return;
	set_current_window(get_next_window());
	update_all_windows();
}

/*
 * swap_next_window:  This swaps the current window with the next hidden
 * window.
 */

void	swap_next_window()
{
	int	flag;
	Window	*tmp;
	int	next = MAXINT;
	int	smallest;

	if (invisible_list == null(Window *))
	{
		say("There are no hidden windows");
		return;
	}
	flag = 1;
	smallest = curr_scr_win->refnum;
	while (tmp = traverse_all_windows(&flag))
	{
		if (!tmp->visible)
		{
			if (tmp->refnum < smallest)
				smallest = tmp->refnum;
			if ((tmp->refnum > curr_scr_win->refnum)
			    && (next > tmp->refnum))
				next = tmp->refnum;
		}
	}
	if (next != MAXINT)
		tmp = get_window_by_refnum(next);
	else
		tmp = get_window_by_refnum(smallest);
	swap_window(curr_scr_win, tmp);
	message_from(null(char *), LOG_CRAP);
	update_all_windows();
	update_all_status();
	cursor_to_input();
}

/*
 * previous_window: This switches the current window to the previous visible
 * window 
 */
void	previous_window()
{
	if (current_screen->visible_windows == 1)
		return;
	set_current_window(get_previous_window());
	update_all_windows();
}

/*
 * swap_previous_window:  This swaps the current window with the next 
 * hidden window.
 */

void	swap_previous_window()
{
	int	flag;
	Window	*tmp;
	int	previous = 0;
	int	largest;

	if (invisible_list == null(Window *))
	{
		say("There are no hidden windows");
		return;
	}
	flag = 1;
	largest = curr_scr_win->refnum;
	while (tmp = traverse_all_windows(&flag))
	{
		if (!tmp->visible)
		{
			if (tmp->refnum > largest)
				largest = tmp->refnum;
			if ((tmp->refnum < curr_scr_win->refnum)
			    && (previous < tmp->refnum))
				previous = tmp->refnum;
		}
	}
	if (previous)
		tmp = get_window_by_refnum(previous);
	else
		tmp = get_window_by_refnum(largest);
	swap_window(curr_scr_win,tmp);
	message_from(null(char *), LOG_CRAP);
	update_all_windows();
	update_all_status();
	cursor_to_input();
}

/*
 * back_window:  goes to the last window that was current.  Swapping the
 * current window if the last window was hidden.
 */

void	back_window()
{
	Window	*tmp;

	tmp = get_window_by_refnum(current_screen->last_window_refnum);
	if (tmp->visible)
		set_current_window(tmp);
	else
	{
		swap_window(curr_scr_win, tmp);
		message_from(null(char *), LOG_CRAP);
		update_all_windows();
		update_all_status();
		cursor_to_input();
	}
}

/*
 * add_to_window_list: This inserts the given window into the visible window
 * list (and thus adds it to the displayed windows on the screen).  The
 * window is added by splitting the current window.  If the current window is
 * too small, the next largest window is used.  The added window is returned
 * as the function value or null is returned if the window couldn't be added 
 */
static	Window	*add_to_window_list(new)
Window	*new;
{
	Window	*biggest = null(Window *),
		*tmp;

	current_screen->visible_windows++;
	if (curr_scr_win == null(Window *))
	{
		current_screen->window_list_end =
				current_screen->window_list = new;
		if (dumb)
		{
#ifdef SCROLL_AFTER_DISPLAY
			new->display_size = 24 - 1;
#else
			new->display_size = 24;	/* what the hell */
#endif /* SCROLL_AFTER_DISPLAY */
			set_current_window(new);
			return (new);
		}
		recalculate_windows();
		/* resize_display(new,0); */
	}
	else
	{
		/* split current window, or find a better window to split */
		if ((curr_scr_win->display_size < 4) ||
				get_int_var(ALWAYS_SPLIT_BIGGEST_VAR))
		{
			int	size = 0;

			for (tmp = current_screen->window_list; tmp;
					tmp = tmp->next)
			{
				if (tmp->display_size > size)
				{
					size = tmp->display_size;
					biggest = tmp;
				}
			}
			if ((biggest == null(Window *)) || (size < 4))
			{
				say("Not enough room for another window!");
				/* Probably a source of memory leaks */
				new_free(&new);
				current_screen->visible_windows--;
				return (null(Window *));
			}
		}
		else
			biggest = curr_scr_win;
		if (new->prev = biggest->prev)
			new->prev->next = new;
		else
			current_screen->window_list = new;
		new->next = biggest;
		biggest->prev = new;
		new->top = biggest->top;
		new->bottom = (biggest->top + biggest->bottom) / 2;
		biggest->top = new->bottom + 1;
#ifdef SCROLL_AFTER_DISPLAY
		new->display_size = new->bottom - new->top - 1;
		biggest->display_size = biggest->bottom - biggest->top -
			biggest->menu.lines - 1;
#else
		new->display_size = new->bottom - new->top;
		biggest->display_size = biggest->bottom - biggest->top -
			biggest->menu.lines;
#endif /* SCROLL_AFTER_DISPLAY */
		new->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
		biggest->update |= REDRAW_DISPLAY_FULL | REDRAW_STATUS;
	}
	return (new);
}

/*
 * remove_from_window_list: this removes the given window from the list of
 * visible windows.  It closes up the hole created by the windows abnsense in
 * a nice way 
 */
void	remove_from_window_list(window)
Window	*window;
{
	Window	*other;

	/* find adjacent visible window to close up the screen */
	for (other = window->next; other; other = other->next)
	{
		if (other->visible)
		{
			other->top = window->top;
			break;
		}
	}
	if (other == null(Window *))
	{
		for (other = window->prev; other; other = other->prev)
		{
			if (other->visible)
			{
				other->bottom = window->bottom;
				break;
			}
		}
	}
	/* remove window from window list */
	if (window->prev)
		window->prev->next = window->next;
	else
		current_screen->window_list = window->next;
	if (window->next)
		window->next->prev = window->prev;
	else
		current_screen->window_list_end = window->prev;
	if (window->visible)
	{
		current_screen->visible_windows--;
#ifdef SCROLL_AFTER_DISPLAY
		other->display_size = other->bottom - other->top -
			other->menu.lines - 1;
#else
		other->display_size = other->bottom - other->top -
			other->menu.lines;
#endif /* SCROLL_AFTER_DISPLAY */
		if (window == curr_scr_win)
			set_current_window(null(Window *));
		if (window->refnum == current_screen->last_window_refnum)
			current_screen->last_window_refnum =
			    curr_scr_win->refnum;
	}
}

/*
 * window_check_servers: this checks the validity of the open servers vs the
 * current window list.  Every open server must have at least one window
 * associated with it.  If a window is associated with a server that's no
 * longer open, that window's server is set to the primary server.  If an
 * open server has no assicatiate windows, that server is closed.  If the
 * primary server is no more, a new primary server is picked from the open
 * servers 
 */
void	window_check_servers()
{
	Window	*tmp;
	int	flag, cnt, max, i, not_connected,
	prime = -1;

	connected_to_server = 0;
	max = server_list_size();
	for (i = 0; i < max; i++)
	{
		not_connected = !is_server_open(i);
		flag = 1;
		cnt = 0;
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->server == i)
			{
				if (not_connected)
					tmp->server = primary_server;
				else
					prime = tmp->server;
				cnt++;
			}
		}
		if (cnt == 0)
			close_server(i);
		else
			connected_to_server++;
	}
	if (!is_server_open(primary_server))
		primary_server = prime;
	update_all_status();
	cursor_to_input();
}

/*
 * delete_window: This deletes the given window.  It frees all data and
 * structures associated with the window, and it adjusts the other windows so
 * they will display correctly on the screen. 
 */
void	delete_window(window)
Window	*window;
{
	if (window == null(Window *))
		window = curr_scr_win;
	if (window->visible && (current_screen->visible_windows == 1))
	{
		if (invisible_list)
		{
			swap_window(window, invisible_list);
			window = invisible_list;
		}
		else
		{
			say("You can't kill the last window!");
			return;
		}
	}
	if (window->name)
		strmcpy(buffer, window->name, BIG_BUFFER_SIZE);
	else
		sprintf(buffer, "%u", window->refnum);
	new_free(&(window->status_line));
	new_free(&(window->query_nick));
	new_free(&(window->current_channel));
	new_free(&(window->waiting_channel));
	new_free(&(window->logfile));
	new_free(&(window->name));
	free_display(window);
	free_hold(window);
	free_lastlog(window);
	free_nicks(window);
	if (window->visible)
		remove_from_window_list(window);
	else
		remove_from_invisible_list(window);
	new_free(&window);
	window_check_servers();
	do_hook(WINDOW_KILL_LIST, "%s", buffer);
}

/* delete_other_windows: zaps all visible windows except the current one */
static	void	delete_other_windows()
{
	Window	*tmp,
		*cur,
		*next;

	cur = curr_scr_win;
	tmp = current_screen->window_list;
	while (tmp)
	{
		next = tmp->next;
		if (tmp != cur)
			delete_window(tmp);
		tmp = next;
	}
}

/*
 * create_refnum: this generates a reference number for a new window that is
 * not currently is use by another window.  A refnum of 0 is reserved (and
 * never returned by this routine).  Using a refnum of 0 in the message_to()
 * routine means no particular window (stuff goes to CRAP) 
 */
static	unsigned int	create_refnum()
{
	unsigned int	new_refnum = 1;
	Window	*tmp;
	int	done = 0,
		flag;

	while (!done)
	{
		done = 1;
		if (new_refnum == 0)
			new_refnum++;

		flag = 1;
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->refnum == new_refnum)
			{
				done = 0;
				new_refnum++;
				break;
			}
		}
	}
	return (new_refnum);
}

/*
 * window_kill_swap:  Swaps with the last window that was hidden, then
 * kills the window that was swapped.  Give the effect of replacing the
 * current window with the last one, and removing it at the same time.
 */

void	window_kill_swap()
{
	if (invisible_list != null(Window *))
	{
		swap_last_window();
		delete_window(get_window_by_refnum(current_screen->last_window_refnum));
	}
	else
		say("There are no hidden windows!");
}

/*
 * new_window: This creates a new window on the screen.  It does so by either
 * splitting the current window, or if it can't do that, it splits the
 * largest window.  The new window is added to the window list and made the
 * current window 
 */
Window	*new_window()
{
	Window	*new;
	static	int	no_screens = 1;

	if (no_screens)

	{
		set_current_screen(create_new_screen());
		main_screen = current_screen;
		no_screens = 0;
	}
	if (dumb && (current_screen->visible_windows == 1))
		return (null(Window *));
	new = (Window *) new_malloc(sizeof(Window));
	new->refnum = create_refnum();
	if (curr_scr_win)
		new->server = curr_scr_win->server;
	else
		new->server = primary_server;
	new->line_cnt = 0;
#if 0   /* I hate having new windows with level ALL automatically */
	if (current_screen->visible_windows == 0)
		new->window_level = LOG_ALL;
	else
#endif
		new->window_level = LOG_NONE;
	new->hold_mode = get_int_var(HOLD_MODE_VAR);
	new->scroll = get_int_var(SCROLL_VAR);
	new->lastlog_head = null(Lastlog *);
	new->lastlog_tail = null(Lastlog *);
	new->nicks = null(NickList *);
	new->lastlog_level = real_lastlog_level();
	new->name = null(char *);
	new->prompt = null(char *);
	new->lastlog_size = 0;
	new->held = OFF;
	new->last_held = OFF;
	new->current_channel = null(char *);
	new->query_nick = null(char *);
	new->hold_on_next_rite = 0;
	new->status_line = null(char *);
	new->top_of_display = null(Display *);
	new->display_ip = null(Display *);
	new->display_size = 1;
	new->old_size = 1;
	new->hold_head = null(Hold *);
	new->hold_tail = null(Hold *);
	new->held_lines = 0;
	new->scrolled_lines = 0;
	new->next = null(Window *);
	new->prev = null(Window *);
	new->cursor = 0;
	new->visible = 1;
	new->screen = current_screen;
	new->waiting_channel = null(char *);
	new->logfile = null(char *);
	new->log = 0;
	new->log_fp = null(FILE *);
	new->miscflags = 0;
	new->update = 0;
	new->menu.lines = 0;
	new->menu.menu = null(Menu *);
	resize_display(new);
	if (add_to_window_list(new))
		set_current_window(new);
	term_flush();
	return (new);
}

/*
 * unhold_windows: This is used by the main io loop to display held
 * information at an appropriate time.  Each time this is called, each
 * windows hold list is checked.  If there is info in the hold list and the
 * window is not held, the first line is displayed and removed from the hold
 * list.  Zero is returned if no infomation is displayed 
 */
int	unhold_windows()
{
	Window	*tmp;
	char	*stuff;
	int	hold_flag = 0,
		flag = 1;
	int	logged;

	while (tmp = traverse_all_windows(&flag))
	{
		if (!hold_output(tmp) && (stuff = hold_queue(tmp)))
		{
			logged = hold_queue_logged(tmp);
			if (rite(tmp, stuff, 1, 0, 0, logged) == 0)
			{
				remove_from_hold_list(tmp);
				hold_flag = 1;
			}
		}
	}
	return (hold_flag);
}

/*
 * update_window_status: This updates the status line for the given window.
 * If the refresh flag is true, the entire status line is redrawn.  If not,
 * only some of the changed portions are redrawn 
 */
void	update_window_status(window, refresh)
Window	*window;
int	refresh;
{
	if (dumb || (!window->visible) || !status_update_flag || never_connected)
		return;
	if (window == null(Window *))
		window = curr_scr_win;
	if (refresh)
		new_free(&(window->status_line));
	make_status(window);
}

/*
 * redraw_all_status: This redraws all of the status lines for all of the
 * windows. 
 */
void	redraw_all_status()
{
	Window	*tmp;

	if (dumb)
		return;
	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
	{
		new_free(&(tmp->status_line));
		make_status(tmp);
	}
	update_input(UPDATE_JUST_CURSOR);
	term_flush();
}

/*
 * update_all_status: This updates all of the status lines for all of the
 * windows.  By updating, it only draws from changed portions of the status
 * line to the right edge of the screen 
 */
/*ARGSUSED*/
void	update_all_status(unused)
int	unused;
{
	Window	*window;
	Screen	*screen;

	if (dumb || !status_update_flag || never_connected)
		return;
	for (screen = screen_list; screen; screen = screen->next)
	{
		if (!screen->alive)
			continue;
		for (window = screen->window_list;window; window = window->next)
			if (window->visible)
				make_status(window);
	}
	update_input(UPDATE_JUST_CURSOR);
	term_flush();
}

/*
 * status_update: sets the status_update_flag to whatever flag is.  This also
 * calls update_all_status(), which will update the status line if the flag
 * was true, otherwise it's just ignored 
 */
void	status_update(flag)
int	flag;
{
	status_update_flag = flag;
	update_all_status();
	cursor_to_input();
}

/*
 * is_current_channel: Returns true is channel is a current channel for any
 * window.  If the delete flag is set, the unset channel as the current
 * channel for any and all windows 
 */
int	is_current_channel(channel, delete)
char	*channel;
char	delete;
{
	Window	*tmp;
	int	found = 0,
		flag = 1;

	while (tmp = traverse_all_windows(&flag))
	{
		if (tmp->current_channel &&
		    !stricmp(channel, tmp->current_channel) &&
		    (tmp->server == from_server))
		{
			found = 1;
			if (delete)
			{
				new_free(&(tmp->current_channel));
				tmp->update |= UPDATE_STATUS;
			}
		}
	}
	return (found);
}

/*
 * get_window_server: returns the server index for the window with the given
 * refnum 
 */
int	get_window_server(refnum)
unsigned int	refnum;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	return (tmp->server);
}

/*
 * set_window_server:  This sets the server of the given window to server. 
 * If refnum is -1 then we are setting the primary server and all windows
 * that are set to the current primary server are changed to server.  The all
 * flag is ignored in this case.  If refnum is not -1, then that window is
 * set to the given server.  If the all flag is set as well, then all windows
 * with the same server as renum are set to the new server as well 
 */
void	set_window_server(refnum, server, all)
int	refnum;
int	server;
int	all;
{
	Window	*tmp,
		*window;
	int	flag = 1,
		old;

	if (refnum == -1)
	{
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->server == primary_server)
				tmp->server = server;
		}
		window_check_servers();
		primary_server = server;
	}
	else
	{
		if ((window = get_window_by_refnum(refnum)) == null(Window *))
			window = curr_scr_win;
		old = window->server;
		if (all)
		{
			while (tmp = traverse_all_windows(&flag))
			{
				if (tmp->server == old)
					tmp->server = server;
			}
		}
		else
			window->server = server;
		flag = 1;
		window_check_servers();
	}
}

/*
 * set_channel_by_refnum: This sets the current channel for the current
 * window. It returns the current channel as it's value.  If channel is null,
 * the * current channel is not changed, but simply reported by the function
 * result.  This treats as a special case setting the current channel to
 * channel "0".  This frees the current_channel for the
 * current_screen->current_window, * setting it to null 
 */
char	*set_channel_by_refnum(refnum, channel)
unsigned int	refnum;
char	*channel;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	if (channel)
		if (strcmp(channel, "0") == 0)
			channel = null(char *);
	malloc_strcpy(&(tmp->current_channel), channel);
	tmp->waiting_channel = null(char *);
	tmp->update |= UPDATE_STATUS;
	set_channel_window(tmp, channel);
	return (channel);
}

/* get_channel_by_refnum: returns the current channel for window refnum */
char	*get_channel_by_refnum(refnum)
unsigned int refnum;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	return (tmp->current_channel);
}

/* current_refnum: returns the reference number for the current window */
unsigned int current_refnum()
{
	return (curr_scr_win->refnum);
}

/* query_nick: Returns the query nick for the current channel */
char	*query_nick()
{
	return (curr_scr_win->query_nick);
}

/* get_prompt_by_refnum: returns the prompt for the given window refnum */
char	*get_prompt_by_refnum(refnum)
unsigned int refnum;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	if (tmp->prompt)
		return (tmp->prompt);
	else
		return (empty_string);
}

/*
 * get_target_by_refnum: returns the target for the window with the given
 * refnum (or for the current window).  The target is either the query nick
 * or current channel for the window 
 */
char	*get_target_by_refnum(refnum)
unsigned int refnum;
{
	Window	*tmp;

	if ((tmp = get_window_by_refnum(refnum)) == null(Window *))
		tmp = curr_scr_win;
	if (tmp->query_nick)
		return (tmp->query_nick);
	else if (tmp->current_channel)
		return (tmp->current_channel);
	else
		return (null(char *));
}

/* set_query_nick: sets the query nick for the current channel to nick */
void	set_query_nick(nick)
char	*nick;
{
	char	*ptr;
	NickList *tmp;

	if (curr_scr_win->query_nick)
	{
		char	*nick;

		nick = curr_scr_win->query_nick;
		while(nick)
		{
			if (ptr = index(nick,','))
				*(ptr++) = null(char);
			if (tmp = (NickList *)
				remove_from_list(&(curr_scr_win->nicks), nick))
			{
				new_free(&tmp->nick);
				new_free(&tmp);
			}
			nick = ptr;
		}
		new_free(&curr_scr_win->query_nick);
	}
	if (nick)
	{
		malloc_strcpy(&(curr_scr_win->query_nick), nick);
		curr_scr_win->update |= UPDATE_STATUS;
		while(nick)
		{
			if (ptr = index(nick,','))
				*(ptr++) = null(char);
			tmp = (NickList *) new_malloc(sizeof(NickList));
			tmp->nick = null(char *);
			malloc_strcpy(&tmp->nick, nick);
			add_to_list(&(curr_scr_win->nicks),tmp);
			nick = ptr;
		}
	}
	update_window_status(curr_scr_win,0);
}

/*
 * goto_window: This will switch the current window to the window numbered
 * "which", where which is 0 through the number of visible windows on the
 * screen.  The which has nothing to do with the windows refnum. 
 */
static	void	goto_window(which)
int	which;
{
	Window	*tmp;
	int	i;


	if (which == 0)
		return;
	if ((which < 0) || (which > current_screen->visible_windows))
	{
		say("GOTO: Illegal value");
		return;
	}
	tmp = current_screen->window_list;
	for (i = 1; tmp && (i != which); tmp = tmp->next, i++)
		;
	set_current_window(tmp);
}

/*
 * hide_window: sets the given window to invisible and recalculates remaing
 * windows to fill the entire screen 
 */
void	hide_window(window)
Window	*window;
{
	if (current_screen->visible_windows == 1)
	{
		say("You can't hide the last window.");
		return;
	}
	if (window->visible)
	{
		remove_from_window_list(window);
		add_to_invisible_list(window);
#ifdef SCROLL_AFTER_DISPLAY
		window->display_size = LI - 3;
#else
		window->display_size = LI - 2;
#endif /* SCROLL_AFTER_DISPLAY */
		set_current_window(null(Window *));
	}
}

/* hide_other_windows: makes all visible windows but the current one hidden */
void	hide_other_windows()
{
	Window	*tmp,
		*cur,
		*next;

	cur = curr_scr_win;
	tmp = current_screen->window_list;
	while (tmp)
	{
		next = tmp->next;
		if (tmp != cur)
			hide_window(tmp);
		tmp = next;
	}
}

#define WIN_FORM "%%-4s %%-%u.%us %%-%u.%us  %%-%u.%us %%-9.9s %%-10.10s %%s%%s"
static	void	list_a_window(window, len)
Window	*window;
int	len;
{
	char	tmp[10];

	sprintf(tmp, "%-4u", window->refnum);
	sprintf(buffer, WIN_FORM, NICKNAME_LEN, NICKNAME_LEN, len,
			len, get_int_var(CHANNEL_NAME_WIDTH_VAR),
			get_int_var(CHANNEL_NAME_WIDTH_VAR));
	say(buffer, tmp, get_server_nickname(window->server),
			window->name?window->name:"<None>",
			window->current_channel ?
				window->current_channel : "<None>",
			window->query_nick ? window->query_nick : "<None>",
			get_server_itsname(window->server),
			bits_to_lastlog_level(window->window_level),
			(window->visible) ? "" : " Hidden");
}

/*
 * list_windows: This Gives a terse list of all the windows, visible or not,
 * by displaying their refnums, current channel, and current nick 
 */
static	void	list_windows()
{
	Window	*tmp;
	int	flag = 1;
	int	len = 4;
#if 0
	Window	*win;
#endif

	while (tmp = traverse_all_windows(&flag))
	{
		if (tmp->name && (strlen(tmp->name) > len))
			len = strlen(tmp->name);
	}
	sprintf(buffer, WIN_FORM, NICKNAME_LEN, NICKNAME_LEN, len, len,
		get_int_var(CHANNEL_NAME_WIDTH_VAR),
		get_int_var(CHANNEL_NAME_WIDTH_VAR));
	say(buffer, "Ref", "Nick", "Name", "Channel", "Query", "Server",
		"Level", empty_string);
	flag = 1;
	while (tmp = traverse_all_windows(&flag))
		list_a_window(tmp, len);
#if 0
	win = NULL
	while (new_window_traverse(&win, &flag))
		list_a_window(win, len);
#endif
}

/* show_window: This makes the given window visible.  */
static	void	show_window(window)
Window	*window;
{
	if (window->visible)
	{
		set_current_window(window);
		return;
	}
	remove_from_invisible_list(window);
	if (add_to_window_list(window))
		set_current_window(window);
	else
		add_to_invisible_list(window);
}

/* push_window_by_refnum:  This pushes the given refnum onto the window stack */
static	void	push_window_by_refnum(refnum)
unsigned int refnum;
{
	WindowStack *new;

	new = (WindowStack *) new_malloc(sizeof(WindowStack));
	new->refnum = refnum;
	new->next = current_screen->window_stack;
	current_screen->window_stack = new;
}

/*
 * pop_window: this pops the top entry off the window stack and sets the
 * current window to that window.  If the top of the window stack is no
 * longer a valid window, then that entry is discarded and the next entry
 * down is tried (as so on).  If the stack is empty, the current window is
 * left unchanged 
 */
static	void	pop_window()
{
	int	refnum;
	WindowStack *tmp;
	Window	*win;

	while (1)
	{
		if (current_screen->window_stack)
		{
			refnum = current_screen->window_stack->refnum;
			tmp = current_screen->window_stack->next;
			new_free(&current_screen->window_stack);
			current_screen->window_stack = tmp;
			if (win = get_window_by_refnum(refnum))
			{
				if (!win->visible)
					show_window(win);
				else
					set_current_window(win);
				break;
			}
		}
		else
		{
			say("The window stack is empty!");
			break;
		}
	}
}

/*
 * show_stack: displays the current window stack.  This also purges out of
 * the stack any window refnums that are no longer valid 
 */
static	void	show_stack()
{
	WindowStack *last = null(WindowStack *),
	    *tmp, *crap;
	Window	*win;
	int	flag = 1;
	int	len = 4;

	while (win = traverse_all_windows(&flag))
	{
		if (win->name && (strlen(win->name) > len))
			len = strlen(win->name);
	}
	say("Window stack:");
	tmp = current_screen->window_stack;
	while (tmp)
	{
		if (win = get_window_by_refnum(tmp->refnum))
		{
			list_a_window(win, len);
			tmp = tmp->next;
		}
		else
		{
			crap = tmp->next;
			new_free(&tmp);
			if (last)
				last->next = crap;
			else
				current_screen->window_stack = crap;
			tmp = crap;
		}
	}
}

/*
 * is_window_name_unique: checks the given name vs the names of all the
 * windows and returns true if the given name is unique, false otherwise 
 */
static	int	is_window_name_unique(name)
char	*name;
{
	Window	*tmp;
	int	flag = 1;

	if (name)
	{
		while (tmp = traverse_all_windows(&flag))
		{
			if (tmp->name && (stricmp(tmp->name, name) == 0))
				return (0);
		}
	}
	return (1);
}

/*
 * add_nicks_by_refnum:  This adds the given str to the nicklist of the
 * window refnum.  Only unique names are added to the list.  If the name is
 * preceeded by a ^ it is removed from the list.  The str may be a comma
 * separated list of nicks, channels, etc 
 */
static	void	add_nicks_by_refnum(refnum, str)
unsigned int refnum;
char	*str;
{
	Window	*tmp;
	char	*ptr;
	NickList *new;

	if (tmp = get_window_by_refnum(refnum))
	{
		while (str)
		{
			if (ptr = index(str, ','))
				*(ptr++) = null(char);
			if (*str == '^')
			{
				if (new = (NickList *)
					remove_from_list(&(tmp->nicks),str + 1))
				{
					say("%s removed from window name list",
						new->nick);
					new_free(&new->nick);
					new_free(&new);
				}
				else
					say("%s is not on the list for this \
window!", str + 1);
			}
			else
			{
				if (!find_in_list(&(tmp->nicks), str,
						!USE_WILDCARDS))
				{
					say("%s add to window name list", str);
					new = (NickList *)
						new_malloc(sizeof(NickList));
					new->nick = null(char *);
					malloc_strcpy(&new->nick, str);
					add_to_list(&(tmp->nicks), new);
				}
				else
					say("%s already on window name list",
						str);
			}
			str = ptr;
		}
	}
	else
		say("No such window!");
}

/* below is stuff used for parsing of WINDOW command */

/*
 * get_window_by_name: returns a pointer to a window with a matching logical
 * name or null if no window matches 
 */
Window	*get_window_by_name(name)
char	*name;
{
	Window	*tmp;
	int	flag = 1;

	while (tmp = traverse_all_windows(&flag))
	{
		if (tmp->name && (stricmp(tmp->name, name) == 0))
			return (tmp);
	}
	return (null(Window *));
}

/*
 * get_window: this parses out any window (visible or not) and returns a
 * pointer to it 
 */
static	Window	*get_window(name, args)
char	*name;
char	**args;
{
	char	*arg;
	Window	*tmp;

	if (arg = next_arg(*args, args))
	{
		if (is_number(arg))
		{
			if (tmp = get_window_by_refnum(atoi(arg)))
				return (tmp);
		}
		if (tmp = get_window_by_name(arg))
			return (tmp);
		say("%s: No such window: %s", name, arg);
	}
	else
		say("%s: Please specify a window refnum or name", name);
	return (null(Window *));
}

/*
 * get_invisible_window: parses out an invisible window by reference number.
 * Returns the pointer to the window, or null.  The args can also be "LAST"
 * indicating the top of the invisible window list (and thus the last window
 * made invisible) 
 */
static	Window	*get_invisible_window(name, args)
char	*name;
char	**args;
{
	char	*arg;
	Window	*tmp;

	if (arg = next_arg(*args, args))
	{
		if (strnicmp(arg, "LAST", strlen(arg)) == 0)
		{
			if (invisible_list == null(Window *))
				say("%s: There are no hidden windows", name);
			return (invisible_list);
		}
		if (tmp = get_window(name, &arg))
		{
			if (!tmp->visible)
				return (tmp);
			else
			{
				if (tmp->name)
					say("%s: Window %s is not hidden!",
						name, tmp->name);
				else
					say("%s: Window %d is not hidden!",
						name, tmp->refnum);
			}
		}
	}
	else
		say("%s: Please specify a window refnum or LAST", name);
	return (null(Window *));
}

#ifdef THIS_NEVER_GETS_USED
/*
 * get_visible_window: parses out a window by reference number.  Returns the
 * pointer to thw window if it exists and if it is visible. 
 */
static	Window	*get_visible_window(name, args)
char	*name;
char	**args;
{
	char	*arg;
	Window	*tmp;

	if (arg = next_arg(*args, args))
	{
		if (strnicmp(arg, "LAST", strlen(arg)) == 0)
			tmp = get_window_by_refnum(current_screen->last_window_refnum);
		else
			tmp = get_window(name, &arg);
		if (tmp)
		{
			if (tmp->visible)
				return (tmp);
			else
			{
				if (tmp->name)
					say("%s: Window %s is hidden!", name, tmp->name);
				else
					say("%s: Window %d is hidden!", name, tmp->refnum);
			}
		}
	}
	else
		say("%s: Please specify a window refnum or LAST", name);
	return (null(Window *));
}
#endif

/* get_number: parses out an integer number and returns it */
static	int	get_number(name, args)
char	*name;
char	**args;
{
	char	*arg;

	if (arg = next_arg(*args, args))
		return (atoi(arg));
	else
		say("%s: You must specify the number of lines", name);
	return (0);
}

/*
 * get_boolean: parses either ON, OFF, or TOGGLE and sets the var
 * accordingly.  Returns 0 if all went well, -1 if a bogus or missing value
 * was specified 
 */
static	int	get_boolean(name, args, var)
char	*name;
char	**args;
int	*var;
{
	char	*arg;

	if (((arg = next_arg(*args, args)) == null(char *)) ||
	    do_boolean(arg, var))
	{
		say("Value for %s must be ON, OFF, or TOGGLE", name);
		return (-1);
	}
	else
	{
		say("Window %s is %s", name, var_settings[*var]);
		return (0);
	}
}

#ifdef WINDOW_CREATE
static	Window	*create_additional_screen()
{
	Window	*win;
	Screen	*screen,
		*oldscreen;
	char	*termvar,
		*styvar;
	int	ScreenType = ST_NOTHING;
	char	PipeName[BIG_BUFFER_SIZE+1];
	struct	sockaddr_un *sockaddr = (struct sockaddr_un *) PipeName,
			NewSock;
	int	NsZ;
	int	s;
	fd_set	fd_read;
	struct	timeval	timeout;

	/*
	 * Environment variable STY has to be set for screen to work..  so it is
	 * the best way to check screen..  regardless of what TERM is, the 
	 * execpl() for screen won't just open a new window if STY isn't set,
	 * it will open a new screen process, and run the wserv in its first
	 * window, not what we want...  -phone
	 */
	if (styvar = getenv("STY"))
		ScreenType = ST_SCREEN;
	else if (termvar = getenv("TERM"))
	{
						/* OpenWindows Brain Damage */
		if (!strncmp(termvar, "sun", 3))
			ScreenType = ST_XTERM;
		else
					/* Mainly for AIX where it's aixterm */
			for(;*termvar; termvar++)
				if (!strncmp(termvar, "xterm", 5))
				{
					ScreenType = ST_XTERM;
					break;
				}
	}
	if (ScreenType == ST_NOTHING)
	{
		say("I don't know how to create new windows for this terminal");
		return NULL;
	}
	say("Opening new screen...");
	sprintf(PipeName + sizeof(sockaddr->sun_family), "/tmp/irc_%08d",
		getpid());
	sockaddr->sun_family = AF_UNIX;
	s = socket(AF_UNIX, SOCK_STREAM, 0);
	bind(s, (struct sockaddr *) sockaddr, strlen(sockaddr->sun_path) +
		sizeof(sockaddr->sun_family));
	listen(s, 1);
	oldscreen = current_screen;
	set_current_screen(create_new_screen());
	if (fork() == 0)
	{
		setuid(getuid());
		setgid(getgid());
	/*
	 * Unlike most other cases, it is important here to close down
	 * *ALL* unneeded file descriptors. Failure to do so can cause
	 * Things like server and DCC connections to fail to close on
	 * request. This isn't a problem with "screen", but is with X.
	 */
		close(s);
		for (screen = screen_list; screen && screen != current_screen;
				screen = screen->next)
			if (screen->alive && screen->fdin != 0)
				new_close(screen->fdin);
		close_all_dcc();
		close_all_exec();
		close_all_server();
		if (ScreenType == ST_SCREEN)
			execlp("screen", "screen", WSERV_PATH,
				sockaddr->sun_path, 0);
		else if (ScreenType == ST_XTERM)
			execlp("xterm", "xterm", "-e", WSERV_PATH,
				sockaddr->sun_path, 0);
		exit(1);
	}
	NsZ = sizeof(NewSock);
	FD_ZERO(&fd_read);
	FD_SET(s, &fd_read);
	timeout.tv_sec = (long) 5;
	switch (select(NFDBITS , &fd_read, null(fd_set *), null(fd_set *),
			&timeout))
	{
	case -1:
	case 0:
		kill_screen(current_screen);
		last_input_screen = oldscreen;
		set_current_screen(oldscreen);
		return null(Window *);
		break;
	case 1:
		current_screen->fdin = current_screen->fdout =
			accept(s, (struct sockaddr *) &NewSock, &NsZ);
		current_screen->fpin = current_screen->fpout =
			fdopen(current_screen->fdin, "r+");
		close(s);
		unlink(sockaddr->sun_path);
		win = new_window();
		(void) refresh_screen(0, null(char *));
		set_current_screen(oldscreen);
		return win;
		break;
	}
}
#endif /* WINDOW_CREATE

/*ARGSUSED*/
void	window(command, args)
char	*command,
	*args;
{
	int	len;
	char	*arg;
	int	no_args = 1;
	Window	*window,
		*tmp;

	in_window_command = 1;
	message_from(null(char *), LOG_CURRENT);
	window = curr_scr_win;
	while (arg = next_arg(args, &args))
	{
		no_args = 0;
		len = strlen(arg);
		if (strnicmp("NEW", arg, len) == 0)
		{
			if (tmp = new_window())
				window = tmp;
		}
#ifdef WINDOW_CREATE
		else if (strnicmp("CREATE", arg, len) == 0)
		{
			if (tmp = create_additional_screen())
				window = tmp;
			else
				say("Cannot create new screen!");
		}
		else if (!strnicmp("DELETE", arg, len))
		{
			kill_screen(current_screen);
		}
#endif /* WINODW_CREATE */
		else if (strnicmp("REFNUM", arg, len) == 0)
		{
			if (tmp = get_window("REFNUM", &args))
			{
				if (tmp->screen && tmp->screen !=window->screen)
					say("Window in another screen!");
				else if (tmp->visible)
				{ 
					set_current_window(tmp);
					window = tmp;
				}
				else
					say("Window not visible!");
			}
			else
			{
				say("No such window!");
				in_window_command = 0;
				return;
			}
		}
		else if (strnicmp("KILL", arg, len) == 0)
			delete_window(window);
		else if (strnicmp("SHRINK", arg, len) == 0)
			grow_window(window, -get_number("SHRINK", &args));
		else if (strnicmp("GROW", arg, len) == 0)
			grow_window(window, get_number("SHRINK", &args));
		else if (strnicmp("SCROLL", arg, len) == 0)
			get_boolean("SCROLL", &args, &(window->scroll));
		else if (strnicmp("LOG", arg, len) == 0)
		{
			if (get_boolean("LOG", &args, &(window->log)))
				break;
			else
			{
				char	*logfile;
				int	add_ext = 1;

				if (logfile = window->logfile)
					add_ext = 0;
				else if (!(logfile = get_string_var(LOGFILE_VAR)))
					logfile = empty_string;
				if (!add_ext)
					sprintf(buffer, "%s", logfile);
				else if (window->current_channel)
					sprintf(buffer, "%s.%s", logfile,
						window->current_channel);
				else if (window->query_nick)
					sprintf(buffer, "%s.%s", logfile,
						window->query_nick);
				else
					sprintf(buffer, "%s.Window_%d", logfile,
						window->refnum);
				window->log_fp = do_log(window->log, buffer,
					window->log_fp);
				if (window->log_fp == null(FILE *))
					window->log = 0;
			}
		}
		else if (strnicmp("HOLD_MODE", arg, len) == 0)
		{
			if (get_boolean("HOLD_MODE",&args,&(window->hold_mode)))
				break;
			else
				set_int_var(HOLD_MODE_VAR, window->hold_mode);
		}
		else if (strnicmp("LASTLOG_LEVEL", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				window->lastlog_level =parse_lastlog_level(arg);
				say("Lastlog level is %s",
				  bits_to_lastlog_level(window->lastlog_level));
			}
			else
				say("Level required");
		}
		else if (strnicmp("LEVEL", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				window->window_level = parse_lastlog_level(arg);
				say("Window level is %s",
				   bits_to_lastlog_level(window->window_level));
				revamp_window_levels(window);
			}
			else
				say("LEVEL: Level required");
		}
		else if (strnicmp("BALANCE", arg, len) == 0)
			recalculate_windows();
		else if (strnicmp("NAME", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				if (is_window_name_unique(arg))
				{
					malloc_strcpy(&(window->name), arg);
					window->update |= UPDATE_STATUS;
				}
				else
					say("%s is not unique!", arg);
			}
			else
				say("You must specify a name for the window!");
		}
		else if (strnicmp("PROMPT", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				malloc_strcpy(&(window->prompt), arg);
				window->update |= UPDATE_STATUS;
			}
			else
			    say("You must specify a prompt for the window!");
		}
		else if (strnicmp("GOTO", arg, len) == 0)
		{
			goto_window(get_number("GOTO", &args));
			window = curr_scr_win;
		}
		else if (strnicmp("LAST", arg, len) == 0)
			set_current_window(null(Window *));
		else if (strnicmp("MOVE", arg, len) == 0)
		{
			move_window(window, get_number("MOVE", &args));
			window = curr_scr_win;
		}
		else if (strnicmp("SWAP", arg, len) == 0)
		{
			if (tmp = get_invisible_window("SWAP", &args))
				swap_window(window, tmp);
		}
		else if (strnicmp("HIDE", arg, len) == 0)
			hide_window(window);
		else if (strnicmp("PUSH", arg, len) == 0)
			push_window_by_refnum(window->refnum);
		else if (strnicmp("POP", arg, len) == 0)
			pop_window();
		else if (strnicmp("ADD", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				add_nicks_by_refnum(window->refnum, arg);
			}
			else
				say("ADD: Do something!  Geez!");
		}
		else if (strnicmp("STACK", arg, len) == 0)
			show_stack();
		else if (strnicmp("LIST", arg, len) == 0)
			list_windows();
		else if (strnicmp("SERVER", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				int	i,
					port_num;
				char	*port,
					*password,
					*nick;

				parse_server_info(arg, &port, &password, &nick);
				if (port)
					port_num = atoi(port);
				else
					port_num = -1;
				if ((i = find_in_server_list(arg, port_num))
						!= -1)
					port_num = server_list[i].port;
				if ((i = parse_server_index(arg))
						== -1)
				{
					if (nick && *nick)
					    malloc_strcpy(&connect_next_nick,
						nick);
					if (!connect_to_server(arg, port_num,
							-1))
					{
					    set_window_server(window->refnum,
						from_server, 0);
					    window->window_level = LOG_ALL;
					    if (window->current_channel)
					      new_free(window->current_channel);
					}
				}
				else
				{
					if (nick && *nick)
						malloc_strcpy(&(
						    server_list[i].nickname),
						    nick);
					if(!connect_to_server(get_server_name(i)
						, port_num, -1))
					{
					    set_window_server(window->refnum,
						from_server, 0);
					    window->window_level = LOG_ALL;
					    clear_channel_list(from_server);
					    if (window->current_channel)
					      new_free(window->current_channel);
					}
				}
			}
			else
				say("SERVER: You must specify a server");
		}
		else if (strnicmp("SHOW", arg, len) == 0)
		{
			if (tmp = get_window("SHOW", &args))
			{
				show_window(tmp);
				window = curr_scr_win;
			}
		}
		else if (strnicmp("HIDE_OTHERS", arg, len) == 0)
			hide_other_windows();
		else if (strnicmp("KILL_OTHERS", arg, len) == 0)
			delete_other_windows();
		else if (strnicmp("NOTIFY", arg, len) == 0)
		{
			window->miscflags ^= WINDOW_NOTIFY;
			say("Notification when hidden set to %s",
			    (window->miscflags &WINDOW_NOTIFY)? "ON" : "OFF");
		}
		else if (strnicmp("CHANNEL", arg, len) == 0)
		{
			if (arg = next_arg(args, &args))
			{
				if (is_current_channel(arg, 1))
				{
					say("You are now talking to channel %s",
						arg);
					set_channel_by_refnum(0, arg);
				}
				else
				{
					if (is_on_channel(arg, nickname))
					{
						say("You are now talking to \
channel %s", arg);
						set_channel_by_refnum(0, arg);
					}
					else if ((*arg == '0') && (*(arg + 1) = 
							null(char)))
						set_channel_by_refnum(0,
							null(char *));
					else
					{
						int	server;

						server = from_server;
						from_server = window->server;
						if (get_server_version(
						  window->server) > Server2_5)
						    send_to_server("JOIN %s",
							arg);
						else
							send_to_server(
							"CHANNEL %s", arg);
						malloc_strcpy(
						    &window->waiting_channel,
							arg);
						from_server = server;
					}
				}
			}
			else
				set_channel_by_refnum(0, "0");
		}
		else if (strnicmp("PREVIOUS", arg, len) == 0)
		{
			swap_previous_window();
		}
		else if (strnicmp("NEXT", arg, len) == 0)
		{
			swap_next_window();
		}
		else if (strnicmp("BACK", arg, len) == 0)
		{
			back_window();
		}
		else if (strnicmp("KILLSWAP", arg, len) == 0)
		{
			window_kill_swap();
		}
		else if (!strnicmp("LOGFILE", arg, len))
		{
			if (arg = next_arg(args, &args))
			{
				malloc_strcpy(&window->logfile, arg);
				say("Window LOGFILE set to %s", arg);
			}
			else
				say("No LOGFILE given");
		}
		else
			say("Unknown WINDOW command: %s", arg);
	}
	if (no_args)
	{
		if (window->name)
			say("Window %s (%u)", window->name, window->refnum);
		else
			say("Window %u", window->refnum);
		if (window->server == -1)
			say("\tServer: <None>");
		else
			say("\tServer: %s", get_server_name(window->server));
		say("\tCurrent channel: %s", window->current_channel ?
			window->current_channel : "<None>");
		say("\tQuery User: %s", (window->query_nick ?
			window->query_nick : "<None>"));
		say("\tPrompt: %s", window->prompt ?
			window->prompt : "<None>");
		say("\tScrolling is %s", var_settings[window->scroll]);
		say("\tLogging is %s", var_settings[window->log]);
		if (window->logfile)
			say("\tLogfile is %s", window->logfile);
		else
			say("\tNo logfile given");
		say("\tNotification is %s",
			var_settings[window->miscflags&WINDOW_NOTIFY]);
		say("\tHold mode is %s", var_settings[window->hold_mode]);
		say("\tWindow level is %s",
			bits_to_lastlog_level(window->window_level));
		say("\tLastlog level is %s",
			bits_to_lastlog_level(window->lastlog_level));
		if (window->nicks)
		{
			NickList *tmp;

			say("\tName list:");
			for (tmp = window->nicks; tmp; tmp = tmp->next)
				say("\t  %s", tmp->nick);
		}
	}
	in_window_command = 0;
	message_from(null(char *), LOG_CRAP);
	update_all_windows();
	cursor_to_input();
}

int	number_of_windows()
{
	return (current_screen->visible_windows);
}

void	unstop_all_windows()
{
	Window	*tmp;

	for (tmp = current_screen->window_list; tmp; tmp = tmp->next)
		hold_mode(tmp, OFF, 1);
}

/* this will make underline toggle between 2 and -1 and never let it get to 0 */
void	set_underline_video(value)
int	value;
{
	if (value == OFF)
		underline = -1;
	else
		underline = 1;
}

static	char	*next_line_back(window)
Window	*window;
{
	static	int	row;
	static	Lastlog	*LogLine;
	char	**TheirLines;
	static	char	*ScreenLines[MAXIMUM_SPLITS] =
	{ 
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *),
		null(char *), null(char *), null(char *), null(char *)
	};

	if (window)

	{
		LogLine = window->lastlog_head;
		row = -1;
	}
	if (row<=0)

	{
		for (row = 0; ScreenLines[row]; row++)
			new_free(&ScreenLines[row]);
		if (!window && LogLine)
			LogLine = LogLine->next;
		if (!LogLine)
			return null(char *);
		TheirLines = split_up_line(LogLine->msg);
		for (row = 0; TheirLines[row]; row++)
		{
			ScreenLines[row] = TheirLines[row];
			TheirLines[row] = NULL;
		}
		if (window)
			return NULL;
	}
	return ScreenLines[--row];
}

static	void	display_lastlog_lines(start, end, window)
int	start,
	end;
Window	*window;
{
	Display	*Disp;
	char	*Line;
	int	i;

	next_line_back(window);
	for (i = 0, Disp = window->top_of_display; i < window->display_size;
			Disp = Disp->next, i++)
		if (Disp->linetype)
			next_line_back(null(Window *));
	for (i = 0; i < start; i++)
		next_line_back(null(Window *));
	for (; i < end; i++)
	{
		if (!(Line = next_line_back(null(Window *))))
			break;
		term_move_cursor(0, window->top + window->menu.lines +
			window->scrolled_lines-i-1);
		rite(window, Line, 0, 0, 1, 0);
	}
}

void	scrollback_backwards_lines(ScrollDist)
int	ScrollDist;
{
	Window	*window;

	window = curr_scr_win;
	if (!window->scrolled_lines && !window->scroll)
	{
		term_beep();
		return;
	}
	window->scrolled_lines += ScrollDist;

	term_scroll(window->top + window->menu.lines,
			window->top + window->menu.lines +
			window->display_size - 1, -ScrollDist);

	display_lastlog_lines(window->scrolled_lines-ScrollDist,
			window->scrolled_lines, window);
	cursor_not_in_display();
	update_input(UPDATE_JUST_CURSOR);
}

void	scrollback_forwards_lines(ScrollDist)
{
	Window	*window;

	window = curr_scr_win;
	if (!window->scrolled_lines)
	{
		term_beep();
		return;
	}
	ScrollDist = window->display_size / 2;
	if (ScrollDist > window->scrolled_lines)
		ScrollDist = window->scrolled_lines;

	window->scrolled_lines -= ScrollDist;
	term_scroll(window->top + window->menu.lines,
	    window->top + window->menu.lines + window->display_size - 1,
	    ScrollDist);

	if (window->scrolled_lines < window->display_size)
		redraw_window(window, ScrollDist + window->scrolled_lines-
				window->display_size);

	display_lastlog_lines(window->scrolled_lines-window->display_size,
	    window->scrolled_lines-window->display_size + ScrollDist,
	    window);
	cursor_not_in_display();
	update_input(UPDATE_JUST_CURSOR);

	if (!window->scrolled_lines)
	{
		if (curr_scr_win->hold_mode)
			hold_mode(curr_scr_win, ON, 1);
		else
			hold_mode(curr_scr_win, OFF, 0);
	}
}

void	scrollback_forwards()
{
	scrollback_forwards_lines(curr_scr_win->display_size/2);
}

void	scrollback_backwards()
{
	scrollback_backwards_lines(curr_scr_win->display_size/2);
}


void	scrollback_end()
{
	if (!curr_scr_win->scrolled_lines)

	{
		term_beep();
		return;
	}
	if (curr_scr_win->scrolled_lines<
	    curr_scr_win->display_size)
		scrollback_forwards(curr_scr_win->scrolled_lines);
	else

	{
		curr_scr_win->scrolled_lines = 0;
		redraw_window(curr_scr_win, 1);
		cursor_not_in_display();
		update_input(UPDATE_JUST_CURSOR);
		if (curr_scr_win->hold_mode)
			hold_mode(curr_scr_win, ON, 1);
		else
			hold_mode(curr_scr_win, OFF, 0);
	}
}
