/*
 *	AX.25 release 029
 *
 *	This is ALPHA test software. This code may break your machine, randomly fail to work with new 
 *	releases, misbehave and/or generally screw up. It might even work. 
 *
 *	This code REQUIRES 1.2.1 or higher/ NET3.029
 *
 *	This module:
 *		This module is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 *	Most of this code is based on the SDL diagrams published in the 7th
 *	ARRL Computer Networking Conference papers. The diagrams have mistakes
 *	in them, but are mostly correct. Before you modify the code could you
 *	read the SDL diagrams as the code is not obvious and probably very
 *	easy to break;
 *
 *	History
 *	AX.25 029	Alan(GW4PTS) Switched to KA9Q constant names. Removed
 *			old BSD code.
 */
 
#include <linux/config.h>
#ifdef CONFIG_AX25
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include "ax25.h"
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include "sock.h"
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/interrupt.h>

/*
 * This routine purges the input queue of frames.
 */
 
void ax25_clear_tx_queue(ax25_socket *sk)
{
	struct sk_buff *skb;

	while ((skb = skb_dequeue(&sk->write_queue)) != NULL) {
		skb->free = 1;
		skb->sk   = sk;
		kfree_skb(skb, FREE_WRITE);
	}

	while ((skb = skb_dequeue(&sk->ax25_ack_queue)) != NULL) {
		skb->free = 1;
		skb->sk   = sk;
		kfree_skb(skb, FREE_WRITE);
	}
}

/*
 * This routine purges the input queue of those frames that have been
 * acknowledged. This replaces the boxes labelled "V(a) <- N(r)" on the
 * SDL diagram.
 */
 
void ax25_frames_acked(ax25_socket *sk, unsigned char nr)
{
	struct sk_buff *skb, *skb_prev = NULL;
	int n = 0;

	/*
	 * Remove all the ack-ed frames from the ack queue.
	 */
	if (sk->ax25_va != nr) 
	{
		while (skb_peek(&sk->ax25_ack_queue) != NULL && sk->ax25_va != nr) {
		        skb = skb_dequeue(&sk->ax25_ack_queue);
			skb->free = 1;
			skb->sk   = sk;
			kfree_skb(skb, FREE_WRITE);
			sk->ax25_va = (sk->ax25_va + 1) % MODULUS;
			n++;
		}

		if (sk->debug)
			printk("S%d: purged %d frame(s) from the ack queue\n", sk->ax25_state, n);
	}

	/*
	 * Requeue all the un-ack-ed frames on the output queue to be picked
	 * up by ax25_kick called from the timer. This arrangement handles the
	 * possibility of an empty output queue.
	 */
	n = 0;

	while ((skb = skb_dequeue(&sk->ax25_ack_queue)) != NULL) {
		if (skb_prev == NULL)
			skb_queue_head(&sk->write_queue, skb);
		else
			skb_append(skb_prev, skb);
		skb_prev = skb;
		n++;
	}

	if (sk->debug && n > 0)
		printk("S%d: requeued %d frame(s) to the tx queue\n", sk->ax25_state, n);
}

/*
 *	Validate that the value of nr is between va and vs. Return true or
 *	false for testing.
 */
 
int ax25_validate_nr(ax25_socket *sk, unsigned char nr)
{
	unsigned char vc = sk->ax25_va;

	while (vc != sk->ax25_vs) {
		if (nr == vc) return 1;
		vc = (vc + 1) % MODULUS;
	}
	
	if (nr == sk->ax25_vs) return 1;

	return 0;
}

int ax25_decode(ax25_socket *sk, unsigned char *frame)
{
	int frametype = ILLEGAL;

	if ((frame[0]&S) == 0)
		frametype = I;		/* I frame - carries NR/NS/PF */
	else if ((frame[0]&U) == 1) 	/* S frame - take out PF/NR */
		frametype=frame[0]&0x0F;
	else if ((frame[0]&U) == 3) 	/* U frame - take out PF */
		frametype=frame[0]&~PF;
	return(frametype);
}

/* 
 *  This routine is called when the HDLC layer internally  generates a
 *  command or  response  for  the remote machine ( eg. RR, UA etc. ). 
 *  Only supervisory or unnumbered frames are processed.
 */
void ax25_send_control(ax25_socket *sk, int frametype, int type)
{
	struct sk_buff     *skb;
	unsigned char      *dptr;
	int len;
	struct device *dev;
	
	if ((dev = sk->ax25_device) == NULL)
		return;	/* Route died */

	if ((skb = alloc_skb(16 + 1 + size_ax25_addr(sk->ax25_digipeat), GFP_ATOMIC)) == NULL)
		return;

	skb->sk = sk;
	
	sk->wmem_alloc += skb->mem_len;
	dptr = skb->data;
	
	dptr += 1 + size_ax25_addr(sk->ax25_digipeat);	/* KISS byte & 2 calls */

	/* Assume a response - address structure for DTE */
	len = 1;		/* Normal size */
	
	if((frametype&U)==S)		/* S frames carry NR */
		frametype|=(sk->ax25_vr<<5);
	*dptr=frametype;
	
	skb->free = 1;
	skb->len  = len + size_ax25_addr(sk->ax25_digipeat) + 1;
	ax25_transmit_buffer(sk, skb, type);
}

/*
 *	Send a 'DM' to an unknown connection attempt, or an invalid caller.
 *
 *	Note: src here is the sender, thus its the target of the DM
 */
 
void ax25_return_dm(struct device *dev, ax25_address *src, ax25_address *dest, ax25_digi *digi)
{
	struct sk_buff *skb;
	char *dptr;
	ax25_digi retdigi;
	int len = 2 + size_ax25_addr(digi);

	if ((skb = alloc_skb(len, GFP_ATOMIC)) == NULL)
		return;	/* Next SABM will get DM'd */

	skb->len = len;

	ax25_digi_invert(digi, &retdigi);

	dptr = skb->data + 1 + size_ax25_addr(digi);
	skb->sk = NULL;
	
	dptr[0] = DM;

	if (dev == NULL)
		return;

	dptr    = skb->data;
	*dptr++ = 0;
	dptr   += build_ax25_addr(dptr, dest, src, &retdigi, C_RESPONSE);

	skb->arp  = 1;
	skb->free = 1;

	dev_queue_xmit(skb, dev, SOPRI_NORMAL);
}
	

/*
 *	Digipeated address processing
 */
 

/*
 *	Given an AX.25 address pull of to, from, digi list, command/response and the start of data
 *
 */

unsigned char *ax25_parse_addr(unsigned char *buf,int len, ax25_address *src, ax25_address *dest, ax25_digi *digi, int *flags)
{
	int d = 0;
	
	if (len < 14) return NULL;
		
	if (flags != NULL) {
		*flags = 0;
	
		if (buf[6] & LAPB_C) {
			*flags = C_COMMAND;
		}
		if (buf[13] & LAPB_C) {
			*flags = C_RESPONSE;
		}
	}
		
	/* Copy to, from */
	if (dest != NULL) memcpy(dest, buf + 0, 7);
	if (src != NULL)  memcpy(src,  buf + 7, 7);
	buf += 14;
	len -= 14;
	digi->lastrepeat = -1;
	digi->ndigi      = 0;
	
	while (!(buf[-1] & LAPB_E))
	{
		if (d >= 6)  return NULL;	/* Max of 6 digis */
		if (len < 7) return NULL;	/* Short packet */

		if (digi != NULL) {
			memcpy(&digi->calls[d], buf, 7);
			digi->ndigi = d + 1;
			if (buf[6] & AX25_REPEATED) {
				digi->repeated[d] = 1;
				digi->lastrepeat  = d;
			} else {
				digi->repeated[d] = 0;
			}
		}

		buf += 7;
		len -= 7;
		d++;
	}

	return buf;
}

/*
 *	Assemble an AX.25 header from the bits
 */
		
int build_ax25_addr(unsigned char *buf, ax25_address *src, ax25_address *dest, ax25_digi *d, int flag)
{
	int len = 0;
	int ct  = 0;

	memcpy(buf, dest, 7);
	
	if (flag != C_COMMAND && flag != C_RESPONSE)
		printk("build_ax25_addr: Bogus flag %d\n!", flag);
	buf[6] &= ~(LAPB_E | LAPB_C);
	buf[6] |= SSID_SPARE;

	if (flag == C_COMMAND) buf[6] |= LAPB_C;

	buf += 7;
	len += 7;
	memcpy(buf, src, 7);
	buf[6] &= ~(LAPB_E | LAPB_C);
	buf[6] |= SSID_SPARE;

	if (flag == C_RESPONSE) buf[6] |= LAPB_C;
	/*
	 *	Fast path the normal digiless path
	 */
	if (d == NULL || d->ndigi == 0) {
		buf[6] |= LAPB_E;
		return 14;
	}	
	
	buf += 7;
	len += 7;
	
	while (ct < d->ndigi) {
		memcpy(buf, &d->calls[ct], 7);
		if (d->repeated[ct])
			buf[6] |= AX25_REPEATED;
		else
			buf[6] &= ~AX25_REPEATED;
		buf[6] &= ~LAPB_E;
		buf[6] |= SSID_SPARE;

		buf += 7;
		len += 7;
		ct++;
	}

	buf[-1] |= LAPB_E;
	
	return len;
}

int size_ax25_addr(ax25_digi *dp)
{
	if (dp == NULL)
		return 14;

	return 14 + (7 * dp->ndigi);
}
	
/* 
 *	Reverse Digipeat List. May not pass both parameters as same struct
 */	

void ax25_digi_invert(ax25_digi *in, ax25_digi *out)
{
	int ct = 0;
	
	/* Invert the digipeaters */
	
	while (ct < in->ndigi) {
		out->calls[ct]    = in->calls[in->ndigi - ct - 1];
		out->repeated[ct] = 0;
		ct++;
	}
	
	/* Copy ndigis */
	out->ndigi = in->ndigi;

	/* Finish off */
	out->lastrepeat = 0;
}

#endif
