/*
-- ----------------------------------------------------------------------------
--
--     Object Name : afserver.c
--     Revision    : 1
--
--     Copyright INMOS Limited, 1987, 1988.
--     All Rights Reserved.
--
--     DESCRIPTION
--         Afserver command line parsing and main entry point procedures.
--
--     NOTES
--         None.
--
--     HISTORY
--         25-Feb-1988    Antony King    Last change.
--         12-Apr-1988    Antony King    Added this comment.
--
-- ----------------------------------------------------------------------------
*/

/* Included files */

#include "version.h"
#include "afserver.h"

/* Static variables */

static char tcom[COMMAND_LINE_LENGTH];  /* Transputer command line buffer */

static int reset_host = TRUE;           /* Reset transputer flag */

static int analyse_host = FALSE;        /* Reset analyse mode flag */

/* Global variables */

long int option_flag = 0;             /* Type of run-time scheme etc */

long int board_size_flag = 0;         /* Size of transputer board */

long int load_address = -1;           /* Load address for code */

int config_boot = FALSE;              /* Boot down with configured code flag */

int old_compatable = FALSE;           /* Server compatability flag. */

int serial_boot_log = FALSE;          /* Produce debugging information */

int test_error_flag = FALSE;          /* Test transputer error flag */

int tcominptr = 0;                    /* Command line input pointer */

int tcomoutptr = 0;                   /* Command line output pointer */

char core_memory[CORE_MEMORY_SIZE];   /* Transputer core dump buffer */

int terminate_code = T_TERMINATED;    /* Server exit result */

int transputer_result = DEF_RESULT;   /* Result from transputer */

long int version_number = VERS_NUMB;  /* Server version number */

long int version_date = VERS_DATE;    /* Server version date */

long int version_state = RESET_STATE; /* Transputer initialisation state */

/* External procedures */

extern void host_end();

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : terminate_message
--
--     Input Parameters :
--         int t_code - terminate code of the afserver.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Report the afserver's reason for termination.
--
-- ----------------------------------------------------------------------------
*/

static void terminate_message (t_code)
    int t_code;
{
    switch (t_code)
    {
        case T_TERMINATED:
            break;
        case T_BAD_BOOT:
            fprintf(stderr, "Server terminated: cannot boot root transputer\n");
            break;
        case T_BAD_HOST_INIT:
            fprintf(stderr, "Server terminated: cannot initialise host\n");
            break;
        case T_BAD_COMMAND:
            fprintf(stderr, "Server terminated: illegal filer command received\n");
            break;
        case T_BAD_COMMAND_LINE:
            fprintf(stderr, "Server terminated: bad command line\n");
            break;
        case T_BAD_RECORD:
            fprintf(stderr, "Server terminated: bad protocol when expecting record\n");
            break;
        case T_BAD_INT32:
            fprintf(stderr, "Server terminated: bad protocol when expecting INT32\n");
            break;
	case T_BAD_LINUX:
	    fprintf(stderr, "Server terminated: Linux isn't a DOS Box\n");
	    break;
        default:
            break;
    }
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : add_to_com
--
--     Input Parameters :
--         char *str - string to be added to the command line buffer.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Add the given string to the command line buffer.
--
-- ----------------------------------------------------------------------------
*/

static void add_to_tcom (str)
    char *str;
{
    char c;

    while ((c = *str++) != '\0')
        tcom[tcominptr++] = c;
    tcom[tcominptr++] = ' ';
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : read_number
--
--     Input Parameters :
--         char *str - string to be converted.
--
--     Output Parameters :
--         None.
--
--     Result :
--         long int  - numerical value of string.
--
--     DESCRIPTION
--         Convert the given string into an integer number.
--
-- ----------------------------------------------------------------------------
*/

static long int read_number (str)
    char *str;
{
    long int result = 0;

    if (*str == '#')
        sscanf(str, "#%lx", &result);
    else
        sscanf(str, "%ld", &result);
    return(result);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : parse_command
--
--     Input Parameters :
--         int argc              - number of argument strings to parse.
--         char *argv[]          - array of the argument strings.
--
--     Output Parameters :
--         char **boot_file_name - pointer to the name of the boot file.
--
--     Result :
--         int                   - result of operation.
--
--     DESCRIPTION
--         Parse the command line typed in by the user.
--
-- ----------------------------------------------------------------------------
*/

static int parse_command (argc, argv, boot_file_name)
    int argc;
    char *argv[], **boot_file_name;
{
    char *ch;
    int i = 1;

    while (i < argc)
    {
        ch = argv[i++];
        switch (*ch++)
        {
            case '-':
            case '/':
                switch (*ch++)
                {
                    case ':':
                        switch (*ch++)
                        {
                            case 'x':
                            case 'X':
                                analyse_host = TRUE;
                                break;
                            case 'n':
                            case 'N':
                                reset_host = FALSE;
                                break;
                            case 'e':
                            case 'E':
                                test_error_flag = TRUE;
                                break;

                            case '#':
                                old_compatable = ! old_compatable;
                                break;
                            case 'a':
                            case 'A':
                                if (i < argc)
                                {
                                    load_address = read_number(argv[i++]);
                                }
                                else
                                {
                                    fprintf(stderr, "Missing load address parameter\n");
                                    return(T_BAD_COMMAND_LINE);
                                }
                                break;
                            case 'd':
                            case 'D':
                                serial_boot_log = TRUE;
                                break;
                            case 'c':
                            case 'C':
                                if (i < argc)
                                {
                                    config_boot = TRUE;
                                    *boot_file_name = argv[i++];
                                }
                                else
                                {
                                    fprintf(stderr, "Missing boot file name parameter\n");
                                    return(T_BAD_COMMAND_LINE);
                                }
                                break;

                            case 'b':
                            case 'B':
                                if (i < argc)
                                {
                                    *boot_file_name = argv[i++];
                                }
                                else
                                {
                                    fprintf(stderr, "Missing boot file name parameter\n");
                                    return(T_BAD_COMMAND_LINE);
                                }
                                break;
                            case 'i':
                            case 'I':
                                fprintf(stderr, VERSION_STRING);
                                fprintf(stderr, COPYRIGHT_STRING);
                                break;
                            case 'o':
                            case 'O':
                                if (i < argc)
                                {
                                    option_flag = read_number(argv[i++]);
                                }
                                else
                                {
                                    fprintf(stderr, "Missing option flag parameter\n");
                                    return(T_BAD_COMMAND_LINE);
                                }
                                break;
                            case 's':
                            case 'S':
                                if (i < argc)
                                {
                                    board_size_flag = read_number(argv[i++]);
                                }
                                else
                                {
                                    fprintf(stderr, "Missing board size parameter\n");
                                    return(T_BAD_COMMAND_LINE);
                                }
                                break;

                            default:
                                add_to_tcom(argv[i-1]);
                                break;
                        }
                        break;
                    default:
                        add_to_tcom(argv[i-1]);
                        break;
                }
                break;
            default:
                add_to_tcom(argv[i-1]);
                break;
        }
    }
    return(T_TERMINATED);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : tcomread
--
--     Input Parameters :
--         int len      - length to be read from command line buffer
--
--     Output Parameters :
--         char *buffer - data block read
--
--     Result :
--         int          - actual length read from command line buffer
--
--     DESCRIPTION
--         Read from the command line buffer the given length.
--
-- ----------------------------------------------------------------------------
*/

int tcomread (buffer, len)
    int len;
    char *buffer;
{
    int bytes_read = 0;

    while ((tcomoutptr < tcominptr) && (bytes_read < len))
    {
        *buffer++ = tcom[tcomoutptr++];
        bytes_read++;
    }
    return(bytes_read);
}

/*
-- ----------------------------------------------------------------------------
--
--     Function Name : main
--
--     Input Parameters :
--         int argc     - number of argument strings.
--         char *argv[] - array of the argument strings.
--         char *envp[] - array of environmental strings.
--
--     Output Parameters :
--         None.
--
--     Result :
--         None.
--
--     DESCRIPTION
--         Main entry point for the afserver.
--
-- ----------------------------------------------------------------------------
*/

void main (argc, argv, envp)
    int argc;
    char *argv[], *envp[];
{
    char *boot_file_name = NULL;

    host_begin();
    terminate_code = parse_command(argc, argv, &boot_file_name);
    if (terminate_code == T_TERMINATED)
    {
        terminate_code = root_init(boot_file_name, analyse_host, reset_host,
                                   &version_state, core_memory);
        if (terminate_code == T_TERMINATED)
        {
            filer_init();
            server_main();
        }
    }
    terminate_message(terminate_code);
    host_end(transputer_result);
}
