/*
-- ----------------------------------------------------------------------------
--
--  Object name : filer2.c
--  Revision    : 1.0
--
--  Copyright (c) Inmos Ltd, 1987.
--  All Rights Reserved.
--
--  DESCRIPTION
--    TDS Server  Filer Auxiliary Routines
--
--  NOTES
--
--  HISTORY
--     ibm-pc   20-Oct-86  CREATION.
--     ibm-pc   17-Feb-88  RJO - Addition of read ahead/ write behind features.
--     ibm-pc   23-Feb-88  RJO - Changed ResultToLink to return BYTEs instead
--              of Ints.
--     ibm-pc   10-Mar-88  RJO - Added TruncateFileName, StripExtension.
--     ibm-pc   14-Mar-88  RJO - Added MakeUniqueFileName.
-- ----------------------------------------------------------------------------
*/
#include "inmos.h"
#include "srvconst.h"
#include "filecoms.h"
#include "errno.h"
#include "stdio.h"
#include "filconst.h"
#include "string.h"
#include "sys/types.h"

/*
-- ----------------------------------------------------------------------------
--                     File attributes table
-- Maps file contents attribute onto file type attributes and filename
-- extensions.
-- ----------------------------------------------------------------------------
*/

struct ATT_TABLE_STRUCT FileAttrTable[MAX_FILE_CONTENTS+1] =
  {
    { ".tcm", 0 },   /* 0 - Comment text         */
    { ".tsr", 0 },   /* 1 - Source text          */
    { ".dcd", 1 },   /* 2 - Code                 */
    { ".cs1", 2 },   /* 3 - Occam 1 Put SC       */
    { ".dds", 1 },   /* 4 - Descriptor           */
    { ".ddb", 1 },   /* 5 - Debug                */
    { ".cp1", 2 },   /* 6 - Occam 1 Put Program  */
    { ".cu1", 2 },   /* 7 - Occam 1 Put Util     */
    { ".ce1", 2 },   /* 8 - Occam 1 Put Exe      */
    { ".dlk", 1 },   /* 9 - Link                 */
    { ".csc", 2 },   /* 10 - Occam 2 Put SC      */
    { ".cpr", 2 },   /* 11 - Occam 2 Put Program */
    { ".cut", 2 },   /* 12 - Occam 2 Put Util    */
    { ".cex", 2 },   /* 13 - Occam 2 Put Exe     */
    { ".   ", 0 },   /* 14 - Occam 2 Library     */
    { ".   ", 0 },   /* 15 - IMP process         */
    { ".   ", 0 },   /* 16 - IMP SC              */
    { ".   ", 0 },   /* 17 - IMP library         */
    { ".   ", 0 },   /* 18 - C process           */
    { ".   ", 0 },   /* 19 - C SC                */
    { ".   ", 0 },   /* 20 - C library           */
    { ".   ", 0 },   /* 21 - Pascal process      */
    { ".   ", 0 },   /* 22 - Pascal SC           */
    { ".   ", 0 },   /* 23 - Pascal library      */
    { ".   ", 0 },   /* 24 - Fortran process     */
    { ".   ", 0 },   /* 25 - Fortran SC          */
    { ".   ", 0 },   /* 26 - Fortran library     */
    { ".dck", 1 },   /* 27 - Checker data        */
    { ".dbn", 1 },   /* 28 - Linked code image   */
    { ".dmp", 1 },   /* 29 - Link map data       */
    { "    ", 0 },   /* 30                       */
    { ".tci", 0 },   /* 31 - Configuration info  */
    { ".tai", 0 }    /* 32 - Analyse info        */
  };

extern void ConvertToLinux( char* String );

/*
-- ----------------------------------------------------------------------------
-- int ReadAhead - Perform read ahead operation.
--
-- Input Parameters:
--    (struct) *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
int ReadAhead (fptr)
struct FILER_SPACE_DEF *fptr;
{
  int BytesRead = 0;

  /* Read some bytes from the file */
  BytesRead = fread (fptr->buffer, sizeof(char), BUFFER_LENGTH, fptr->handle);

  if (BytesRead == F_OK)
    {
      /* There is no more to read */
      fptr -> rwresult = F_EOF;
      fptr -> buffer_count = BytesRead;
    }
  else if (BytesRead < F_OK)
    {
       /* There has been a well bad error */
       fptr -> host_error = BytesRead;
       fptr -> rwresult = BytesRead;
       fptr -> buffer_count = 0;
       fptr -> buffer_ptr = 0;
    }
  else
    {
      /* Its a perfect read */
      fptr -> rwresult = F_OK;
      fptr -> buffer_count = BytesRead;
      fptr -> buffer_ptr = 0;
    }
}
/*
-- ----------------------------------------------------------------------------
-- int FlushBuffer - Flush buffer whose data space is pointed to by fptr.
--
-- Input Parameters:
--    (struct) *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
int FlushBuffer (fptr)
struct FILER_SPACE_DEF *fptr;
{
  if (fptr -> buffer_ptr != 0)
    {
      /* There is something eerie in the buffer */
      int ChCount;
      int NoInBuffer = fptr -> buffer_count;

      ChCount = fwrite (fptr->buffer, sizeof(char), fptr->buffer_count,
                        fptr->handle);

      if (ChCount < F_OK)
        {
          /* The Write has failed with a tokally crucial bad error */
          fptr -> rwresult = ChCount;
          fptr -> result = F_NOT_ALL_DATA_WRITTEN;
          fptr -> host_error = ChCount;
        }
      else
        {
          /* Write was OK.. or was it ?? */
          if (ChCount == NoInBuffer)
            fptr -> rwresult = F_OK;    /* All bytes were sent */
          else
            fptr -> rwresult = F_NOT_ALL_DATA_WRITTEN;
        };
      fptr -> buffer_ptr = 0;
      fptr -> buffer_count = 0;
    }
}
/*
-- ----------------------------------------------------------------------------
-- void StripExtension - Return the filename with no extn on it.
--
-- Input Parameters:
--   (BYTE *) FileName      - Supplied filename with extension.
--   (BYTE *) StrippedName  - Return filename without extension.
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void StripExtension (FileName, StrippedName)
BYTE *FileName;
BYTE *StrippedName;
{
  register int i;
  int StrLen = strlen (FileName);

  for (i=0; ((i < StrLen) && (FileName[i] != '.')); i++)
    *StrippedName++ = FileName[i];
  *StrippedName = '\0'; /* That will terminate nicely */
}
/*
-- ----------------------------------------------------------------------------
-- void TruncateFileName - Make a valid filename currently eight chars.
--
-- Input Parameters:
--   (BYTE *) FileId  - Name of the file.
--   (BYTE *) TruncatedName - Return name of the file.
-- Output Parameters:
--   Returns truncated filename if longer than eight chars.
-- ----------------------------------------------------------------------------
*/
void TruncateFileName (FileId, TruncatedName)
BYTE *FileId;
BYTE *TruncatedName;
{
  BYTE TempName[FILE_NAME_LENGTH];
  register int i;
  int StartPtr;
  int DotPtr;
  int FileNameLen;
  strcpy (TempName, FileId);         /* Save the filename           */
  FileNameLen = strlen (TempName);
  StartPtr = 0;                      /* Point to start of string    */
  DotPtr = 0;

  for (i=0; (i < FileNameLen); i++)
    {
      if ((TempName[i] == '\\') || (TempName[i] == '/'))
        StartPtr = i+1;              /* point to next */
      else if (TempName[i] == '.')
        DotPtr = i;
    };
  for (i=0; (i < StartPtr) ; i++)    /* Copy everything up to start */
    *TruncatedName++ = TempName[i];
  if ((DotPtr - StartPtr) > F_LENGTH)
    /* This filename is just too long */
    {
      int EndPoint;

      EndPoint = StartPtr + F_LENGTH;
      for (i=StartPtr; i < EndPoint; i++)    /* Copy only eight characters */
        *TruncatedName++ = TempName[i];
      for (i=DotPtr; i < FileNameLen; i++)   /* Copy the rest over         */
        *TruncatedName++ = TempName[i];
    }
  else
    {
      for (i=StartPtr; i < FileNameLen ; i++)
        *TruncatedName++ = TempName[i];
    }
  *TruncatedName = '\0';
}
/*
-- ----------------------------------------------------------------------------
-- void MakeUniqueFileName - Make up a totally unique filename.
--
-- Input Parameters:
--   (BYTE *) FileName - The unique filename.
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void MakeUniqueFileName (FileName)
BYTE *FileName;
{
  register int i;
  long int TimeNow;

/*  (void) Time (&TimeNow);  */    /* Luvly piece of C to get the time */
  TimeNow = time(NULL);
  for (i=0 ; i < 8; i++)
    {
      FileName[i] = (TimeNow & 0xf) + 'A';
      TimeNow = TimeNow >> 4;
    };
  FileName[8] = '\0';
}
/*
-- ----------------------------------------------------------------------------
-- void FilerInitFile
-- Initialise a filer whose data space is pointed to by fptr.
--
-- Input Parameters:
--    (struct) *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FilerInitFile (fptr)
struct FILER_SPACE_DEF *fptr;
{
  fptr -> state = FS_IDLE;
  fptr -> result = F_OK;
  fptr -> rwresult = F_OK;
  fptr -> host_error = 0;
  fptr -> buffer_count = 0;
  fptr -> buffer_ptr = 0;
  fptr -> sr_stack_ptr = 0;
}
/*
-- ----------------------------------------------------------------------------
--void FilerCloseFile
--Close a file on the filer whose data space is pointed to by fptr.
--
-- Input Parameters:
--    (struct) *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FilerCloseFile (fptr)
struct FILER_SPACE_DEF *fptr;
{
  switch(fptr->state)
    {
      case FS_OPEN_FOR_READING:
      case FS_OPEN_FOR_WRITING:
      case FS_OPEN_FOR_READING_TEXT:
      case FS_OPEN_FOR_WRITING_TEXT:
      case FS_OPEN_BLOCK_READ:
      case FS_OPEN_BLOCK_WRITE:
      case FS_OPEN_BLOCK_UPDATE:
        if (fclose (fptr->handle) == EOF)
          fptr->result = F_CANNOT_CLOSE_FILE;
        else
          fptr->state = FS_IDLE;
        break;
      default: break;
    };
}
/*
-- ----------------------------------------------------------------------------
-- void StrFromLink
-- Read a str from the link.
--
-- Input Parameters:
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void StrFromLink (Str)
BYTE *Str;
{
  int length = TruncSliceFromLink(FILE_NAME_LENGTH-1,Str);
  Str[length] = '\0';
}
/*
-- ----------------------------------------------------------------------------
-- void BufferFromLink
-- Read a record from the link into a filer's buffer.
--
-- Input Parameters:
--    (struct) *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void BufferFromLink (fptr)
struct FILER_SPACE_DEF *fptr;
{
  int NoOfBytesRead;
  NoOfBytesRead = TruncSliceFromLink (BUFFER_LENGTH,
                                      &(fptr->buffer[fptr->buffer_ptr]));
  fptr->buffer_ptr += NoOfBytesRead;
  fptr->buffer_count += NoOfBytesRead;
}
/*
-- ----------------------------------------------------------------------------
-- void FileIdFromLink - Read a FileId from the link.
--
-- Input Parameters:
--  (struct) FILE_ID_STRUCT *FileIdPtr
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FileIdFromLink (FileIdPtr)
struct FILE_ID_STRUCT *FileIdPtr;
{
  int length = TruncSliceFromLink (FILE_NAME_LENGTH-1, FileIdPtr->name);
  FileIdPtr->name[length] = '\0';  /* convert to C string */
  FileIdPtr->type_attr = WordFromLink();
  FileIdPtr->contents_attr = WordFromLink();
}
/*
-- ----------------------------------------------------------------------------
-- void AsciiIdFromLink - Read an ascii_id from the link.
--
-- Input Parameters:
--  (struct) ASCII_ID_STRUCT *asciiIdPtr
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void AsciiIdFromLink (ascii_id_ptr)
struct ASCII_ID_STRUCT *ascii_id_ptr;
{
  int length = TruncSliceFromLink(FILE_NAME_LENGTH-1, ascii_id_ptr->name);
  ascii_id_ptr->name[length] = '\0';  /* convert to C string */
  ascii_id_ptr->type_attr = WordFromLink();
  ascii_id_ptr->contents_attr = WordFromLink();
}
/*
-- ----------------------------------------------------------------------------
-- void AttrFromLink - Read an attr from the link.
--
-- Input Parameters:
--   (struct) ATTR_STRUCT *attr - Attributes.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void AttrFromLink (attr)
struct ATTR_STRUCT *attr;
{
  attr->type = WordFromLink ();
  attr->contents = WordFromLink ();
}
/*
-- ----------------------------------------------------------------------------
-- void NullSliceToLink - Send a zero-length slice to the link.
--
-- Input Parameters:
--    None.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void NullSliceToLink ()
{
  WordToLink(0);  /* Send a zero length */
}
/*
-- ----------------------------------------------------------------------------
-- void ResultToLink - Send the result of a filer operation to the link.
--
-- Input Parameters:
--   (int)  Channel  - Channel Number.
--   (int)  Command  - Command to be sent.
--   (int)  Result   - Result of operation.
--    None.
--
-- Output Parameters:
--    None.
-- ----------------------------------------------------------------------------
*/
void ResultToLink (Channel, Command, Result)
int Channel, Command, Result;
{
  ByteToLink(FROM_FILER_0_CHAN + Channel);
  WordToLink(Command);
  WordToLink(Result);
}
/*
-- ----------------------------------------------------------------------------
-- void FileNameToLink
--
-- Input Parameters:
--   (struct) FILER_SPACE_DEF *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FileNameToLink (fptr)
struct FILER_SPACE_DEF *fptr;
{
  SliceToLink(strlen(fptr->file_name), fptr->file_name);
}
/*
-- ----------------------------------------------------------------------------
-- void StrToLink - Write a str to the link.
--
-- Input Parameters:
--   (BYTE)  *str  - String to be sent.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void StrToLink (str)
BYTE *str;
{
  SliceToLink (strlen(str), str);
}
/*
-- ----------------------------------------------------------------------------
-- void BufferToLink - Write a record in the Buffer of the Filer Data Space
-- pointed to by Fptr, to the link.
--
-- Input Parameters:
--   (struct) FILER_SPACE_DEF *fptr - Filers data area.
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void BufferToLink (fptr)
struct FILER_SPACE_DEF *fptr;
{
  int NoOfBytesToSend = fptr->buffer_count;
  int StartBuffer = fptr->buffer_ptr;
  if (NoOfBytesToSend > RECORD_LENGTH)
    NoOfBytesToSend = RECORD_LENGTH;

  SliceToLink(NoOfBytesToSend, &fptr->buffer[StartBuffer]);
  fptr->buffer_count -= NoOfBytesToSend;
  fptr->buffer_ptr += NoOfBytesToSend;
}
/*
-- ----------------------------------------------------------------------------
-- void FileIdToLink - Write a file id to the link.
--
-- Input Parameters:
--   (struct) FILE_ID_STRUCT *FileIdPtr
--
-- Output Parameters:
--    None.
--
-- ----------------------------------------------------------------------------
*/
void FileIdToLink (FileIdPtr)
struct FILE_ID_STRUCT *FileIdPtr;
{
  SliceToLink(strlen(FileIdPtr->name), FileIdPtr->name);
  WordToLink(FileIdPtr->type_attr);
  WordToLink(FileIdPtr->contents_attr);
}
/*
-- ----------------------------------------------------------------------------
-- int LengthToBuffer - Write the integer length, to the buffer from bufferPtr
-- on, in OPS format.
--
-- Input Parameters:
--   (int) Length
--   (BYTE*) Buffer
--   (int) BufferPtr
--
-- Output Parameters:
--   (int) BufferPtr
--
-- ----------------------------------------------------------------------------
*/
int LengthToBuffer (Length, Buffer, BufferPtr)
int Length;
BYTE *Buffer;
int BufferPtr;
{
  if ((Length >> DATA_FIELD_SIZE) != 0)
    Buffer[BufferPtr++] = ((Length >> DATA_FIELD_SIZE) | PREFIX_BYTE);
  Buffer[BufferPtr++] = (Length & (~STRING_BYTE));
  return (BufferPtr);
}
/*
-- ----------------------------------------------------------------------------
-- int LengthToBuffer -
-- Write the string name, to the buffer from bufferPtr on, as an OPS string
-- record.
--
-- Input Parameters:
--   (int) Length
--   (BYTE*) Buffer
--   (int) BufferPtr
--
-- Output Parameters:
--   (int) BufferPtr
-- ----------------------------------------------------------------------------
*/
int FileNameToBuffer (Name, Buffer, BufferPtr)
BYTE *Name;
BYTE *Buffer;
int  BufferPtr;
{
  int i;
  BufferPtr = LengthToBuffer (strlen(Name), Buffer, BufferPtr);
  for (i=0; (Name[i] != '\0'); i++)
    Buffer[BufferPtr++] = Name[i];
  return (BufferPtr);
}
/*
-- ----------------------------------------------------------------------------
-- void TopLevelToLink
--
-- Input Parameters:
--   (BYTE *) TopLevelFileName
--
-- Output Parameters:
--   (int) BufferPtr
--
-- NOTES:
-- Fabricate a record containing the contents of an imaginary top level file
--   and send it to the link as a slice.
--   The record is built up in TempSlice.
--   The record has the following structure:
--     StartList
--       StartFiled
--         StartList
--           Record ( the name of the top level file )
--           Attribute ( 0 )
--           Attribute ( 1 )
--           Attribute ( 0 )
--         EndList
--       EndFiled
--     EndList
-- ----------------------------------------------------------------------------
*/
void TopLevelToLink (TopLevelFile)
BYTE *TopLevelFile;
{
  BYTE Buffer[RECORD_LENGTH];
  int BufferPtr = 0;

  Buffer[BufferPtr++] = START_LIST_BYTE;
  Buffer[BufferPtr++] = START_FILED_BYTE;
  Buffer[BufferPtr++] = START_LIST_BYTE;
  BufferPtr = FileNameToBuffer (TopLevelFile, Buffer, BufferPtr);
  Buffer[BufferPtr++] = TOP_LEVEL_ATTR1;
  Buffer[BufferPtr++] = TOP_LEVEL_ATTR2;
  Buffer[BufferPtr++] = TOP_LEVEL_ATTR3;
  Buffer[BufferPtr++] = END_LIST_BYTE;
  BufferPtr = FileNameToBuffer (TopLevelFile, Buffer, BufferPtr);
  Buffer[BufferPtr++] = END_FILED_BYTE;
  Buffer[BufferPtr++] = END_LIST_BYTE;
  SliceToLink (BufferPtr, Buffer);
}
/*
-- ----------------------------------------------------------------------------
-- int FixTopLevelFile
--
-- Input Parameters:
--   (int) Channel
--   (struct) FILER_SPACE_DEF *fptr
--   (BYTE *) TopLevelFileName
--
-- Output Parameters:
--   (int) Result
--
-- NOTES:
-- Enforce the protocol to read the contents of the top level file.
-- This procedure enforces the following communication on the link:
--     to.link ! Channel; fkf.lock.read; nullslice
--     from.link ? Channel; tkf.open.read; nullslice
--     to.link ! Channel; fkf.open.read; f.ok
--     from.link ? Channel; tkf.read
--     to.link ! Channel; fkf.read; f.ok; record
--     from.link ? Channel; tkf.close
--     to.link ! Channel; fkf.close; f.ok; nullslice
--  If the protocol is not obeyed then the server terminates with an error
--  message.
-- ----------------------------------------------------------------------------
*/
int FixTopLevelFile (Channel, fptr, TopLevelFile)
int Channel;
struct FILER_SPACE_DEF *fptr;
BYTE *TopLevelFile;
{
  register int CarryOn = TRUE;
  ResultToLink (Channel, FKF_LOCK_READ, fptr->result);
  NullSliceToLink();
  CarryOn = (ByteFromLink() == (Channel + TO_FILER_0_CHAN));
  if (CarryOn) CarryOn = (WordFromLink() == TKF_OPEN_READ);
  if (CarryOn) CarryOn = (WordFromLink() == 0);
  if (CarryOn)
    {
      ResultToLink (Channel, FKF_OPEN_READ, F_OK);
      CarryOn = (ByteFromLink() == (Channel + TO_FILER_0_CHAN));
    };

  if (CarryOn) CarryOn = (WordFromLink() == TKF_READ);
  if (CarryOn)
    {
      ResultToLink (Channel, FKF_READ, F_OK);
      TopLevelToLink (TopLevelFile);
      CarryOn = (ByteFromLink() == (Channel + TO_FILER_0_CHAN));
    };
  if (CarryOn) CarryOn = (WordFromLink() == TKF_CLOSE);
  if (CarryOn)
    {
      ResultToLink (Channel, FKF_CLOSE, F_OK);
      NullSliceToLink();
    };
  if (CarryOn)
    return (T_OK);
  else
    return (T_INVALID_TOPFILE_READ);
}
/*
-- ----------------------------------------------------------------------------
-- BOOL DeriveExtension
-- Find the filename extension corresponding to the attributes and write
-- it at *ch.
--
-- Input Parameters:
--   (BYTE) *Ch;
--   (int) TypeAttr;
--   (int) ContentsAttr;
--
-- Output Parameters:
--   (BOOL) Result
-- ----------------------------------------------------------------------------
*/
BOOL DeriveExtension (Ch, TypeAttr, ContentsAttr)
BYTE *Ch;
int TypeAttr;
int ContentsAttr;
{
  BOOL Result = FALSE;
  if ((ContentsAttr >= 0) && (ContentsAttr <= MAX_FILE_CONTENTS))
    {
      if ((FileAttrTable[ContentsAttr]).type == TypeAttr)
        {
          strcpy (Ch, (FileAttrTable[ContentsAttr]).extn);
          Result = TRUE;
        }
    };
  return (Result);
}
/*
-- ----------------------------------------------------------------------------
-- BOOL ValidExtension
-- Take the string at *Ch and treat it as a Filename extension.
-- If this Filename extension is valid return TRUE, otherwise return FALSE.
--
-- Input Parameters:
--   (BYTE) *Ch       - Pointer to a string containing the extension;
--
-- Output Parameters:
--   (BOOL) Result    - return result
--   (int) TypeAttr   - Return attr.
--   (int) ContentsAttr - Return contents.
--
-- ----------------------------------------------------------------------------
*/
BOOL ValidExtension (Ch, TypeAttr, ContentsAttr)
BYTE *Ch;
int *TypeAttr, *ContentsAttr;
{
  int i;
  BOOL Result = FALSE;
  for (i=0; (i<=MAX_FILE_CONTENTS); i++)
   if (strcmp(Ch, (FileAttrTable[i]).extn) == 0)
     {
       *TypeAttr = (FileAttrTable[i]).type;
       *ContentsAttr = i;
       Result = TRUE;
       break;
     };
  return (Result);
}
/*
-- ----------------------------------------------------------------------------
-- void PushFile
-- Save information on filer's currently open file on filer's
-- suspend/resume stack.
--
-- Input Parameters:
--   (struct) FILER_SPACE_DEF *fptr - Filers data area.
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void PushFile (fptr)
struct FILER_SPACE_DEF *fptr;
{
  struct SR_STACK_ENTRY *stack_entry;
  /* Point stack_entry to top entry of filer's suspend/resume stack */
  stack_entry = &(fptr->sr_stack[fptr->sr_stack_ptr]);

  stack_entry->file_mode = fptr->state;
  strcpy (stack_entry->file_str, fptr->file_name);
  (fptr->sr_stack_ptr)++;
}
/*
-- ----------------------------------------------------------------------------
-- void PullFile
-- Restore information from top entry of filer's suspend/resume stack.
--
-- Input Parameters:
--   (struct) FILER_SPACE_DEF *fptr - Filers data area.
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void PullFile (fptr)
struct FILER_SPACE_DEF *fptr;
{
  struct SR_STACK_ENTRY *stack_entry;
  --(fptr->sr_stack_ptr);
  stack_entry = &(fptr->sr_stack[fptr->sr_stack_ptr]);
  fptr->state = stack_entry->file_mode;
  strcpy (fptr->file_name, stack_entry->file_str);
  fptr->buffer_count = 0;
  fptr->buffer_ptr = 0;
}
/*
-- ----------------------------------------------------------------------------
-- void Ins
-- Insert a hole at posn in a string pointed to by absid_str.
--
-- Input Parameters:
--  (BYTE *) AbsIdStr
--  (int)  Posn
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void Ins (AbsIdStr, Posn)
BYTE *AbsIdStr;
int Posn;
{
  int i;
  for (i = strlen(AbsIdStr); i >= Posn; i--)
    AbsIdStr[i+1] = AbsIdStr[i];
  AbsIdStr[Posn] = '0';
}
/*
-- ----------------------------------------------------------------------------
-- void CreateVariant
-- Create a copy-file variant in AbsIdStr the variant of a sequence of ascii
-- '0'.
--
-- Input Parameters:
--  (BYTE *) AbsIdStr
--  (int)  FNameStart
--  (int)  ExtnStart
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void CreateVariant (AbsIdStr, FNameStart, ExtnStart)
BYTE *AbsIdStr;
int FNameStart;
int ExtnStart;
{
  int i, VariantStart;
  for (i = 0; i < VARIANT_LENGTH; i++)
    if ((ExtnStart - FNameStart) < F_LENGTH)
      Ins(AbsIdStr, ExtnStart++);
    else
      break;
  if (ExtnStart - FNameStart <= F_LENGTH)
    VariantStart = ExtnStart - VARIANT_LENGTH;
  else
    VariantStart = FNameStart + F_LENGTH - VARIANT_LENGTH;
  for (i = VariantStart; (i < VariantStart + VARIANT_LENGTH); i++)
    AbsIdStr[i] = '0';
}
/*
-- ----------------------------------------------------------------------------
-- void IncVariant
-- Increment the copy-file variant at VariantStart in AbsIdStr.
--
-- Input Parameters:
--  (BYTE *) AbsIdStr
--  (int) VariantStart
--
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void IncVariant (AbsIdStr, VariantStart)
BYTE *AbsIdStr;
int VariantStart;
{
  int i = VariantStart + VARIANT_LENGTH - 1;
  while ((AbsIdStr[i] == '9') && (i >= VariantStart))
    AbsIdStr[i--] = '0';
  if (i >= VariantStart)
    AbsIdStr[i]++;
}
/*
-- ----------------------------------------------------------------------------
-- BOOL VariantExists
-- Return TRUE if the string pointed to by str begins with a copy-file
-- variant ( a sequence of digits).
--
-- Input Parameters:
--  (BYTE *) Str     - The string
--
-- Output Parameters:
--  (BOOL) Result
-- ----------------------------------------------------------------------------
*/
BOOL VariantExists (Str)
BYTE *Str;
{
  int i = 0;
  BOOL Result;

  while ((*Str <= '9') && (*Str >= '0') && (i < VARIANT_LENGTH))
    {
      Str++;
      i++;
    };
  if (i == VARIANT_LENGTH)
    Result = TRUE;
  else
    Result = FALSE;
  return (Result);
}
/*
-- ----------------------------------------------------------------------------
-- BOOL FileExists
-- Return TRUE if the file whose name is in FileStr exists,
-- FALSE otherwise.
--
-- Input Parameters:
--  (BYTE *) FileStr     - The file name
--
-- Output Parameters:
--  (BOOL) Result
-- ----------------------------------------------------------------------------
*/
BOOL FileExists (FileStr)
BYTE *FileStr;
{
  FILE *Stream;
  BOOL Result;

  ConvertToLinux(FileStr);

  if ( (Stream = fopen (FileStr, "r")) != NULL )
    {
      fclose (Stream);
      Result = TRUE;
    }
  else
    Result = FALSE;
  return (Result);
}
/*
-- ----------------------------------------------------------------------------
-- BYTE *FindExtension
-- Looks for a filename extension in the string str.
-- If found, returns a pointer to the '.' preceding the the extension,
-- otherwise returns NULL.
--
-- Input Parameters:
--  (BYTE *) Str     - The string
--
-- Output Parameters:
--  (BYTE *) Str     - The extn
-- ----------------------------------------------------------------------------
*/
BYTE *FindExtension (Str)
BYTE *Str;
{
  BYTE *Strptr;
  BOOL Found = FALSE;

  for ( Strptr = (strchr(Str, '\0') - 1); Strptr >= Str; Strptr--)
    {
      if (*Strptr == '.')
        {
          Found = TRUE;
          break;
        }
      if ((*Strptr == '\\') || (*Strptr == ':'))
        {
          Found = FALSE;
          break;
        }
    };

  if (Found)
    return (Strptr);
  else
    return (NULL);
}
/*
-- ----------------------------------------------------------------------------
-- int SeekFileLength
--
-- Input Parameters:
--  (struct) FILER_SPACE_DEF *fptr - Filers data area.
--  (long *) Length                - Return length.
--
-- Output Parameters:
--  (int) Result
-- ----------------------------------------------------------------------------
*/
int SeekFileLength (fptr, Length)
struct FILER_SPACE_DEF *fptr;
long *Length;
{
  long Curpos;
  int Result = F_OK;

  Curpos = ftell(fptr->handle);              /* Remember where we are    */
  if (fseek(fptr->handle, 0L, S_END) == 0)   /* Go to the end to find    */
    *Length = ftell(fptr->handle);           /* where it is              */
  else
    Result = F_CANNOT_SEEK;
  if (fseek(fptr->handle, Curpos, S_BEGINNING) != 0) /* Go back to where */
    Result = F_CANNOT_SEEK;                   /* We were                 */
  return (Result);
}

/*
--
--     End of File.
--
*/
