#include "kernel.h"
#include "ether.h"

#if (ETHERNET == ETH_3C503)

/* LOCAL VARIABLES */
PRIVATE unsigned char thin_bit=ECNTRL_THIN;	/* default to thin cable */
PRIVATE phys_bytes membase;			/* shared memory "base" */

/* LOCAL FUNCTIONS */
FORWARD void rcopy_subr();
FORWARD void do_reset();

/*===========================================================================*
 *				ram_enable				     * 
 *===========================================================================*/
PUBLIC void ram_enable()
{
  output(E33G_GACFR, EGACFR_NORM);	/* make sure gate array is set up
					 * and the RAM is enabled (not EPROM)
					 */
}

/*===========================================================================*
 *				reset_8390				     * 
 *===========================================================================*/
PUBLIC void reset_8390()
{
  output(E33G_CNTRL, thin_bit|ECNTRL_RESET);	/* turn on board reset bit */
  output(E33G_CNTRL, thin_bit);			/* turn off board reset bit */
  do_reset();
}

/*===========================================================================*
 *				block_output				     * 
 *===========================================================================*/
PUBLIC int block_output(src_addr, sm_offset, count)
phys_bytes src_addr;		/* source buffer location */
vir_bytes sm_offset;		/* board shared memory offset */
vir_bytes count;		/* byte count */
{
  phys_bytes dst_addr;

  dst_addr=membase+(phys_bytes)sm_offset;	/* compute dst physical addr */
  ram_enable();			/* make sure the RAM is actually there */
  rcopy_subr(src_addr, dst_addr, count);
  return OK;
}

/*===========================================================================*
 *				block_input				     * 
 *===========================================================================*/
PUBLIC void block_input(sm_offset, dst_addr, count)
vir_bytes sm_offset;		/* board shared memory offset */
phys_bytes dst_addr;		/* dest buffer location */
vir_bytes count;		/* byte count */
{
  vir_bytes count1;
  phys_bytes src_addr;

  if ((sm_offset+count) > (sm_rstop_ptr<<8))	/* over the top of the ring? */
  {			/* yes, copy in two pieces due to buffer wraparound. */
    count1=sm_rstop_ptr<<8;	/* compute length of first part... */
    count1-=sm_offset;		/* ...as all the bytes up to wrap point */
    src_addr=membase+(phys_bytes)sm_offset;	/* compute src physical addr */
    rcopy_subr(src_addr, dst_addr, count1);
    sm_offset=SM_RSTART_PG<<8;	/* offset to start of first receive page */
    dst_addr+=(phys_bytes)count1;  /* new buffer location */
    count-=count1;		/* bytes left to move */
  }
  src_addr=membase+(phys_bytes)sm_offset;	/* compute src physical addr */
  rcopy_subr(src_addr, dst_addr, count);
}

/*===========================================================================*
 *				rcopy_subr				     * 
 *===========================================================================*/
PRIVATE void rcopy_subr(src_addr, dst_addr, count)
phys_bytes src_addr;		/* source address */
phys_bytes dst_addr;		/* destination address */
vir_bytes count;		/* byte count */
{
  phys_copy(src_addr, dst_addr, (phys_bytes) count);	/* do copy */
}

/*===========================================================================*
 *				do_reset				     * 
 *===========================================================================*/
PRIVATE void do_reset()
{
  int i;

  lock();			/* protect the E33G_CNTRL contents */
  output(E33G_CNTRL, thin_bit|ECNTRL_SAPROM);	/* switch control bits
						 * to enable SA PROM */
  for(i=0 ; i < sizeof(Eth_addr) ; i++)	/* get board address */
    curr_hw_addr.ea[i]=input(E33_SAPROM+i);
  output(E33G_CNTRL, thin_bit);			/* switch control bits
						 * to turn off SA PROM
						 * windowing */
  unlock();			/* Ok for E33G_CNTRL to change now */
  set_8390_eaddr();
}

/*===========================================================================*
 *				init_card				     * 
 *===========================================================================*/
PUBLIC void init_card()
{
  int i;
  unsigned char memcfg, irq, irq_bit;

/* Now get the board's physical address from on-board PROM into curr_hw_addr */
  lock();			/* protect the E33G_CNTRL contents */
  output(E33G_CNTRL, thin_bit|ECNTRL_SAPROM);	/* switch control bits
						 * to enable SA PROM */
  for(i=0 ; i < sizeof(Eth_addr) ; i++)	/* get board address */
    curr_hw_addr.ea[i]=input(E33_SAPROM+i);
  output(E33G_CNTRL, thin_bit);			/* switch control bits
						 * to turn off SA PROM
						 * windowing */
  unlock();			/* Ok for E33G_CNTRL to change now */

/* Point the "Vector Pointer" registers off into the boonies so we
 * don't get the shared RAM disabled on us while we're using it.
 * Ideally a warm boot should reset this too, to get to ROM on this card,
 * but a guaranteed way to determine that value is unknown.
 */
  output(E33G_VP2, 0xff);	/* point at the ROM restart location: ffff0h */
  output(E33G_VP1, 0xff);
  output(E33G_VP0, 0x00);

/* Make sure shared memory is jumpered on and verify its address */
  memcfg=input(E33G_ROMBASE);	/* read rom/ram cfg register */
  if (memcfg & 0xf0)	/* any bits on? */
  {	
    membase=0xC6000L;		/* build memory "base" here */
    if (memcfg & 0xc0)		/* DC00 or D800 ? */
      membase+=0x10000L;	/* yes, make Dx00 */
    if (memcfg & 0xa0)		/* DC00 or CC00 ? */
      membase+=0x4000L;		/* yes, make xC00 */
  }
  if ( membase != (ETHER_BASE-(phys_bytes)(SM_TSTART_PG<<8)) )
    panic("3C503 Ethernet board not found at specified address", NO_NUM);

/* Set up Gate Array's Config Reg to enable and size the RAM */
  output(E33G_GACFR, EGACFR_IRQOFF);	/* make sure gate array is set up
					 * and the RAM is enabled (not EPROM)
					 */

/* On-board RAM check stuff would go here.
 * For now, let's assume that memory is ok.
 */

/* Set up control of shared memory, buffer ring, etc. */
  output(E33G_STARTPG, SM_RSTART_PG);	/* set ASIC copy of rx's
					 * first buffer page
					 */
  output(E33G_STOPPG, sm_rstop_ptr);	/* and ASIC copy of rx's
					 * last buffer page + 1
					 */

/* Set up interrupt/DMA control register in ASIC.
 * We won't use the DMA, so B0-B3 are zero.
 */
  irq=ETHER_IRQ;
  if (irq == 9)		/* if converted to 9, make it back into 2 */
    irq=2;		/* now card thinks it's IRQ2 */
  irq-=2;		/* make 0-3 for tables */
  if (irq > (5-2))	/* out of range ? */
    panic("3C503 Ethernet board IRQ out of range", NO_NUM);
  irq_bit=(0x10<<irq);	/* make the bit for ASIC */
  output(E33G_IDCFR, irq_bit);	/* set bit in ASIC reg for IRQ level */
  output(E33G_NBURST, 8);	/* set burst size to 8 */
  output(E33G_DMAAH, SM_TSTART_PG);	/* set up transmit bfr in DMA addr */
  output(E33G_DMAAL, 0);		/* ... */
}

#endif /* ETHERNET - no code after this line ! */
