/* db.c */

#include "const.h"
#include "type.h"
#include "chars.h"
#include "/usr/include/a.out.h"		/* full pathname to get right one! */
#undef EXTERN
#define EXTERN
#include "var.h"

#define NBREAKPOINTS 16
#define VECTOR_SEGMENT 0
#define makevectoff( intnum ) (4 * (intnum))

/* command keys for db */

#define BREAK1 'b'
#define BREAK2 ETOA( F9 )
#define CASEFLIP1 'c'
#define DUMP1 'd'
#define ENTER1 'e'
#define FORK1 'f'
#define GO1 'g'
#define GO2 ETOA( F5 )
#define FLIP1 'x'
#define FLIP2 ETOA( F4 )
#define HELP 'h'
#define INPORT1 'i'
#define ITRACEFLIP1 'j'
#define MAPDUMP1 'm'
#define MAPDUMP2 ETOA( F2 )
#define OUTPORT1 'o'
#define PROCDUMP1 'p'
#define PROCDUMP2 ETOA( F1 )
#define QUIT1 'q'
#define QUIT2 ESCAPE
#define REG1 'r'
#define REG2 ETOA( F3 )
#define SLEVEL1 's'
#define SYMBOL1 'n'
#define TRACE1 't'
#define TRACE2 ETOA( F8 )
#define UNASM1 'u'
#define VERBOSE_ENTRY_FLIP 'v'
PRIVATE struct
{
	bool_t armed;
	struct breakentry *end;
	struct breakentry
	{
		struct adr eadr;
		opcode_pt oldval;
	}
		entry[NBREAKPOINTS];
}
	breaktable = { 0, breaktable.entry };

FORWARD struct breakentry *findbreakentry();
#define getbigcount_t( x ) getoffset_t( (x) )
FORWARD void getch();
#define getcount_t( x ) getopcode_pt( (x) )
#define getport_t( x ) getopcode_pt( (x) )
FORWARD void getsym();
FORWARD physoff_t linearadr();
FORWARD offset_t pop();

PRIVATE unsigned base = 16;
PRIVATE char_pt ch;
PRIVATE count_t dlength = 8;
PRIVATE struct adr dptr;
PRIVATE count_t dwidth = 16;
PRIVATE bool_t gotrace = FALSE;
PRIVATE port_t inportnum = 0;
PRIVATE bool_t ints_while_tracing = FALSE;
PRIVATE char *lineptr;
PRIVATE char_pt lowch;
PRIVATE struct adr mptr = { 0, 0 };
PRIVATE flags_t pretrace_f;
PRIVATE bigcount_t tracecount = 0;
PRIVATE bool_t traceprint = FALSE;
PRIVATE struct adr tracestack = { 0, 0 };
PRIVATE bigcount_t traceremainder;
PRIVATE count_t ulength = 16;
PRIVATE bool_t verbose_entry = TRUE;

PUBLIC void db_main( is_sstep, regs )
bool_pt is_sstep;
struct regs regs;
{
	char_pt key;

	if ( is_sstep )
	{
		if ( traceremainder == 0 )
		{
			/* false alarm */
			regs.f &= ~TF;
			return;
		}
		regs.f = regs.f & ~(IF | TF) | pretrace_f & (IF | TF);
trace_entry:
		++tracecount;
		/*
			count-down only if stack is above current level
			print the trace only if finished or printing enabled as well
		*/
		if ( regs.sp >= tracestack.off &&
			 (tracestack.off == 0 || regs.ss == tracestack.seg) &&
			 ((--traceremainder == 0 || traceprint)) && !gotrace )
		{
			if ( !showentry( &regs ) )
				traceremainder = 0;
			if ( traceremainder != 0 )
				closeio();
		}
		if ( traceremainder != 0 )
		{
			if ( trace1( &regs ) )
				/* trace was faked */
				goto trace_entry;
			/* continue tracing */
			return;
		}
		if ( gotrace )
		{
			gotrace = FALSE;
			armbreakpoints();
			return;
		}
		if ( tracecount > 1 )
		{
			outh32( tracecount );
			outnstr( " instructions traced" );
		}
	}
	else
	{
		/* breakpoint interrupt */
		--regs.ip;
		if ( !breaktable.armed ||
		     findbreakentry( (struct adr *) &regs.ip ) == NULL )
			++regs.ip;
		disarmbreakpoints();
		showentry( &regs );
	}
	traceprint = FALSE;
	dptr.seg = regs.ds;
	while ( TRUE )
	{
		nextline();
		key = lowch;
		getsym();
		if ( ch == ':' || ch == ';' )
		{
			setproc( key, &dptr, &mptr );
			continue;
		}
		switch( key )
		{
		case BREAK1:
			if ( ch == '-' )
			{
				getsym();
				clearbreakpoint();
			}
			else
				setbreakpoint();
		case BREAK2:
			showbreakpoints();
			break;
		case CASEFLIP1:		/* case CLOSE: overloaded */
			switch( lowch )
			{
			case 'k': can_keyboard(); break;
			case 's': can_screen(); break;
			case 't':
				getch();
				if ( lowch == 'i' )
					can_itty();
				else if ( lowch == 'i' )
					can_otty();
				else
				{
					can_itty();
					can_otty();
				}
				break;
			default: flipcase(); break;
			}
			break;
		case DUMP1:
			if ( lowch == 'l' )
			{
				getsym();
				getcount_t( &dlength );
				break;
			}
			if ( lowch == 'w' )
			{
				getsym();
				getcount_t( &dwidth );
				break;
			}
			getadr( &dptr );
/* case DUMP2: */
			showmem();
			break;
		case ENTER1:
			getadr( &mptr );
			setmem();
			break;
		case FLIP1:
		case FLIP2:
			show_user_screen();
			while ( testchar() == EOF )	/* inchar() messes cursor */
				;
			show_db_screen();
			break;
		case FORK1:
			--regs.ip;
			if ( peek8 ( (struct adr *) &regs.ip ) != INT_BREAKPOINT )
				++regs.ip;
			else
			{
				mptr.off = regs.ip;
				mptr.seg = regs.cs;
				setmem();
			}
			break;
		case GO1:
			setbreakpoint();
		case GO2:
			if ( findbreakentry( (struct adr *) &regs.ip ) != NULL )
			{
				tracestack.off = 0;
				ints_while_tracing = FALSE;
				gotrace = TRUE;
				goto trace;
			}
			armbreakpoints();
			/* fall into QUIT1 */
		case QUIT1:
		case QUIT2:
			tracestack.off = 0;
			ints_while_tracing = FALSE;
			closeio();
			return;
		case HELP:
			info();
			break;
		case INPORT1:
			showport();
			break;
		case ITRACEFLIP1:
			if ( (ints_while_tracing = !ints_while_tracing) == FALSE )
				outnstr( "Interrupts disabled while tracing" );
			else
				outnstr( "Interrupts enabled while tracing" );
			break;
#ifndef DOS
		case MAPDUMP1:
		case MAPDUMP2:
			map_dmp();
			break;
#endif
		case OUTPORT1:		/* case OPEN: overloaded */
			switch( lowch )
			{
			case 'k': enab_keyboard(); break;
			case 's': enab_screen(); break;
			case 't':
				getch();
				if ( lowch == 'i' )
					enab_itty();
				else if ( lowch == 'i' )
					enab_otty();
				else
				{
					enab_itty();
					enab_otty();
				}
				break;
			}
			setport();
			break;
#ifndef DOS
		case PROCDUMP1:
		case PROCDUMP2:
			p_dmp();
			break;
#endif
		case REG1:
		case REG2:
			showregs( &regs );
			break;
		case SLEVEL1:		/* case STTY: is overloaded */
			if ( lowch == 't' )
			{
				set_tty();
				break;
			}
			if ( !getoffset_t( &tracestack.off ) )
			{
				if ( tracestack.off == 0 )
					tracestack.off = regs.sp;
				else
					tracestack.off = 0;
			}
			outh32( tracestack.off );
			outnl();
			break;
		case SYMBOL1:
			if ( ch != 0 )
				--lineptr;
			findsname( lineptr, CSEG, TRUE );
			break;
		case TRACE1:
			if ( lowch == 'p' )
			{
				traceprint = TRUE;
				getsym();
			}
			if ( getbigcount_t( &traceremainder ) && traceremainder == 0 )
			{
				traceprint = FALSE;
				break;
			}
		case TRACE2:
trace:
			if ( traceremainder == 0 )
				++traceremainder;
			tracestack.seg = regs.ss;
			closeio();
			tracecount = 0;
			if ( trace1( &regs ) )
				goto trace_entry;
			return;
		case UNASM1:
			if ( lowch == 'l' )
			{
				getsym();
				getcount_t( &ulength );
				break;
			}
			getadr( &uptr );
/* case UNASM2: */
			showinstructions();
			break;
		case VERBOSE_ENTRY_FLIP:
			verbose_entry = !verbose_entry;
			break;
		}
	}
}

PRIVATE void armbreakpoints()
{
	struct breakentry *entry;

	for ( entry = breaktable.entry; entry < breaktable.end; ++entry )
	{
		entry->oldval = peek8( &entry->eadr );
		poke8( &entry->eadr, INT_BREAKPOINT );
	}
	breaktable.armed = TRUE;
}

PRIVATE void clearbreakpoint()
{
	struct breakentry *entry;
	struct adr adr;

	if ( lowch == '-' )
		breaktable.end = breaktable.entry;
	else if ( adr.seg = uptr.seg, getadr( &adr ) )
	{
		if ( (entry = findbreakentry( &adr )) == NULL )
			error( "No Breakpoint Set" );
		else
		{
			memcpy( (char *) entry, (char *) (entry + 1),
			        (unsigned) ((char *) breaktable.end - (char *) entry) );
			--breaktable.end;
		}
	}
}

PRIVATE void disarmbreakpoints()
{
	struct breakentry *entry;

	if ( breaktable.armed )
	{
		for ( entry = breaktable.entry; entry < breaktable.end; ++entry )
			poke8( &entry->eadr, entry->oldval );
		breaktable.armed = FALSE;
	}
}

PRIVATE void error( s )
char *s;
{
	outnstr( s );
}

PRIVATE void fakeint( intnum, regs )
unsigned intnum;
struct regs *regs;
{
	push( regs->f, regs );
	regs->f &= ~(IF | TF);
	push( (offset_t) regs->cs, regs );
	regs->cs = peekw( VECTOR_SEGMENT, makevectoff( intnum ) + 2 );
	push( regs->ip, regs );
	regs->ip = peekw( VECTOR_SEGMENT, makevectoff( intnum ) );
}

PRIVATE struct breakentry *findbreakentry( adr )
struct adr *adr;
{
	struct breakentry *entry;

	for ( entry = breaktable.entry; entry < breaktable.end; ++entry )
		if ( linearadr( &entry->eadr ) == linearadr( adr ) )
			return entry;
	return NULL;
}

PRIVATE void get1char()
{
	static char line[2] = { 0, 0 };

	line[0] = ch = inchar();
	lineptr = line;
	getch();
}


PUBLIC void info()
{
	outstr( "MINIX db v1.0" );
	outnl();
	outstr( "Processor type is " );
	switch( db_processor )
	{
	case 86: outstr( "8088 or 8086" ); break;
	case 186: outstr( "80188 or 80186" ); break;
	case 286: outstr( "80286" ); break;
	case 386: outstr( "80386" ); break;
	}
	outnl();
}

PRIVATE physoff_t linearadr( adr )
struct adr *adr;
{
	return CLICK_SIZE * (physoff_t) adr->seg + adr->off;
}
 
PRIVATE offset_t pop( regs )
struct regs *regs;
{
	offset_t value;

	/* need mods here for 32 bits and protected mode */
	value = peek16( (struct adr *) &regs->sp );
	regs->sp += 2;
	return value;
}

PRIVATE void push( value, regs )
offset_t value;
struct regs *regs;
{
	regs->sp -= 2;
	/* need mods here for 32 bits and protected mode */
	poke16( (struct adr *) &regs->sp, (u16_t) value );
}

PRIVATE void setbreakpoint()
{
	struct breakentry *entry;

	if ( (entry = breaktable.end) >= (breaktable.entry + NBREAKPOINTS) )
	{
		error( "Breakpoint Table Full" );
	}
	else if ( entry->eadr.seg = uptr.seg, getadr( &entry->eadr ) )
	{
		if ( findbreakentry( entry ) != NULL )
			error( "Breakpoint Already Set" );
		else
		{
			entry->oldval = peek8( &entry->eadr );
			if ( (poke8( &entry->eadr, 0x55 ), peek8( &entry->eadr ) != 0x55) ||
				 (poke8( &entry->eadr, 0xAA ), peek8( &entry->eadr ) != 0xAA) )
				error( "Breakpoint in ROM?" );
			else
				++breaktable.end;
			poke8( &entry->eadr, entry->oldval );
		}
	}
}

PRIVATE void setmem()
{
	opcode_pt byte;
	char_pt digit;

	outssegadr( &mptr );
	while ( TRUE )
	{
		outh8( peek8( &mptr ) );
		outbyte( '-' );
		get1char();
		if ( getdigit( &digit ) )
		{
			outbyte( lineptr[-1] );
			byte = digit;
			while ( get1char(), getdigit( &digit ) )
			{
				outbyte( lineptr[-1] );
				byte = byte * base + digit;
			}
			poke8( &mptr, byte );
		}
		if ( ch == ' ' )
			++mptr.off;
		else if ( ch == '-' )
		{
			--mptr.off;
			outnl();
			outsegadr( &mptr );
		}
		else
			break;
		out2space();
	}
	outnl();
}

PRIVATE void setport()
{
	opcode_pt byte;
	port_t outportnum;

	if ( getport_t( &outportnum ) && getopcode_pt( &byte ) )
		/* don't display port, may be side affect */
		oportb( outportnum, byte );
}

PRIVATE bool_pt show1instruction()
{
	bool_pt idone;
	static char line[81];
	struct adr newuptr;
	struct adr olduptr;
	bool_pt outnlstatus;
	struct nlist *sp;

	outbyte( '\r' );
	do
	{
		if ( (sp = findsval( uptr.off, CSEG )) != NULL && sp->n_value == uptr.off )
		{
			outsym( sp, uptr.off );
			outcolon();
			outnl();
		}
		olduptr = uptr;
		openstring( line );
		idone = puti();
		line[stringpos()] = 0;
		closestring();
		newuptr = uptr;
		uptr = olduptr;
		outssegadr( &uptr );
		while ( uptr.off != newuptr.off )
			outget8();
		if ( newuptr.off - olduptr.off < ((db_processor == 386) ? 5 : 3) )
			outtab();					/* to column 16 (24 for 386) */
		outtab();						/* to column 24 (32 for 386) */
		outnlstatus = outnstr( line );
	}
	while ( !idone );			/* eat all prefixes */
	return outnlstatus;
}

PRIVATE void showbreakpoints()
{
	struct breakentry *entry;
	unsigned n;

	for ( entry = breaktable.entry, n = 0; entry < breaktable.end; ++entry, ++n )
	{
		outssegadr( &entry->eadr );
		if ( n == 6  )
		{
			outnl();
			n = -1;
		}
	}
	if ( n != 0 )
		outnl();
}

PRIVATE bool_pt showentry( regs )
struct regs *regs;
{
	openio();
	syminit();
	if ( verbose_entry )
		return showregs( regs );
	else
	{
		uptr.off = regs->ip;
		uptr.seg = regs->cs;
		return show1instruction();
	}
}

PRIVATE void showinstructions()
{
	count_t row;

	for ( row = 0; row < ulength; ++row )
		if ( !show1instruction() )
			break;
}

PRIVATE void showmem()
{
	u8_pt byte;
	count_t column;
	offset_t oldoffset;
	count_t row;

	for ( column = (db_processor == 386 ? 16 : 12); column != 0; --column )
		/* spaces on title line above address */
		outspace();
	for ( column = 0; column < dwidth; ++column )
	{
		outh4( (u4_pt) (dptr.off + column) );
		out2space();
	}
	for ( row = 0; row < dlength; ++row )
	{
		if ( !outnl() )
			break;
		outssegadr( &dptr );
		for ( column = 0, oldoffset = dptr.off; column < dwidth; ++column )
		{
			outh8s( peek8( &dptr ) );
			++dptr.off;
		}
		for ( column = 0, dptr.off = oldoffset; column < dwidth; ++column )
		{
			byte = peek8( &dptr );
			++dptr.off;
			if ( byte >= 0x7F || byte < ' ' )
				byte = '.';
			outbyte( (int) byte );
		}
	}
	outnl();
}

PRIVATE void showport()
{
	if ( !getport_t( &inportnum ) )
	{
		outh16( inportnum );
		outcolon();
		outspace();
	}
	outh8( inportb( inportnum ) );
	outnl();
}

PRIVATE bool_pt showregs( regs )
struct regs *regs;
{
	char *nameptr;
	static char *regnames =
		"ax\0bx\0cx\0dx\0\nsi\0di\0bp\0\nds\0es\0sp\0ss\0\nip\0cs\0 f\0\n";
	offset_t *regptr;

	outbyte( '\r' );
	for ( nameptr = regnames, regptr = (offset_t *) regs; *nameptr != 0; )
	{
		outustr( nameptr );
		outstr( " = " );
		if ( nameptr[1] == 's' )
		{
			if ( db_processor == 386 )
			{
				out2space();
				out2space();
			}
			outh16( *(u16_t *) regptr );
			regptr = (offset_t *) ((u16_t *) regptr + 1);
		}
		else if ( db_processor == 386 )
			outh32( *regptr++ );
		else
			outh16( (u16_t) *regptr++ );
		nameptr += 3;
		if ( *nameptr == '\n' )
		{
			++nameptr;
			outnl();
		}
		else
			outstr( "  " );
	}
	uptr.off = regs->ip;
	uptr.seg = regs->cs;
	return show1instruction();
}

/* set up for trace of 1 instruction, maybe fake the instruction */

PRIVATE bool_pt trace1( regs )
struct regs *regs;
{
	unsigned intnum;

	/*
		The 386 clears the trap flag for software interrupts.
		This is undesirable since we want to trace them, both to debug and to
		get proper instruction counts.
		So fake these instructions (real mode only).
		Division by zero and other 386 exceptions require another method.
		Also fake all instructions which change the interrupt flag:
			CLI, IRET, POPF, PUSHF, STI.
		These fakes are not quite right yet, since prefixed instructions are
		not caught.
		INT and IRET are extremely complicated in protected mode and it will
		be a big job to emulate them.
	*/
#define BREAKPOINT_VECTOR 3
#define CLI 0xFA
#define INT_GENERAL 0xCD
#define INT_OVERFLOW 0xCE
#define INTO_VECTOR 4
#define IRET 0xCF
#define POPF 0x9D
#define PUSHF 0x9C
#define STI 0xFB

	switch( peek8( (struct adr *) &regs->ip ) )
	{
	case CLI:
		regs->f &= ~IF;
		break;
	case INT_GENERAL:
		if ( protected )
			goto toohard;
		++regs->ip;
		intnum = peek8( (struct adr *) &regs->ip );
		++regs->ip;
		fakeint( intnum, regs );
		return TRUE;
	case INT_BREAKPOINT:
		if ( protected )
			goto toohard;
		fakeint( BREAKPOINT_VECTOR, regs );
		return TRUE;
	case INT_OVERFLOW:
		if ( protected )
			goto toohard;
		fakeint( INTO_VECTOR, regs );
		return TRUE;
	case IRET:
		if ( protected )
			goto toohard;
		regs->ip = pop( regs );
		regs->cs = pop( regs );
		regs->f = pop( regs );
		return TRUE;
	case POPF:
		if ( protected )
			goto toohard;
		regs->f = pop( regs );
		break;
	case PUSHF:
		if ( protected )
			goto toohard;
		push( regs->f, regs );
		break;
	case STI:
		regs->f |= IF;
		break;
	default:
toohard:
		pretrace_f = regs->f;
		if ( !ints_while_tracing )
			regs->f &= ~IF;
		regs->f |= TF;
		return FALSE;
	}
	++regs->ip;
	return TRUE;
}

#ifdef DOS

/* various dummy routines */

PUBLIC struct nlist *findsname()
{
	return NULL;
}

PUBLIC struct nlist *findsval()
{
	return NULL;
}

PUBLIC void get_con_state()
{
}

PUBLIC void outsym()
{
}

PUBLIC void reboot()
{
}

PUBLIC void reset_con_state()
{
}

PUBLIC void setproc()
{
}

PUBLIC void syminit()
{
	prompt = '?';
}

#endif

/* parsing routines */

/* test input for being an address expression of the form [segment:;]offset */
/* where segment and offset are numbers in the current base, fetch if so */

PRIVATE bool_pt getadr( adr )
struct adr *adr;
{
	struct adr tempadr;

	if ( !getoffset_t( &tempadr.off ) )
		return FALSE;
	if ( ch == ':' || ch == ';' )
	{
		getsym();
		tempadr.seg = tempadr.off;
		if ( !getoffset_t( &tempadr.off ) )
			return FALSE;
		adr->seg = tempadr.seg;
	}
	adr->off = tempadr.off;
	return TRUE;
}

PRIVATE char_pt getch()
{
	/* convert to unsigned char so ETOA macro works */
	if ( (ch = UCHAR( *lineptr )) != 0 )
		++lineptr;
	lowch = mytolower( ch );
	return ch;
}

/* test input for being an opcode_pt in the current base, fetch if so */

PRIVATE bool_pt getopcode_pt( num )
opcode_pt *num;
{
	offset_t tempnum;

	if ( !getoffset_t( &tempnum ) )
		return FALSE;
	*num = tempnum;
	return TRUE;
}

/* test current char for being a digit in the current base, fetch if so */

PRIVATE bool_pt getdigit( digit )
char_pt *digit;
{
	char_pt tempdigit;

	if ( (tempdigit = lowch) >= '0' && tempdigit <= '9' )
		tempdigit = tempdigit - '0';
	else if ( tempdigit >= 'a' && tempdigit <= 'f' )
		tempdigit = tempdigit + (10 - 'a');
	else
		return FALSE;
	if ( tempdigit >= base )
		return FALSE;
	*digit = tempdigit;
	getch();
	return TRUE;
}

/* test input for being an offset_t in the current base, fetch if so */

PRIVATE bool_pt getoffset_t( num )
offset_t *num;
{
	char_pt digit;
	offset_t tempnum;

	if ( !getdigit( &digit ) )
		return FALSE;
	tempnum = digit;
	while ( getdigit( &digit ) )
		tempnum = tempnum * base + digit;
	*num = tempnum;
	if ( ch == ' ' )
		getsym();
	return TRUE;
}

PRIVATE void getsym()
{
	while( getch() == ' ' )
		;
}

PRIVATE void nextline()
{
		lineptr = getline();
		if ( UCHAR( *lineptr ) <= '~' )
			outnl();
		getsym();
}
