

;*****************--******************--******************--******************
									     ;

  ;	  Assembler STRUCTURE FOR RESIDENT UNINSTALLABLE PROGRAMS
  ;	------------------------------------------------------------
  ;					      by Antonio Musarra, Pisa, Italy.
  ;							Tel. (+ 39) 50 575991.

  ; This structure is useful to create resident programs, which can be
  ;  repeatedly deactivated and reactivated by simply reexecuting the program.

  ; Also, this program can be uninstalled using option /u .
  ;  Deactivation leaves program resident in memory, but it has the advantage
  ;  to be always possible, whilest uninstallation not always.

  ; Installation gains 164 resident bytes; this is obtained overwriting
  ;  unnecessary portion of PSP. It means that eventual parameters passed to
  ;  the program through command-line are lost during installation.
  ; Globally, memory occupation is reduced by about half a Kbyte: this is
  ;  obtained freeing, in addition to PSP, also Environment space.


  ; Program structure is exposed so as to be easily adaptable to other
  ;  resident programs (having 64 Kbyte maximum length).

  ; The functionality of this particular program consists in intercepting
  ;  the AT BIOS routine which manages Sys Req keystroke (Alt-PrtScr,
  ;  almost never used by other programs) and printing on screen a value
  ;  proportional to the resistance that a resistor (for example an
  ;  electronic thermometer) connected to Game port pins 1 and 3 has
  ;  when you hit that keystroke.
  ; (In absence of Game port, program reads an unexisting port and it can
  ;  furnish the value 65535).
  ; Program is based on the fact that Game port four lower bits implement
  ;  such an analog to digital converter for electric resistance measures,
  ;  though the obtained result depends on computer speed.
  ; With the interface used for tests, measures appeared correct inside the
  ;  interval 1 kohm - 3 Mohm.

  ; In this case, the only intercepted interrupt is interrupt 15h.

  ; The areas in which modifications must be made each time are only those
  ;  enclosed between pairs of 77 or 78 characters long asterisk lines
  ;  (and you can ignore everything else).
  ; Particularly: if they are discontinuous asterisk lines (like those
  ;  surrounding this introduction), everything must be put which is needed
  ;  by user program,
  ;  whilest between continuous asterisk lines you must EXCLUSIVELY substitute
  ;  each number 15 ("h" omitted) appearing in this case by the appropriate
  ;  number of interrupt or interrupts used by program
  ;  (if more than one interrupt is present, some sections between asterisks
  ;   must also be duplicated, according to the indications put in their
  ;   comments; for multi-instruction sections, duplication is intended as
  ;   copying the section as a whole to obtain some sections contiguous each
  ;   other).

  ; If you use this structure with no instruction code modification outside
  ;  long asterisk line pairs, please indicate
  ;  "Resident structure by Antonio Musarra"  in your source program.


  ; To obtain a .COM file, execute (in this order) the commands
  ;  TASM /s/c/la/mu/m3/w2/x/z/zi name.ASM
  ;  TLINK /m/l/s/t name.OBJ
  ; or
  ;  TASM /s/c/la/mu/m3/w2/x/z/zi name.ASM
  ;  TLINK /m/l/s/v name.OBJ
  ;  EXE2BIN name.EXE name.COM
  ;
  ; If, during assembling, "Instruction can be compacted with override"
  ;  warning is emitted, it means that value 3 of option /M3 must be
  ;  incremented.


  ; Since is not rare to find out a program which contains with no doubt
  ;  a particular routine that can be considered useful for your own resident,
  ;  if you are successful in find it and redirect to a file its unassembled
  ;  listing produced by DEBUG, you may employ the following DEB2ASM.BTM batch
  ;  (running under 4DOS, an incomparable shareware COMMAND.COM substitute),
  ;  to transform unassembled listing obtained by DEBUG in a handier Assembler
  ;  listing:
  ;
  ; @echo off
  ; iff %# == 1 then
  ;   iff exist %1 then
  ;     setlocal
  ;     erase jmp.lst deb.asm >& nul
  ;     for %R in (@%1) gosub PossibleJmpLine
  ;     for %R in (@%1) gosub AsmLine
  ;     QUIT
  ;   endiff
  ; endiff
  ;
  ; text
  ; Use:     DEB2ASM  fileName
  ;   in which fileName contains a (redirected) output produced by DEBUG,
  ;   consisting in an unassembled listing (command lines beginning with
  ;   typical "-" prompt are ignored).
  ; It produces the file of jumps  JMP.LST  and the Assembler listing
  ;   DEB.ASM  (in which, however, numbers lack final "h" and eventual
  ;   initial "0").
  ; endtext
  ; QUIT
  ; 
  ; :PossibleJmpLine
  ; if %@index["%R",j]==-1 .and. %@index["%R",loop]==-1 .and. %@index["%R",call]==-1 RETURN
  ; set W=%@word[3,%R]
  ; set NewJmp=Y
  ; iff exist jmp.lst then
  ;   for %J in (@jmp.lst) (
  ;     if %W==%J set NewJmp=N
  ;   )
  ; endiff
  ; if %NewJmp==Y echo %W >>! jmp.lst
  ; RETURN
  ; 
  ; :AsmLine
  ; if %@len[%R]==0 .or. %@ascii[%R]==45 RETURN		^ rem	Char(45) = "-"
  ; set Label=
  ; set JmpOffset=%@substr["%R",6,4]
  ; for %J in (@jmp.lst) (
  ;   if %[JmpOffset].==%J. set Label=%[Label]%J:
  ; )
  ; echo %Label	%@lower[%@substr["%R",25,%@eval[%@len[%R]-24]]] >>! deb.asm
  ; RETURN

;									     *
;*****************--******************--******************--******************


	jumps			; Directive which permits conditional jumps
				;  farther than -128..+127 bytes.
				; TASM option /M3 (or /Mn) minimizes memory
				;  occupation.

	; extrn Label		; Was only employed to find out  JMP FAR
				;  instruction code, assembling the
				;  instruction   JMP far ptr Label .

JmpOldInt	macro	I		; I = number of interrupt.
			 db	0EAh	; JMP FAR instruction code.
	OldOffsetInt_&I  dw	0000h	; Space for jump offset.
					;  "&I" indicates that parameter "I"
					;  is used as string rather than
					;  number.
	OldSegmentInt_&I dw	0000h	; Space for jump segment.
	endm

RetFar		macro	N		; N = number of bytes to be flushed
					;  from stack (not counting return
					;  address four bytes).
	ifb	<N>			; If N is an absent parameter,
		db	0CBh		;  RET FAR instruction code is
					;  assembled.
	else
		db	0CAh		; Instruction code of  RET FAR n .
		dw	N
	endif
	endm
.sall				; Shows only macro names inside listings.
				;  This directive is bypassed by TASM option
				;  /LA .


;*****************--******************--******************--******************
									     ;
KeywordLengthInWords = 5	; Number of words constituting a keyword used
				;  to recognize if program was already
				;  installed. Advisably, a number between
				;  4 and 10.
				; Obviously, program code cannot be
				;  automodifying along entire keyword length.
;									     *
;*****************--******************--******************--******************


	if	KeywordLengthInWords lt 1
	  %out  Keyword must be at least one word long.
	  .err
	endif


;**********************
Cseg    segment
;**********************

        assume	cs:Cseg, ds:Cseg, ss:Cseg, es:Cseg

        org	0100h		; To be able to transform .EXE file into .COM
				;  file.


  ; Program code begins with a jump to initialization; after the backward
  ;  shift (more than 164 bytes wide) of program code to be left resident,
  ;  also the space occupied by this instruction will be regained.

Start:
	jmp	Initialization

Zero	=	Start - 0100h	; = CSEG:0000 .
				;  Utility definition for expressions in which
				;  an absolute (numeric) result is needed
				;  rather than relative to "CSEG:" (address).
				;  It is sufficient to subtract "Zero" from
				;  the address that appears in expression
				;  to obtain a number.

	even			; To align "OldResident" at next word (so
OldResident:			;  successive  REP MOVSW  will be correct).

NewResident =	005Ch		; Whole resident section will be moved
				;  backward starting from offset 005Ch
				;  to overwrite unnecessary part of PSP.
				; Instead, using "org 005Ch" a wrong .EXE
				;  program would be produced.

Shift =	OldResident - Zero - NewResident
				; That is "OldResident - NewResident".
				;  It is greater than 0.
				; Every reference to the resident (or, inside
				;  installator, to any shifted instructions)
				;  must contain " - Shift" .



;************************************************
  ; Begin of custom section: user routines for
  ;  one or more interrupts are put hereafter.
;************************************************

;*****************
FirstInterrupt:
;*****************

;****************************************************************************
									    ;
Int_15:				; Label used during installation.
;									    *
;****************************************************************************


	nop			; Two code bytes that are substituted by
	nop			;  JMP $+7  during installation and each time
				;  the interrupt is activated; or by
				;  JMP $+2  each time the interrupt is
				;  deactivated.
				;  ("$" indicates present Location Counter).

  ; After installation, instead of  NOP NOP  the program code will contain
	; jmp	$+7		; which jumps to beginning of actual user
				;  code of INT 15h (7 bytes farther).

  ; After installation and deactivation, instead of  NOP NOP  there will be
	; jmp	$+2		; which jumps to label "OldInt_15".


;*****************************************************************************
									     ;
OldInt_15:			; Label not used during installation.
	JmpOldInt 15		; Contains jump segment and offset, which are
				;  set during installation and found thanks
				;  to labels "OldOffsetInt_15" and
				;  "OldSegmentInt_15" (labels created by the
				;  macro).
;									     *
;*****************************************************************************


AlmostBeginKeyword:		; The keyword used to recognize if program is
				;  already installed begins from second byte
				;  of next instruction to start from even
				;  address.
				; Keyword is unique: even if more than one
				;  interrupt were present, keyword is defined
				;  only over first interrupt.
				; Keyword length is already defined by
				;  "KeywordLengthInWords".

BeginKeyword =	offset AlmostBeginKeyword + 1


;*****************--******************--******************--******************
									     ;
  ; Start of actual user code of INT 15h.

	cmp	ah, 85h		; System Request routine inside AT BIOS.
	jne	OldInt_15	; If running interrupt does not regards
				;  Sys Req routine, interrupt goes on
				;  undisturbed as if TSR did not exist.
	cmp	al, 0		; Sys Req key pressed down.
	jne	Exit		; Sys Req key release is not considered.

	push	ax		; Saving of registers that must be used.
	push	bx		;
	push	cx		;
	push	dx		;


  ; Measure of resistance connected to pin 3 of Game interface.

	mov	dx, 201h	; Game port address.
	cli			; It is necessary to avoid hardware interrupts
				;  during measure. Otherwise, if meanwhile
				;  an interrupt occured, a lower value would
				;  be obtained.
	out	dx, al		; Game port activation.
	xor	cx, cx		; Counter initialization (CX = 0).
Measuring:
	in	al, dx		; Game port read.
	test	al, 1		; Rightmost bit test; that bit concers Game
				;  port pin 3.
	loopnz	Measuring	; If the bit is not 0, measure is still
				;  in progress.
				;  Further, decrements counter "CX"
				;  (after first decrement, CX = 0FFFFh).
	sti			; After measure end, hardware interrupts
				;  can be reenabled.
	not	cx		; It makes result scale to be ascending
				;  (CX = 0 or successive values).

  ; Value contained in CX (proportional to measured resistance) is written
  ;  on right side of screen.

	push	bp		; Saving of registers that will be modified
	push	si		;  by INT 10h executions.
	push	di		;

	push	cx		; CX = value to be written.
	  mov	ah, 3
	  xor	bh, bh
	  int	10h		; Present cursor position reading.
	pop	cx

	push	dx		; Storing cursor position.
	  mov	dx, 10*256+78	; The final digit (written as first) will be
				;  written in row 10, column 78.
NextDigit:
	  push	dx		; Storing digit position.
	    mov    ah, 2
	    int    10h		; Cursor put where the digit must be written.
	    mov    ax, cx
	    xor    dx, dx
	    mov    bx, 10	; Divisor 10 produces decimal digits.
	    div    bx		; Calculus of one decimal digit.
	    push   ax		; Storing quotient.
	      mov  al, dl	; Digit value.
	      or   al, 30h
	      mov  ah, 0Ah
	      xor  bh, bh
	      mov  cx, 1
	      int  10h		; Writes one digit.
	    pop    cx		; Quotient retrieving.
	  pop	dx 		; Digit position retrieving.
	  dec	dx
	  cmp	dx, 10*256+74	; Initial digit (last written) in row 10,
				;  column 74.
	  jae	NextDigit	; Repeats cycle for 5 digits.

	pop	dx		; Retrieves original cursor position.
	xor	bh, bh
	mov	ah, 2		; Brings back the cursor in the original
	int	10h		;  position.

	pop	di		; Retrieving of registers modified by
	pop	si		;  INT 10h executions.
	pop	bp		;

  ; Exit from Sys Req management routine.

	pop	dx		; Retrieving of used registers.
	pop	cx		;
	pop	bx		;
	pop	ax		;
Exit:

  ; It is not possible to finish an interrupt using following two instructions
  ;  since  IRET  reads also the flags from stack, so making useless
  ;  preceding  CLC  instruction.

	; clc			; AT BIOS requires carry = 0 for the benefit
				;  of eventual system programs that wish
				;  to be informed about if System Request
				;  function be supported or not.
	; iret			; INT 15h finishes and returns to caller
				;  routine (INT 9h, in this case).

  ; A substituting for  CLC IRET  instruction sequence is the following:

;	push	bp		; Storing of BP (which occupies one word
				;  onto stack).
;	  mov	bp, sp
;	  push	[ss: bp + 3*2]	; Puts onto stack a copy of original flags
				;  (those previous to interrupt call), which
				;  stay 3 words above [SS: BP] . The 3 words
				;  store, in POP's order, BP and offset and
				;  segment return address (necessary during
				;  interrupt exit).
;	  popf			; Reads, and flushes from stack, flags copy.
				; It is not advisable to execute POPF after
				;  simply executing an  ADD SP,4  since,
				;  after a POPF, hardware interrupts become
				;  generally enabled (because Interrupt Flag
				;  is loaded) and therefore a risk exists
				;  to overwrite return address.
;	  clc			; Puts carry = 0.
;	  pushf			; Puts onto stack new flags (whose unique
	  			;  difference with respect to previous flags
				;  is carry equal to zero).
;	  pop	[ss: bp + 3*2]	; Reads, and flushes from stack, new flags
	  			;  putting them in place of original flags
				;  3 words above [SS: BP] .
;	pop	bp		; Retrieves BP.
;	iret			; Interrupt finish.
				;  From stack are read, in POP's order, offset
				;  and segment return address and the flags.

  ;  or, with identical functionality (directly acting onto bits):

	push	bp
	  mov	bp, sp
	  and	[ss: bp + 3*2], 0FFFFh - 1
				; Carry is flags register rightmost bit.
	pop	bp
	iret


  ; In this particular user routine, no reference was present pointing inside
  ;  resident; any eventual reference of this kind would have had to add
  ;  " - Shift" to its addressing expression.

;									     *
;*****************--******************--******************--******************


MaxLenKeyword =	$ - BeginKeyword
				;  "$" indicates present Location Counter.

	if	KeywordLengthInWords gt MaxLenKeyword / 2
	  %out  First interrupt cannot be shorter than keyword.
	  .err
	endif


;*****************************
  ;  Successive Interrupts:
;*****************************


;*****************--******************--******************--******************
									     ;
  ; Put here eventual code of any other interrupts, each of them beginning
  ;  as below indicated
  ;  ("xx" represents the number of interrupt, "h" omitted):

; Int_xx:
;	nop
;	nop
; OldInt_xx:
;	JmpOldInt xx
;	...			; Actual body of the interrupt.
;	...			;

  ; If inside interrupt body (or at interrupt end) the execution of user
  ;  routine must stop and you must go on with original routine (that is,
  ;  interrupt routine existing before resident installation), the following
  ;  jump instruction must be used (or analogous in case of conditional
  ;  jumps):

;	jmp	OldInt_xx

  ;  whilest, if at interrupt begin (or inside interrupt body) user routine
  ;  must invoke original routine and then must continue its own execution,
  ;  the following two instructions (simulating an INT instruction) will be
  ;  used:

;	pushf
;	call	dword ptr [cs: OldInt_xx + 1 - Shift]

  ;  otherwise, if you must finish interrupt definitively, you will use the
  ;  proper instruction

;	iret

  ; If interrupt must finish transmitting to caller routine some informations
  ;  through flags, it is necessary to consider that  IRET  restores flags
  ;  as they were previous to interrupt call.
  ; If, at interrupt end, those original values have no more interest for
  ;  caller program, you obtain transmission substituting  IRET  instruction
  ;  by the following (constituted by a macro):

;	RetFar	2

  ;  whilest, if original flag values must be preserved (apart from flags
  ;  to be transmitted), interrupt can finish in the following manner:

;	push	bp
;	  mov	bp, sp
;	  push	[ss: bp + 3*2]
;	  popf
;	  ...			; Instructions that establish flag values
;	  ...			;  to be transmitted.
;	  pushf
;	  pop	[ss: bp + 3*2]
;	pop	bp
;	iret

  ;  or in the following manner:

;	push	bp
;	  mov	bp, sp
;	  ...			; Instructions which directly modify some bits
;	  ...			;  of flags register in [SS: BP+3*2] .
;	pop	bp
;	iret


  ; If any reference is needed which points inside resident, " - Shift" must
  ;  be added to its addressing expression.
  ; For instance

;	mov	[cs: Buffer - Shift + bx], al

  ;  in which "Buffer" is declared inside resident as

; Buffer db	LenBuffer dup (?)

;									     *
;*****************--******************--******************--******************


;****************************
  ; End of custom section.
;****************************



EndNewResident =	$ - Zero - Shift
				; That is "$ - Shift".
				;  "$" indicates present Location Counter.
WholeParagraphs =	EndNewResident / 10h
RemainingBytes =	EndNewResident - WholeParagraphs * 10h

	if	RemainingBytes eq 0
ParagraphsToBeResident =	WholeParagraphs
	else
ParagraphsToBeResident =	WholeParagraphs + 1
	endif

	even			; Aligns "ResidentLength" at next word;
				;  otherwise  REP MOVSW  could lose one byte.
ResidentLength =	$ - Zero - Shift - NewResident
				; Present "$" is equal to previous "$", or
				;  greater by 1.



Initialization:

	; mov	ax, cs		; Unnecessary settings because this is a .COM
	; mov	ds, ax		;  file. Until DS and ES are modified, they
	; mov	es, ax		;  point inside this installer.

  ; Finds out uninstallation option onto command-line.

	cld			; Ascending addresses in string instructions.
	mov	si, offset FirstUninstallCommandLine
	mov	di, 0080h	; Command-line offset inside PSP.
	mov	cx, LenFirstUninstallCommandLine
	repe  cmpsb
	je	SetUninstallFlag
	mov	si, offset SecondUninstallCommandLine
	mov	di, 0080h
	mov	cx, LenSecondUninstallCommandLine
	repe  cmpsb
	jne	NoSetUninstallFlag

SetUninstallFlag:
	mov	[ds:UninstallFlag], 1
NoSetUninstallFlag:

  ; More than 164 byte backward shift of resident section.
  ; Eventual parameters passed through command-line to the program are lost
  ;  after this operation.

	mov	si, offset OldResident
				; "OldResident" is an address (CSEG:offset).
	mov	di, NewResident
				; "NewResident" is simply a number.
	mov	cx, ResidentLength / 2
	rep  movsw

  ; Search of an earlier installation of the program.

	mov	dx, BeginKeyword - Shift
	mov	ax, cs
NextCompare:
	  dec	ax
	  mov	ds, ax		; DS = segment for keyword search.
	  mov	si, dx
	  mov	di, dx
	  mov	cx, KeywordLengthInWords
	  repe cmpsw		; ES is still the installer segment.
	  jne	ContinueSearch
	  			; Generally,  REPE CMPSW  ends after only one
				;  compare.

  ; Recognition of already installed program, if the two NOP's at the
  ;  beginning of first user interrupt are not present.
  ;  Because of this instruction it is necessary that, during first
  ;  installation, the first interrupt be activated.

	  cmp	word ptr [ds:FirstInterrupt - Shift], 9090h
				; 9090h = two NOP's code.

	  je	ContinueSearch
	  jmp	InstalledFound
ContinueSearch:
	  cmp	ax, 0050h	; Start segment for DOS dedicated RAM.
	jne	NextCompare


;*******************************
;	First installation:
;*******************************

	push	cs
	pop	ds

  ; Program ends if uninstallation option was present.

	cmp	[ds:UninstallFlag], 0
	jz	Install
	mov	dx, offset NotInstalledMessage
	mov	cx, LenNotInstalledMessage
	call	Print
	mov	ax, 4C01h	; Exit with ErrorLevel indicating an error.
	int	21h		; End of program.

Install:

;*****************************************************************************
									     ;
  ; Int 15h activation, overwriting  JMP $+7  in place of  NOP NOP .

  ;;;  THIS SECTION MUST BE DUPLICATED for each interrupt TO BE ACTIVATED
  ;;;	during first installation; AMONG THEM also THE FIRST user INTERRUPT
  ;;;	must be necessarily present.

	mov	word ptr [ds:Int_15 - Shift], 05EBh
				; 05EBh =  JMP $+7 instruction code.
;									     *
;*****************************************************************************

  ; For eventual interrupts to leave deactivated during first installation,
  ;  it is necessary no instruction since NOP NOP is equivalent to JMP $+2 .

;****************************************************************************
									    ;
  ; Int 15h vector setting, using the three reference labels "Int_15",
  ;  "OldOffsetInt_15" and "OldSegmentInt_15".

  ;;;  THIS SECTION MUST BE DUPLICATED for each interrupt to be put resident;
  ;;;	eventually, interrupts to be put resident may also be more than those
  ;;;	to be activated, but first user interrupt must always belong to the
  ;;;	group of interrupts activated during first installation.

	mov	ax, 3515h
	int	21h		; Puts original INT 15h vector into es:bx.
	mov	[ds:OldOffsetInt_15 - Shift], bx
	mov	[ds:OldSegmentInt_15 - Shift], es
				; Puts original INT 15h vector into
				;  JMP FAR macro;
				;  "Shift" is necessary since execution
				;  follows the  REP MOVSW .
	mov	dx, offset Int_15 - Shift
	mov	ax, 2515h
	int	21h		; Copies ds:dx into INT 15h vector.
;									    *
;****************************************************************************


  ; Environment space is freed, for the benefit of successive programs
  ;  run from DOS.

	mov	ax, [ds:002Ch]	; Contains environment segment.
	or	ax,ax
	jz	AbsentEnvironment
	  mov	es, ax
	  mov	ah, 49h
	  int	21h		; Frees previously allocated memory.
AbsentEnvironment:


  ; Prints installation message.

	mov	dx, offset ActivationMessage
	mov	cx, LenActivationMessage
	call	Print


  ; Establishes the section to put resident and exits.

	mov	dx, ParagraphsToBeResident
	mov	ax, 3100h
	int	21h		; TSR function (terminate and stay resident).



;*************************
InstalledFound:			; DS is resident segment (installed program).
;*************************

  ; Search for uninstallability conditions: beyond an active value in the
  ;  proper flag, all interrupt vectors must still have the same value set
  ;  during installation.

	cmp	[es:UninstallFlag], 0
				; ES is installer segment.
	jz	DeactivateOrReactivate

; Uninstall:

;*****************************************************************************
									     ;
  ; Compare between present interrupt vector and the interrupt vector set
  ;  during installation.

  ;;;  THIS SECTION MUST BE DUPLICATED for each resident interrupt.

	mov	ax, 3515h
	int	21h		; Puts present INT 15h vector into es:bx.
	cmp	bx, offset Int_15 - Shift
	jne	NotUninstallable
	mov	ax, es
	mov	bx, ds
	cmp	ax, bx
	jne	NotUninstallable
;									     *
;*****************************************************************************

;****************************************************************************
									    ;
  ; Restores original interrupt vector (previous to installation) using the
  ;  value stored inside macro "JmpOldInt 15".

  ;;;  THIS SECTION MUST BE DUPLICATED for each resident interrupt.

	mov	dx, [ds:OldOffsetInt_15 - Shift]
	mov	ax, [ds:OldSegmentInt_15 - Shift]
	push	ds		; DS is resident program segment.
	  mov	ds, ax
	  mov	ax, 2515h
	  int	21h		; Copies ds:dx into INT 15h vector.
	pop	ds		; Retrieves resident program segment.
;									    *
;****************************************************************************

  ; Invalidation of the keyword left inside uninstalled program:
  ;  in fact, reexecuting the program, it must appear not installed.
  ; It is sufficient to invert all bits of the first word costituting the
  ;  keyword.

	not	word ptr [ds: BeginKeyword - Shift]


  ; Frees memory occupied by the program; freed space is certainly reusable
  ;  if program was last resident, otherwise it is usable only when DOS will
  ;  have to allocate an enough small environment space.

	mov	ax, ds		; DS is resident program segment.
	mov	es, ax
	mov	ah, 49h
	int	21h		; Frees allocated memory.

  ; End of uninstallation.

	push	cs
	pop	ds		; Load installer segment (containing
				;  messages).
	mov	dx, offset WellUninstalledMessage
	mov	cx, LenWellUninstalledMessage
	call	Print
	mov	ax, 4C00h	; Exit with ErrorLevel indicating no error.
	int	21h		; End of program.

NotUninstallable:
	push	cs
	pop	ds		; Load installer segment (containing
				;  messages).
	mov	dx, offset NotUninstallableMessage
	mov	cx, LenNotUninstallableMessage
	call	Print
	mov	ax, 4C01h	; Exit with ErrorLevel indicating an error.
	int	21h		; End of program.


;*************************
DeactivateOrReactivate:
;*************************

  ; Inspection of present status of first user interrupt, to determine about
  ;  if to activate or deactivate the program.

	mov	ax, 00EBh	; 00EBh =  JMP $+2  instruction code.
	cmp	ax, word ptr [ds:FirstInterrupt - Shift]
				; DS is resident program segment.
	je	ToReactivate

;*****************
; ToDeactivate:
;*****************

;*****************************************************************************
									     ;
  ; Program DEACTIVATION, thanks to deactivation of Int 15h, obtained
  ;  overwriting  JMP $+2  in place of  JMP $+7 .
  ; In fact, each time the program is reexecuted, its activation
  ;  status is inverted; that is, (generally all) interrupts of the program
  ;  pass from activation to deactivation or vice-versa.

  ;;;  THIS SECTION MUST BE DUPLICATED for each resident interrupt,
  ;;;	exactly as it appears here if interrupt must BE DEACTIVATED during
  ;;;	each deactivation of the program;
  ;;;  if eventually, to deactivate the program, some interrupts
  ;;;	had to be activated instead of deactivated, for those interrupts
  ;;;	"ax" would have to be substituted by "05EBh".

	mov	word ptr [ds:Int_15 - Shift], ax
;									     *
;*****************************************************************************


  ; Deactivation message is emitted.

	mov	dx, offset DeactivationMessage
				; Message segment will be loaded before
				;  executing "CALL Print" instruction.
	mov	cx, LenDeactivationMessage
	jmp	ToggleEnd


;***************
ToReactivate:
;***************

;****************************************************************************
									    ;
  ; Program REACTIVATION, thanks to activation of Int 15h, obtained
  ;  overwriting  JMP $+7  in place of  JMP $+2 .
  ; The same matter: each time the program is reexecuted, its activation
  ;  status is inverted; that is, (generally all) interrupts of the program
  ;  pass from activation to deactivation or vice-versa.

  ;;;  THIS SECTION MUST BE DUPLICATED for each resident interrupt,
  ;;;	exactly as it appears here if interrupt must BE ACTIVATED during
  ;;;	each reactivation of the program;
  ;;;  if eventually, to reactivate the program, some interrupts
  ;;;	had to be deactivated instead of activated, for those interrupts
  ;;;	"05EBh" would have to be substituted by "00EBh".

	mov	word ptr [ds:Int_15 - Shift], 05EBh
;									    *
;****************************************************************************


  ; Activation message is emitted.

	mov	dx, offset ActivationMessage
				; Message segment will be loaded before
				;  executing "CALL Print" instruction.
	mov	cx, LenActivationMessage

ToggleEnd:
	push	cs
	pop	ds		; Loads installer segment (containing the
				;  messages).
	call	Print

	mov	ax, 4C00h	; Exit with ErrorLevel indicating no error.
	int	21h		; End of program.



  ; Subroutine for printing on video (or on another output device).

Print:
	mov	ah, 40h		; Print onto device.
	mov	bx, 0001	; Onto standard device.
	int	21h
	ret


;*****************--******************--******************--******************
									     ;
  ; Put here activation and deactivation messages:

ActivationMessage:
  db 'SysReq (Alt-PrtScr) activated to measure a resistance between Game port'
  db 0Dh, 0Ah
  db 'pins 1 and 3.'
  db 0Dh, 0Ah

LenActivationMessage =	$ - ActivationMessage


DeactivationMessage:
  db 'Sys Req deactivation for resistance measures.'
  db 0Dh, 0Ah

LenDeactivationMessage =	$ - DeactivationMessage

;									     *
;*****************--******************--******************--******************


  ; Flag and messages regarding uninstallation.

UninstallFlag	db	0


NotInstalledMessage:
  db 'Program not installed: invalid option.'
  db 0Dh, 0Ah

LenNotInstalledMessage =	$ - NotInstalledMessage


NotUninstallableMessage:
  db 'Program not uninstallable because of other resident programs.'
  db 0Dh, 0Ah

LenNotUninstallableMessage =	$ - NotUninstallableMessage


WellUninstalledMessage:
  db 'Program removed from memory.'
  db 0Dh, 0Ah

LenWellUninstalledMessage =	$ - WellUninstalledMessage


  ; Command-line patterns recognized for uninstallation option.

FirstUninstallCommandLine:
  db 03h, ' /u', 0Dh

LenFirstUninstallCommandLine =	 $ - FirstUninstallCommandLine


SecondUninstallCommandLine:
  db 03h, ' /U', 0Dh

LenSecondUninstallCommandLine = $ - SecondUninstallCommandLine


;*************************
Cseg    ends
;*************************
        end Start
